#Requires -RunAsAdministrator
<#
.SYNOPSIS
    ABC&D Systems Backup and Migration Tools v4.0 - Unified Migration GUI
.DESCRIPTION
    Complete Windows server/workstation migration with SOURCE and DESTINATION modes
    Works on: Windows 10/11, Windows Server 2016/2019/2022/2025
.NOTES
    Version: 4.0
    Company: ABC&D Systems
    Requires: PowerShell 5.1+, Administrator privileges
#>

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

$ErrorActionPreference = 'Continue'
$Script:InstallPath = "C:\ABCD-Migration"
$Script:LogFile = $null
$Script:Version = "4.0"
$Script:BootstrapLog = $null

# License validation variables (set by Python wrapper or Authentication module)
$Script:LicenseValid = $true  # Will be set by wrapper
$Script:LicenseExpiry = $null
$Script:LicenseTier = "trial"
$Script:UserEmail = ""

#region Early Bootstrap Logging (before modules load)
# Create early log to capture startup issues
function Write-BootstrapLog {
    param([string]$Message, [string]$Level = "INFO")
    
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $entry = "[$timestamp] [$Level] $Message"
    
    # Write to console
    switch ($Level) {
        "ERROR" { Write-Host $entry -ForegroundColor Red }
        "WARNING" { Write-Host $entry -ForegroundColor Yellow }
        "DEBUG" { Write-Host $entry -ForegroundColor Cyan }
        default { Write-Host $entry -ForegroundColor White }
    }
    
    # Write to bootstrap log file
    if ($Script:BootstrapLog) {
        try {
            $entry | Out-File -FilePath $Script:BootstrapLog -Append -Encoding UTF8
        } catch { }
    }
}

# Initialize bootstrap log immediately
try {
    $logDir = "$Script:InstallPath\Logs"
    if (!(Test-Path $logDir)) {
        New-Item -Path $logDir -ItemType Directory -Force | Out-Null
    }
    
    $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
    $Script:BootstrapLog = "$logDir\Bootstrap_$timestamp.log"
    
    # Write initial header
    $header = @"
========================================
ABC&D Migration Tool v$Script:Version - Bootstrap Log
========================================
Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')
Computer: $env:COMPUTERNAME
User: $env:USERNAME
PowerShell Version: $($PSVersionTable.PSVersion)
Working Directory: $(Get-Location)
Script Path: $PSCommandPath
PSScriptRoot: $PSScriptRoot
========================================

"@
    $header | Out-File -FilePath $Script:BootstrapLog -Encoding UTF8
    Write-BootstrapLog "Bootstrap logging initialized: $Script:BootstrapLog"
}
catch {
    Write-Host "WARNING: Could not create bootstrap log: $_" -ForegroundColor Yellow
}

# Clean up old bootstrap logs (keep 7 days)
try {
    $cutoffDate = (Get-Date).AddDays(-7)
    Get-ChildItem -Path "$Script:InstallPath\Logs" -Filter "Bootstrap_*.log" -ErrorAction SilentlyContinue | 
        Where-Object { $_.LastWriteTime -lt $cutoffDate } | 
        Remove-Item -Force -ErrorAction SilentlyContinue
    
    Get-ChildItem -Path "$Script:InstallPath\Logs" -Filter "ABCD-Migration_*.log" -ErrorAction SilentlyContinue | 
        Where-Object { $_.LastWriteTime -lt $cutoffDate } | 
        Remove-Item -Force -ErrorAction SilentlyContinue
        
    Write-BootstrapLog "Old logs cleaned up (retention: 7 days)"
}
catch {
    Write-BootstrapLog "Could not clean old logs: $_" -Level "WARNING"
}
#endregion

#region Prerequisite Checks
function Test-Prerequisites {
    $issues = @()
    $warnings = @()
    $capabilities = @{
        HasPrintbrm = $false
        HasSmbCmdlets = $false
        IsHomeEdition = $false
        IsServer = $false
    }
    
    # Check PowerShell version
    if ($PSVersionTable.PSVersion.Major -lt 5) {
        $issues += "PowerShell 5.1 or higher required (found: $($PSVersionTable.PSVersion))"
    }
    
    # Check Windows version
    $os = Get-CimInstance -ClassName Win32_OperatingSystem -ErrorAction SilentlyContinue
    if ($os) {
        $build = [int]$os.BuildNumber
        # Windows 10 is build 10240+, Server 2016 is 14393+
        if ($build -lt 10240) {
            $issues += "Windows 10/Server 2016 or higher required"
        }
        
        # Check edition
        if ($os.Caption -match "Server") {
            $capabilities.IsServer = $true
        }
        if ($os.Caption -match "Home") {
            $capabilities.IsHomeEdition = $true
        }
    }
    
    # Check for required cmdlets
    $requiredCmdlets = @(
        @{Name='Get-LocalUser'; Module='Microsoft.PowerShell.LocalAccounts'; Critical=$true},
        @{Name='Get-SmbShare'; Module='SmbShare'; Critical=$false},
        @{Name='Get-Printer'; Module='PrintManagement'; Critical=$false}
    )
    
    foreach ($cmdlet in $requiredCmdlets) {
        $cmd = Get-Command $cmdlet.Name -ErrorAction SilentlyContinue
        if (-not $cmd) {
            if ($cmdlet.Critical) {
                $issues += "Required cmdlet not found: $($cmdlet.Name)"
            } else {
                if ($cmdlet.Name -eq 'Get-SmbShare') {
                    $warnings += "SMB cmdlets not found - will use net share fallback"
                }
            }
        } else {
            if ($cmdlet.Name -eq 'Get-SmbShare') {
                $capabilities.HasSmbCmdlets = $true
            }
        }
    }
    
    # Check for printbrm.exe (printer migration)
    $printbrmPaths = @(
        "$env:SystemRoot\System32\spool\tools\printbrm.exe",
        "$env:SystemRoot\System32\printbrm.exe"
    )
    foreach ($path in $printbrmPaths) {
        if (Test-Path $path) { 
            $capabilities.HasPrintbrm = $true
            break 
        }
    }
    if (-not $capabilities.HasPrintbrm) {
        $warnings += "printbrm.exe not found - printer migration will use legacy mode"
    }
    
    # Windows Home specific warnings
    if ($capabilities.IsHomeEdition) {
        $warnings += "Windows Home edition detected:"
        $warnings += "  - Printer migration will use legacy mode (may require manual driver install)"
        $warnings += "  - SMB shares will use net share command"
    }
    
    return @{
        Issues = $issues
        Warnings = $warnings
        CanContinue = ($issues.Count -eq 0)
        Capabilities = $capabilities
    }
}

# Run prerequisite checks
Write-BootstrapLog "Running prerequisite checks..."
$prereqResult = Test-Prerequisites
$Script:Capabilities = $prereqResult.Capabilities

Write-BootstrapLog "Capabilities: Home=$($Script:Capabilities.IsHomeEdition), Server=$($Script:Capabilities.IsServer), Printbrm=$($Script:Capabilities.HasPrintbrm), SmbCmdlets=$($Script:Capabilities.HasSmbCmdlets)"

if (-not $prereqResult.CanContinue) {
    $msg = "ABC&D Migration Tool cannot start due to missing requirements:`n`n"
    $msg += ($prereqResult.Issues -join "`n")
    Write-BootstrapLog "STARTUP FAILED: $($prereqResult.Issues -join '; ')" -Level "ERROR"
    [System.Windows.Forms.MessageBox]::Show($msg, "Prerequisites Missing", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    exit 1
}

if ($prereqResult.Warnings.Count -gt 0) {
    # Log warnings but continue
    $Script:StartupWarnings = $prereqResult.Warnings
    foreach ($warning in $prereqResult.Warnings) {
        Write-BootstrapLog "Startup Warning: $warning" -Level "WARNING"
    }
}
Write-BootstrapLog "Prerequisite checks passed"
#endregion

#region Module Loading
Write-BootstrapLog "Starting module loading..."

# Load modules - Fix path detection for VBS launcher
$modulePath = $PSScriptRoot
Write-BootstrapLog "Initial PSScriptRoot: '$PSScriptRoot'"

if (-not $modulePath -or $modulePath -eq "") { 
    # Try MyInvocation path
    $modulePath = Split-Path -Parent $MyInvocation.MyCommand.Definition -ErrorAction SilentlyContinue
    Write-BootstrapLog "Fallback to MyInvocation path: '$modulePath'"
}
if (-not $modulePath -or $modulePath -eq "" -or -not (Test-Path "$modulePath\Modules")) {
    # Fall back to install path
    $modulePath = $Script:InstallPath
    Write-BootstrapLog "Fallback to InstallPath: '$modulePath'"
}

Write-BootstrapLog "Final module path: '$modulePath'"
Write-BootstrapLog "Modules folder exists: $(Test-Path "$modulePath\Modules")"

# Debug output to console
Write-Host "DEBUG: PSScriptRoot = '$PSScriptRoot'" -ForegroundColor Yellow
Write-Host "DEBUG: modulePath = '$modulePath'" -ForegroundColor Yellow
Write-Host "DEBUG: Modules folder exists = $(Test-Path "$modulePath\Modules")" -ForegroundColor Yellow

# Verify modules exist before loading
$requiredModules = @(
    "Logging.ps1",
    "User-Migration.ps1",
    "Printer-Migration.ps1",
    "Share-Migration.ps1",
    "File-Migration.ps1",
    "AppSettings-Migration.ps1",
    "ScheduledTask-Migration.ps1",
    "Service-Management.ps1",
    "VHD-Migration.ps1",
    "Help-System.ps1",
    "SystemTray.ps1",
    "Rollback.ps1",
    "VMExport.ps1",
    "Backup-Management.ps1",
    "Backup-Dialog.ps1"
)

$missingModules = @()
foreach ($module in $requiredModules) {
    $moduleFile = "$modulePath\Modules\$module"
    $exists = Test-Path $moduleFile
    Write-BootstrapLog "  Module $module : $(if ($exists) { 'FOUND' } else { 'MISSING' })"
    if (!$exists) {
        $missingModules += $module
    }
}

if ($missingModules.Count -gt 0) {
    $msg = "Missing required module files in $modulePath\Modules\:`n`n"
    $msg += ($missingModules -join "`n")
    $msg += "`n`nPlease reinstall the application."
    Write-BootstrapLog "STARTUP FAILED: Missing modules: $($missingModules -join ', ')" -Level "ERROR"
    [System.Windows.Forms.MessageBox]::Show($msg, "Missing Modules", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    exit 1
}

Write-BootstrapLog "All required modules found, loading..."

try {
    Write-BootstrapLog "  Loading Logging.ps1..."
    . "$modulePath\Modules\Logging.ps1"
    
    Write-BootstrapLog "  Loading User-Migration.ps1..."
    . "$modulePath\Modules\User-Migration.ps1"
    
    Write-BootstrapLog "  Loading Printer-Migration.ps1..."
    . "$modulePath\Modules\Printer-Migration.ps1"
    
    Write-BootstrapLog "  Loading Share-Migration.ps1..."
    . "$modulePath\Modules\Share-Migration.ps1"
    
    Write-BootstrapLog "  Loading File-Migration.ps1..."
    . "$modulePath\Modules\File-Migration.ps1"
    
    Write-BootstrapLog "  Loading AppSettings-Migration.ps1..."
    . "$modulePath\Modules\AppSettings-Migration.ps1"
    
    Write-BootstrapLog "  Loading ScheduledTask-Migration.ps1..."
    . "$modulePath\Modules\ScheduledTask-Migration.ps1"
    
    Write-BootstrapLog "  Loading Service-Management.ps1..."
    . "$modulePath\Modules\Service-Management.ps1"
    
    Write-BootstrapLog "  Loading VHD-Migration.ps1..."
    . "$modulePath\Modules\VHD-Migration.ps1"
    
    Write-BootstrapLog "  Loading Help-System.ps1..."
    . "$modulePath\Modules\Help-System.ps1"
    
    Write-BootstrapLog "  Loading SystemTray.ps1..."
    . "$modulePath\Modules\SystemTray.ps1"
    
    Write-BootstrapLog "  Loading Rollback.ps1..."
    . "$modulePath\Modules\Rollback.ps1"
    
    Write-BootstrapLog "  Loading VMExport.ps1..."
    . "$modulePath\Modules\VMExport.ps1"
    
    Write-BootstrapLog "  Loading Backup-Management.ps1..."
    . "$modulePath\Modules\Backup-Management.ps1"
    
    Write-BootstrapLog "  Loading Backup-Dialog.ps1..."
    . "$modulePath\Modules\Backup-Dialog.ps1"
    
    # Load optional modules (Licensing, QuickBooks, Authentication, GoogleDrive, WindowsUpdate)
    $licensingModulePath = "$modulePath\Modules\Licensing.ps1"
    if (Test-Path $licensingModulePath) {
        Write-BootstrapLog "  Loading Licensing.ps1..."
        . $licensingModulePath
    } else {
        Write-BootstrapLog "  Licensing module not found (optional)" -Level "WARNING"
    }
    
    $qbModulePath = "$modulePath\Modules\QuickBooks-Management.ps1"
    if (Test-Path $qbModulePath) {
        Write-BootstrapLog "  Loading QuickBooks-Management.ps1..."
        . $qbModulePath
    } else {
        Write-BootstrapLog "  QuickBooks module not found (optional)" -Level "WARNING"
    }
    
    $authModulePath = "$modulePath\Modules\Authentication.ps1"
    if (Test-Path $authModulePath) {
        Write-BootstrapLog "  Loading Authentication.ps1..."
        . $authModulePath
    } else {
        Write-BootstrapLog "  Authentication module not found (optional)" -Level "WARNING"
    }
    
    $gdriveModulePath = "$modulePath\Modules\GoogleDrive-Backup.ps1"
    if (Test-Path $gdriveModulePath) {
        Write-BootstrapLog "  Loading GoogleDrive-Backup.ps1..."
        . $gdriveModulePath
    } else {
        Write-BootstrapLog "  GoogleDrive-Backup module not found (optional)" -Level "WARNING"
    }
    
    $winUpdateModulePath = "$modulePath\Modules\WindowsUpdate-Management.ps1"
    if (Test-Path $winUpdateModulePath) {
        Write-BootstrapLog "  Loading WindowsUpdate-Management.ps1..."
        . $winUpdateModulePath
    } else {
        Write-BootstrapLog "  WindowsUpdate-Management module not found (optional)" -Level "WARNING"
    }
    
    Write-BootstrapLog "All modules loaded successfully"
}
catch {
    Write-BootstrapLog "FATAL ERROR loading modules: $($_.Exception.Message)" -Level "ERROR"
    Write-BootstrapLog "  Line: $($_.InvocationInfo.ScriptLineNumber)" -Level "ERROR"
    Write-BootstrapLog "  Command: $($_.InvocationInfo.Line)" -Level "ERROR"
    [System.Windows.Forms.MessageBox]::Show("Error loading modules: $($_.Exception.Message)`n`nCheck log: $Script:BootstrapLog", "Module Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    exit 1
}

# Load KeyFinder module if exists
$keyFinderPath = "$modulePath\Modules\KeyFinder.ps1"
if (Test-Path $keyFinderPath) {
    Write-BootstrapLog "  Loading KeyFinder.ps1..."
    . $keyFinderPath
    Write-BootstrapLog "  KeyFinder module loaded"
} else {
    Write-BootstrapLog "  KeyFinder module not found (optional)" -Level "WARNING"
}

#endregion

#region Initialize Main Logging
Write-BootstrapLog "Initializing main logging system..."

# Initialize logging (with 7 day retention)
if (!(Test-Path "$Script:InstallPath\Logs")) {
    New-Item -Path "$Script:InstallPath\Logs" -ItemType Directory -Force | Out-Null
}
$Script:LogFile = Initialize-Logging -LogPath "$Script:InstallPath\Logs" -RetentionDays 7

Write-BootstrapLog "Main log file created: $Script:LogFile"
Write-BootstrapLog "Bootstrap complete, switching to main log"
#endregion

# Create main form - Optimized for 720p (1280x720)
$form = New-Object System.Windows.Forms.Form
$form.Text = "ABC&D Systems Backup and Migration Tools v$Script:Version"
$form.Size = New-Object System.Drawing.Size(1350, 720)
$form.StartPosition = "CenterScreen"
$form.FormBorderStyle = "FixedSingle"
$form.MaximizeBox = $false
$form.MinimizeBox = $true
$form.BackColor = [System.Drawing.Color]::FromArgb(20, 20, 20)
$form.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)

# Load custom icon
$iconPath = "$Script:InstallPath\Assets\ABCD-Icon.ico"
if (Test-Path $iconPath) {
    $form.Icon = New-Object System.Drawing.Icon($iconPath)
} else {
    $form.Icon = [System.Drawing.SystemIcons]::Application
}

# Initialize system tray
$trayIcon = Initialize-SystemTray -Form $form -IconPath $iconPath -TooltipText "ABC&D Migration Tool v$Script:Version"

# Handle minimize to tray
$form.Add_Resize({
    if ($form.WindowState -eq [System.Windows.Forms.FormWindowState]::Minimized) {
        Minimize-ToTray
    }
})

# Handle form closing - minimize to tray instead of closing if running
$Script:AllowClose = $false
$form.Add_FormClosing({
    param($sender, $e)
    if (-not $Script:AllowClose) {
        $result = [System.Windows.Forms.MessageBox]::Show(
            "Do you want to minimize to system tray or exit completely?`n`nYes = Minimize to tray`nNo = Exit application",
            "Close Application",
            [System.Windows.Forms.MessageBoxButtons]::YesNoCancel,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            $e.Cancel = $true
            Minimize-ToTray
        }
        elseif ($result -eq [System.Windows.Forms.DialogResult]::Cancel) {
            $e.Cancel = $true
        }
        else {
            # No - exit completely
            Remove-SystemTray
        }
    }
})

# Help button (top right)
$btnHelp = New-Object System.Windows.Forms.Button
$btnHelp.Text = "? Help"
$btnHelp.Location = New-Object System.Drawing.Point(1100, 8)
$btnHelp.Size = New-Object System.Drawing.Size(70, 28)
$btnHelp.BackColor = [System.Drawing.Color]::LightBlue
$btnHelp.ForeColor = [System.Drawing.Color]::Black
$btnHelp.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnHelp.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
$btnHelp.Add_Click({
    Show-HelpWindow -InitialTopic "Overview"
})
$form.Controls.Add($btnHelp)

# EULA button (next to Help)
$btnEULA = New-Object System.Windows.Forms.Button
$btnEULA.Text = "EULA"
$btnEULA.Location = New-Object System.Drawing.Point(1175, 8)
$btnEULA.Size = New-Object System.Drawing.Size(60, 28)
$btnEULA.BackColor = [System.Drawing.Color]::LightGray
$btnEULA.ForeColor = [System.Drawing.Color]::Black
$btnEULA.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnEULA.Font = New-Object System.Drawing.Font("Segoe UI", 9)
$btnEULA.Add_Click({
    Show-EULADialog
})
$form.Controls.Add($btnEULA)

# License button (left of Help)
$btnLicense = New-Object System.Windows.Forms.Button
$btnLicense.Text = "License"
$btnLicense.Location = New-Object System.Drawing.Point(1020, 8)
$btnLicense.Size = New-Object System.Drawing.Size(75, 28)
$btnLicense.BackColor = [System.Drawing.Color]::Gold
$btnLicense.ForeColor = [System.Drawing.Color]::Black
$btnLicense.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnLicense.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
$btnLicense.Add_Click({
    if (Get-Command Show-LicenseStatusDialog -ErrorAction SilentlyContinue) {
        Show-LicenseStatusDialog
    } else {
        [System.Windows.Forms.MessageBox]::Show("License Status`n`nTier: Trial`nStatus: Active`n`nFor full license management, ensure Authentication module is loaded.", "License Info", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
    }
})
$form.Controls.Add($btnLicense)

#region Mode Selection
$grpMode = New-Object System.Windows.Forms.GroupBox
$grpMode.Text = "Migration Mode"
$grpMode.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
$grpMode.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
$grpMode.Location = New-Object System.Drawing.Point(10, 5)
$grpMode.Size = New-Object System.Drawing.Size(1320, 45)
$form.Controls.Add($grpMode)

$rbSource = New-Object System.Windows.Forms.RadioButton
$rbSource.Text = "SOURCE (Export data from this computer)"
$rbSource.Location = New-Object System.Drawing.Point(20, 18)
$rbSource.Size = New-Object System.Drawing.Size(350, 20)
$rbSource.Checked = $true
$grpMode.Controls.Add($rbSource)

$rbDestination = New-Object System.Windows.Forms.RadioButton
$rbDestination.Text = "DESTINATION (Import data to this computer)"
$rbDestination.Location = New-Object System.Drawing.Point(400, 18)
$rbDestination.Size = New-Object System.Drawing.Size(350, 20)
$grpMode.Controls.Add($rbDestination)
#endregion

#region Source Tabs (TabControl for multiple sources)
$Script:SourceTabs = @{}  # Store source tab controls

# Create TabControl for sources
$tabSources = New-Object System.Windows.Forms.TabControl
$tabSources.Location = New-Object System.Drawing.Point(10, 52)
$tabSources.Size = New-Object System.Drawing.Size(1320, 68)
$tabSources.Font = New-Object System.Drawing.Font("Segoe UI", 9)
$form.Controls.Add($tabSources)
$tabSources.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)

# Function to create a source tab
function Add-SourceTabToMain {
    param(
        [string]$SourceName,
        [string]$ComputerName = "",
        [string]$Username = ".\Administrator",
        [bool]$IsLocal = $false
    )
    
    $tabPage = New-Object System.Windows.Forms.TabPage
    $tabPage.Text = if ($SourceName -eq "Local") { "Local (This PC)" } else { $SourceName }
    $tabPage.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)
    $tabPage.Tag = $SourceName
    
    # All controls on single row
    $lblComp = New-Object System.Windows.Forms.Label
    $lblComp.Text = "SOURCE:"
    $lblComp.Location = New-Object System.Drawing.Point(8, 10)
    $lblComp.Size = New-Object System.Drawing.Size(55, 18)
    $tabPage.Controls.Add($lblComp)
    
    $txtComp = New-Object System.Windows.Forms.TextBox
    $txtComp.Location = New-Object System.Drawing.Point(65, 7)
    $txtComp.Size = New-Object System.Drawing.Size(180, 22)
    $txtComp.Text = if ($IsLocal) { $env:COMPUTERNAME } else { $ComputerName }
    $txtComp.Name = "txtComputer"
    $txtComp.Enabled = -not $IsLocal
    $tabPage.Controls.Add($txtComp)
    
    $chkLoc = New-Object System.Windows.Forms.CheckBox
    $chkLoc.Text = "Local"
    $chkLoc.Location = New-Object System.Drawing.Point(250, 8)
    $chkLoc.Size = New-Object System.Drawing.Size(50, 20)
    $chkLoc.Checked = $IsLocal
    $chkLoc.Name = "chkLocal"
    $tabPage.Controls.Add($chkLoc)
    
    $lblUser = New-Object System.Windows.Forms.Label
    $lblUser.Text = "User:"
    $lblUser.Location = New-Object System.Drawing.Point(305, 10)
    $lblUser.Size = New-Object System.Drawing.Size(35, 18)
    $tabPage.Controls.Add($lblUser)
    
    $txtUser = New-Object System.Windows.Forms.TextBox
    $txtUser.Location = New-Object System.Drawing.Point(342, 7)
    $txtUser.Size = New-Object System.Drawing.Size(140, 22)
    $txtUser.Text = $Username
    $txtUser.Name = "txtUsername"
    $txtUser.Enabled = -not $IsLocal
    $tabPage.Controls.Add($txtUser)
    
    $lblPass = New-Object System.Windows.Forms.Label
    $lblPass.Text = "Pass:"
    $lblPass.Location = New-Object System.Drawing.Point(488, 10)
    $lblPass.Size = New-Object System.Drawing.Size(35, 18)
    $tabPage.Controls.Add($lblPass)
    
    $txtPass = New-Object System.Windows.Forms.TextBox
    $txtPass.Location = New-Object System.Drawing.Point(525, 7)
    $txtPass.Size = New-Object System.Drawing.Size(120, 22)
    $txtPass.PasswordChar = [char]'*'
    $txtPass.Name = "txtPassword"
    $txtPass.Enabled = -not $IsLocal
    $tabPage.Controls.Add($txtPass)
    
    $btnShow = New-Object System.Windows.Forms.Button
    $btnShow.Text = "Show"
    $btnShow.Location = New-Object System.Drawing.Point(650, 6)
    $btnShow.Size = New-Object System.Drawing.Size(45, 22)
    $btnShow.Name = "btnShow"
    $btnShow.Enabled = -not $IsLocal
    $btnShow.BackColor = [System.Drawing.Color]::LightGray
    $btnShow.ForeColor = [System.Drawing.Color]::Black
    $tabPage.Controls.Add($btnShow)
    
    $btnTest = New-Object System.Windows.Forms.Button
    $btnTest.Text = "Test"
    $btnTest.Location = New-Object System.Drawing.Point(700, 6)
    $btnTest.Size = New-Object System.Drawing.Size(45, 22)
    $btnTest.BackColor = [System.Drawing.Color]::LightBlue
    $btnTest.ForeColor = [System.Drawing.Color]::Black
    $btnTest.Name = "btnTest"
    $tabPage.Controls.Add($btnTest)
    
    # Status on same row, after buttons
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Text = if ($IsLocal) { "Ready (Local)" } else { "Enter credentials and test" }
    $lblStatus.Location = New-Object System.Drawing.Point(755, 10)
    $lblStatus.Size = New-Object System.Drawing.Size(350, 18)
    $lblStatus.ForeColor = if ($IsLocal) { [System.Drawing.Color]::Green } else { [System.Drawing.Color]::Gray }
    $lblStatus.Name = "lblStatus"
    $tabPage.Controls.Add($lblStatus)
    
    # Browse button for DESTINATION mode (browse for local backup folder)
    $btnBrowseSource = New-Object System.Windows.Forms.Button
    $btnBrowseSource.Text = "Browse..."
    $btnBrowseSource.Location = New-Object System.Drawing.Point(1050, 6)
    $btnBrowseSource.Size = New-Object System.Drawing.Size(65, 22)
    $btnBrowseSource.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
    $btnBrowseSource.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $btnBrowseSource.Name = "btnBrowseSource"
    $btnBrowseSource.Visible = $false
    $btnBrowseSource.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $tabPage.Controls.Add($btnBrowseSource)
    
    $btnBrowseSource.Add_Click({
        $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
        $fbd.Description = "Select backup folder to restore from (local drive, USB, mapped drive, or Google Drive sync folder)"
        $fbd.ShowNewFolderButton = $false
        if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $tab = $tabSources.SelectedTab
            if ($tab) {
                $comp = $tab.Controls["txtComputer"]
                if ($comp) { $comp.Text = $fbd.SelectedPath }
            }
        }
    })
    
    # Remove source button (not for Local tab)
    if ($SourceName -ne "Local") {
        $btnRemove = New-Object System.Windows.Forms.Button
        $btnRemove.Text = "X"
        $btnRemove.Location = New-Object System.Drawing.Point(1120, 6)
        $btnRemove.Size = New-Object System.Drawing.Size(25, 22)
        $btnRemove.BackColor = [System.Drawing.Color]::LightCoral
        $btnRemove.ForeColor = [System.Drawing.Color]::Black
        $btnRemove.Name = "btnRemove"
        $btnRemove.Tag = $SourceName
        $tabPage.Controls.Add($btnRemove)
        
        $btnRemove.Add_Click({
            $sourceName = $this.Tag
            $result = [System.Windows.Forms.MessageBox]::Show(
                "Remove source '$sourceName'?",
                "Confirm Remove",
                [System.Windows.Forms.MessageBoxButtons]::YesNo,
                [System.Windows.Forms.MessageBoxIcon]::Question
            )
            if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
                # Find and remove the tab
                $tabToRemove = $null
                foreach ($tab in $tabSources.TabPages) {
                    if ($tab.Tag -eq $sourceName) {
                        $tabToRemove = $tab
                        break
                    }
                }
                if ($tabToRemove) {
                    $tabSources.TabPages.Remove($tabToRemove)
                    $Script:SourceTabs.Remove($sourceName)
                    # Delete config file
                    $configPath = "$Script:InstallPath\Settings\source-$sourceName.json"
                    if (Test-Path $configPath) { Remove-Item $configPath -Force }
                }
            }
        })
    }
    
    # Local checkbox toggle
    $chkLoc.Add_CheckedChanged({
        $tab = $this.Parent
        $isLocal = $this.Checked
        foreach ($ctrl in $tab.Controls) {
            if ($ctrl.Name -in @("txtComputer", "txtUsername", "txtPassword", "btnShow")) {
                $ctrl.Enabled = -not $isLocal
            }
        }
        if ($isLocal) {
            $tab.Controls["txtComputer"].Text = $env:COMPUTERNAME
        }
    })
    
    # Show password button
    $btnShow.Add_Click({
        $tab = $this.Parent
        $pwdBox = $tab.Controls["txtPassword"]
        if ($pwdBox.PasswordChar -eq [char]'*') {
            $pwdBox.PasswordChar = [char]0
            $this.Text = "Hide"
        } else {
            $pwdBox.PasswordChar = [char]'*'
            $this.Text = "Show"
        }
    })
    
    # Test connection button
    $btnTest.Add_Click({
        $tab = $this.Parent
        $lblStat = $tab.Controls["lblStatus"]
        $computer = $tab.Controls["txtComputer"].Text
        $isLocal = $tab.Controls["chkLocal"].Checked
        
        $lblStat.Text = "Testing connection..."
        $lblStat.ForeColor = [System.Drawing.Color]::Orange
        $tab.Refresh()
        
        try {
            if ($isLocal) {
                $lblStat.Text = "Status: Local mode - Ready"
                $lblStat.ForeColor = [System.Drawing.Color]::Green
            } else {
                if ([string]::IsNullOrWhiteSpace($computer)) {
                    throw "Enter computer name or IP"
                }
                $pingResult = Test-Connection -ComputerName $computer -Count 1 -Quiet -ErrorAction Stop
                if ($pingResult) {
                    # Try to access admin share
                    $testPath = "\\$computer\C$"
                    if (Test-Path $testPath -ErrorAction SilentlyContinue) {
                        $lblStat.Text = "Status: Connected to $computer - Ready"
                        $lblStat.ForeColor = [System.Drawing.Color]::Green
                    } else {
                        $lblStat.Text = "Status: $computer reachable but admin share access denied"
                        $lblStat.ForeColor = [System.Drawing.Color]::Orange
                    }
                } else {
                    $lblStat.Text = "Status: Cannot reach $computer"
                    $lblStat.ForeColor = [System.Drawing.Color]::Red
                }
            }
        } catch {
            $lblStat.Text = "Status: Error - $_"
            $lblStat.ForeColor = [System.Drawing.Color]::Red
        }
    })
    
    $tabSources.TabPages.Add($tabPage)
    $Script:SourceTabs[$SourceName] = $tabPage
    return $tabPage
}

# Add Source tab will be added AFTER Load-SourceTabsAtStartup
$tabAddSource = New-Object System.Windows.Forms.TabPage
$tabAddSource.Text = "+ Add Source"
$tabAddSource.BackColor = [System.Drawing.Color]::FromArgb(32, 32, 32)

$lblAddInfo = New-Object System.Windows.Forms.Label
$lblAddInfo.Text = "Add SOURCE:"
$lblAddInfo.Location = New-Object System.Drawing.Point(8, 10)
$lblAddInfo.Size = New-Object System.Drawing.Size(80, 18)
$tabAddSource.Controls.Add($lblAddInfo)

$txtNewSourceName = New-Object System.Windows.Forms.TextBox
$txtNewSourceName.Location = New-Object System.Drawing.Point(90, 7)
$txtNewSourceName.Size = New-Object System.Drawing.Size(120, 22)
$txtNewSourceName.Text = "Source2"
$tabAddSource.Controls.Add($txtNewSourceName)

$btnAddNewSource = New-Object System.Windows.Forms.Button
$btnAddNewSource.Text = "Add"
$btnAddNewSource.Location = New-Object System.Drawing.Point(215, 6)
$btnAddNewSource.Size = New-Object System.Drawing.Size(50, 22)
$btnAddNewSource.BackColor = [System.Drawing.Color]::LightGreen
$btnAddNewSource.ForeColor = [System.Drawing.Color]::Black
$tabAddSource.Controls.Add($btnAddNewSource)

$btnAddNewSource.Add_Click({
    $newName = $txtNewSourceName.Text.Trim()
    if ([string]::IsNullOrWhiteSpace($newName)) {
        [System.Windows.Forms.MessageBox]::Show("Enter a source name", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
        return
    }
    if ($Script:SourceTabs.ContainsKey($newName)) {
        [System.Windows.Forms.MessageBox]::Show("Source '$newName' already exists", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
        return
    }
    
    # Remove "+ Add Source" tab, add new source, re-add "+ Add Source"
    $tabSources.TabPages.Remove($tabAddSource)
    Add-SourceTabToMain -SourceName $newName -ComputerName "" -Username ".\Administrator" -IsLocal $false
    if (-not $rbSource.Checked) {
        $tabSources.TabPages.Add($tabAddSource)
    }
    $tabSources.SelectedIndex = $tabSources.TabPages.Count - 2  # Select new tab
    $txtNewSourceName.Text = "Source" + ($Script:SourceTabs.Count + 1)
})

# Load existing sources at startup
function Load-SourceTabsAtStartup {
    # Always add Local tab first
    Add-SourceTabToMain -SourceName "Local" -IsLocal $true
    
    # Load additional sources from JSON files
    $configPath = "$Script:InstallPath\Settings"
    if (Test-Path $configPath) {
        Get-ChildItem -Path $configPath -Filter "source-*.json" -ErrorAction SilentlyContinue | ForEach-Object {
            try {
                $config = Get-Content $_.FullName -Raw | ConvertFrom-Json
                if ($config.Name -and $config.Name -ne "Local") {
                    Add-SourceTabToMain -SourceName $config.Name -ComputerName $config.ComputerName -Username $config.Username -IsLocal $false
                }
            } catch {
                Write-Host "Error loading source config: $_"
            }
        }
    }
    
    # Add "+ Add Source" tab at the end
    if (-not $rbSource.Checked) {
        $tabSources.TabPages.Add($tabAddSource)
    }
}

# Call load function (will be executed when form loads)
Load-SourceTabsAtStartup

# Helper function to get current source connection info
function Get-CurrentSourceConnection {
    $selectedTab = $tabSources.SelectedTab
    if (-not $selectedTab -or $selectedTab.Text -eq "+ Add Source") {
        # Default to first tab (Local)
        $selectedTab = $tabSources.TabPages[0]
    }
    
    return @{
        Computer = $selectedTab.Controls["txtComputer"].Text
        Username = $selectedTab.Controls["txtUsername"].Text
        Password = $selectedTab.Controls["txtPassword"].Text
        IsLocal = $selectedTab.Controls["chkLocal"].Checked
        SourceName = $selectedTab.Tag
    }
}

# Create wrapper variables pointing to Local tab controls (for backwards compatibility)
$Script:LocalTab = $Script:SourceTabs["Local"]
$txtComputer = $Script:LocalTab.Controls["txtComputer"]
$txtUsername = $Script:LocalTab.Controls["txtUsername"]
$txtPassword = $Script:LocalTab.Controls["txtPassword"]
$chkLocal = $Script:LocalTab.Controls["chkLocal"]
$btnShowPass = $Script:LocalTab.Controls["btnShow"]
$btnTestConnection = $Script:LocalTab.Controls["btnTest"]
$lblConnectionStatus = $Script:LocalTab.Controls["lblStatus"]

# Create a dummy grpConnection for code that references it
$grpConnection = New-Object System.Windows.Forms.GroupBox
$grpConnection.Text = "SOURCE Server Connection"
$grpConnection.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
$grpConnection.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
$grpConnection.Visible = $false
#endregion

#region Components Selection
$grpComponents = New-Object System.Windows.Forms.GroupBox
$grpComponents.Text = "Migration Components"
$grpComponents.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
$grpComponents.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
$grpComponents.Location = New-Object System.Drawing.Point(10, 122)
$grpComponents.Size = New-Object System.Drawing.Size(600, 280)
$form.Controls.Add($grpComponents)

$yPos = 18
$chkUsers = New-Object System.Windows.Forms.CheckBox
$chkUsers.Text = "Users and Groups"
$chkUsers.Location = New-Object System.Drawing.Point(15, $yPos)
$chkUsers.Size = New-Object System.Drawing.Size(280, 18)
$chkUsers.Checked = $true
$grpComponents.Controls.Add($chkUsers)

$yPos += 18
$chkCreateProfiles = New-Object System.Windows.Forms.CheckBox
$chkCreateProfiles.Text = "  - Create user profiles on destination"
$chkCreateProfiles.Location = New-Object System.Drawing.Point(30, $yPos)
$chkCreateProfiles.Size = New-Object System.Drawing.Size(260, 18)
$chkCreateProfiles.Enabled = $false
$grpComponents.Controls.Add($chkCreateProfiles)

$yPos += 20
$chkShares = New-Object System.Windows.Forms.CheckBox
$chkShares.Text = "Network Shares"
$chkShares.Location = New-Object System.Drawing.Point(15, $yPos)
$chkShares.Size = New-Object System.Drawing.Size(280, 18)
$chkShares.Checked = $true
$grpComponents.Controls.Add($chkShares)

$yPos += 18
$chkPrinters = New-Object System.Windows.Forms.CheckBox
$chkPrinters.Text = "Printers (with drivers via printbrm)"
$chkPrinters.Location = New-Object System.Drawing.Point(15, $yPos)
$chkPrinters.Size = New-Object System.Drawing.Size(280, 18)
$chkPrinters.Checked = $true
$grpComponents.Controls.Add($chkPrinters)

$yPos += 20
$chkUserData = New-Object System.Windows.Forms.CheckBox
$chkUserData.Text = "User Data (Desktop, Documents, Downloads, etc.)"
$chkUserData.Location = New-Object System.Drawing.Point(15, $yPos)
$chkUserData.Size = New-Object System.Drawing.Size(280, 18)
$chkUserData.Checked = $true
$grpComponents.Controls.Add($chkUserData)

# Users button right next to User Data
$btnUserDataUsers = New-Object System.Windows.Forms.Button
$btnUserDataUsers.Text = "Users..."
$btnUserDataUsers.Location = New-Object System.Drawing.Point(300, ($yPos - 2))
$btnUserDataUsers.Size = New-Object System.Drawing.Size(55, 20)
$btnUserDataUsers.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnUserDataUsers.BackColor = [System.Drawing.Color]::LightGreen
$btnUserDataUsers.ForeColor = [System.Drawing.Color]::Black
$grpComponents.Controls.Add($btnUserDataUsers)

# Wire up Users button (shares handler with App Settings Users button)
# Click handler wired up after btnConfigureUsers is defined

# AppData checkbox with warning - NOT checked by default
$yPos += 18
$chkAppData = New-Object System.Windows.Forms.CheckBox
$chkAppData.Text = "  Include AppData (may cause errors!)"
$chkAppData.Location = New-Object System.Drawing.Point(30, $yPos)
$chkAppData.Size = New-Object System.Drawing.Size(220, 18)
$chkAppData.Checked = $false
$chkAppData.ForeColor = [System.Drawing.Color]::DarkRed
$grpComponents.Controls.Add($chkAppData)

# Warning when AppData is checked
$chkAppData.Add_CheckedChanged({
    if ($chkAppData.Checked) {
        $result = [System.Windows.Forms.MessageBox]::Show(
            "WARNING: Copying AppData can cause errors!`n`nMany apps store machine-specific data, licenses, and cache that will not work on another computer.`n`nCommon issues:`n- Browser profiles may not work`n- App licenses may be invalidated`n- Cached data may cause conflicts`n`nRecommended: Only copy specific app folders you need.`n`nContinue anyway?",
            "AppData Warning",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Warning
        )
        if ($result -eq [System.Windows.Forms.DialogResult]::No) {
            $chkAppData.Checked = $false
        }
    }
})

# Store AppData setting in script scope
$Script:IncludeAppData = $false
$chkAppData.Add_CheckedChanged({ $Script:IncludeAppData = $chkAppData.Checked })

$yPos += 18
$chkProgramFiles = New-Object System.Windows.Forms.CheckBox
$chkProgramFiles.Text = "Program Files (excludes Windows/Microsoft)"
$chkProgramFiles.Location = New-Object System.Drawing.Point(15, $yPos)
$chkProgramFiles.Size = New-Object System.Drawing.Size(280, 18)
$grpComponents.Controls.Add($chkProgramFiles)

$yPos += 18
$chkProgramFilesX86 = New-Object System.Windows.Forms.CheckBox
$chkProgramFilesX86.Text = "Program Files (x86)"
$chkProgramFilesX86.Location = New-Object System.Drawing.Point(15, $yPos)
$chkProgramFilesX86.Size = New-Object System.Drawing.Size(280, 18)
$grpComponents.Controls.Add($chkProgramFilesX86)

$yPos += 18
$chkProgramData = New-Object System.Windows.Forms.CheckBox
$chkProgramData.Text = "ProgramData (settings/templates, no licenses)"
$chkProgramData.Location = New-Object System.Drawing.Point(15, $yPos)
$chkProgramData.Size = New-Object System.Drawing.Size(290, 18)
$grpComponents.Controls.Add($chkProgramData)

$yPos += 20
$chkRegistry = New-Object System.Windows.Forms.CheckBox
$chkRegistry.Text = "Extract License Keys (to report)"
$chkRegistry.Location = New-Object System.Drawing.Point(15, $yPos)
$chkRegistry.Size = New-Object System.Drawing.Size(250, 18)
$chkRegistry.Checked = $true
$grpComponents.Controls.Add($chkRegistry)

$yPos += 18
$chkAppSettings = New-Object System.Windows.Forms.CheckBox
$chkAppSettings.Text = "App Settings"
$chkAppSettings.Location = New-Object System.Drawing.Point(15, $yPos)
$chkAppSettings.Size = New-Object System.Drawing.Size(90, 18)
$chkAppSettings.Checked = $true
$grpComponents.Controls.Add($chkAppSettings)

$btnConfigureApps = New-Object System.Windows.Forms.Button
$btnConfigureApps.Text = "Apps..."
$btnConfigureApps.Location = New-Object System.Drawing.Point(110, ($yPos - 2))
$btnConfigureApps.Size = New-Object System.Drawing.Size(55, 20)
$btnConfigureApps.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnConfigureApps.BackColor = [System.Drawing.Color]::LightSteelBlue
$btnConfigureApps.ForeColor = [System.Drawing.Color]::Black
$grpComponents.Controls.Add($btnConfigureApps)

$btnConfigureUsers = New-Object System.Windows.Forms.Button
$btnConfigureUsers.Text = "Users..."
$btnConfigureUsers.Location = New-Object System.Drawing.Point(170, ($yPos - 2))
$btnConfigureUsers.Size = New-Object System.Drawing.Size(55, 20)
$btnConfigureUsers.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnConfigureUsers.BackColor = [System.Drawing.Color]::LightGreen
$btnConfigureUsers.ForeColor = [System.Drawing.Color]::Black
$grpComponents.Controls.Add($btnConfigureUsers)

$lblAppUserSummary = New-Object System.Windows.Forms.Label
$lblAppUserSummary.Text = "(All users)"
$lblAppUserSummary.Location = New-Object System.Drawing.Point(230, $yPos)
$lblAppUserSummary.Size = New-Object System.Drawing.Size(100, 18)
$lblAppUserSummary.ForeColor = [System.Drawing.Color]::Gray
$grpComponents.Controls.Add($lblAppUserSummary)

# Additional Drives section
$yPos += 22
$lblDrives = New-Object System.Windows.Forms.Label
$lblDrives.Text = "Additional Drives:"
$lblDrives.Location = New-Object System.Drawing.Point(15, $yPos)
$lblDrives.Size = New-Object System.Drawing.Size(110, 18)
$lblDrives.Font = New-Object System.Drawing.Font("Arial", 9, [System.Drawing.FontStyle]::Bold)
$grpComponents.Controls.Add($lblDrives)

# Create drive checkboxes dynamically - detect all available drives D-Z
$Script:DriveCheckboxes = @{}

# Get all fixed/removable drives (exclude C: and network drives)
$availableDrives = @()
try {
    $drives = Get-CimInstance -ClassName Win32_LogicalDisk -ErrorAction SilentlyContinue | 
        Where-Object { $_.DeviceID -ne "C:" -and $_.DriveType -in @(2, 3) }  # 2=Removable, 3=Fixed
    $availableDrives = $drives | ForEach-Object { $_.DeviceID.TrimEnd(':') }
}
catch {
    # Fallback - check each drive letter
    foreach ($letter in @('D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z')) {
        if (Test-Path "${letter}:\") {
            $availableDrives += $letter
        }
    }
}

# Create checkboxes for D through L (first row)
$driveXPos = 130
$driveYPos = $yPos
$driveCount = 0

foreach ($driveLetter in @('D','E','F','G','H','I','J','K','L')) {
    $drivePath = "${driveLetter}:\"
    $chkDrive = New-Object System.Windows.Forms.CheckBox
    $chkDrive.Text = "${driveLetter}:"
    $chkDrive.Location = New-Object System.Drawing.Point($driveXPos, $driveYPos)
    $chkDrive.Size = New-Object System.Drawing.Size(42, 18)
    $chkDrive.Tag = $driveLetter
    
    # Enable only if drive exists and is accessible
    $driveExists = $driveLetter -in $availableDrives -or (Test-Path $drivePath)
    if ($driveExists) {
        $chkDrive.Enabled = $true
        $chkDrive.ForeColor = [System.Drawing.Color]::Black
        # Get drive label for tooltip
        try {
            $driveInfo = Get-CimInstance -ClassName Win32_LogicalDisk -Filter "DeviceID='${driveLetter}:'" -ErrorAction SilentlyContinue
            if ($driveInfo -and $driveInfo.VolumeName) {
                $tooltip = New-Object System.Windows.Forms.ToolTip
                $tooltip.SetToolTip($chkDrive, "$($driveInfo.VolumeName) - $([math]::Round($driveInfo.Size/1GB, 1)) GB")
            }
        } catch { }
    } else {
        $chkDrive.Enabled = $false
        $chkDrive.ForeColor = [System.Drawing.Color]::LightGray
    }
    
    $grpComponents.Controls.Add($chkDrive)
    $Script:DriveCheckboxes[$driveLetter] = $chkDrive
    $driveXPos += 42
    $driveCount++
    
    # Wrap to second row after 6 checkboxes
    if ($driveCount -eq 6) {
        $driveXPos = 130
        $driveYPos += 18
    }
}

# Update yPos if we used two rows
if ($driveCount -gt 6) {
    $yPos += 18
}
#endregion

#region Settings Panel
$grpSettings = New-Object System.Windows.Forms.GroupBox
$grpSettings.Text = "Migration Settings"
$grpSettings.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
$grpSettings.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
$grpSettings.Location = New-Object System.Drawing.Point(620, 122)
$grpSettings.Size = New-Object System.Drawing.Size(710, 280)
$form.Controls.Add($grpSettings)

# Overwrite existing files checkbox
$chkOverwrite = New-Object System.Windows.Forms.CheckBox
$chkOverwrite.Text = "Overwrite existing files (if unchecked, skip existing)"
$chkOverwrite.Location = New-Object System.Drawing.Point(15, 18)
$chkOverwrite.Size = New-Object System.Drawing.Size(320, 18)
$chkOverwrite.Checked = $false
$grpSettings.Controls.Add($chkOverwrite)

# Sub-checkbox: Only copy newer files (only enabled when Overwrite is checked)
$chkNewerOnly = New-Object System.Windows.Forms.CheckBox
$chkNewerOnly.Text = "Only replace if source is newer"
$chkNewerOnly.Location = New-Object System.Drawing.Point(35, 36)
$chkNewerOnly.Size = New-Object System.Drawing.Size(220, 18)
$chkNewerOnly.Checked = $false
$chkNewerOnly.Enabled = $false
$chkNewerOnly.ForeColor = [System.Drawing.Color]::Gray
$grpSettings.Controls.Add($chkNewerOnly)

# Generate button (for master password - DESTINATION mode)
$btnGenerate = New-Object System.Windows.Forms.Button
$btnGenerate.Text = "Generate"
$btnGenerate.Location = New-Object System.Drawing.Point(340, 15)
$btnGenerate.Size = New-Object System.Drawing.Size(70, 22)
$btnGenerate.BackColor = [System.Drawing.Color]::LightGreen
$btnGenerate.ForeColor = [System.Drawing.Color]::Black
$btnGenerate.Visible = $false
$grpSettings.Controls.Add($btnGenerate)

# Enable/disable NewerOnly based on Overwrite
$chkOverwrite.Add_CheckedChanged({
    $chkNewerOnly.Enabled = $chkOverwrite.Checked
    if ($chkOverwrite.Checked) {
        $chkNewerOnly.ForeColor = [System.Drawing.Color]::Black
    } else {
        $chkNewerOnly.Checked = $false
        $chkNewerOnly.ForeColor = [System.Drawing.Color]::Gray
    }
})

# Stop services for locked files checkbox
$chkStopServices = New-Object System.Windows.Forms.CheckBox
$chkStopServices.Text = "Stop services for locked files (QuickBooks, SQL, etc.)"
$chkStopServices.Location = New-Object System.Drawing.Point(15, 55)
$chkStopServices.Size = New-Object System.Drawing.Size(320, 18)
$chkStopServices.Checked = $false
$tooltip = New-Object System.Windows.Forms.ToolTip
$tooltip.SetToolTip($chkStopServices, "Temporarily stops database services to copy locked files, then restarts them")
$grpSettings.Controls.Add($chkStopServices)

# Enable file versioning (rollback) checkbox with days selector
$chkRollback = New-Object System.Windows.Forms.CheckBox
$chkRollback.Text = "Enable file versioning"
$chkRollback.Location = New-Object System.Drawing.Point(15, 73)
$chkRollback.Size = New-Object System.Drawing.Size(130, 18)
$chkRollback.Checked = $false
$tooltipRollback = New-Object System.Windows.Forms.ToolTip
$tooltipRollback.SetToolTip($chkRollback, "Creates dated backup copies before overwriting files, allows recovery")
$grpSettings.Controls.Add($chkRollback)

# Retention days selector
$numRollbackDays = New-Object System.Windows.Forms.NumericUpDown
$numRollbackDays.Location = New-Object System.Drawing.Point(150, 71)
$numRollbackDays.Size = New-Object System.Drawing.Size(45, 20)
$numRollbackDays.Minimum = 1
$numRollbackDays.Maximum = 30
$numRollbackDays.Value = 7
$numRollbackDays.Enabled = $false
$grpSettings.Controls.Add($numRollbackDays)

$lblRollbackDays = New-Object System.Windows.Forms.Label
$lblRollbackDays.Text = "days rollback"
$lblRollbackDays.Location = New-Object System.Drawing.Point(198, 74)
$lblRollbackDays.Size = New-Object System.Drawing.Size(80, 18)
$lblRollbackDays.ForeColor = [System.Drawing.Color]::FromArgb(180, 180, 180)
$grpSettings.Controls.Add($lblRollbackDays)

# Enable/disable days selector based on checkbox
$chkRollback.Add_CheckedChanged({
    $numRollbackDays.Enabled = $chkRollback.Checked
})

# Master Password (only shown in DESTINATION mode)
$lblMasterPassword = New-Object System.Windows.Forms.Label
$lblMasterPassword.Text = "Master Password:"
$lblMasterPassword.Location = New-Object System.Drawing.Point(15, 95)
$lblMasterPassword.Size = New-Object System.Drawing.Size(100, 18)
$lblMasterPassword.Visible = $false
$grpSettings.Controls.Add($lblMasterPassword)

$txtMasterPassword = New-Object System.Windows.Forms.TextBox
$txtMasterPassword.Location = New-Object System.Drawing.Point(120, 92)
$txtMasterPassword.Size = New-Object System.Drawing.Size(180, 20)
$txtMasterPassword.PasswordChar = [char]'*'
$txtMasterPassword.Font = New-Object System.Drawing.Font("Consolas", 9)
$txtMasterPassword.Visible = $false
$grpSettings.Controls.Add($txtMasterPassword)

$btnShowMaster = New-Object System.Windows.Forms.Button
$btnShowMaster.Text = "Show"
$btnShowMaster.Location = New-Object System.Drawing.Point(305, 91)
$btnShowMaster.Size = New-Object System.Drawing.Size(45, 22)
$btnShowMaster.Visible = $false
$grpSettings.Controls.Add($btnShowMaster)

# Password Enforcement (DESTINATION mode)
$chkPasswordMonitoring = New-Object System.Windows.Forms.CheckBox
$chkPasswordMonitoring.Text = "Install Password Change Monitoring"
$chkPasswordMonitoring.Location = New-Object System.Drawing.Point(15, 115)
$chkPasswordMonitoring.Size = New-Object System.Drawing.Size(280, 18)
$chkPasswordMonitoring.Checked = $true
$chkPasswordMonitoring.Visible = $false
$grpSettings.Controls.Add($chkPasswordMonitoring)

$lblMonitoringInfo = New-Object System.Windows.Forms.Label
$lblMonitoringInfo.Text = "  Admin GUI + User reminders"
$lblMonitoringInfo.Location = New-Object System.Drawing.Point(15, 133)
$lblMonitoringInfo.Size = New-Object System.Drawing.Size(200, 16)
$lblMonitoringInfo.ForeColor = [System.Drawing.Color]::Gray
$lblMonitoringInfo.Visible = $false
$grpSettings.Controls.Add($lblMonitoringInfo)

# Robocopy Threads
$lblThreads = New-Object System.Windows.Forms.Label
$lblThreads.Text = "Robocopy Threads:"
$lblThreads.Location = New-Object System.Drawing.Point(15, 95)
$lblThreads.Size = New-Object System.Drawing.Size(110, 18)
$grpSettings.Controls.Add($lblThreads)

$numThreads = New-Object System.Windows.Forms.NumericUpDown
$numThreads.Location = New-Object System.Drawing.Point(130, 92)
$numThreads.Size = New-Object System.Drawing.Size(55, 20)
$numThreads.Minimum = 1
$numThreads.Maximum = 128
$numThreads.Value = 8
$grpSettings.Controls.Add($numThreads)

# Create Shares (SOURCE only)
$chkCreateShares = New-Object System.Windows.Forms.CheckBox
$chkCreateShares.Text = "Create Migration Shares (SOURCE mode)"
$chkCreateShares.Location = New-Object System.Drawing.Point(15, 115)
$chkCreateShares.Size = New-Object System.Drawing.Size(280, 18)
$chkCreateShares.Checked = $true
$grpSettings.Controls.Add($chkCreateShares)

$lblSharesInfo = New-Object System.Windows.Forms.Label
$lblSharesInfo.Text = "  Creates hidden shares for migration access"
$lblSharesInfo.Location = New-Object System.Drawing.Point(15, 133)
$lblSharesInfo.Size = New-Object System.Drawing.Size(280, 16)
$lblSharesInfo.ForeColor = [System.Drawing.Color]::Gray
$grpSettings.Controls.Add($lblSharesInfo)

# Recovery Tools section (SOURCE mode only)
$lblRecoveryTools = New-Object System.Windows.Forms.Label
$lblRecoveryTools.Text = "Recovery Tools:"
$lblRecoveryTools.Location = New-Object System.Drawing.Point(285, 55)
$lblRecoveryTools.Size = New-Object System.Drawing.Size(130, 18)
$lblRecoveryTools.Font = New-Object System.Drawing.Font("Arial", 9, [System.Drawing.FontStyle]::Bold)
$grpSettings.Controls.Add($lblRecoveryTools)

$btnKeyFinder = New-Object System.Windows.Forms.Button
$btnKeyFinder.Text = "Keys"
$btnKeyFinder.Location = New-Object System.Drawing.Point(305, 73)
$btnKeyFinder.Size = New-Object System.Drawing.Size(55, 22)
$btnKeyFinder.BackColor = [System.Drawing.Color]::LightBlue
$btnKeyFinder.ForeColor = [System.Drawing.Color]::Black
$btnKeyFinder.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$grpSettings.Controls.Add($btnKeyFinder)

$btnLaZagne = New-Object System.Windows.Forms.Button
$btnLaZagne.Text = "Passwords"
$btnLaZagne.Location = New-Object System.Drawing.Point(365, 73)
$btnLaZagne.Size = New-Object System.Drawing.Size(70, 22)
$btnLaZagne.BackColor = [System.Drawing.Color]::LightYellow
$btnLaZagne.ForeColor = [System.Drawing.Color]::Black
$btnLaZagne.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$grpSettings.Controls.Add($btnLaZagne)

$btnViewReport = New-Object System.Windows.Forms.Button
$btnViewReport.Text = "Report"
$btnViewReport.Location = New-Object System.Drawing.Point(440, 73)
$btnViewReport.Size = New-Object System.Drawing.Size(55, 22)
$btnViewReport.BackColor = [System.Drawing.Color]::WhiteSmoke
$btnViewReport.ForeColor = [System.Drawing.Color]::Black
$btnViewReport.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$grpSettings.Controls.Add($btnViewReport)

# Scheduling section
$lblSchedule = New-Object System.Windows.Forms.Label
$lblSchedule.Text = "Scheduled Tasks:"
$lblSchedule.Location = New-Object System.Drawing.Point(15, 155)
$lblSchedule.Size = New-Object System.Drawing.Size(120, 18)
$lblSchedule.Font = New-Object System.Drawing.Font("Arial", 9, [System.Drawing.FontStyle]::Bold)
$grpSettings.Controls.Add($lblSchedule)

$chkScheduleEnabled = New-Object System.Windows.Forms.CheckBox
$chkScheduleEnabled.Text = "Enable daily scheduled run"
$chkScheduleEnabled.Location = New-Object System.Drawing.Point(15, 175)
$chkScheduleEnabled.Size = New-Object System.Drawing.Size(180, 18)
$grpSettings.Controls.Add($chkScheduleEnabled)

$lblScheduleTime = New-Object System.Windows.Forms.Label
$lblScheduleTime.Text = "Time:"
$lblScheduleTime.Location = New-Object System.Drawing.Point(200, 175)
$lblScheduleTime.Size = New-Object System.Drawing.Size(35, 18)
$grpSettings.Controls.Add($lblScheduleTime)

$cboScheduleHour = New-Object System.Windows.Forms.ComboBox
$cboScheduleHour.Location = New-Object System.Drawing.Point(238, 172)
$cboScheduleHour.Size = New-Object System.Drawing.Size(45, 21)
$cboScheduleHour.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
for ($h = 0; $h -le 23; $h++) { $cboScheduleHour.Items.Add($h.ToString("00")) | Out-Null }
$cboScheduleHour.SelectedIndex = 21
$grpSettings.Controls.Add($cboScheduleHour)

$lblColon = New-Object System.Windows.Forms.Label
$lblColon.Text = ":"
$lblColon.Location = New-Object System.Drawing.Point(285, 175)
$lblColon.Size = New-Object System.Drawing.Size(8, 18)
$grpSettings.Controls.Add($lblColon)

$cboScheduleMinute = New-Object System.Windows.Forms.ComboBox
$cboScheduleMinute.Location = New-Object System.Drawing.Point(295, 172)
$cboScheduleMinute.Size = New-Object System.Drawing.Size(45, 21)
$cboScheduleMinute.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
for ($m = 0; $m -le 55; $m += 5) { $cboScheduleMinute.Items.Add($m.ToString("00")) | Out-Null }
$cboScheduleMinute.SelectedIndex = 0
$grpSettings.Controls.Add($cboScheduleMinute)

$lblScheduleStatus = New-Object System.Windows.Forms.Label
$lblScheduleStatus.Text = ""
$lblScheduleStatus.Location = New-Object System.Drawing.Point(15, 195)
$lblScheduleStatus.Size = New-Object System.Drawing.Size(350, 16)
$lblScheduleStatus.ForeColor = [System.Drawing.Color]::Green
$grpSettings.Controls.Add($lblScheduleStatus)

# Schedule buttons row
$btnSaveSchedule = New-Object System.Windows.Forms.Button
$btnSaveSchedule.Text = "Save + Schedule"
$btnSaveSchedule.Location = New-Object System.Drawing.Point(15, 213)
$btnSaveSchedule.Size = New-Object System.Drawing.Size(100, 24)
$btnSaveSchedule.BackColor = [System.Drawing.Color]::LightGreen
$btnSaveSchedule.ForeColor = [System.Drawing.Color]::Black
$btnSaveSchedule.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$grpSettings.Controls.Add($btnSaveSchedule)

$btnSaveSettings = New-Object System.Windows.Forms.Button
$btnSaveSettings.Text = "Save"
$btnSaveSettings.Location = New-Object System.Drawing.Point(120, 213)
$btnSaveSettings.Size = New-Object System.Drawing.Size(60, 24)
$btnSaveSettings.BackColor = [System.Drawing.Color]::LightGreen
$btnSaveSettings.ForeColor = [System.Drawing.Color]::Black
$btnSaveSettings.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$grpSettings.Controls.Add($btnSaveSettings)

$btnRemoveSchedule = New-Object System.Windows.Forms.Button
$btnRemoveSchedule.Text = "Remove"
$btnRemoveSchedule.Location = New-Object System.Drawing.Point(185, 213)
$btnRemoveSchedule.Size = New-Object System.Drawing.Size(65, 24)
$btnRemoveSchedule.BackColor = [System.Drawing.Color]::LightCoral
$btnRemoveSchedule.ForeColor = [System.Drawing.Color]::Black
$btnRemoveSchedule.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$grpSettings.Controls.Add($btnRemoveSchedule)

# Tool buttons row
$btnCreateVHD = New-Object System.Windows.Forms.Button
$btnCreateVHD.Text = "Disk to VHDX..."
$btnCreateVHD.Location = New-Object System.Drawing.Point(15, 242)
$btnCreateVHD.Size = New-Object System.Drawing.Size(95, 24)
$btnCreateVHD.BackColor = [System.Drawing.Color]::LightSteelBlue
$btnCreateVHD.ForeColor = [System.Drawing.Color]::Black
$btnCreateVHD.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$btnCreateVHD.Visible = $true
$tooltipVHD = New-Object System.Windows.Forms.ToolTip
$tooltipVHD.SetToolTip($btnCreateVHD, "Convert physical disk to VHDX using Disk2VHD (for VM creation)")
$grpSettings.Controls.Add($btnCreateVHD)

# VHD button click handler
$btnCreateVHD.Add_Click({
    Write-GuiLog "Opening VHD creation dialog..."
    
    # Initialize Disk2VHD
    $disk2vhd = Initialize-Disk2VHD -ToolsPath "$Script:InstallPath\Tools"
    
    if (-not $disk2vhd) {
        $downloadResult = [System.Windows.Forms.MessageBox]::Show(
            "Disk2VHD is not available and could not be downloaded automatically.`n`nWould you like to download it manually from Microsoft?",
            "Disk2VHD Required",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($downloadResult -eq [System.Windows.Forms.DialogResult]::Yes) {
            Start-Process "https://docs.microsoft.com/sysinternals/downloads/disk2vhd"
        }
        return
    }
    
    # Show the VHD creation dialog
    $targetComputer = $env:COMPUTERNAME
    
    # If in DESTINATION mode, offer to create VHD on SOURCE
    if ($radDest.Checked -and $txtComputer.Text) {
        $choice = [System.Windows.Forms.MessageBox]::Show(
            "Create VHD on:`n`n- This computer ($env:COMPUTERNAME)`n- Source computer ($($txtComputer.Text))`n`nClick YES for this computer, NO for source computer.",
            "Select Computer",
            [System.Windows.Forms.MessageBoxButtons]::YesNoCancel,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($choice -eq [System.Windows.Forms.DialogResult]::Cancel) {
            return
        }
        elseif ($choice -eq [System.Windows.Forms.DialogResult]::No) {
            $targetComputer = $txtComputer.Text
        }
    }
    
    $vhdSettings = Show-Disk2VHDDialog -DefaultOutputPath "$Script:InstallPath\Backup" -ComputerName $targetComputer
    
    if ($vhdSettings) {
        Write-GuiLog "Creating VHD backup..."
        Write-GuiLog "Drives: $($vhdSettings.Drives -join ', ')"
        Write-GuiLog "Output: $($vhdSettings.OutputPath)"
        
        # Disable form during VHD creation
        $form.Enabled = $false
        $form.Cursor = [System.Windows.Forms.Cursors]::WaitCursor
        
        try {
            if ($targetComputer -eq $env:COMPUTERNAME) {
                # Local VHD creation
                $result = New-DiskImage -Drives $vhdSettings.Drives -OutputPath $vhdSettings.OutputPath `
                    -UseVHDX $vhdSettings.UseVHDX -UseVSS $vhdSettings.UseVSS
            }
            else {
                # Remote VHD creation
                $securePass = ConvertTo-SecureString $txtPassword.Text -AsPlainText -Force
                $cred = New-Object System.Management.Automation.PSCredential($txtUsername.Text, $securePass)
                
                $result = New-RemoteDiskImage -ComputerName $targetComputer -Drives $vhdSettings.Drives `
                    -OutputPath $vhdSettings.OutputPath -Credential $cred `
                    -UseVHDX $vhdSettings.UseVHDX -UseVSS $vhdSettings.UseVSS
            }
            
            if ($result) {
                Write-GuiLog "VHD backup created successfully!"
                [System.Windows.Forms.MessageBox]::Show(
                    "VHD backup created successfully!`n`nLocation: $($vhdSettings.OutputPath)",
                    "VHD Created",
                    [System.Windows.Forms.MessageBoxButtons]::OK,
                    [System.Windows.Forms.MessageBoxIcon]::Information
                )
            }
            else {
                Write-GuiLog "ERROR: VHD creation failed"
                [System.Windows.Forms.MessageBox]::Show(
                    "VHD creation failed. Check the log for details.",
                    "VHD Error",
                    [System.Windows.Forms.MessageBoxButtons]::OK,
                    [System.Windows.Forms.MessageBoxIcon]::Error
                )
            }
        }
        finally {
            $form.Enabled = $true
            $form.Cursor = [System.Windows.Forms.Cursors]::Default
        }
    }
})

# v4.0: Removed redundant VM Export button (consolidated into VHD Backup dialog)

# Rollback/Restore button
$btnRollback = New-Object System.Windows.Forms.Button
$btnRollback.Text = "Rollback..."
$btnRollback.Location = New-Object System.Drawing.Point(115, 242)
$btnRollback.Size = New-Object System.Drawing.Size(70, 24)
$btnRollback.BackColor = [System.Drawing.Color]::LightYellow
$btnRollback.ForeColor = [System.Drawing.Color]::Black
$btnRollback.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$tooltipRB = New-Object System.Windows.Forms.ToolTip
$tooltipRB.SetToolTip($btnRollback, "Restore files from previous versions")
$grpSettings.Controls.Add($btnRollback)

$btnRollback.Add_Click({
    Show-RollbackDialog
})

# Backup Settings button (v4.0: now the single entry point for all VM/VHD backups)
$btnBackupSettings = New-Object System.Windows.Forms.Button
$btnBackupSettings.Text = "VHD Backup..."
$btnBackupSettings.Location = New-Object System.Drawing.Point(190, 242)
$btnBackupSettings.Size = New-Object System.Drawing.Size(105, 24)
$btnBackupSettings.BackColor = [System.Drawing.Color]::LightCyan
$btnBackupSettings.ForeColor = [System.Drawing.Color]::Black
$btnBackupSettings.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$tooltipBS = New-Object System.Windows.Forms.ToolTip
$tooltipBS.SetToolTip($btnBackupSettings, "Configure VM export, staggered backup, and file backup settings")
$grpSettings.Controls.Add($btnBackupSettings)

$btnBackupSettings.Add_Click({
    Write-GuiLog "Opening VHD Backup Settings dialog (v4.0)..."
    Show-BackupSettingsDialog
})

# QuickBooks Settings button
$btnQuickBooks = New-Object System.Windows.Forms.Button
$btnQuickBooks.Text = "QuickBooks..."
$btnQuickBooks.Location = New-Object System.Drawing.Point(300, 242)
$btnQuickBooks.Size = New-Object System.Drawing.Size(85, 24)
$btnQuickBooks.BackColor = [System.Drawing.Color]::FromArgb(46, 139, 87)
$btnQuickBooks.ForeColor = [System.Drawing.Color]::Black
$btnQuickBooks.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$tooltipQB = New-Object System.Windows.Forms.ToolTip
$tooltipQB.SetToolTip($btnQuickBooks, "QuickBooks updates, payroll, and backup settings")
$grpSettings.Controls.Add($btnQuickBooks)

$btnQuickBooks.Add_Click({
    Write-GuiLog "Opening QuickBooks Settings dialog..."
    try {
        if (Get-Command Show-QuickBooksSettingsDialog -ErrorAction SilentlyContinue) {
            Show-QuickBooksSettingsDialog
        } else {
            [System.Windows.Forms.MessageBox]::Show("QuickBooks module not loaded.`nCheck that QuickBooks-Management.ps1 exists in the Modules folder.", "Module Not Found", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
        }
    } catch {
        Write-GuiLog "QuickBooks dialog error: $($_.Exception.Message)"
        [System.Windows.Forms.MessageBox]::Show("Error: $($_.Exception.Message)", "QuickBooks Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
})

# Windows Update button
$btnWindowsUpdate = New-Object System.Windows.Forms.Button
$btnWindowsUpdate.Text = "Win Updates..."
$btnWindowsUpdate.Location = New-Object System.Drawing.Point(475, 242)
$btnWindowsUpdate.Size = New-Object System.Drawing.Size(90, 24)
$btnWindowsUpdate.BackColor = [System.Drawing.Color]::FromArgb(0, 120, 215)
$btnWindowsUpdate.ForeColor = [System.Drawing.Color]::Black
$btnWindowsUpdate.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$tooltipWU = New-Object System.Windows.Forms.ToolTip
$tooltipWU.SetToolTip($btnWindowsUpdate, "Schedule Windows Update checks and installations")
$grpSettings.Controls.Add($btnWindowsUpdate)

$btnWindowsUpdate.Add_Click({
    Write-GuiLog "Opening Windows Update Settings dialog..."
    try {
        if (Get-Command Show-WindowsUpdateDialog -ErrorAction SilentlyContinue) {
            Show-WindowsUpdateDialog
        } else {
            [System.Windows.Forms.MessageBox]::Show("Windows Update module not loaded.`nCheck that WindowsUpdate-Management.ps1 exists in the Modules folder.", "Module Not Found", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
        }
    } catch {
        Write-GuiLog "Windows Update dialog error: $($_.Exception.Message)"
        [System.Windows.Forms.MessageBox]::Show("Error: $($_.Exception.Message)", "Windows Update Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
})

# License Status button
# REMOVED: $btnLicense = New-Object System.Windows.Forms.Button
# REMOVED: $btnLicense.Text = "License"
# REMOVED: $btnLicense.Location = New-Object System.Drawing.Point(570, 242)
# REMOVED: $btnLicense.Size = New-Object System.Drawing.Size(60, 24)
# REMOVED: $btnLicense.BackColor = [System.Drawing.Color]::Gold
# REMOVED: $btnLicense.ForeColor = [System.Drawing.Color]::Black
# REMOVED: $btnLicense.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
# REMOVED: $tooltipLic = New-Object System.Windows.Forms.ToolTip
# REMOVED: $tooltipLic.SetToolTip($btnLicense, "View license status and subscription details")
# REMOVED: $grpSettings.Controls.Add($btnLicense)
# REMOVED: 
# REMOVED: $btnLicense.Add_Click({
# REMOVED:     if (Get-Command Show-LicenseStatusDialog -ErrorAction SilentlyContinue) {
# REMOVED:         Show-LicenseStatusDialog
# REMOVED:     } else {
# REMOVED:         [System.Windows.Forms.MessageBox]::Show("License Status`n`nTier: Trial`nStatus: Active`n`nFor full license management, ensure Authentication module is loaded.", "License Info", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
# REMOVED:     }
# REMOVED: })
# REMOVED: #endregion

#region Progress
# Google Drive Login button
$btnGoogleDrive = New-Object System.Windows.Forms.Button
$btnGoogleDrive.Text = "G-Drive..."
$btnGoogleDrive.Location = New-Object System.Drawing.Point(570, 242)
$btnGoogleDrive.Size = New-Object System.Drawing.Size(75, 24)
$btnGoogleDrive.BackColor = [System.Drawing.Color]::FromArgb(66, 133, 244)
$btnGoogleDrive.ForeColor = [System.Drawing.Color]::Black
$btnGoogleDrive.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
$tooltipGD = New-Object System.Windows.Forms.ToolTip
$tooltipGD.SetToolTip($btnGoogleDrive, "Google Drive login for cloud migration - SOURCE pushes, DESTINATION pulls")
$grpSettings.Controls.Add($btnGoogleDrive)

$btnGoogleDrive.Add_Click({
    Write-GuiLog "Opening Google Drive Settings..."
    $mode = if ($rbSource.Checked) { "SOURCE" } else { "DESTINATION" }
    try {
        Show-GoogleDriveDialog -Mode $mode
    } catch {
        Write-GuiLog "Google Drive dialog error: $($_.Exception.Message)"
        [System.Windows.Forms.MessageBox]::Show("Error opening Google Drive settings:`n`n$($_.Exception.Message)", "Google Drive Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
})
$grpProgress = New-Object System.Windows.Forms.GroupBox
$grpProgress.Text = "Progress and Log"
$grpProgress.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
$grpProgress.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
$grpProgress.Location = New-Object System.Drawing.Point(10, 410)
$grpProgress.Size = New-Object System.Drawing.Size(1320, 215)
$form.Controls.Add($grpProgress)

# Overall Progress
$lblOverallProgress = New-Object System.Windows.Forms.Label
$lblOverallProgress.Text = "Overall Progress:"
$lblOverallProgress.Location = New-Object System.Drawing.Point(20, 25)
$lblOverallProgress.Size = New-Object System.Drawing.Size(120, 20)
$grpProgress.Controls.Add($lblOverallProgress)

$progressOverall = New-Object System.Windows.Forms.ProgressBar
$progressOverall.Location = New-Object System.Drawing.Point(130, 20)
$progressOverall.Size = New-Object System.Drawing.Size(1020, 20)
$grpProgress.Controls.Add($progressOverall)

$lblOverallPercent = New-Object System.Windows.Forms.Label
$lblOverallPercent.Text = "0%"
$lblOverallPercent.Location = New-Object System.Drawing.Point(1160, 22)
$lblOverallPercent.Size = New-Object System.Drawing.Size(50, 18)
$grpProgress.Controls.Add($lblOverallPercent)

# Current Step
$lblCurrentProgress = New-Object System.Windows.Forms.Label
$lblCurrentProgress.Text = "Current Step:"
$lblCurrentProgress.Location = New-Object System.Drawing.Point(20, 55)
$lblCurrentProgress.Size = New-Object System.Drawing.Size(120, 20)
$grpProgress.Controls.Add($lblCurrentProgress)

$progressCurrent = New-Object System.Windows.Forms.ProgressBar
$progressCurrent.Location = New-Object System.Drawing.Point(130, 45)
$progressCurrent.Size = New-Object System.Drawing.Size(1020, 20)
$grpProgress.Controls.Add($progressCurrent)

$lblCurrentPercent = New-Object System.Windows.Forms.Label
$lblCurrentPercent.Text = "0%"
$lblCurrentPercent.Location = New-Object System.Drawing.Point(1160, 47)
$lblCurrentPercent.Size = New-Object System.Drawing.Size(50, 18)
$grpProgress.Controls.Add($lblCurrentPercent)

# Log
$txtLog = New-Object System.Windows.Forms.TextBox
$txtLog.Location = New-Object System.Drawing.Point(15, 70)
$txtLog.Size = New-Object System.Drawing.Size(1200, 135)
$txtLog.Multiline = $true
$txtLog.ScrollBars = "Vertical"
$txtLog.ReadOnly = $true
$txtLog.BackColor = [System.Drawing.Color]::Black
$txtLog.ForeColor = [System.Drawing.Color]::Lime
$txtLog.Font = New-Object System.Drawing.Font("Consolas", 9)
$grpProgress.Controls.Add($txtLog)
#endregion

#region Action Buttons
$btnStart = New-Object System.Windows.Forms.Button
$btnStart.Text = "RUN"
$btnStart.Location = New-Object System.Drawing.Point(450, 633)
$btnStart.Size = New-Object System.Drawing.Size(200, 35)
$btnStart.BackColor = [System.Drawing.Color]::LightGreen
$btnStart.ForeColor = [System.Drawing.Color]::Black
$btnStart.Font = New-Object System.Drawing.Font("Arial", 11, [System.Drawing.FontStyle]::Bold)
$form.Controls.Add($btnStart)

$btnViewLog = New-Object System.Windows.Forms.Button
$btnViewLog.Text = "View Full Log"
$btnViewLog.Location = New-Object System.Drawing.Point(670, 618)
$btnViewLog.Size = New-Object System.Drawing.Size(120, 35)
$btnViewLog.BackColor = [System.Drawing.Color]::LightGray
$btnViewLog.ForeColor = [System.Drawing.Color]::Black
$form.Controls.Add($btnViewLog)

$btnClose = New-Object System.Windows.Forms.Button
$btnClose.Text = "Close"
$btnClose.Location = New-Object System.Drawing.Point(810, 633)
$btnClose.Size = New-Object System.Drawing.Size(120, 35)
$btnClose.BackColor = [System.Drawing.Color]::LightCoral
$btnClose.ForeColor = [System.Drawing.Color]::Black
$form.Controls.Add($btnClose)
#endregion

#region Helper Functions
function Write-GuiLog {
    param([string]$Message)
    $timestamp = Get-Date -Format "HH:mm:ss"
    $txtLog.AppendText("[$timestamp] $Message`r`n")
    $txtLog.SelectionStart = $txtLog.Text.Length
    $txtLog.ScrollToCaret()
    if ($Script:LogFile) {
        Write-LogMessage -Message $Message -Level Info
    }
}

function Update-ProgressBars {
    param([int]$Overall, [int]$Current)
    $progressOverall.Value = [Math]::Min($Overall, 100)
    $progressCurrent.Value = [Math]::Min($Current, 100)
    $lblOverallPercent.Text = "$Overall%"
    $lblCurrentPercent.Text = "$Current%"
    $form.Refresh()
}

# Initialize selected apps - all safe apps selected by default
$Script:SelectedApps = @{}
if ($Script:CommonApps -and $Script:CommonApps.Count -gt 0) {
    foreach ($app in $Script:CommonApps) {
        # Select by default if Warning level is 0 or 1 (safe or caution)
        $Script:SelectedApps[$app.Name] = ($app.Warning -le 1)
    }
    Write-GuiLog "Loaded $($Script:CommonApps.Count) apps from configuration"
} else {
    Write-GuiLog "WARNING: CommonApps not loaded from module - app selection will be empty"
}

# Initialize selected users - all users selected by default
$Script:SelectedUsers = @{}
$Script:AllUsersSelected = $true  # Flag for "All Users" mode

function Get-MigratableUsers {
    $excludeUsers = @("Public", "Default", "Default User", "All Users", "TEMP")
    $users = Get-ChildItem "C:\Users" -Directory -ErrorAction SilentlyContinue | Where-Object {
        $_.Name -notin $excludeUsers -and 
        (Test-Path (Join-Path $_.FullName "AppData"))
    }
    return $users
}

function Show-EULADialog {
    <#
    .SYNOPSIS
        Displays the End User License Agreement dialog
    #>
    
    $eulaForm = New-Object System.Windows.Forms.Form
    $eulaForm.Text = "End User License Agreement - ABC&D Systems"
    $eulaForm.Size = New-Object System.Drawing.Size(700, 600)
    $eulaForm.StartPosition = "CenterParent"
    $eulaForm.FormBorderStyle = "FixedDialog"
    $eulaForm.MaximizeBox = $false
    $eulaForm.MinimizeBox = $false
    $eulaForm.BackColor = [System.Drawing.Color]::White
    
    # Header
    $lblHeader = New-Object System.Windows.Forms.Label
    $lblHeader.Text = "ABC&D Systems Backup and Migration Tools"
    $lblHeader.Location = New-Object System.Drawing.Point(20, 15)
    $lblHeader.Size = New-Object System.Drawing.Size(650, 25)
    $lblHeader.Font = New-Object System.Drawing.Font("Segoe UI", 12, [System.Drawing.FontStyle]::Bold)
    $eulaForm.Controls.Add($lblHeader)
    
    $lblSubHeader = New-Object System.Windows.Forms.Label
    $lblSubHeader.Text = "End User License Agreement (EULA)"
    $lblSubHeader.Location = New-Object System.Drawing.Point(20, 40)
    $lblSubHeader.Size = New-Object System.Drawing.Size(650, 20)
    $lblSubHeader.Font = New-Object System.Drawing.Font("Segoe UI", 10)
    $lblSubHeader.ForeColor = [System.Drawing.Color]::Gray
    $eulaForm.Controls.Add($lblSubHeader)
    
    # EULA Text
    $txtEULA = New-Object System.Windows.Forms.RichTextBox
    $txtEULA.Location = New-Object System.Drawing.Point(20, 70)
    $txtEULA.Size = New-Object System.Drawing.Size(645, 440)
    $txtEULA.ReadOnly = $true
    $txtEULA.BackColor = [System.Drawing.Color]::FromArgb(250, 250, 250)
    $txtEULA.Font = New-Object System.Drawing.Font("Segoe UI", 9)
    $txtEULA.Text = @"
ABC&D SYSTEMS BACKUP AND MIGRATION TOOLS
END USER LICENSE AGREEMENT

Last Updated: January 2025

IMPORTANT: PLEASE READ THIS AGREEMENT CAREFULLY BEFORE USING THIS SOFTWARE.

1. LICENSE GRANT
ABC&D Systems grants you a limited, non-exclusive, non-transferable license to use this software ("Software") subject to the terms of this Agreement and your valid subscription.

2. RESTRICTIONS
You may NOT:
- Reverse engineer, decompile, or disassemble the Software
- Copy, modify, or distribute the Software
- Remove or alter any proprietary notices
- Use the Software for competitive analysis
- Share your license with unauthorized users
- Use the Software after your subscription expires

3. PROPRIETARY RIGHTS
The Software contains trade secrets and proprietary information owned by ABC&D Systems. All rights not expressly granted are reserved.

4. SUBSCRIPTION AND VALIDATION
- This Software requires a valid subscription
- The Software may validate your license periodically
- Features may be limited based on subscription tier
- Subscriptions are billed monthly or annually

5. THIRD-PARTY COMPONENTS
This Software includes third-party tools under their respective licenses:

Microsoft Sysinternals Tools (Disk2VHD, Autologon)
- Copyright (c) Microsoft Corporation
- Licensed under Sysinternals Software License Terms
- https://learn.microsoft.com/en-us/sysinternals/license-terms

LaZagne (Optional Password Recovery)
- Copyright (c) Alessandro Zanni
- Licensed under LGPL v3.0
- https://github.com/AlessandroZ/LaZagne

6. DISCLAIMER OF WARRANTIES
THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ABC&D SYSTEMS DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

7. LIMITATION OF LIABILITY
IN NO EVENT SHALL ABC&D SYSTEMS BE LIABLE FOR ANY INDIRECT, INCIDENTAL, SPECIAL, OR CONSEQUENTIAL DAMAGES ARISING FROM USE OF THE SOFTWARE.

8. TERMINATION
This license terminates automatically if you:
- Fail to comply with any term of this Agreement
- Allow your subscription to expire
- Attempt to circumvent license validation

Upon termination, you must destroy all copies of the Software.

9. DATA HANDLING
- The Software processes system data locally
- Migration data is transferred via secure protocols
- No data is sent to ABC&D Systems except for license validation
- You are responsible for compliance with data protection laws

10. GOVERNING LAW
This Agreement shall be governed by the laws of the United States.

11. ENTIRE AGREEMENT
This Agreement constitutes the entire agreement between you and ABC&D Systems regarding the Software.

By using this Software, you acknowledge that you have read, understood, and agree to be bound by this Agreement.

Copyright (c) 2024-2025 ABC&D Systems. All Rights Reserved.
"@
    $eulaForm.Controls.Add($txtEULA)
    
    # Close button
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(560, 520)
    $btnClose.Size = New-Object System.Drawing.Size(100, 30)
    $btnClose.BackColor = [System.Drawing.Color]::LightGray
    $btnClose.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnClose.Add_Click({ $eulaForm.Close() })
    $eulaForm.Controls.Add($btnClose)
    
    $eulaForm.ShowDialog() | Out-Null
}

function Show-UserSelectionDialog {
    $dialogForm = New-Object System.Windows.Forms.Form
    $dialogForm.Text = "Select Users for App Settings Migration"
    $dialogForm.Size = New-Object System.Drawing.Size(450, 400)
    $dialogForm.StartPosition = "CenterParent"
    $dialogForm.FormBorderStyle = "FixedDialog"
    $dialogForm.MaximizeBox = $false
    $dialogForm.MinimizeBox = $false
    
    # Info label
    $lblInfo = New-Object System.Windows.Forms.Label
    $lblInfo.Text = "Select which users' application settings to migrate:"
    $lblInfo.Location = New-Object System.Drawing.Point(10, 10)
    $lblInfo.Size = New-Object System.Drawing.Size(420, 20)
    $dialogForm.Controls.Add($lblInfo)
    
    # All Users checkbox
    $chkAllUsers = New-Object System.Windows.Forms.CheckBox
    $chkAllUsers.Text = "All Users (migrate settings for every user profile)"
    $chkAllUsers.Location = New-Object System.Drawing.Point(15, 35)
    $chkAllUsers.Size = New-Object System.Drawing.Size(400, 20)
    $chkAllUsers.Checked = $Script:AllUsersSelected
    $chkAllUsers.Font = New-Object System.Drawing.Font("Arial", 9, [System.Drawing.FontStyle]::Bold)
    $dialogForm.Controls.Add($chkAllUsers)
    
    # Separator
    $lblSep = New-Object System.Windows.Forms.Label
    $lblSep.Text = "--- Or select specific users ---"
    $lblSep.Location = New-Object System.Drawing.Point(15, 60)
    $lblSep.Size = New-Object System.Drawing.Size(400, 20)
    $lblSep.ForeColor = [System.Drawing.Color]::Gray
    $dialogForm.Controls.Add($lblSep)
    
    # Scrollable panel for user checkboxes
    $pnlUsers = New-Object System.Windows.Forms.Panel
    $pnlUsers.Location = New-Object System.Drawing.Point(10, 85)
    $pnlUsers.Size = New-Object System.Drawing.Size(415, 220)
    $pnlUsers.AutoScroll = $true
    $pnlUsers.BorderStyle = [System.Windows.Forms.BorderStyle]::FixedSingle
    $dialogForm.Controls.Add($pnlUsers)
    
    # Get users and create checkboxes
    $users = Get-MigratableUsers
    $userCheckboxes = @{}
    $yPos = 5
    
    foreach ($user in $users) {
        $chk = New-Object System.Windows.Forms.CheckBox
        $chk.Text = $user.Name
        $chk.Location = New-Object System.Drawing.Point(10, $yPos)
        $chk.Size = New-Object System.Drawing.Size(380, 22)
        $chk.Tag = $user.Name
        
        # Check if user was previously selected
        if ($Script:SelectedUsers.ContainsKey($user.Name)) {
            $chk.Checked = $Script:SelectedUsers[$user.Name]
        } else {
            $chk.Checked = $true  # Default to selected
        }
        
        # Add profile size info
        $profilePath = $user.FullName
        $appDataPath = Join-Path $profilePath "AppData"
        if (Test-Path $appDataPath) {
            $chk.Text = "$($user.Name)"
        }
        
        # Disable individual checkboxes when "All Users" is checked
        $chk.Enabled = -not $chkAllUsers.Checked
        
        $pnlUsers.Controls.Add($chk)
        $userCheckboxes[$user.Name] = $chk
        $yPos += 25
    }
    
    # Update count label
    $lblCount = New-Object System.Windows.Forms.Label
    $lblCount.Location = New-Object System.Drawing.Point(15, 310)
    $lblCount.Size = New-Object System.Drawing.Size(200, 20)
    $dialogForm.Controls.Add($lblCount)
    
    $updateCount = {
        if ($chkAllUsers.Checked) {
            $lblCount.Text = "Mode: All Users ($($users.Count) profiles)"
        } else {
            $selected = ($userCheckboxes.Values | Where-Object { $_.Checked }).Count
            $lblCount.Text = "Selected: $selected of $($users.Count) users"
        }
    }
    
    # Toggle individual checkboxes when All Users changes
    $chkAllUsers.Add_CheckedChanged({
        foreach ($chk in $userCheckboxes.Values) {
            $chk.Enabled = -not $chkAllUsers.Checked
            if ($chkAllUsers.Checked) {
                $chk.Checked = $true
            }
        }
        & $updateCount
    })
    
    # Update count when individual checkbox changes
    foreach ($chk in $userCheckboxes.Values) {
        $chk.Add_CheckedChanged($updateCount)
    }
    
    & $updateCount
    
    # Buttons
    $btnOK = New-Object System.Windows.Forms.Button
    $btnOK.Text = "OK"
    $btnOK.Location = New-Object System.Drawing.Point(260, 330)
    $btnOK.Size = New-Object System.Drawing.Size(75, 25)
    $btnOK.DialogResult = [System.Windows.Forms.DialogResult]::OK
    $dialogForm.Controls.Add($btnOK)
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = New-Object System.Drawing.Point(345, 330)
    $btnCancel.Size = New-Object System.Drawing.Size(75, 25)
    $btnCancel.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
    $dialogForm.Controls.Add($btnCancel)
    
    $dialogForm.AcceptButton = $btnOK
    $dialogForm.CancelButton = $btnCancel
    
    $result = $dialogForm.ShowDialog()
    
    if ($result -eq [System.Windows.Forms.DialogResult]::OK) {
        $Script:AllUsersSelected = $chkAllUsers.Checked
        $Script:SelectedUsers = @{}
        
        foreach ($userName in $userCheckboxes.Keys) {
            $Script:SelectedUsers[$userName] = $userCheckboxes[$userName].Checked
        }
        
        if ($Script:AllUsersSelected) {
            return @{Mode="All"; Count=$users.Count}
        } else {
            $selectedCount = ($Script:SelectedUsers.Values | Where-Object { $_ }).Count
            return @{Mode="Selected"; Count=$selectedCount}
        }
    }
    
    return $null
}

function Show-AppSelectionDialog {
    # Check if CommonApps is loaded
    if (-not $Script:CommonApps -or $Script:CommonApps.Count -eq 0) {
        [System.Windows.Forms.MessageBox]::Show(
            "Application list not loaded.`n`nThe AppSettings-Migration module may not have loaded correctly.",
            "No Apps Available",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Warning
        )
        return -1
    }
    
    $dialogForm = New-Object System.Windows.Forms.Form
    $dialogForm.Text = "Select Applications to Migrate"
    $dialogForm.Size = New-Object System.Drawing.Size(750, 650)
    $dialogForm.StartPosition = "CenterParent"
    $dialogForm.FormBorderStyle = "FixedDialog"
    $dialogForm.MaximizeBox = $false
    $dialogForm.MinimizeBox = $false
    
    # Warning legend at top
    $lblLegend = New-Object System.Windows.Forms.Label
    $lblLegend.Text = "Legend:  [GREEN] Safe  |  [YELLOW] Caution - may need reconfiguration  |  [ORANGE] Risky - likely issues  |  [RED] Excluded - DO NOT migrate"
    $lblLegend.Location = New-Object System.Drawing.Point(10, 10)
    $lblLegend.Size = New-Object System.Drawing.Size(720, 20)
    $lblLegend.Font = New-Object System.Drawing.Font("Arial", 8)
    $dialogForm.Controls.Add($lblLegend)
    
    # Button panel at top
    $pnlButtons = New-Object System.Windows.Forms.Panel
    $pnlButtons.Location = New-Object System.Drawing.Point(10, 35)
    $pnlButtons.Size = New-Object System.Drawing.Size(720, 35)
    $dialogForm.Controls.Add($pnlButtons)
    
    $btnSelectAll = New-Object System.Windows.Forms.Button
    $btnSelectAll.Text = "Select All Safe"
    $btnSelectAll.Location = New-Object System.Drawing.Point(0, 5)
    $btnSelectAll.Size = New-Object System.Drawing.Size(100, 25)
    $pnlButtons.Controls.Add($btnSelectAll)
    
    $btnSelectNone = New-Object System.Windows.Forms.Button
    $btnSelectNone.Text = "Select None"
    $btnSelectNone.Location = New-Object System.Drawing.Point(110, 5)
    $btnSelectNone.Size = New-Object System.Drawing.Size(100, 25)
    $pnlButtons.Controls.Add($btnSelectNone)
    
    $btnSelectAllRisky = New-Object System.Windows.Forms.Button
    $btnSelectAllRisky.Text = "Include Risky"
    $btnSelectAllRisky.Location = New-Object System.Drawing.Point(220, 5)
    $btnSelectAllRisky.Size = New-Object System.Drawing.Size(100, 25)
    $btnSelectAllRisky.BackColor = [System.Drawing.Color]::LightSalmon
    $btnSelectAllRisky.ForeColor = [System.Drawing.Color]::Black
    $pnlButtons.Controls.Add($btnSelectAllRisky)
    
    $lblCount = New-Object System.Windows.Forms.Label
    $lblCount.Text = "Selected: 0"
    $lblCount.Location = New-Object System.Drawing.Point(550, 8)
    $lblCount.Size = New-Object System.Drawing.Size(150, 20)
    $lblCount.Font = New-Object System.Drawing.Font("Arial", 9, [System.Drawing.FontStyle]::Bold)
    $pnlButtons.Controls.Add($lblCount)
    
    # Create scrollable panel for checkboxes
    $pnlScroll = New-Object System.Windows.Forms.Panel
    $pnlScroll.Location = New-Object System.Drawing.Point(10, 75)
    $pnlScroll.Size = New-Object System.Drawing.Size(715, 480)
    $pnlScroll.AutoScroll = $true
    $pnlScroll.BorderStyle = [System.Windows.Forms.BorderStyle]::FixedSingle
    $dialogForm.Controls.Add($pnlScroll)
    
    # Group apps by category
    $categories = $Script:CommonApps | Group-Object -Property Category | Sort-Object Name
    
    $checkboxes = @{}
    $yPos = 5
    
    foreach ($category in $categories) {
        # Category header
        $lblCat = New-Object System.Windows.Forms.Label
        $lblCat.Text = $category.Name
        $lblCat.Location = New-Object System.Drawing.Point(5, $yPos)
        $lblCat.Size = New-Object System.Drawing.Size(680, 18)
        $lblCat.Font = New-Object System.Drawing.Font("Arial", 9, [System.Drawing.FontStyle]::Bold)
        $lblCat.BackColor = [System.Drawing.Color]::LightGray
        $pnlScroll.Controls.Add($lblCat)
        $yPos += 22
        
        # Apps in category
        foreach ($app in ($category.Group | Sort-Object Name)) {
            $chk = New-Object System.Windows.Forms.CheckBox
            $chk.Text = "$($app.Name)"
            $chk.Location = New-Object System.Drawing.Point(15, $yPos)
            $chk.Size = New-Object System.Drawing.Size(200, 20)
            $chk.Tag = $app
            
            # Set color based on warning level
            switch ($app.Warning) {
                0 { $chk.ForeColor = [System.Drawing.Color]::DarkGreen }
                1 { $chk.ForeColor = [System.Drawing.Color]::DarkGoldenrod }
                2 { $chk.ForeColor = [System.Drawing.Color]::OrangeRed }
                3 { 
                    $chk.ForeColor = [System.Drawing.Color]::Red
                    $chk.Font = New-Object System.Drawing.Font("Arial", 8, [System.Drawing.FontStyle]::Strikeout)
                }
            }
            
            # Set checked state from saved selection
            if ($Script:SelectedApps.ContainsKey($app.Name)) {
                $chk.Checked = $Script:SelectedApps[$app.Name]
            } else {
                $chk.Checked = ($app.Warning -le 1)
            }
            
            # Disable excluded apps
            if ($app.Warning -eq 3) {
                $chk.Enabled = $false
                $chk.Checked = $false
            }
            
            $pnlScroll.Controls.Add($chk)
            $checkboxes[$app.Name] = $chk
            
            # Note label
            $lblNote = New-Object System.Windows.Forms.Label
            $lblNote.Text = $app.Note
            $lblNote.Location = New-Object System.Drawing.Point(220, $yPos)
            $lblNote.Size = New-Object System.Drawing.Size(470, 18)
            $lblNote.Font = New-Object System.Drawing.Font("Arial", 8)
            $lblNote.ForeColor = [System.Drawing.Color]::Gray
            $pnlScroll.Controls.Add($lblNote)
            
            $yPos += 22
        }
        $yPos += 5
    }
    
    # Update count function
    $updateCount = {
        $count = ($checkboxes.Values | Where-Object { $_.Checked }).Count
        $lblCount.Text = "Selected: $count / $($checkboxes.Count)"
    }
    
    # Add change event to all checkboxes
    foreach ($chk in $checkboxes.Values) {
        $chk.Add_CheckedChanged($updateCount)
    }
    
    # Initial count
    & $updateCount
    
    # Button actions
    $btnSelectAll.Add_Click({
        foreach ($chk in $checkboxes.Values) {
            if ($chk.Enabled -and $chk.Tag.Warning -le 1) {
                $chk.Checked = $true
            }
        }
    })
    
    $btnSelectNone.Add_Click({
        foreach ($chk in $checkboxes.Values) {
            $chk.Checked = $false
        }
    })
    
    $btnSelectAllRisky.Add_Click({
        $confirm = [System.Windows.Forms.MessageBox]::Show(
            "Include RISKY apps?`n`nThese apps may cause errors or require reactivation:`n- Adobe products (license issues)`n- Microsoft Office (reactivation needed)`n- Docker (VM state issues)`n- QuickBooks/Sage (license bound)`n`nContinue?",
            "Include Risky Apps",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Warning
        )
        if ($confirm -eq [System.Windows.Forms.DialogResult]::Yes) {
            foreach ($chk in $checkboxes.Values) {
                if ($chk.Enabled) {
                    $chk.Checked = $true
                }
            }
        }
    })
    
    # OK/Cancel buttons
    $btnOK = New-Object System.Windows.Forms.Button
    $btnOK.Text = "OK"
    $btnOK.Location = New-Object System.Drawing.Point(550, 565)
    $btnOK.Size = New-Object System.Drawing.Size(80, 30)
    $btnOK.DialogResult = [System.Windows.Forms.DialogResult]::OK
    $dialogForm.Controls.Add($btnOK)
    $dialogForm.AcceptButton = $btnOK
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = New-Object System.Drawing.Point(640, 565)
    $btnCancel.Size = New-Object System.Drawing.Size(80, 30)
    $btnCancel.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
    $dialogForm.Controls.Add($btnCancel)
    $dialogForm.CancelButton = $btnCancel
    
    # Show dialog
    $result = $dialogForm.ShowDialog()
    
    if ($result -eq [System.Windows.Forms.DialogResult]::OK) {
        # Save selections
        foreach ($appName in $checkboxes.Keys) {
            $Script:SelectedApps[$appName] = $checkboxes[$appName].Checked
        }
        
        $selectedCount = ($Script:SelectedApps.Values | Where-Object { $_ }).Count
        return $selectedCount
    }
    
    return -1  # Cancelled
}
#endregion

#region Event Handlers

# Configure Apps button
$btnConfigureApps.Add_Click({
    $result = Show-AppSelectionDialog
    if ($result -ge 0) {
        Write-GuiLog "App selection updated: $result apps selected"
        # Update summary label
        if ($Script:AllUsersSelected) {
            $lblAppUserSummary.Text = "($result apps, all users)"
        } else {
            $userCount = ($Script:SelectedUsers.Values | Where-Object { $_ }).Count
            $lblAppUserSummary.Text = "($result apps, $userCount users)"
        }
    }
})

# Configure Users button
$btnConfigureUsers.Add_Click({
    $result = Show-UserSelectionDialog
    if ($result) {
        if ($result.Mode -eq "All") {
            Write-GuiLog "User selection: All users ($($result.Count) profiles)"
            $appCount = ($Script:SelectedApps.Values | Where-Object { $_ }).Count
            $lblAppUserSummary.Text = "($appCount apps, all users)"
        } else {
            Write-GuiLog "User selection: $($result.Count) users selected"
            $appCount = ($Script:SelectedApps.Values | Where-Object { $_ }).Count
            $lblAppUserSummary.Text = "($appCount apps, $($result.Count) users)"
        }
    }
})

# Wire up User Data Users button to same handler
$btnUserDataUsers.Add_Click({
    $result = Show-UserSelectionDialog
    if ($result) {
        if ($result.Mode -eq "All") {
            Write-GuiLog "User selection: All users ($($result.Count) profiles)"
        } else {
            Write-GuiLog "User selection: $($result.Count) users selected"
        }
    }
})

# Mode selection
$rbSource.Add_CheckedChanged({
    if ($rbSource.Checked) {
        # Hide Add Source tab in SOURCE mode (only for DESTINATION)
        if ($tabSources.TabPages.Contains($tabAddSource)) {
            $tabSources.TabPages.Remove($tabAddSource)
        }
        $chkCreateProfiles.Enabled = $false
        $chkCreateProfiles.Checked = $false
        $chkCreateShares.Enabled = $true
        $chkLocal.Enabled = $true
        
        # Hide master password controls in SOURCE mode (not needed)
        $lblMasterPassword.Visible = $false
        $txtMasterPassword.Visible = $false
        $btnShowMaster.Visible = $false
        $btnGenerate.Visible = $false
        $chkPasswordMonitoring.Visible = $false
        $lblMonitoringInfo.Visible = $false
        
        # Show recovery tools (SOURCE only)
        $lblRecoveryTools.Visible = $true
        $btnKeyFinder.Visible = $true
        $btnLaZagne.Visible = $true
        $btnViewReport.Visible = $true
        
        $grpConnection.Text = "Connection Settings (LOCAL - This Computer)"
        # Using Local tab for SOURCE mode
        $tabSources.SelectedIndex = 0  # Select Local tab
        $txtComputer.Text = $env:COMPUTERNAME
        $chkLocal.Checked = $true
        # Hide Browse button in SOURCE mode
        foreach ($tp in $tabSources.TabPages) {
            $browse = $tp.Controls["btnBrowseSource"]
            if ($browse) { $browse.Visible = $false }
        }
        
        # Load saved SOURCE settings
        try {
            $savedSettings = Load-MigrationSettings -Mode "SOURCE"
            if ($savedSettings) {
                if ($null -ne $savedSettings.Users) { $chkUsers.Checked = $savedSettings.Users }
                if ($null -ne $savedSettings.Shares) { $chkShares.Checked = $savedSettings.Shares }
                if ($null -ne $savedSettings.Printers) { $chkPrinters.Checked = $savedSettings.Printers }
                if ($null -ne $savedSettings.UserData) { $chkUserData.Checked = $savedSettings.UserData }
                if ($null -ne $savedSettings.ProgramFiles) { $chkProgramFiles.Checked = $savedSettings.ProgramFiles }
                if ($null -ne $savedSettings.ProgramFilesX86) { $chkProgramFilesX86.Checked = $savedSettings.ProgramFilesX86 }
                if ($null -ne $savedSettings.ProgramData) { $chkProgramData.Checked = $savedSettings.ProgramData }
                if ($null -ne $savedSettings.Registry) { $chkRegistry.Checked = $savedSettings.Registry }
                if ($null -ne $savedSettings.AppSettings) { $chkAppSettings.Checked = $savedSettings.AppSettings }
                if ($null -ne $savedSettings.CreateShares) { $chkCreateShares.Checked = $savedSettings.CreateShares }
                if ($null -ne $savedSettings.Overwrite) { $chkOverwrite.Checked = $savedSettings.Overwrite }
                if ($null -ne $savedSettings.NewerOnly) { $chkNewerOnly.Checked = $savedSettings.NewerOnly; $chkNewerOnly.Enabled = $chkOverwrite.Checked }
                if ($null -ne $savedSettings.Threads) { $numThreads.Value = $savedSettings.Threads }
                
                # Load additional drives
                if ($savedSettings.AdditionalDrives) {
                    foreach ($drive in $savedSettings.AdditionalDrives) {
                        if ($Script:DriveCheckboxes.ContainsKey($drive)) {
                            $Script:DriveCheckboxes[$drive].Checked = $true
                        }
                    }
                }
                
                # Load selected apps
                if ($savedSettings.SelectedApps) {
                    $Script:SelectedApps = @{}
                    foreach ($key in $savedSettings.SelectedApps.PSObject.Properties.Name) {
                        $Script:SelectedApps[$key] = $savedSettings.SelectedApps.$key
                    }
                }
                
                # Load selected users
                if ($null -ne $savedSettings.AllUsersSelected) {
                    $Script:AllUsersSelected = $savedSettings.AllUsersSelected
                }
                if ($savedSettings.SelectedUsers) {
                    $Script:SelectedUsers = @{}
                    foreach ($key in $savedSettings.SelectedUsers.PSObject.Properties.Name) {
                        $Script:SelectedUsers[$key] = $savedSettings.SelectedUsers.$key
                    }
                }
                
                # Update summary label
                $selectedAppCount = ($Script:SelectedApps.Values | Where-Object { $_ }).Count
                if ($Script:AllUsersSelected) {
                    $lblAppUserSummary.Text = "($selectedAppCount apps, all users)"
                } else {
                    $selectedUserCount = ($Script:SelectedUsers.Values | Where-Object { $_ }).Count
                    $lblAppUserSummary.Text = "($selectedAppCount apps, $selectedUserCount users)"
                }
                
                Write-GuiLog "Loaded saved SOURCE settings"
            }
        } catch {
            Write-GuiLog "Note: Could not load SOURCE settings"
        }
        
        Update-ScheduleStatusUI
        Write-GuiLog "Mode: SOURCE selected"
    }
})

$rbDestination.Add_CheckedChanged({
    if ($rbDestination.Checked) {
        # Show Add Source tab in DESTINATION mode
        if (-not $tabSources.TabPages.Contains($tabAddSource)) {
            if (-not $rbSource.Checked) {
        $tabSources.TabPages.Add($tabAddSource)
    }
        }
        $chkCreateProfiles.Enabled = $true
        $chkCreateProfiles.Checked = $true
        $chkCreateShares.Enabled = $false
        $chkCreateShares.Checked = $false
        
        # Show master password controls in DESTINATION mode (required for user creation)
        $lblMasterPassword.Visible = $true
        $txtMasterPassword.Visible = $true
        $btnShowMaster.Visible = $true
        $btnGenerate.Visible = $true
        $chkPasswordMonitoring.Visible = $true
        $lblMonitoringInfo.Visible = $true
        
        # Hide recovery tools (SOURCE only)
        $lblRecoveryTools.Visible = $false
        $btnKeyFinder.Visible = $false
        $btnLaZagne.Visible = $false
        $btnViewReport.Visible = $false
        
        $grpConnection.Text = "SOURCE Server Connection (Enter SOURCE computer details)"
        # In DESTINATION mode, show all source tabs
        if ($tabSources.TabPages.Count -gt 1) {
            $tabSources.SelectedIndex = 0  # Select first source tab
        }
        $txtComputer.Text = ""
        $chkLocal.Checked = $false
        $chkLocal.Enabled = $false
        $txtComputer.Enabled = $true
        $txtUsername.Enabled = $true
        $txtPassword.Enabled = $true
        $btnShowPass.Enabled = $true
        # Show Browse button for local path selection
        foreach ($tp in $tabSources.TabPages) {
            $browse = $tp.Controls["btnBrowseSource"]
            if ($browse) { $browse.Visible = $true }
        }
        $lblConnectionStatus.Text = "Status: Enter SOURCE server IP/name or local backup path (e.g. D:\ABCD-Migration\Backup)"
        $lblConnectionStatus.ForeColor = [System.Drawing.Color]::Gray
        
        # Load saved DESTINATION settings
        try {
            $savedSettings = Load-MigrationSettings -Mode "DESTINATION"
            if ($savedSettings) {
                if ($null -ne $savedSettings.Users) { $chkUsers.Checked = $savedSettings.Users }
                if ($null -ne $savedSettings.Shares) { $chkShares.Checked = $savedSettings.Shares }
                if ($null -ne $savedSettings.Printers) { $chkPrinters.Checked = $savedSettings.Printers }
                if ($null -ne $savedSettings.UserData) { $chkUserData.Checked = $savedSettings.UserData }
                if ($null -ne $savedSettings.ProgramFiles) { $chkProgramFiles.Checked = $savedSettings.ProgramFiles }
                if ($null -ne $savedSettings.ProgramFilesX86) { $chkProgramFilesX86.Checked = $savedSettings.ProgramFilesX86 }
                if ($null -ne $savedSettings.ProgramData) { $chkProgramData.Checked = $savedSettings.ProgramData }
                if ($null -ne $savedSettings.Registry) { $chkRegistry.Checked = $savedSettings.Registry }
                if ($null -ne $savedSettings.AppSettings) { $chkAppSettings.Checked = $savedSettings.AppSettings }
                if ($null -ne $savedSettings.Overwrite) { $chkOverwrite.Checked = $savedSettings.Overwrite }
                if ($null -ne $savedSettings.NewerOnly) { $chkNewerOnly.Checked = $savedSettings.NewerOnly; $chkNewerOnly.Enabled = $chkOverwrite.Checked }
                if ($null -ne $savedSettings.Threads) { $numThreads.Value = $savedSettings.Threads }
                if ($null -ne $savedSettings.CreateProfiles) { $chkCreateProfiles.Checked = $savedSettings.CreateProfiles }
                if ($null -ne $savedSettings.PasswordMonitoring) { $chkPasswordMonitoring.Checked = $savedSettings.PasswordMonitoring }
                
                # Load connection settings
                if ($savedSettings.SourceComputer) { $txtComputer.Text = $savedSettings.SourceComputer }
                if ($savedSettings.NetworkUsername) { $txtUsername.Text = $savedSettings.NetworkUsername }
                if ($savedSettings.NetworkPassword) { $txtPassword.Text = $savedSettings.NetworkPassword }
                if ($savedSettings.MasterPassword) { $txtMasterPassword.Text = $savedSettings.MasterPassword }
                
                # Load selected apps
                if ($savedSettings.SelectedApps) {
                    $Script:SelectedApps = @{}
                    foreach ($key in $savedSettings.SelectedApps.PSObject.Properties.Name) {
                        $Script:SelectedApps[$key] = $savedSettings.SelectedApps.$key
                    }
                }
                
                # Load selected users
                if ($null -ne $savedSettings.AllUsersSelected) {
                    $Script:AllUsersSelected = $savedSettings.AllUsersSelected
                }
                if ($savedSettings.SelectedUsers) {
                    $Script:SelectedUsers = @{}
                    foreach ($key in $savedSettings.SelectedUsers.PSObject.Properties.Name) {
                        $Script:SelectedUsers[$key] = $savedSettings.SelectedUsers.$key
                    }
                }
                
                # Update summary label
                $selectedAppCount = ($Script:SelectedApps.Values | Where-Object { $_ }).Count
                if ($Script:AllUsersSelected) {
                    $lblAppUserSummary.Text = "($selectedAppCount apps, all users)"
                } else {
                    $selectedUserCount = ($Script:SelectedUsers.Values | Where-Object { $_ }).Count
                    $lblAppUserSummary.Text = "($selectedAppCount apps, $selectedUserCount users)"
                }
                
                Write-GuiLog "Loaded saved DESTINATION settings"
            }
        } catch {
            Write-GuiLog "Note: Could not load DESTINATION settings"
        }
        
        Update-ScheduleStatusUI
        Write-GuiLog "Mode: DESTINATION selected"
    }
})

# Show/Hide master password button
$btnShowMaster.Add_Click({
    if ($txtMasterPassword.PasswordChar -eq [char]'*') {
        $txtMasterPassword.PasswordChar = [char]0
        $btnShowMaster.Text = "Hide"
    } else {
        $txtMasterPassword.PasswordChar = [char]'*'
        $btnShowMaster.Text = "Show"
    }
})

# Generate master password
$btnGenerate.Add_Click({
    $chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
    $special = "!@#%"
    $password = -join ((1..12) | ForEach-Object { $chars[(Get-Random -Maximum $chars.Length)] })
    $password += $special[(Get-Random -Maximum $special.Length)]
    $password += (Get-Random -Minimum 10 -Maximum 99)
    $password += "Aa"
    
    $txtMasterPassword.Text = $password
    $txtMasterPassword.PasswordChar = [char]0
    $btnShowMaster.Text = "Hide"
    
    [System.Windows.Forms.MessageBox]::Show(
        "Master Password Generated:`n`n$password`n`nPlease save this password securely!`n`nYou will need it on the DESTINATION computer.",
        "Master Password",
        [System.Windows.Forms.MessageBoxButtons]::OK,
        [System.Windows.Forms.MessageBoxIcon]::Information
    )
})

# KeyFinder button
$btnKeyFinder.Add_Click({
    if (-not $rbSource.Checked) {
        [System.Windows.Forms.MessageBox]::Show(
            "License Key Finder is only available in SOURCE mode.`n`nSwitch to SOURCE mode to scan this computer for license keys.",
            "SOURCE Mode Required",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        return
    }
    
    Write-GuiLog "Starting License Key Finder..."
    $form.Cursor = [System.Windows.Forms.Cursors]::WaitCursor
    
    try {
        $backupPath = "$Script:InstallPath\Backup"
        if (!(Test-Path $backupPath)) {
            New-Item -Path $backupPath -ItemType Directory -Force | Out-Null
        }
        
        $result = Export-AllLicenseKeys -OutputPath $backupPath -IncludeWifi
        
        Write-GuiLog "KeyFinder complete! Found $($result.Keys.Count) license keys"
        Write-GuiLog "Report saved to Desktop: License_Keys_Report.html"
        
        [System.Windows.Forms.MessageBox]::Show(
            "License Key Finder Complete!`n`nFound: $($result.Keys.Count) license keys`nWiFi Networks: $($result.WifiPasswords.Count)`n`nReport opened in browser - Print to PDF!`n`nFile saved to Desktop",
            "KeyFinder Complete",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
    }
    catch {
        Write-GuiLog "KeyFinder ERROR: $($_.Exception.Message)"
        [System.Windows.Forms.MessageBox]::Show("Error: $($_.Exception.Message)", "Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
    }
    finally {
        $form.Cursor = [System.Windows.Forms.Cursors]::Default
    }
})

# LaZagne button
$btnLaZagne.Add_Click({
    if (-not $rbSource.Checked) {
        [System.Windows.Forms.MessageBox]::Show(
            "Password Finder is only available in SOURCE mode.",
            "SOURCE Mode Required",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        return
    }
    
    $lazagnePath = "$Script:InstallPath\Tools\LaZagne.exe"
    $backupPath = "$Script:InstallPath\Backup"
    
    if (!(Test-Path $lazagnePath)) {
        # LaZagne is NOT bundled (often triggers antivirus, LGPL license)
        # User must manually download and place in Tools folder
        $result = [System.Windows.Forms.MessageBox]::Show(
            "LaZagne.exe not found in Tools folder.`n`nThis tool is NOT bundled due to antivirus false positives.`n`nTo use password recovery:`n1. Download LaZagne.exe from GitHub`n2. Place it in: $Script:InstallPath\Tools\`n`nOpen download page now?",
            "LaZagne Not Installed",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            Start-Process "https://github.com/AlessandroZ/LaZagne/releases"
        }
        return
    }
    
    Write-GuiLog "Running LaZagne password recovery..."
    $form.Cursor = [System.Windows.Forms.Cursors]::WaitCursor
    
    try {
        if (!(Test-Path $backupPath)) { New-Item -Path $backupPath -ItemType Directory -Force | Out-Null }
        
        $textFile = "$backupPath\passwords_lazagne.txt"
        Start-Process -FilePath $lazagnePath -ArgumentList "all" -Wait -NoNewWindow -RedirectStandardOutput $textFile
        
        Write-GuiLog "LaZagne complete! Results saved to $textFile"
        Start-Process "notepad.exe" -ArgumentList $textFile
        
        [System.Windows.Forms.MessageBox]::Show("Password recovery complete!`n`nResults opened in Notepad.", "Complete", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
    }
    catch {
        Write-GuiLog "LaZagne ERROR: $($_.Exception.Message)"
    }
    finally {
        $form.Cursor = [System.Windows.Forms.Cursors]::Default
    }
})

# View Report button
$btnViewReport.Add_Click({
    $backupPath = "$Script:InstallPath\Backup"
    if (Test-Path $backupPath) {
        Start-Process "explorer.exe" -ArgumentList $backupPath
    } else {
        [System.Windows.Forms.MessageBox]::Show("No reports found. Run recovery tools first.", "No Reports", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
    }
})

# Test connection button (uses currently selected source tab)
$btnTestConnection.Add_Click({
    $conn = Get-CurrentSourceConnection
    
    if ($conn.IsLocal) {
        $lblConnectionStatus.Text = "Status: Local computer - Connection OK"
        $lblConnectionStatus.ForeColor = [System.Drawing.Color]::Green
        Write-GuiLog "Connection test: Local computer OK"
    } else {
        $computer = $conn.Computer.Trim() -replace '^\\\\', '' -replace '\\.*$', ''
        $username = $conn.Username.Trim()
        $password = $conn.Password
        
        if ([string]::IsNullOrWhiteSpace($computer) -or [string]::IsNullOrWhiteSpace($username) -or [string]::IsNullOrWhiteSpace($password)) {
            [System.Windows.Forms.MessageBox]::Show("Please enter computer, username, and password.", "Validation", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        Write-GuiLog "Testing connection to $computer..."
        $lblConnectionStatus.Text = "Status: Testing..."
        $lblConnectionStatus.ForeColor = [System.Drawing.Color]::Orange
        $form.Refresh()
        
        try {
            $shareName = 'ABCD-Migration$'
            net use "\\$computer\$shareName" /delete 2>&1 | Out-Null
            $result = net use "\\$computer\$shareName" /user:$username $password 2>&1
            
            if ($LASTEXITCODE -eq 0) {
                $lblConnectionStatus.Text = "Status: Connected to $computer"
                $lblConnectionStatus.ForeColor = [System.Drawing.Color]::Green
                Write-GuiLog "Connection successful!"
            } else {
                $lblConnectionStatus.Text = "Status: Connection failed"
                $lblConnectionStatus.ForeColor = [System.Drawing.Color]::Red
                Write-GuiLog "Connection failed: $result"
            }
        }
        catch {
            $lblConnectionStatus.Text = "Status: Error - $($_.Exception.Message)"
            $lblConnectionStatus.ForeColor = [System.Drawing.Color]::Red
        }
    }
})

# View Log button
$btnViewLog.Add_Click({
    $logPath = "$Script:InstallPath\Logs"
    if (Test-Path $logPath) {
        $latestLog = Get-ChildItem $logPath -Filter "*.log" | Sort-Object LastWriteTime -Descending | Select-Object -First 1
        if ($latestLog) {
            Start-Process "notepad.exe" -ArgumentList $latestLog.FullName
        }
    }
})

# Close button
$btnClose.Add_Click({
    $form.Close()
})

# Update schedule status label
function Update-ScheduleStatusUI {
    $mode = if ($rbSource.Checked) { "SOURCE" } else { "DESTINATION" }
        # Hide Add Source tab in SOURCE mode (only for DESTINATION)
        if ($tabSources.TabPages.Contains($tabAddSource)) {
            $tabSources.TabPages.Remove($tabAddSource)
        }
    $taskInfo = Get-MigrationScheduledTask -Mode $mode
    
    if ($taskInfo.Exists) {
        $chkScheduleEnabled.Checked = $true
        
        # Parse time and set combo boxes
        if ($taskInfo.Time -match "(\d{2}):(\d{2})") {
            $hour = $Matches[1]
            $minute = $Matches[2]
            
            # Set hour combo
            $hourIndex = $cboScheduleHour.Items.IndexOf($hour)
            if ($hourIndex -ge 0) { $cboScheduleHour.SelectedIndex = $hourIndex }
            
            # Set minute combo (rounds to nearest 5)
            $minuteInt = [int]$minute
            $roundedMinute = [Math]::Round($minuteInt / 5) * 5
            if ($roundedMinute -ge 60) { $roundedMinute = 55 }
            $minuteIndex = $cboScheduleMinute.Items.IndexOf($roundedMinute.ToString("00"))
            if ($minuteIndex -ge 0) { $cboScheduleMinute.SelectedIndex = $minuteIndex }
        }
        
        $nextRun = if ($taskInfo.NextRun) { $taskInfo.NextRun.ToString("yyyy-MM-dd HH:mm") } else { "Unknown" }
        $lblScheduleStatus.Text = "Scheduled: Daily at $($taskInfo.Time) | Next: $nextRun"
        $lblScheduleStatus.ForeColor = [System.Drawing.Color]::Green
    } else {
        $chkScheduleEnabled.Checked = $false
        $lblScheduleStatus.Text = "No scheduled task configured for $mode mode"
        $lblScheduleStatus.ForeColor = [System.Drawing.Color]::Gray
    }
}

# Save Settings button
$btnSaveSettings.Add_Click({
    $mode = if ($rbSource.Checked) { "SOURCE" } else { "DESTINATION" }
        # Hide Add Source tab in SOURCE mode (only for DESTINATION)
        if ($tabSources.TabPages.Contains($tabAddSource)) {
            $tabSources.TabPages.Remove($tabAddSource)
        }
    
    # Collect current settings
    $settings = @{
        Users = $chkUsers.Checked
        Shares = $chkShares.Checked
        Printers = $chkPrinters.Checked
        UserData = $chkUserData.Checked
        ProgramFiles = $chkProgramFiles.Checked
        ProgramFilesX86 = $chkProgramFilesX86.Checked
        ProgramData = $chkProgramData.Checked
        Registry = $chkRegistry.Checked
        AppSettings = $chkAppSettings.Checked
        CreateShares = $chkCreateShares.Checked
        Overwrite = $chkOverwrite.Checked
        NewerOnly = $chkNewerOnly.Checked
        Threads = [int]$numThreads.Value
        CreateProfiles = $chkCreateProfiles.Checked
        PasswordMonitoring = $chkPasswordMonitoring.Checked
        ScheduleEnabled = $chkScheduleEnabled.Checked
        ScheduleTime = "$($cboScheduleHour.SelectedItem):$($cboScheduleMinute.SelectedItem)"
        SelectedApps = $Script:SelectedApps
        SelectedUsers = $Script:SelectedUsers
        AllUsersSelected = $Script:AllUsersSelected
    }
    
    # Mode-specific settings
    if ($mode -eq "SOURCE") {
        # Collect additional drives
        $additionalDrives = @()
        foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
            if ($Script:DriveCheckboxes[$driveLetter].Checked) {
                $additionalDrives += $driveLetter
            }
        }
        $settings.AdditionalDrives = $additionalDrives
    } else {
        # DESTINATION needs source computer, credentials, and master password (if Users selected)
        $sourceInput = $txtComputer.Text.Trim()
        if ($sourceInput -match '^[A-Za-z]:\\') {
            $settings.SourceComputer = $sourceInput
        } else {
            $settings.SourceComputer = $sourceInput -replace '^\\\\', '' -replace '\\.*$', ''
        }
        $settings.NetworkUsername = $txtUsername.Text.Trim()
        $settings.NetworkPassword = $txtPassword.Text
        $settings.MasterPassword = $txtMasterPassword.Text
        
        if ([string]::IsNullOrWhiteSpace($settings.SourceComputer)) {
            [System.Windows.Forms.MessageBox]::Show("Please enter the SOURCE computer name/IP or local backup path.", "Validation", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        # Only require master password if Users is checked
        if ($chkUsers.Checked -and [string]::IsNullOrWhiteSpace($settings.MasterPassword)) {
            [System.Windows.Forms.MessageBox]::Show("Please enter or generate a Master Password for scheduled imports.`n`n(Required because Users is selected)", "Validation", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
    }
    
    # Save settings
    Save-MigrationSettings -Mode $mode -Settings $settings
    Write-GuiLog "Settings saved for $mode mode"
    
    # Handle scheduled task
    if ($chkScheduleEnabled.Checked) {
        $scheduleTime = "$($cboScheduleHour.SelectedItem):$($cboScheduleMinute.SelectedItem)"
        $sourceComp = if ($mode -eq "DESTINATION") { $settings.SourceComputer } else { "" }
        
        $result = New-MigrationScheduledTask -Mode $mode -Time $scheduleTime -SourceComputer $sourceComp
        
        if ($result) {
            Write-GuiLog "Scheduled task created: Daily at $scheduleTime"
            [System.Windows.Forms.MessageBox]::Show(
                "Settings Saved!`n`nScheduled Task Created:`n- Mode: $mode`n- Time: $scheduleTime (daily)`n`nThe migration will run automatically.",
                "Settings Saved",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Information
            )
        } else {
            Write-GuiLog "ERROR: Could not create scheduled task"
            [System.Windows.Forms.MessageBox]::Show("Settings saved but scheduled task creation failed.", "Warning", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
        }
    } else {
        # Remove task if exists and checkbox unchecked
        Remove-MigrationScheduledTask -Mode $mode
        Write-GuiLog "Settings saved (no scheduled task)"
        [System.Windows.Forms.MessageBox]::Show("Settings Saved!`n`n(No scheduled task configured)", "Settings Saved", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
    }
    
    Update-ScheduleStatusUI
})

# Remove Schedule button
$btnRemoveSchedule.Add_Click({
    $mode = if ($rbSource.Checked) { "SOURCE" } else { "DESTINATION" }
        # Hide Add Source tab in SOURCE mode (only for DESTINATION)
        if ($tabSources.TabPages.Contains($tabAddSource)) {
            $tabSources.TabPages.Remove($tabAddSource)
        }
    
    $confirm = [System.Windows.Forms.MessageBox]::Show(
        "Remove the scheduled task for $mode mode?`n`nThis will stop automatic daily runs.",
        "Confirm Remove",
        [System.Windows.Forms.MessageBoxButtons]::YesNo,
        [System.Windows.Forms.MessageBoxIcon]::Question
    )
    
    if ($confirm -eq [System.Windows.Forms.DialogResult]::Yes) {
        $removed = Remove-MigrationScheduledTask -Mode $mode
        if ($removed) {
            Write-GuiLog "Scheduled task removed for $mode mode"
            $chkScheduleEnabled.Checked = $false
            [System.Windows.Forms.MessageBox]::Show("Scheduled task removed.", "Removed", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
        } else {
            Write-GuiLog "No scheduled task found to remove"
            [System.Windows.Forms.MessageBox]::Show("No scheduled task was configured.", "Info", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
        }
        Update-ScheduleStatusUI
    }
})

# Save Settings button - save current settings to JSON file
$btnSaveSettings.Add_Click({
    $mode = if ($rbSource.Checked) { "SOURCE" } else { "DESTINATION" }
        # Hide Add Source tab in SOURCE mode (only for DESTINATION)
        if ($tabSources.TabPages.Contains($tabAddSource)) {
            $tabSources.TabPages.Remove($tabAddSource)
        }
    
    Write-GuiLog "Saving $mode settings..."
    
    # Collect current settings
    $settings = @{
        Computer = $txtComputer.Text
        Username = $txtUsername.Text
        # Don't save password in plain text - it's handled by credential manager for scheduled tasks
        Overwrite = $chkOverwrite.Checked
        NewerOnly = $chkNewerOnly.Checked
        StopServices = $chkStopServices.Checked
        Rollback = $chkRollback.Checked
        Threads = [int]$numThreads.Value
        Users = $chkUsers.Checked
        Shares = $chkShares.Checked
        Printers = $chkPrinters.Checked
        ProgramFiles = $chkProgramFiles.Checked
        ProgramFilesX86 = $chkProgramFilesX86.Checked
        ProgramData = $chkProgramData.Checked
        UserData = $chkUserData.Checked
        AppSettings = $chkAppSettings.Checked
        AdditionalDrives = @()
    }
    
    # Collect additional drives
    foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
        if ($Script:DriveCheckboxes[$driveLetter].Checked) {
            $settings.AdditionalDrives += $driveLetter
        }
    }
    
    # Save settings
    $result = Save-MigrationSettings -Mode $mode -Settings $settings
    
    if ($result) {
        Write-GuiLog "$mode settings saved successfully"
        [System.Windows.Forms.MessageBox]::Show(
            "$mode settings saved successfully!`n`nSettings will be automatically loaded when you start the application.",
            "Settings Saved",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
    }
    else {
        Write-GuiLog "ERROR: Failed to save settings"
        [System.Windows.Forms.MessageBox]::Show(
            "Failed to save settings. Check the log for details.",
            "Save Error",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Error
        )
    }
})

# Update status when mode changes
$rbSource.Add_CheckedChanged({ Update-ScheduleStatusUI })
$rbDestination.Add_CheckedChanged({ Update-ScheduleStatusUI })

# Start Migration button
$btnStart.Add_Click({
    
    if ($rbSource.Checked) {
        # Hide Add Source tab in SOURCE mode (only for DESTINATION)
        if ($tabSources.TabPages.Contains($tabAddSource)) {
            $tabSources.TabPages.Remove($tabAddSource)
        }
        # SOURCE MODE - No master password needed
        Write-GuiLog "=========================================="
        Write-GuiLog "Starting SOURCE Export..."
        Write-GuiLog "=========================================="
        
        # Ask user for backup destination
        $needsPortable = ($chkUserData.Checked -or $chkProgramFiles.Checked -or $chkProgramFilesX86.Checked -or $chkProgramData.Checked)
        $driveChecked = $false
        foreach ($dl in $Script:DriveCheckboxes.Keys) { if ($Script:DriveCheckboxes[$dl].Checked) { $driveChecked = $true; break } }
        if ($driveChecked) { $needsPortable = $true }
        
        $defaultPath = "$Script:InstallPath\Backup"
        $portableMode = $false
        
        $destForm = New-Object System.Windows.Forms.Form
        $destForm.Text = "Backup Destination - ABCD Systems"
        $destForm.Size = New-Object System.Drawing.Size(520, 300)
        $destForm.StartPosition = "CenterScreen"
        $destForm.FormBorderStyle = "FixedDialog"
        $destForm.MaximizeBox = $false
        $destForm.BackColor = [System.Drawing.Color]::FromArgb(20, 20, 20)
        $destForm.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        
        $lblDestTitle = New-Object System.Windows.Forms.Label
        $lblDestTitle.Text = "Where should the backup be saved?"
        $lblDestTitle.Location = New-Object System.Drawing.Point(15, 15)
        $lblDestTitle.Size = New-Object System.Drawing.Size(480, 22)
        $lblDestTitle.Font = New-Object System.Drawing.Font("Segoe UI", 10, [System.Drawing.FontStyle]::Bold)
        $destForm.Controls.Add($lblDestTitle)
        
        $lblDestPath = New-Object System.Windows.Forms.Label
        $lblDestPath.Text = "Backup Path:"
        $lblDestPath.Location = New-Object System.Drawing.Point(15, 50)
        $lblDestPath.Size = New-Object System.Drawing.Size(90, 20)
        $destForm.Controls.Add($lblDestPath)
        
        $txtDestPath = New-Object System.Windows.Forms.TextBox
        $txtDestPath.Text = $defaultPath
        $txtDestPath.Location = New-Object System.Drawing.Point(110, 47)
        $txtDestPath.Size = New-Object System.Drawing.Size(290, 22)
        $txtDestPath.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
        $txtDestPath.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $destForm.Controls.Add($txtDestPath)
        
        $btnDestBrowse = New-Object System.Windows.Forms.Button
        $btnDestBrowse.Text = "Browse..."
        $btnDestBrowse.Location = New-Object System.Drawing.Point(410, 46)
        $btnDestBrowse.Size = New-Object System.Drawing.Size(80, 24)
        $btnDestBrowse.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
        $btnDestBrowse.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnDestBrowse.Add_Click({
            $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
            $fbd.Description = "Select backup destination (local drive, USB, mapped drive, or network path)"
            $fbd.SelectedPath = $txtDestPath.Text
            $fbd.ShowNewFolderButton = $true
            if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
                $txtDestPath.Text = $fbd.SelectedPath
            }
        })
        $destForm.Controls.Add($btnDestBrowse)
        
        $chkPortable = New-Object System.Windows.Forms.CheckBox
        $chkPortable.Text = "Portable backup (include all file data for USB / Google Drive / offline restore)"
        $chkPortable.Location = New-Object System.Drawing.Point(15, 85)
        $chkPortable.Size = New-Object System.Drawing.Size(480, 20)
        $chkPortable.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $chkPortable.Checked = $(if ($txtDestPath.Text -ne $defaultPath) { $true } else { $false })
        $destForm.Controls.Add($chkPortable)
        
        $lblPortableInfo = New-Object System.Windows.Forms.Label
        $lblPortableInfo.Text = "When enabled, copies User Data, Program Files, and other selected data into the backup folder so it can be restored from any location without network access."
        $lblPortableInfo.Location = New-Object System.Drawing.Point(35, 108)
        $lblPortableInfo.Size = New-Object System.Drawing.Size(460, 36)
        $lblPortableInfo.ForeColor = [System.Drawing.Color]::FromArgb(140, 140, 140)
        $lblPortableInfo.Font = New-Object System.Drawing.Font("Segoe UI", 8)
        $destForm.Controls.Add($lblPortableInfo)
        
        $lblPortableWarn = New-Object System.Windows.Forms.Label
        $lblPortableWarn.Text = "Note: Portable backups can be very large (10-100+ GB). Ensure your destination has enough space."
        $lblPortableWarn.Location = New-Object System.Drawing.Point(35, 145)
        $lblPortableWarn.Size = New-Object System.Drawing.Size(460, 18)
        $lblPortableWarn.ForeColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
        $lblPortableWarn.Font = New-Object System.Drawing.Font("Segoe UI", 8)
        $destForm.Controls.Add($lblPortableWarn)
        
        $chkSharesToo = New-Object System.Windows.Forms.CheckBox
        $chkSharesToo.Text = "Also create network shares (for live network migration)"
        $chkSharesToo.Location = New-Object System.Drawing.Point(15, 175)
        $chkSharesToo.Size = New-Object System.Drawing.Size(480, 20)
        $chkSharesToo.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $chkSharesToo.Checked = $chkCreateShares.Checked
        $destForm.Controls.Add($chkSharesToo)
        
        # Auto-check portable when path changes away from default
        $txtDestPath.Add_TextChanged({
            if ($txtDestPath.Text -ne $defaultPath) { $chkPortable.Checked = $true }
        })
        
        $Script:DestDialogResult = $false
        
        $btnDestOK = New-Object System.Windows.Forms.Button
        $btnDestOK.Text = "Start Backup"
        $btnDestOK.Location = New-Object System.Drawing.Point(280, 215)
        $btnDestOK.Size = New-Object System.Drawing.Size(110, 32)
        $btnDestOK.BackColor = [System.Drawing.Color]::FromArgb(76, 175, 80)
        $btnDestOK.ForeColor = [System.Drawing.Color]::Black
        $btnDestOK.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnDestOK.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
        $btnDestOK.Add_Click({ $Script:DestDialogResult = $true; $destForm.Close() })
        $destForm.Controls.Add($btnDestOK)
        
        $btnDestCancel = New-Object System.Windows.Forms.Button
        $btnDestCancel.Text = "Cancel"
        $btnDestCancel.Location = New-Object System.Drawing.Point(400, 215)
        $btnDestCancel.Size = New-Object System.Drawing.Size(90, 32)
        $btnDestCancel.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
        $btnDestCancel.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnDestCancel.Add_Click({ $Script:DestDialogResult = $false; $destForm.Close() })
        $destForm.Controls.Add($btnDestCancel)
        
        $destForm.ShowDialog() | Out-Null
        
        if (-not $Script:DestDialogResult) {
            Write-GuiLog "Backup cancelled by user"
            return
        }
        
        $backupPath = $txtDestPath.Text
        $portableMode = $chkPortable.Checked
        $createShares = $chkSharesToo.Checked
        
        Write-GuiLog "Backup destination: $backupPath"
        Write-GuiLog "Portable mode: $portableMode"
        Write-GuiLog "Create shares: $createShares"
        
        # Stop services and log off users for clean backup
        $stoppedLocalServices = @()
        if ($chkStopServices.Checked) {
            Write-GuiLog "Stopping services and logging off users for clean backup..."
            try {
                $stoppedLocalServices = Stop-LocalServicesForBackup
                Write-GuiLog "Stopped $($stoppedLocalServices.Count) service(s)"
            } catch {
                Write-GuiLog "WARNING: Could not stop some services: $($_.Exception.Message)"
            }
        }
        Write-GuiLog "=========================================="
        
        if (!(Test-Path $backupPath)) {
            New-Item -Path $backupPath -ItemType Directory -Force | Out-Null
        }
        
        $totalSteps = 0
        if ($chkUsers.Checked) { $totalSteps++ }
        if ($chkShares.Checked) { $totalSteps++ }
        if ($chkPrinters.Checked) { $totalSteps++ }
        if ($chkRegistry.Checked) { $totalSteps++ }
        if ($chkAppSettings.Checked) { $totalSteps++ }
        if ($createShares) { $totalSteps++ }
        # Portable backup steps
        if ($portableMode) {
            if ($chkUserData.Checked) { $totalSteps++ }
            if ($chkProgramFiles.Checked) { $totalSteps++ }
            if ($chkProgramFilesX86.Checked) { $totalSteps++ }
            if ($chkProgramData.Checked) { $totalSteps++ }
            foreach ($dl in $Script:DriveCheckboxes.Keys) { if ($Script:DriveCheckboxes[$dl].Checked) { $totalSteps++ } }
        }
        if ($totalSteps -eq 0) { $totalSteps = 1 }
        
        $currentStep = 0
        
        # Export Users
        if ($chkUsers.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Exporting users..."
            
            try {
                $userResult = Export-LocalUsers
                # Save to JSON file
                $userResult | ConvertTo-Json -Depth 5 | Out-File "$backupPath\users.json" -Encoding UTF8
                Write-GuiLog "Exported $($userResult.Count) users to users.json"
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR exporting users: $($_.Exception.Message)"
            }
        }
        
        # Export Shares
        if ($chkShares.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Exporting network shares..."
            
            try {
                $shareResult = Export-NetworkShares
                # Save to JSON file
                $shareResult | ConvertTo-Json -Depth 5 | Out-File "$backupPath\shares.json" -Encoding UTF8
                Write-GuiLog "Exported $($shareResult.Count) shares to shares.json"
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR exporting shares: $($_.Exception.Message)"
            }
        }
        
        # Export Printers
        if ($chkPrinters.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Exporting printers with drivers..."
            
            try {
                Export-PrinterConfiguration -BackupPath $backupPath
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR exporting printers: $($_.Exception.Message)"
            }
        }
        
        # Export Registry/Keys
        if ($chkRegistry.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Extracting license keys..."
            
            try {
                Export-RegistryComplete -BackupPath $backupPath
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR extracting keys: $($_.Exception.Message)"
            }
        }
        
        # Export App Settings
        if ($chkAppSettings.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Exporting application settings (100+ apps)..."
            
            try {
                # Get selected users list
                $usersToExport = @()
                if ($Script:AllUsersSelected) {
                    $usersToExport = (Get-MigratableUsers).Name
                    Write-GuiLog "Exporting for ALL users ($($usersToExport.Count) profiles)"
                } else {
                    $usersToExport = $Script:SelectedUsers.Keys | Where-Object { $Script:SelectedUsers[$_] }
                    Write-GuiLog "Exporting for $($usersToExport.Count) selected users"
                }
                
                $appResult = Export-AppSettings -BackupPath $backupPath -SelectedUsers $usersToExport -SelectedApps $Script:SelectedApps
                Write-GuiLog "Exported settings for $($appResult.Count) applications"
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR exporting app settings: $($_.Exception.Message)"
            }
        }
        
        # Create Shares (only if user opted in via destination dialog)
        if ($createShares) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Creating migration shares..."
            
            try {
                $sharesToCreate = @()
                $sharesToCreate += @{Name='ABCD-Migration$'; Path=$backupPath; Required=$true}
                if ($chkProgramFiles.Checked) { $sharesToCreate += @{Name='ProgramFiles$'; Path='C:\Program Files'; Required=$true} }
                if ($chkProgramFilesX86.Checked) { $sharesToCreate += @{Name='ProgramFilesX86$'; Path='C:\Program Files (x86)'; Required=$true} }
                if ($chkProgramData.Checked) { $sharesToCreate += @{Name='ProgramData$'; Path='C:\ProgramData'; Required=$true} }
                if ($chkUserData.Checked) { $sharesToCreate += @{Name='Users$'; Path='C:\Users'; Required=$true} }
                
                # Add additional drive shares
                foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
                    if ($Script:DriveCheckboxes[$driveLetter].Checked) {
                        $drivePath = "${driveLetter}:\"
                        if (Test-Path $drivePath) {
                            $sharesToCreate += @{Name="${driveLetter}Drive$"; Path=$drivePath; Required=$false}
                        }
                    }
                }
                
                # Use net share for Home editions, New-SmbShare for others
                $useSmbCmdlets = $Script:Capabilities.HasSmbCmdlets
                
                foreach ($share in $sharesToCreate) {
                    if ($useSmbCmdlets) {
                        $existingShare = Get-SmbShare -Name $share.Name -ErrorAction SilentlyContinue
                        if ($existingShare) {
                            Write-GuiLog "Share already exists: $($share.Name)"
                        } else {
                            New-SmbShare -Name $share.Name -Path $share.Path -FullAccess "Administrators" -Description "ABCD Migration" | Out-Null
                            Write-GuiLog "Created share: $($share.Name)"
                        }
                    } else {
                        # Use net share for Home editions
                        $existingShare = net share $share.Name 2>&1
                        if ($existingShare -match "does not exist") {
                            $result = net share "$($share.Name)=$($share.Path)" "/grant:Administrators,FULL" "/remark:ABCD Migration" 2>&1
                            if ($LASTEXITCODE -eq 0) {
                                Write-GuiLog "Created share: $($share.Name)"
                            } else {
                                Write-GuiLog "WARNING: Could not create share: $($share.Name)"
                            }
                        } else {
                            Write-GuiLog "Share already exists: $($share.Name)"
                        }
                    }
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR creating shares: $($_.Exception.Message)"
            }
        }
        
        # ============================================================
        # PORTABLE BACKUP - Copy bulk data into backup folder
        # ============================================================
        if ($portableMode) {
            Write-GuiLog "=========================================="
            Write-GuiLog "Starting Portable Backup (copying file data)..."
            Write-GuiLog "=========================================="
            
            # Copy User Data
            if ($chkUserData.Checked) {
                $currentStep++
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
                Write-GuiLog "Copying User Data to portable backup..."
                $userDataDest = "$backupPath\UserData"
                if (!(Test-Path $userDataDest)) { New-Item -Path $userDataDest -ItemType Directory -Force | Out-Null }
                try {
                    $robArgs = "`"C:\Users`" `"$userDataDest`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /XD `"C:\Users\Default`" `"C:\Users\Public`" /XJ"
                    $proc = Start-Process -FilePath "robocopy.exe" -ArgumentList $robArgs -Wait -PassThru -NoNewWindow
                    Write-GuiLog "User Data copy complete (exit code: $($proc.ExitCode))"
                } catch {
                    Write-GuiLog "WARNING: User Data copy error: $($_.Exception.Message)"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            
            # Copy Program Files
            if ($chkProgramFiles.Checked) {
                $currentStep++
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
                Write-GuiLog "Copying Program Files to portable backup..."
                $pfDest = "$backupPath\ProgramFiles"
                if (!(Test-Path $pfDest)) { New-Item -Path $pfDest -ItemType Directory -Force | Out-Null }
                try {
                    $robArgs = "`"C:\Program Files`" `"$pfDest`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /XJ"
                    $proc = Start-Process -FilePath "robocopy.exe" -ArgumentList $robArgs -Wait -PassThru -NoNewWindow
                    Write-GuiLog "Program Files copy complete (exit code: $($proc.ExitCode))"
                } catch {
                    Write-GuiLog "WARNING: Program Files copy error: $($_.Exception.Message)"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            
            # Copy Program Files (x86)
            if ($chkProgramFilesX86.Checked) {
                $currentStep++
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
                Write-GuiLog "Copying Program Files (x86) to portable backup..."
                $pfx86Dest = "$backupPath\ProgramFilesX86"
                if (!(Test-Path $pfx86Dest)) { New-Item -Path $pfx86Dest -ItemType Directory -Force | Out-Null }
                try {
                    $robArgs = "`"C:\Program Files (x86)`" `"$pfx86Dest`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /XJ"
                    $proc = Start-Process -FilePath "robocopy.exe" -ArgumentList $robArgs -Wait -PassThru -NoNewWindow
                    Write-GuiLog "Program Files (x86) copy complete (exit code: $($proc.ExitCode))"
                } catch {
                    Write-GuiLog "WARNING: Program Files (x86) copy error: $($_.Exception.Message)"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            
            # Copy ProgramData
            if ($chkProgramData.Checked) {
                $currentStep++
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
                Write-GuiLog "Copying ProgramData to portable backup..."
                $pdDest = "$backupPath\ProgramData"
                if (!(Test-Path $pdDest)) { New-Item -Path $pdDest -ItemType Directory -Force | Out-Null }
                try {
                    $robArgs = "`"C:\ProgramData`" `"$pdDest`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /XJ /XD `"C:\ProgramData\Microsoft\Windows\WER`" `"C:\ProgramData\Package Cache`""
                    $proc = Start-Process -FilePath "robocopy.exe" -ArgumentList $robArgs -Wait -PassThru -NoNewWindow
                    Write-GuiLog "ProgramData copy complete (exit code: $($proc.ExitCode))"
                } catch {
                    Write-GuiLog "WARNING: ProgramData copy error: $($_.Exception.Message)"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            
            # Copy Additional Drives
            foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
                if ($Script:DriveCheckboxes[$driveLetter].Checked) {
                    $currentStep++
                    Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
                    $driveSrc = "${driveLetter}:\"
                    $driveDest = "$backupPath\Drives\$driveLetter"
                    Write-GuiLog "Copying drive ${driveLetter}: to portable backup..."
                    if (!(Test-Path $driveDest)) { New-Item -Path $driveDest -ItemType Directory -Force | Out-Null }
                    try {
                        $robArgs = "`"$driveSrc`" `"$driveDest`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /XJ"
                        $proc = Start-Process -FilePath "robocopy.exe" -ArgumentList $robArgs -Wait -PassThru -NoNewWindow
                        Write-GuiLog "Drive ${driveLetter}: copy complete (exit code: $($proc.ExitCode))"
                    } catch {
                        Write-GuiLog "WARNING: Drive ${driveLetter}: copy error: $($_.Exception.Message)"
                    }
                    Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
                }
            }
            
            Write-GuiLog "Portable backup data copy complete"
        }
        
        # Save config and manifest
        $selectedDrives = @()
        foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
            if ($Script:DriveCheckboxes[$driveLetter].Checked) {
                $selectedDrives += $driveLetter
            }
        }
        
        $config = @{
            ExportDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
            SourceComputer = $env:COMPUTERNAME
            Version = $Script:Version
            AdditionalDrives = $selectedDrives
        }
        $config | ConvertTo-Json | Out-File "$backupPath\MigrationConfig.json" -Encoding UTF8
        
        # Write backup manifest
        $manifest = @{
            FormatVersion = 2
            PortableBackup = $portableMode
            SourceComputer = $env:COMPUTERNAME
            ExportDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
            BackupPath = $backupPath
            ToolVersion = $Script:Version
            Contents = @{
                Users = $chkUsers.Checked
                Shares = $chkShares.Checked
                Printers = $chkPrinters.Checked
                Registry = $chkRegistry.Checked
                AppSettings = $chkAppSettings.Checked
                UserData = ($chkUserData.Checked -and $portableMode)
                ProgramFiles = ($chkProgramFiles.Checked -and $portableMode)
                ProgramFilesX86 = ($chkProgramFilesX86.Checked -and $portableMode)
                ProgramData = ($chkProgramData.Checked -and $portableMode)
                AdditionalDrives = $selectedDrives
            }
            Paths = @{
                UserData = "UserData"
                ProgramFiles = "ProgramFiles"
                ProgramFilesX86 = "ProgramFilesX86"
                ProgramData = "ProgramData"
                Drives = "Drives"
            }
        }
        $manifest | ConvertTo-Json -Depth 3 | Out-File "$backupPath\backup-manifest.json" -Encoding UTF8
        Write-GuiLog "Backup manifest written"
        
        Update-ProgressBars -Overall 100 -Current 100
        Write-GuiLog "=========================================="
        # Restart services after backup
        if ($stoppedLocalServices.Count -gt 0) {
            Write-GuiLog "Restarting services..."
            try {
                Start-LocalServicesAfterBackup -ServiceNames $stoppedLocalServices
                Write-GuiLog "Restarted $($stoppedLocalServices.Count) service(s)"
            } catch {
                Write-GuiLog "WARNING: Could not restart some services: $($_.Exception.Message)"
            }
        }

        Write-GuiLog "SOURCE Export Complete!"
        Write-GuiLog "Backup location: $backupPath"
        Write-GuiLog "=========================================="
        
        $completeMsg = "SOURCE Export Complete!`n`nBackup saved to: $backupPath"
        if ($portableMode) {
            $completeMsg += "`n`nPortable backup: All data included.`nCopy this folder to USB, Google Drive, or any destination.`nRestore by pointing DESTINATION at this folder path."
        }
        if ($createShares) {
            $completeMsg += "`n`nNetwork shares created for live migration."
        }
        $completeMsg += "`n`nRun this tool on the DESTINATION computer to import."
        
        [System.Windows.Forms.MessageBox]::Show(
            $completeMsg,
            "Export Complete",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        
    } else {
        # DESTINATION MODE
        
        # Validate master password only if Users checkbox is checked
        $masterPassword = $txtMasterPassword.Text
        if ($chkUsers.Checked -and [string]::IsNullOrWhiteSpace($masterPassword)) {
            [System.Windows.Forms.MessageBox]::Show("Please enter or generate a Master Password.`n`nThis password will be set for all imported users.", "Validation", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        $sourceInput = $txtComputer.Text.Trim()
        
        if ([string]::IsNullOrWhiteSpace($sourceInput)) {
            [System.Windows.Forms.MessageBox]::Show("Please enter the SOURCE computer name/IP or a local backup path (e.g. D:\ABCD-Migration\Backup).", "Validation", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        # Detect local path vs network computer name
        $isLocalPath = $false
        $sourcePath = ""
        $sourceComputer = ""
        
        if ($sourceInput -match '^[A-Za-z]:\\' -or ($sourceInput.StartsWith("\\") -and ($sourceInput.Split("\").Count -gt 4))) {
            # Local drive path (C:\...) or deep UNC path
            $isLocalPath = $true
            $sourcePath = $sourceInput
            # Check for Backup subfolder
            $testBackup = Join-Path $sourceInput "Backup"
            if ((Test-Path $testBackup) -and (Test-Path (Join-Path $testBackup "users.json"))) {
                $sourcePath = $testBackup
            }
            Write-GuiLog "=========================================="
            Write-GuiLog "Starting DESTINATION Import from local path: $sourcePath"
            Write-GuiLog "=========================================="
        } else {
            # Computer name - construct UNC share path
            $sourceComputer = $sourceInput -replace '^\\\\', '' -replace '\\.*$', ''
            $sourcePath = "\\$sourceComputer\ABCD-Migration$"
            # Also check Backup subfolder on share
            $testBackup = "$sourcePath\Backup"
            if ((Test-Path $testBackup) -and (Test-Path "$testBackup\users.json")) {
                $sourcePath = $testBackup
            }
            Write-GuiLog "=========================================="
            Write-GuiLog "Starting DESTINATION Import from $sourceComputer ($sourcePath)..."
            Write-GuiLog "=========================================="
        }
        
        # Clear any previous copy issues
        Clear-CopyIssues
        
        # Initialize rollback if enabled
        if ($chkRollback.Checked) {
            Write-GuiLog "Initializing file versioning (rollback)..."
            Initialize-Rollback -Enabled $true -RetentionDays $numRollbackDays.Value
        }
        
        # Update tray status
        Update-TrayStatus -Status "Running DESTINATION Import"
        
        # Create credential for remote service management
        $networkCredential = $null
        if ($txtUsername.Text -and $txtPassword.Text) {
            try {
                $securePass = ConvertTo-SecureString $txtPassword.Text -AsPlainText -Force
                $networkCredential = New-Object System.Management.Automation.PSCredential($txtUsername.Text, $securePass)
            }
            catch {
                Write-GuiLog "WARNING: Could not create credential for remote services"
            }
        }
        
        # sourcePath is already set above (local or network)
        
        if (!(Test-Path $sourcePath)) {
            [System.Windows.Forms.MessageBox]::Show("Cannot access SOURCE: $sourcePath`n`nPlease check the path and try again.`n`nFor local restore: enter the folder path (e.g. D:\ABCD-Migration\Backup)`nFor network: enter the SOURCE computer name or IP", "Connection Error", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Error)
            return
        }
        
        $totalSteps = 0
        # Stop services on SOURCE for clean file copy (network mode only)
        $stoppedRemoteServices = @()
        if ($chkStopServices.Checked -and (-not $isLocalPath) -and $sourceComputer) {
            Write-GuiLog "Stopping services and logging off users on SOURCE..."
            try {
                $stoppedRemoteServices = Stop-RemoteServicesForCopy -ComputerName $sourceComputer -Credential $networkCredential
                if ($stoppedRemoteServices.Count -gt 0) {
                    Write-GuiLog "Stopped $($stoppedRemoteServices.Count) service(s) on SOURCE"
                }
            } catch {
                Write-GuiLog "WARNING: Could not stop remote services: $($_.Exception.Message)"
            }
        }

        if ($chkUsers.Checked) { $totalSteps++ }
        if ($chkShares.Checked) { $totalSteps++ }
        if ($chkPrinters.Checked) { $totalSteps++ }
        if ($chkProgramFiles.Checked) { $totalSteps++ }
        if ($chkProgramFilesX86.Checked) { $totalSteps++ }
        if ($chkProgramData.Checked) { $totalSteps++ }
        if ($chkUserData.Checked) { $totalSteps++ }
        if ($chkAppSettings.Checked) { $totalSteps++ }
        
        # Count additional drives
        foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
            if ($Script:DriveCheckboxes[$driveLetter].Checked) { $totalSteps++ }
        }
        
        if ($totalSteps -eq 0) { $totalSteps = 1 }
        
        $currentStep = 0
        
        # Convert master password to SecureString (only if Users selected)
        $securePassword = $null
        if ($chkUsers.Checked -and $masterPassword) {
            $securePassword = ConvertTo-SecureString -String $masterPassword -AsPlainText -Force
        }
        
        # Import Users
        if ($chkUsers.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Importing users..."
            
            try {
                $usersFile = "$sourcePath\users.json"
                if (Test-Path $usersFile) {
                    $users = Get-Content $usersFile -Raw | ConvertFrom-Json
                    $result = Import-LocalUsers -Users $users -MasterPassword $securePassword
                    Write-GuiLog "Imported $($result.Imported) users, skipped $($result.Skipped)"
                    
                    # Create RDS profiles if enabled
                    if ($chkCreateProfiles.Checked) {
                        Write-GuiLog "Creating user profiles via RDS..."
                        foreach ($user in $users) {
                            $profilePath = "C:\Users\$($user.Name)"
                            if (!(Test-Path $profilePath)) {
                                try {
                                    Write-GuiLog "Creating profile for: $($user.Name)"
                                    New-UserProfileViaRDS -Username $user.Name -MasterPassword $securePassword
                                }
                                catch {
                                    Write-GuiLog "WARNING: Could not create profile for $($user.Name)"
                                }
                            } else {
                                Write-GuiLog "Profile already exists: $($user.Name)"
                            }
                        }
                    }
                } else {
                    Write-GuiLog "WARNING: users.json not found at $usersFile"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR importing users: $($_.Exception.Message)"
            }
        }
        
        # Import Shares
        if ($chkShares.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Importing network shares..."
            
            try {
                $sharesFile = "$sourcePath\shares.json"
                if (Test-Path $sharesFile) {
                    $shares = Get-Content $sharesFile -Raw | ConvertFrom-Json
                    Import-NetworkShares -Shares $shares -CreateFolders
                    Write-GuiLog "Imported $($shares.Count) network shares"
                } else {
                    Write-GuiLog "WARNING: shares.json not found at $sharesFile"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR importing shares: $($_.Exception.Message)"
            }
        }
        
        # Import Printers
        if ($chkPrinters.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Importing printers..."
            
            try {
                Import-PrinterConfiguration -SourcePath $sourcePath
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR importing printers: $($_.Exception.Message)"
            }
        }
        
        # Copy Program Files
        if ($chkProgramFiles.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Copying Program Files..."
            
            try {
                # Resolve path: portable backup subfolder or network share
                $progFilesShare = ""
                if ($isLocalPath) {
                    if (Test-Path "$sourcePath\ProgramFiles") { $progFilesShare = "$sourcePath\ProgramFiles" }
                }
                if (-not $progFilesShare -and $sourceComputer) { $progFilesShare = "\\$sourceComputer\ProgramFiles$" }
                
                if ($progFilesShare -and (Test-Path $progFilesShare)) {
                    Copy-ProgramFilesWithACL -SourceShare $progFilesShare -Threads $numThreads.Value -Overwrite:$chkOverwrite.Checked -NewerOnly:$chkNewerOnly.Checked
                    Write-GuiLog "Program Files copy complete"
                } else {
                    Write-GuiLog "WARNING: Program Files not found in backup (checked: $progFilesShare)"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR copying Program Files: $($_.Exception.Message)"
            }
        }
        
        # Copy Program Files (x86)
        if ($chkProgramFilesX86.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Copying Program Files (x86)..."
            
            try {
                $progFilesX86Share = ""
                if ($isLocalPath) {
                    if (Test-Path "$sourcePath\ProgramFilesX86") { $progFilesX86Share = "$sourcePath\ProgramFilesX86" }
                }
                if (-not $progFilesX86Share -and $sourceComputer) { $progFilesX86Share = "\\$sourceComputer\ProgramFilesX86$" }
                
                if ($progFilesX86Share -and (Test-Path $progFilesX86Share)) {
                    Copy-ProgramFilesX86WithACL -SourceShare $progFilesX86Share -Threads $numThreads.Value -Overwrite:$chkOverwrite.Checked -NewerOnly:$chkNewerOnly.Checked
                    Write-GuiLog "Program Files (x86) copy complete"
                } else {
                    Write-GuiLog "WARNING: Program Files (x86) not found in backup"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR copying Program Files (x86): $($_.Exception.Message)"
            }
        }
        
        # Copy ProgramData
        if ($chkProgramData.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Copying ProgramData..."
            
            try {
                $progDataShare = ""
                if ($isLocalPath) {
                    if (Test-Path "$sourcePath\ProgramData") { $progDataShare = "$sourcePath\ProgramData" }
                }
                if (-not $progDataShare -and $sourceComputer) { $progDataShare = "\\$sourceComputer\ProgramData$" }
                
                if ($progDataShare -and (Test-Path $progDataShare)) {
                    Copy-ProgramDataWithACL -SourceShare $progDataShare -Threads $numThreads.Value -Overwrite:$chkOverwrite.Checked -NewerOnly:$chkNewerOnly.Checked
                } else {
                    Write-GuiLog "WARNING: ProgramData not found in backup"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR copying ProgramData: $($_.Exception.Message)"
            }
        }
        
        # Copy User Data
        if ($chkUserData.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Copying user data..."
            
            try {
                $usersShare = ""
                if ($isLocalPath) {
                    if (Test-Path "$sourcePath\UserData") { $usersShare = "$sourcePath\UserData" }
                }
                if (-not $usersShare -and $sourceComputer) { $usersShare = "\\$sourceComputer\Users$" }
                
                if ($usersShare -and (Test-Path $usersShare)) {
                    $usersFile = "$sourcePath\users.json"
                    if (Test-Path $usersFile) {
                        $users = Get-Content $usersFile -Raw | ConvertFrom-Json
                        foreach ($user in $users) {
                            Write-GuiLog "Copying data for: $($user.Name)"
                            Copy-UserDataWithACL -SourceShare $usersShare -Username $user.Name -Threads $numThreads.Value -Overwrite:$chkOverwrite.Checked -NewerOnly:$chkNewerOnly.Checked -IncludeAppData:$chkAppData.Checked
                        }
                    }
                } else {
                    Write-GuiLog "WARNING: User Data not found in backup"
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR copying user data: $($_.Exception.Message)"
            }
        }
        
        # Import App Settings
        if ($chkAppSettings.Checked) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Importing application settings (100+ apps)..."
            
            try {
                # Get selected users list for import
                $usersToImport = @()
                if ($Script:AllUsersSelected) {
                    $usersToImport = $null  # null means import all available
                    Write-GuiLog "Importing for ALL available users"
                } else {
                    $usersToImport = @($Script:SelectedUsers.Keys | Where-Object { $Script:SelectedUsers[$_] })
                    Write-GuiLog "Importing for $($usersToImport.Count) selected users"
                }
                
                $appResult = Import-AppSettings -SourcePath $sourcePath -Overwrite:$chkOverwrite.Checked -SelectedUsers $usersToImport
                Write-GuiLog "Imported $($appResult.Imported) app settings, skipped $($appResult.Skipped)"
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR importing app settings: $($_.Exception.Message)"
            }
        }
        
        # Copy Additional Drives
        $stoppedRemoteServices = @()
        $drivesToCopy = @()
        
        foreach ($driveLetter in $Script:DriveCheckboxes.Keys) {
            if ($Script:DriveCheckboxes[$driveLetter].Checked) {
                $drivesToCopy += $driveLetter
            }
        }
        
        
        foreach ($driveLetter in $drivesToCopy) {
            $currentStep++
            Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 0
            Write-GuiLog "Copying drive ${driveLetter}:..."
            
            try {
                if ($isLocalPath) {
                    # Check all portable backup naming conventions
                    $localDriveData = ""
                    $checkPaths = @(
                        (Join-Path $sourcePath "Drives\$driveLetter"),
                        (Join-Path $sourcePath "Drives\${driveLetter}_Drive"),
                        (Join-Path $sourcePath "${driveLetter}_Drive"),
                        (Join-Path $sourcePath "${driveLetter}Drive")
                    )
                    foreach ($checkPath in $checkPaths) {
                        if (Test-Path $checkPath) { $localDriveData = $checkPath; break }
                    }
                    if ($localDriveData) {
                        Copy-AdditionalDrive -SourceShare $localDriveData -DestinationDrive "${driveLetter}:" -Threads $numThreads.Value -Overwrite:$chkOverwrite.Checked -NewerOnly:$chkNewerOnly.Checked
                        Write-GuiLog "Drive ${driveLetter}: restored from $localDriveData"
                    } else {
                        Write-GuiLog "WARNING: No drive ${driveLetter} data found in backup path"
                    }
                } else {
                    $driveShare = "\\$sourceComputer\${driveLetter}Drive$"
                    if (Test-Path $driveShare) {
                        Copy-AdditionalDrive -SourceShare $driveShare -DestinationDrive "${driveLetter}:" -Threads $numThreads.Value -Overwrite:$chkOverwrite.Checked -NewerOnly:$chkNewerOnly.Checked
                        Write-GuiLog "Drive ${driveLetter}: copy complete"
                    } else {
                        Write-GuiLog "WARNING: Drive share not accessible: $driveShare"
                    }
                }
                Update-ProgressBars -Overall ([int]($currentStep / $totalSteps * 100)) -Current 100
            }
            catch {
                Write-GuiLog "ERROR copying drive ${driveLetter}: $($_.Exception.Message)"
            }
        }
        
        # Restart services on SOURCE if we stopped any (network mode only)
        if ($stoppedRemoteServices.Count -gt 0 -and (-not $isLocalPath) -and $sourceComputer) {
            Write-GuiLog "Restarting services on SOURCE..."
            try {
                Start-RemoteStoppedServices -ComputerName $sourceComputer -Credential $networkCredential -ServiceNames $stoppedRemoteServices
                Write-GuiLog "Restarted $($stoppedRemoteServices.Count) service(s) on SOURCE"
            }
            catch {
                Write-GuiLog "WARNING: Could not restart some services: $($_.Exception.Message)"
                Write-GuiLog "You may need to manually restart services on $sourceComputer"
            }
        }
        
        Update-ProgressBars -Overall 100 -Current 100
        Write-GuiLog "=========================================="
        Write-GuiLog "DESTINATION Import Complete!"
        Write-GuiLog "=========================================="
        
        # Show summary of any files that couldn't be copied
        $hadIssues = Show-CopyIssuesSummary
        
        if ($hadIssues) {
            [System.Windows.Forms.MessageBox]::Show(
                "DESTINATION Import Complete!`n`nSome files could not be copied (see previous popup for details).`n`nUsers can log in with the master password.`n`nRemember to reactivate licensed software with your keys.",
                "Import Complete",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Information
            )
        } else {
            [System.Windows.Forms.MessageBox]::Show(
                "DESTINATION Import Complete!`n`nUsers can log in with the master password.`n`nRemember to reactivate licensed software with your keys.",
                "Import Complete",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Information
            )
        }
    }
})

#endregion

# Apply capability-based UI adjustments
if ($Script:Capabilities.IsHomeEdition) {
    # Gray out printer option on Home edition (no printbrm)
    $chkPrinters.Text = "Printers (limited - Windows Home)"
    $chkPrinters.ForeColor = [System.Drawing.Color]::Gray
    
    # Add tooltip explaining limitation
    $tooltip = New-Object System.Windows.Forms.ToolTip
    $tooltip.SetToolTip($chkPrinters, "Windows Home does not include printbrm.exe.`nPrinter config will be exported but drivers may need manual installation.")
}

if (-not $Script:Capabilities.HasPrintbrm -and -not $Script:Capabilities.IsHomeEdition) {
    # Non-Home edition but printbrm missing (unusual)
    $chkPrinters.Text = "Printers (legacy mode - printbrm not found)"
    $chkPrinters.ForeColor = [System.Drawing.Color]::DarkOrange
}

# Set initial state for SOURCE mode (default)
# Hide master password controls (not needed for SOURCE)
$lblMasterPassword.Visible = $false
$txtMasterPassword.Visible = $false
$btnShowMaster.Visible = $false
$btnGenerate.Visible = $false
$chkPasswordMonitoring.Visible = $false
$lblMonitoringInfo.Visible = $false

# Show recovery tools (SOURCE only)
$lblRecoveryTools.Visible = $true
$btnKeyFinder.Visible = $true
$btnLaZagne.Visible = $true
$btnViewReport.Visible = $true

# Show the form
Write-GuiLog "ABC&D Systems Backup and Migration Tools v$Script:Version started"

# Log any startup warnings
if ($Script:StartupWarnings -and $Script:StartupWarnings.Count -gt 0) {
    foreach ($warning in $Script:StartupWarnings) {
        Write-GuiLog "WARNING: $warning"
    }
}

# Log capabilities
Write-GuiLog "Capabilities: Home=$($Script:Capabilities.IsHomeEdition), Printbrm=$($Script:Capabilities.HasPrintbrm), SmbCmdlets=$($Script:Capabilities.HasSmbCmdlets)"

# Load saved settings for current mode and update schedule status
try {
    $savedSettings = Load-MigrationSettings -Mode $currentMode
    
    if ($savedSettings) {
        Write-GuiLog "Loading saved settings for $currentMode mode..."
        
        # Apply saved checkbox states
        if ($null -ne $savedSettings.Users) { $chkUsers.Checked = $savedSettings.Users }
        if ($null -ne $savedSettings.Shares) { $chkShares.Checked = $savedSettings.Shares }
        if ($null -ne $savedSettings.Printers) { $chkPrinters.Checked = $savedSettings.Printers }
        if ($null -ne $savedSettings.UserData) { $chkUserData.Checked = $savedSettings.UserData }
        if ($null -ne $savedSettings.ProgramFiles) { $chkProgramFiles.Checked = $savedSettings.ProgramFiles }
        if ($null -ne $savedSettings.ProgramFilesX86) { $chkProgramFilesX86.Checked = $savedSettings.ProgramFilesX86 }
        if ($null -ne $savedSettings.ProgramData) { $chkProgramData.Checked = $savedSettings.ProgramData }
        if ($null -ne $savedSettings.Registry) { $chkRegistry.Checked = $savedSettings.Registry }
        if ($null -ne $savedSettings.AppSettings) { $chkAppSettings.Checked = $savedSettings.AppSettings }
        if ($null -ne $savedSettings.CreateShares) { $chkCreateShares.Checked = $savedSettings.CreateShares }
        if ($null -ne $savedSettings.Overwrite) { $chkOverwrite.Checked = $savedSettings.Overwrite }
        if ($null -ne $savedSettings.NewerOnly) { $chkNewerOnly.Checked = $savedSettings.NewerOnly; $chkNewerOnly.Enabled = $chkOverwrite.Checked }
        if ($null -ne $savedSettings.Threads) { $numThreads.Value = $savedSettings.Threads }
        
        # Load additional drives
        if ($savedSettings.AdditionalDrives) {
            foreach ($drive in $savedSettings.AdditionalDrives) {
                if ($Script:DriveCheckboxes.ContainsKey($drive)) {
                    $Script:DriveCheckboxes[$drive].Checked = $true
                }
            }
        }
        
        # Load selected apps
        if ($savedSettings.SelectedApps) {
            $Script:SelectedApps = @{}
            foreach ($key in $savedSettings.SelectedApps.PSObject.Properties.Name) {
                $Script:SelectedApps[$key] = $savedSettings.SelectedApps.$key
            }
        }
        
        # Load selected users
        if ($null -ne $savedSettings.AllUsersSelected) {
            $Script:AllUsersSelected = $savedSettings.AllUsersSelected
        }
        if ($savedSettings.SelectedUsers) {
            $Script:SelectedUsers = @{}
            foreach ($key in $savedSettings.SelectedUsers.PSObject.Properties.Name) {
                $Script:SelectedUsers[$key] = $savedSettings.SelectedUsers.$key
            }
        }
        
        Write-GuiLog "Settings loaded successfully"
    }
    
    # Update schedule status display
    Update-ScheduleStatusUI
}
catch {
    Write-GuiLog "Note: Could not load saved settings: $($_.Exception.Message)"
}

# Update summary label with app/user counts
$selectedAppCount = ($Script:SelectedApps.Values | Where-Object { $_ }).Count
if ($selectedAppCount -eq 0 -and $Script:CommonApps) {
    # Default count (all safe apps)
    $selectedAppCount = @($Script:CommonApps | Where-Object { $_.Warning -le 1 }).Count
}
if ($null -eq $selectedAppCount) { $selectedAppCount = 0 }

if ($Script:AllUsersSelected) {
    $lblAppUserSummary.Text = "($selectedAppCount apps, all users)"
} else {
    $selectedUserCount = ($Script:SelectedUsers.Values | Where-Object { $_ }).Count
    if ($null -eq $selectedUserCount) { $selectedUserCount = 0 }
    if ($selectedUserCount -eq 0) {
        $lblAppUserSummary.Text = "($selectedAppCount apps, all users)"
    } else {
        $lblAppUserSummary.Text = "($selectedAppCount apps, $selectedUserCount users)"
    }
}

# Final startup logging
Write-GuiLog "=========================================="
Write-GuiLog "Startup complete - GUI ready"
Write-GuiLog "Log file: $Script:LogFile"
Write-GuiLog "Bootstrap log: $Script:BootstrapLog"
Write-GuiLog "=========================================="

# Load saved settings for default mode (SOURCE)
Write-GuiLog "Loading saved settings..."
try {
    $savedSettings = Load-MigrationSettings -Mode "SOURCE"
    if ($savedSettings) {
        Write-GuiLog "Applying saved SOURCE settings..."
        
        # Apply saved settings to controls
        if ($savedSettings.PSObject.Properties.Name -contains 'Computer') { $txtComputer.Text = $savedSettings.Computer }
        if ($savedSettings.PSObject.Properties.Name -contains 'Username') { $txtUsername.Text = $savedSettings.Username }
        if ($savedSettings.PSObject.Properties.Name -contains 'Overwrite') { $chkOverwrite.Checked = $savedSettings.Overwrite }
        if ($savedSettings.PSObject.Properties.Name -contains 'NewerOnly') { $chkNewerOnly.Checked = $savedSettings.NewerOnly }
        if ($savedSettings.PSObject.Properties.Name -contains 'StopServices') { $chkStopServices.Checked = $savedSettings.StopServices }
        if ($savedSettings.PSObject.Properties.Name -contains 'Rollback') { $chkRollback.Checked = $savedSettings.Rollback }
        if ($savedSettings.PSObject.Properties.Name -contains 'Threads') { $numThreads.Value = [Math]::Min([Math]::Max($savedSettings.Threads, 1), 32) }
        if ($savedSettings.PSObject.Properties.Name -contains 'Users') { $chkUsers.Checked = $savedSettings.Users }
        if ($savedSettings.PSObject.Properties.Name -contains 'Shares') { $chkShares.Checked = $savedSettings.Shares }
        if ($savedSettings.PSObject.Properties.Name -contains 'Printers') { $chkPrinters.Checked = $savedSettings.Printers }
        if ($savedSettings.PSObject.Properties.Name -contains 'ProgramFiles') { $chkProgramFiles.Checked = $savedSettings.ProgramFiles }
        if ($savedSettings.PSObject.Properties.Name -contains 'ProgramFilesX86') { $chkProgramFilesX86.Checked = $savedSettings.ProgramFilesX86 }
        if ($savedSettings.PSObject.Properties.Name -contains 'ProgramData') { $chkProgramData.Checked = $savedSettings.ProgramData }
        if ($savedSettings.PSObject.Properties.Name -contains 'UserData') { $chkUserData.Checked = $savedSettings.UserData }
        if ($savedSettings.PSObject.Properties.Name -contains 'AppSettings') { $chkAppSettings.Checked = $savedSettings.AppSettings }
        
        # Apply additional drives
        if ($savedSettings.PSObject.Properties.Name -contains 'AdditionalDrives' -and $savedSettings.AdditionalDrives) {
            foreach ($drive in $savedSettings.AdditionalDrives) {
                if ($Script:DriveCheckboxes.ContainsKey($drive)) {
                    $Script:DriveCheckboxes[$drive].Checked = $true
                }
            }
        }
        
        Write-GuiLog "Saved settings applied"
    }
} catch {
    Write-GuiLog "Note: Could not load saved settings: $($_.Exception.Message)"
}

[void]$form.ShowDialog()

# Log when GUI closes
Write-GuiLog "GUI closed - session ended"
