#Requires -RunAsAdministrator
<#
.SYNOPSIS
    ABCD Migration Tool - Scheduled/Headless Execution
.DESCRIPTION
    Runs migration in unattended mode using saved settings
    Called by Windows Task Scheduler
.PARAMETER Mode
    SOURCE or DESTINATION
.PARAMETER SourceComputer
    (DESTINATION only) Computer name/IP to pull from
.EXAMPLE
    .\ABCD-Migration-Scheduled.ps1 -Mode SOURCE
    .\ABCD-Migration-Scheduled.ps1 -Mode DESTINATION -SourceComputer "192.168.1.100"
#>

param(
    [Parameter(Mandatory=$true)]
    [ValidateSet("SOURCE", "DESTINATION")]
    [string]$Mode,
    
    [string]$SourceComputer = ""
)

$ErrorActionPreference = 'Continue'
$Script:InstallPath = "C:\ABCD-Migration"
$Script:SettingsFile = "$Script:InstallPath\Settings\migration-settings.json"
$Script:Version = "2.2"

# Initialize logging
$logFolder = "$Script:InstallPath\Logs"
if (!(Test-Path $logFolder)) {
    New-Item -Path $logFolder -ItemType Directory -Force | Out-Null
}
$logFile = "$logFolder\scheduled-$(Get-Date -Format 'yyyyMMdd-HHmmss')-$Mode.log"

function Write-Log {
    param([string]$Message)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] $Message"
    Add-Content -Path $logFile -Value $logMessage
    Write-Host $logMessage
}

Write-Log "=========================================="
Write-Log "ABCD Migration Tool v$Script:Version - Scheduled Execution"
Write-Log "Mode: $Mode"
Write-Log "=========================================="

# Load modules
$modulePath = $Script:InstallPath
try {
    . "$modulePath\Modules\Logging.ps1"
    . "$modulePath\Modules\User-Migration.ps1"
    . "$modulePath\Modules\Printer-Migration.ps1"
    . "$modulePath\Modules\Share-Migration.ps1"
    . "$modulePath\Modules\File-Migration.ps1"
    . "$modulePath\Modules\AppSettings-Migration.ps1"
    . "$modulePath\Modules\ScheduledTask-Migration.ps1"
    
    $keyFinderPath = "$modulePath\Modules\KeyFinder.ps1"
    if (Test-Path $keyFinderPath) {
        . $keyFinderPath
    }
    
    Write-Log "Modules loaded successfully"
}
catch {
    Write-Log "ERROR loading modules: $($_.Exception.Message)"
    exit 1
}

# Load saved settings
$settings = Load-MigrationSettings -Mode $Mode
if (!$settings) {
    Write-Log "ERROR: No saved settings found for $Mode mode"
    Write-Log "Please run the GUI and save settings first"
    exit 1
}

Write-Log "Settings loaded from: $Script:SettingsFile"

# Detect capabilities
$Script:Capabilities = @{
    HasPrintbrm = (Get-Command printbrm.exe -ErrorAction SilentlyContinue) -ne $null
    HasSmbCmdlets = (Get-Command New-SmbShare -ErrorAction SilentlyContinue) -ne $null
    IsHomeEdition = (Get-CimInstance Win32_OperatingSystem).Caption -match "Home"
    IsServer = (Get-CimInstance Win32_OperatingSystem).Caption -match "Server"
}

Write-Log "Capabilities: Home=$($Script:Capabilities.IsHomeEdition), SMB=$($Script:Capabilities.HasSmbCmdlets)"

if ($Mode -eq "SOURCE") {
    # ==========================================
    # SOURCE MODE - Export/Backup
    # ==========================================
    Write-Log "Starting SOURCE backup..."
    
    $backupPath = "$Script:InstallPath\Backup"
    if (!(Test-Path $backupPath)) {
        New-Item -Path $backupPath -ItemType Directory -Force | Out-Null
    }
    
    # Export Users
    if ($settings.Users) {
        Write-Log "Exporting users..."
        try {
            $userResult = Export-LocalUsers
            $userResult | ConvertTo-Json -Depth 5 | Out-File "$backupPath\users.json" -Encoding UTF8
            Write-Log "Exported $($userResult.Count) users"
        }
        catch {
            Write-Log "ERROR exporting users: $($_.Exception.Message)"
        }
    }
    
    # Export Shares
    if ($settings.Shares) {
        Write-Log "Exporting shares..."
        try {
            $shareResult = Export-NetworkShares
            $shareResult | ConvertTo-Json -Depth 5 | Out-File "$backupPath\shares.json" -Encoding UTF8
            Write-Log "Exported $($shareResult.Count) shares"
        }
        catch {
            Write-Log "ERROR exporting shares: $($_.Exception.Message)"
        }
    }
    
    # Export Printers
    if ($settings.Printers) {
        Write-Log "Exporting printers..."
        try {
            Export-PrinterConfiguration -BackupPath $backupPath
            Write-Log "Printers exported"
        }
        catch {
            Write-Log "ERROR exporting printers: $($_.Exception.Message)"
        }
    }
    
    # Export Registry/Keys
    if ($settings.Registry) {
        Write-Log "Extracting license keys..."
        try {
            Export-RegistryComplete -BackupPath $backupPath
            Write-Log "License keys extracted"
        }
        catch {
            Write-Log "ERROR extracting keys: $($_.Exception.Message)"
        }
    }
    
    # Export App Settings
    if ($settings.AppSettings) {
        Write-Log "Exporting application settings..."
        try {
            $selectedApps = $null
            if ($settings.SelectedApps) {
                $selectedApps = @{}
                foreach ($prop in $settings.SelectedApps.PSObject.Properties) {
                    $selectedApps[$prop.Name] = $prop.Value
                }
            }
            $appResult = Export-AppSettings -BackupPath $backupPath -AllUsers -SelectedApps $selectedApps
            Write-Log "Exported settings for $($appResult.Count) applications"
        }
        catch {
            Write-Log "ERROR exporting app settings: $($_.Exception.Message)"
        }
    }
    
    # Create/Update Shares
    if ($settings.CreateShares) {
        Write-Log "Creating/updating migration shares..."
        try {
            $sharesToCreate = @()
            $sharesToCreate += @{Name='ABCD-Migration$'; Path=$backupPath}
            if ($settings.ProgramFiles) { $sharesToCreate += @{Name='ProgramFiles$'; Path='C:\Program Files'} }
            if ($settings.ProgramFilesX86) { $sharesToCreate += @{Name='ProgramFilesX86$'; Path='C:\Program Files (x86)'} }
            if ($settings.ProgramData) { $sharesToCreate += @{Name='ProgramData$'; Path='C:\ProgramData'} }
            if ($settings.UserData) { $sharesToCreate += @{Name='Users$'; Path='C:\Users'} }
            
            # Additional drives
            if ($settings.AdditionalDrives) {
                foreach ($drive in $settings.AdditionalDrives) {
                    $drivePath = "${drive}:\"
                    if (Test-Path $drivePath) {
                        $sharesToCreate += @{Name="${drive}Drive$"; Path=$drivePath}
                    }
                }
            }
            
            foreach ($share in $sharesToCreate) {
                if ($Script:Capabilities.HasSmbCmdlets) {
                    $existing = Get-SmbShare -Name $share.Name -ErrorAction SilentlyContinue
                    if (!$existing) {
                        New-SmbShare -Name $share.Name -Path $share.Path -FullAccess "Administrators" -Description "ABCD Migration" | Out-Null
                        Write-Log "Created share: $($share.Name)"
                    }
                } else {
                    $result = net share "$($share.Name)=$($share.Path)" "/grant:Administrators,FULL" 2>&1
                    if ($LASTEXITCODE -eq 0) {
                        Write-Log "Created share: $($share.Name)"
                    }
                }
            }
        }
        catch {
            Write-Log "ERROR creating shares: $($_.Exception.Message)"
        }
    }
    
    # Save config
    $config = @{
        ExportDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        SourceComputer = $env:COMPUTERNAME
        Version = $Script:Version
        ScheduledRun = $true
    }
    $config | ConvertTo-Json | Out-File "$backupPath\MigrationConfig.json" -Encoding UTF8
    
    Write-Log "=========================================="
    Write-Log "SOURCE Backup Complete!"
    Write-Log "=========================================="
}
else {
    # ==========================================
    # DESTINATION MODE - Import
    # ==========================================
    
    # Get source computer/path from settings or parameter
    if (!$SourceComputer -and $settings.SourceComputer) {
        $SourceComputer = $settings.SourceComputer
    }
    
    if (!$SourceComputer) {
        Write-Log "ERROR: No source computer/path specified"
        exit 1
    }
    
    # Detect local path vs network computer name
    $isLocalPath = $SourceComputer -match '^[A-Za-z]:\\'
    
    if ($isLocalPath) {
        Write-Log "Starting DESTINATION import from local path: $SourceComputer"
        $sourcePath = $SourceComputer
        # Check for Backup subfolder
        $testBackup = Join-Path $SourceComputer "Backup"
        if ((Test-Path $testBackup) -and (Test-Path (Join-Path $testBackup "users.json"))) {
            $sourcePath = $testBackup
        }
    } else {
        Write-Log "Starting DESTINATION import from $SourceComputer..."
        
        # Setup network credentials if provided (needed when running as SYSTEM)
        $networkUsername = $settings.NetworkUsername
        $networkPassword = $settings.NetworkPassword
        
        if ($networkUsername -and $networkPassword) {
            Write-Log "Setting up network connection with stored credentials..."
            
            # Disconnect any existing connections first
            $shares = @('ABCD-Migration$', 'ProgramFiles$', 'ProgramFilesX86$', 'ProgramData$', 'Users$')
            foreach ($share in $shares) {
                net use "\\$SourceComputer\$share" /delete 2>&1 | Out-Null
            }
            
            # Connect to main share with credentials
            $connectResult = net use "\\$SourceComputer\ABCD-Migration$" /user:$networkUsername $networkPassword 2>&1
            if ($LASTEXITCODE -ne 0) {
                Write-Log "WARNING: Network connection may have failed: $connectResult"
            } else {
                Write-Log "Network connection established"
            }
        }
        
        $sourcePath = "\\$SourceComputer\ABCD-Migration$"
        # Check for Backup subfolder
        $testBackup = "$sourcePath\Backup"
        if ((Test-Path $testBackup) -and (Test-Path "$testBackup\users.json")) {
            $sourcePath = $testBackup
        }
    }
    
    # Test connection
    if (!(Test-Path $sourcePath)) {
        Write-Log "ERROR: Cannot access source: $sourcePath"
        Write-Log "Make sure network credentials are saved and SOURCE shares are created"
        exit 1
    }
    
    Write-Log "Connected to source: $sourcePath"
    
    # Get master password from settings (only required if Users is selected)
    $masterPassword = $settings.MasterPassword
    $securePassword = $null
    
    if ($settings.Users) {
        if (!$masterPassword) {
            Write-Log "ERROR: No master password in settings (required for user import)"
            exit 1
        }
        $securePassword = ConvertTo-SecureString -String $masterPassword -AsPlainText -Force
    }
    
    $overwrite = $settings.Overwrite -eq $true
    $threads = if ($settings.Threads) { $settings.Threads } else { 8 }
    
    # Import Users
    if ($settings.Users) {
        Write-Log "Importing users..."
        try {
            $usersFile = "$sourcePath\users.json"
            if (Test-Path $usersFile) {
                $users = Get-Content $usersFile -Raw | ConvertFrom-Json
                $result = Import-LocalUsers -Users $users -MasterPassword $securePassword
                Write-Log "Imported $($result.Imported) users, skipped $($result.Skipped)"
            }
        }
        catch {
            Write-Log "ERROR importing users: $($_.Exception.Message)"
        }
    }
    
    # Import Shares
    if ($settings.Shares) {
        Write-Log "Importing network shares..."
        try {
            $sharesFile = "$sourcePath\shares.json"
            if (Test-Path $sharesFile) {
                $shares = Get-Content $sharesFile -Raw | ConvertFrom-Json
                Import-NetworkShares -Shares $shares -CreateFolders
                Write-Log "Imported network shares"
            }
        }
        catch {
            Write-Log "ERROR importing shares: $($_.Exception.Message)"
        }
    }
    
    # Import Printers
    if ($settings.Printers) {
        Write-Log "Importing printers..."
        try {
            Import-PrinterConfiguration -SourcePath $sourcePath
            Write-Log "Printers imported"
        }
        catch {
            Write-Log "ERROR importing printers: $($_.Exception.Message)"
        }
    }
    
    # Copy Program Files
    if ($settings.ProgramFiles) {
        Write-Log "Copying Program Files..."
        try {
            $pfShare = ""
            if ($isLocalPath) {
                if (Test-Path "$sourcePath\ProgramFiles") { $pfShare = "$sourcePath\ProgramFiles" }
            }
            if (-not $pfShare -and (-not $isLocalPath)) { $pfShare = "\\$SourceComputer\ProgramFiles$" }
            if ($pfShare -and (Test-Path $pfShare)) {
                Copy-ProgramFilesWithACL -SourceShare $pfShare -Threads $threads -Overwrite:$overwrite
                Write-Log "Program Files copy complete"
            } else {
                Write-Log "WARNING: Program Files not found in backup"
            }
        }
        catch {
            Write-Log "ERROR copying Program Files: $($_.Exception.Message)"
        }
    }
    
    # Copy Program Files (x86)
    if ($settings.ProgramFilesX86) {
        Write-Log "Copying Program Files (x86)..."
        try {
            $pfx86Share = ""
            if ($isLocalPath) {
                if (Test-Path "$sourcePath\ProgramFilesX86") { $pfx86Share = "$sourcePath\ProgramFilesX86" }
            }
            if (-not $pfx86Share -and (-not $isLocalPath)) { $pfx86Share = "\\$SourceComputer\ProgramFilesX86$" }
            if ($pfx86Share -and (Test-Path $pfx86Share)) {
                Copy-ProgramFilesX86WithACL -SourceShare $pfx86Share -Threads $threads -Overwrite:$overwrite
                Write-Log "Program Files (x86) copy complete"
            } else {
                Write-Log "WARNING: Program Files (x86) not found in backup"
            }
        }
        catch {
            Write-Log "ERROR copying Program Files (x86): $($_.Exception.Message)"
        }
    }
    
    # Copy ProgramData
    if ($settings.ProgramData) {
        Write-Log "Copying ProgramData..."
        try {
            $pdShare = ""
            if ($isLocalPath) {
                if (Test-Path "$sourcePath\ProgramData") { $pdShare = "$sourcePath\ProgramData" }
            }
            if (-not $pdShare -and (-not $isLocalPath)) { $pdShare = "\\$SourceComputer\ProgramData$" }
            if ($pdShare -and (Test-Path $pdShare)) {
                Copy-ProgramDataWithACL -SourceShare $pdShare -Threads $threads -Overwrite:$overwrite
                Write-Log "ProgramData copy complete"
            } else {
                Write-Log "WARNING: ProgramData not found in backup"
            }
        }
        catch {
            Write-Log "ERROR copying ProgramData: $($_.Exception.Message)"
        }
    }
    
    # Copy User Data
    if ($settings.UserData) {
        Write-Log "Copying user data..."
        try {
            $usersShare = ""
            if ($isLocalPath) {
                if (Test-Path "$sourcePath\UserData") { $usersShare = "$sourcePath\UserData" }
            }
            if (-not $usersShare -and (-not $isLocalPath)) { $usersShare = "\\$SourceComputer\Users$" }
            if ($usersShare -and (Test-Path $usersShare)) {
                $usersFile = "$sourcePath\users.json"
                if (Test-Path $usersFile) {
                    $users = Get-Content $usersFile -Raw | ConvertFrom-Json
                    foreach ($user in $users) {
                        Write-Log "Copying data for: $($user.Name)"
                        Copy-UserDataWithACL -SourceShare $usersShare -Username $user.Name -Threads $threads -Overwrite:$overwrite
                    }
                }
            } else {
                Write-Log "WARNING: User Data not found in backup"
            }
        }
        catch {
            Write-Log "ERROR copying user data: $($_.Exception.Message)"
        }
    }
    
    # Import App Settings
    if ($settings.AppSettings) {
        Write-Log "Importing application settings..."
        try {
            $appResult = Import-AppSettings -SourcePath $sourcePath -Overwrite:$overwrite
            Write-Log "Imported $($appResult.Imported) app settings"
        }
        catch {
            Write-Log "ERROR importing app settings: $($_.Exception.Message)"
        }
    }
    
    # Copy Additional Drives
    if ($settings.AdditionalDrives) {
        foreach ($drive in $settings.AdditionalDrives) {
            Write-Log "Copying drive ${drive}:..."
            try {
                $driveShare = "\\$SourceComputer\${drive}Drive$"
                if (Test-Path $driveShare) {
                    Copy-AdditionalDrive -SourceShare $driveShare -DestinationDrive "${drive}:" -Threads $threads -Overwrite:$overwrite
                    Write-Log "Drive ${drive}: copy complete"
                }
            }
            catch {
                Write-Log "ERROR copying drive ${drive}: $($_.Exception.Message)"
            }
        }
    }
    
    Write-Log "=========================================="
    Write-Log "DESTINATION Import Complete!"
    Write-Log "=========================================="
    
    # Cleanup network connections
    if ($networkUsername) {
        Write-Log "Cleaning up network connections..."
        $shares = @('ABCD-Migration$', 'ProgramFiles$', 'ProgramFilesX86$', 'ProgramData$', 'Users$')
        foreach ($share in $shares) {
            net use "\\$SourceComputer\$share" /delete 2>&1 | Out-Null
        }
    }
}

Write-Log "Log saved to: $logFile"
exit 0
