#Requires -RunAsAdministrator
<#
.SYNOPSIS
    ABCD-Migration Password Monitor - Admin GUI
.DESCRIPTION
    Monitors which users have changed their passwords from the master password
#>

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

$Script:FlagPath = "C:\ABCD-Migration\PasswordFlags"
$Script:RefreshInterval = 5000 # 5 seconds

# Create main form
$form = New-Object System.Windows.Forms.Form
$form.Text = "ABCD-Migration - Password Change Monitor"
$form.Size = New-Object System.Drawing.Size(600, 450)
$form.StartPosition = "CenterScreen"
$form.FormBorderStyle = "FixedDialog"
$form.MaximizeBox = $false
$form.BackColor = [System.Drawing.Color]::White

# Header
$lblHeader = New-Object System.Windows.Forms.Label
$lblHeader.Text = "User Password Status Monitor"
$lblHeader.Location = New-Object System.Drawing.Point(20, 20)
$lblHeader.Size = New-Object System.Drawing.Size(550, 30)
$lblHeader.Font = New-Object System.Drawing.Font("Arial", 14, [System.Drawing.FontStyle]::Bold)
$lblHeader.ForeColor = [System.Drawing.Color]::DarkBlue
$form.Controls.Add($lblHeader)

# Info label
$lblInfo = New-Object System.Windows.Forms.Label
$lblInfo.Text = "This monitor shows which users have changed their password from the master password."
$lblInfo.Location = New-Object System.Drawing.Point(20, 55)
$lblInfo.Size = New-Object System.Drawing.Size(550, 40)
$lblInfo.Font = New-Object System.Drawing.Font("Arial", 9)
$form.Controls.Add($lblInfo)

# User list view
$listView = New-Object System.Windows.Forms.ListView
$listView.Location = New-Object System.Drawing.Point(20, 100)
$listView.Size = New-Object System.Drawing.Size(550, 250)
$listView.View = [System.Windows.Forms.View]::Details
$listView.FullRowSelect = $true
$listView.GridLines = $true

$listView.Columns.Add("Username", 150) | Out-Null
$listView.Columns.Add("Status", 150) | Out-Null
$listView.Columns.Add("Changed Date", 150) | Out-Null
$listView.Columns.Add("Reminder Status", 90) | Out-Null

$form.Controls.Add($listView)

# Status bar
$lblStatus = New-Object System.Windows.Forms.Label
$lblStatus.Text = "Monitoring..."
$lblStatus.Location = New-Object System.Drawing.Point(20, 360)
$lblStatus.Size = New-Object System.Drawing.Size(400, 20)
$lblStatus.ForeColor = [System.Drawing.Color]::Gray
$form.Controls.Add($lblStatus)

# Refresh button
$btnRefresh = New-Object System.Windows.Forms.Button
$btnRefresh.Text = "Refresh Now"
$btnRefresh.Location = New-Object System.Drawing.Point(430, 355)
$btnRefresh.Size = New-Object System.Drawing.Size(140, 30)
$btnRefresh.BackColor = [System.Drawing.Color]::LightBlue
$form.Controls.Add($btnRefresh)

# Close button
$btnClose = New-Object System.Windows.Forms.Button
$btnClose.Text = "Close"
$btnClose.Location = New-Object System.Drawing.Point(430, 390)
$btnClose.Size = New-Object System.Drawing.Size(140, 25)
$form.Controls.Add($btnClose)

# Timer for auto-refresh
$timer = New-Object System.Windows.Forms.Timer
$timer.Interval = $Script:RefreshInterval
$timer.Enabled = $true

# Refresh function
function Refresh-UserList {
    $listView.Items.Clear()
    
    # Get all local users (excluding built-in)
    $excludeUsers = @("Administrator", "Guest", "DefaultAccount", "WDAGUtilityAccount")
    $users = Get-LocalUser | Where-Object { 
        $_.Enabled -eq $true -and $_.Name -notin $excludeUsers
    }
    
    foreach ($user in $users) {
        $item = New-Object System.Windows.Forms.ListViewItem($user.Name)
        
        # Check if user has password change flag
        $flagFile = Join-Path $Script:FlagPath "$($user.Name).flag"
        
        if (Test-Path $flagFile) {
            try {
                $flagData = Get-Content $flagFile -Raw | ConvertFrom-Json
                $item.SubItems.Add("Changed") | Out-Null
                $item.SubItems.Add($flagData.ChangedDate) | Out-Null
                $item.SubItems.Add("Removed") | Out-Null
                $item.BackColor = [System.Drawing.Color]::LightGreen
                
                # Check if reminder script still exists
                $reminderScript = "C:\Users\$($user.Name)\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\Startup\PasswordChangeReminder.ps1"
                if (Test-Path $reminderScript) {
                    # Script should have been removed, clean it up
                    Remove-Item $reminderScript -Force -ErrorAction SilentlyContinue
                }
            } catch {
                $item.SubItems.Add("Unknown") | Out-Null
                $item.SubItems.Add("N/A") | Out-Null
                $item.SubItems.Add("Error") | Out-Null
            }
        } else {
            $item.SubItems.Add("Not Changed") | Out-Null
            $item.SubItems.Add("N/A") | Out-Null
            
            # Check if reminder script exists
            $reminderScript = "C:\Users\$($user.Name)\AppData\Roaming\Microsoft\Windows\Start Menu\Programs\Startup\PasswordChangeReminder.ps1"
            if (Test-Path $reminderScript) {
                $item.SubItems.Add("Active") | Out-Null
            } else {
                $item.SubItems.Add("Missing") | Out-Null
            }
            
            $item.BackColor = [System.Drawing.Color]::LightYellow
        }
        
        $listView.Items.Add($item) | Out-Null
    }
    
    $changedCount = ($listView.Items | Where-Object { $_.SubItems[1].Text -eq "Changed" }).Count
    $totalCount = $listView.Items.Count
    
    $lblStatus.Text = "Last updated: $(Get-Date -Format 'HH:mm:ss') | Users: $totalCount | Changed: $changedCount"
}

# Event handlers
$btnRefresh.Add_Click({ Refresh-UserList })
$btnClose.Add_Click({ $form.Close() })
$timer.Add_Tick({ Refresh-UserList })

# Initial load
Refresh-UserList

# Show form
$form.Add_Shown({ $form.Activate() })
$form.ShowDialog() | Out-Null
$timer.Dispose()
