#Requires -RunAsAdministrator
<#
.SYNOPSIS
    Application Settings Migration Module
.DESCRIPTION
    Backs up and restores AppData and Registry settings for 100+ common applications
    With warnings for apps that may cause issues when copied
#>

# Common applications with their AppData and Registry paths
# Warning levels: 0=Safe, 1=Caution (may need reconfig), 2=Risky (likely issues), 3=Exclude (don't copy)
$Script:CommonApps = @(
    # Browsers - Generally safe
    @{Name="Google Chrome"; Category="Browsers"; AppData=@("Local\Google\Chrome"); Registry=@("HKCU:\Software\Google\Chrome"); Warning=0; Note="Safe to migrate"}
    @{Name="Mozilla Firefox"; Category="Browsers"; AppData=@("Roaming\Mozilla\Firefox", "Local\Mozilla\Firefox"); Registry=@("HKCU:\Software\Mozilla"); Warning=0; Note="Safe to migrate"}
    @{Name="Microsoft Edge"; Category="Browsers"; AppData=@("Local\Microsoft\Edge"); Registry=@("HKCU:\Software\Microsoft\Edge"); Warning=1; Note="May need to re-sync"}
    @{Name="Brave"; Category="Browsers"; AppData=@("Local\BraveSoftware\Brave-Browser"); Registry=@("HKCU:\Software\BraveSoftware"); Warning=0; Note="Safe to migrate"}
    @{Name="Opera"; Category="Browsers"; AppData=@("Roaming\Opera Software", "Local\Opera Software"); Registry=@("HKCU:\Software\Opera Software"); Warning=0; Note="Safe to migrate"}
    @{Name="Vivaldi"; Category="Browsers"; AppData=@("Local\Vivaldi"); Registry=@("HKCU:\Software\Vivaldi"); Warning=0; Note="Safe to migrate"}
    
    # Communication - Mixed
    @{Name="Microsoft Teams"; Category="Communication"; AppData=@("Roaming\Microsoft\Teams", "Local\Microsoft\Teams"); Registry=@("HKCU:\Software\Microsoft\Teams"); Warning=1; Note="May need re-login"}
    @{Name="Slack"; Category="Communication"; AppData=@("Roaming\Slack", "Local\Slack"); Registry=@("HKCU:\Software\Slack"); Warning=1; Note="May need re-login"}
    @{Name="Discord"; Category="Communication"; AppData=@("Roaming\discord", "Local\Discord"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Zoom"; Category="Communication"; AppData=@("Roaming\Zoom"); Registry=@("HKCU:\Software\Zoom"); Warning=1; Note="May need re-login"}
    @{Name="Skype"; Category="Communication"; AppData=@("Roaming\Skype", "Local\Packages\Microsoft.SkypeApp*"); Registry=@("HKCU:\Software\Skype"); Warning=1; Note="UWP app may have issues"}
    @{Name="Telegram"; Category="Communication"; AppData=@("Roaming\Telegram Desktop"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="WhatsApp"; Category="Communication"; AppData=@("Local\WhatsApp"); Registry=@(); Warning=1; Note="May need re-verification"}
    @{Name="Signal"; Category="Communication"; AppData=@("Roaming\Signal"); Registry=@(); Warning=0; Note="Safe to migrate"}
    
    # Email Clients - Caution
    @{Name="Microsoft Outlook"; Category="Email"; AppData=@("Roaming\Microsoft\Outlook", "Local\Microsoft\Outlook"); Registry=@("HKCU:\Software\Microsoft\Office\*\Outlook"); Warning=2; Note="PST files OK, profiles may break"}
    @{Name="Mozilla Thunderbird"; Category="Email"; AppData=@("Roaming\Thunderbird", "Local\Thunderbird"); Registry=@("HKCU:\Software\Mozilla\Thunderbird"); Warning=0; Note="Safe to migrate"}
    @{Name="eM Client"; Category="Email"; AppData=@("Roaming\eM Client"); Registry=@("HKCU:\Software\eM Client"); Warning=1; Note="May need license reactivation"}
    @{Name="Mailbird"; Category="Email"; AppData=@("Local\Mailbird"); Registry=@("HKCU:\Software\Mailbird"); Warning=1; Note="May need license reactivation"}
    
    # Office & Productivity - Mixed
    @{Name="Microsoft Office"; Category="Office"; AppData=@("Roaming\Microsoft\Office", "Local\Microsoft\Office"); Registry=@("HKCU:\Software\Microsoft\Office"); Warning=2; Note="Settings only - reactivation needed"}
    @{Name="LibreOffice"; Category="Office"; AppData=@("Roaming\LibreOffice"); Registry=@("HKCU:\Software\LibreOffice"); Warning=0; Note="Safe to migrate"}
    @{Name="Notepad++"; Category="Office"; AppData=@("Roaming\Notepad++"); Registry=@("HKCU:\Software\Notepad++"); Warning=0; Note="Safe to migrate"}
    @{Name="Sublime Text"; Category="Office"; AppData=@("Roaming\Sublime Text", "Roaming\Sublime Text 3"); Registry=@("HKCU:\Software\Sublime Text"); Warning=1; Note="May need license"}
    @{Name="Visual Studio Code"; Category="Office"; AppData=@("Roaming\Code"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Obsidian"; Category="Office"; AppData=@("Roaming\obsidian"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Notion"; Category="Office"; AppData=@("Roaming\Notion", "Local\Notion"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Evernote"; Category="Office"; AppData=@("Local\Evernote", "Roaming\Evernote"); Registry=@("HKCU:\Software\Evernote"); Warning=1; Note="May need re-sync"}
    
    # Development - Generally safe
    @{Name="Visual Studio"; Category="Development"; AppData=@("Local\Microsoft\VisualStudio"); Registry=@("HKCU:\Software\Microsoft\VisualStudio"); Warning=1; Note="Extensions may need reinstall"}
    @{Name="JetBrains IDEs"; Category="Development"; AppData=@("Roaming\JetBrains", "Local\JetBrains"); Registry=@("HKCU:\Software\JetBrains"); Warning=1; Note="May need license"}
    @{Name="Git"; Category="Development"; AppData=@("Roaming\Git"); Registry=@("HKCU:\Software\Git"); Warning=0; Note="Safe to migrate"}
    @{Name="GitHub Desktop"; Category="Development"; AppData=@("Roaming\GitHub Desktop", "Local\GitHubDesktop"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="SourceTree"; Category="Development"; AppData=@("Local\Atlassian\SourceTree"); Registry=@("HKCU:\Software\Atlassian\SourceTree"); Warning=0; Note="Safe to migrate"}
    @{Name="Node.js/npm"; Category="Development"; AppData=@("Roaming\npm", "Local\npm-cache"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Python"; Category="Development"; AppData=@("Local\pip", "Roaming\Python"); Registry=@("HKCU:\Software\Python"); Warning=0; Note="Safe to migrate"}
    @{Name="Docker Desktop"; Category="Development"; AppData=@("Roaming\Docker", "Local\Docker"); Registry=@("HKCU:\Software\Docker"); Warning=2; Note="May need reinstall"}
    @{Name="Postman"; Category="Development"; AppData=@("Roaming\Postman"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="HeidiSQL"; Category="Development"; AppData=@("Roaming\HeidiSQL"); Registry=@("HKCU:\Software\HeidiSQL"); Warning=0; Note="Safe to migrate"}
    @{Name="DBeaver"; Category="Development"; AppData=@("Roaming\DBeaverData"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="WinSCP"; Category="Development"; AppData=@("Roaming\WinSCP"); Registry=@("HKCU:\Software\Martin Prikryl\WinSCP 2"); Warning=0; Note="Safe to migrate"}
    @{Name="PuTTY"; Category="Development"; AppData=@(); Registry=@("HKCU:\Software\SimonTatham\PuTTY"); Warning=0; Note="Safe to migrate"}
    @{Name="FileZilla"; Category="Development"; AppData=@("Roaming\FileZilla"); Registry=@(); Warning=0; Note="Safe to migrate"}
    
    # Graphics & Design - License issues common
    @{Name="Adobe Creative Cloud"; Category="Graphics"; AppData=@("Roaming\Adobe", "Local\Adobe"); Registry=@("HKCU:\Software\Adobe"); Warning=2; Note="Requires reactivation"}
    @{Name="Adobe Photoshop"; Category="Graphics"; AppData=@("Roaming\Adobe\Adobe Photoshop*"); Registry=@("HKCU:\Software\Adobe\Photoshop"); Warning=2; Note="Requires reactivation"}
    @{Name="Adobe Illustrator"; Category="Graphics"; AppData=@("Roaming\Adobe\Adobe Illustrator*"); Registry=@("HKCU:\Software\Adobe\Illustrator"); Warning=2; Note="Requires reactivation"}
    @{Name="Adobe Premiere Pro"; Category="Graphics"; AppData=@("Roaming\Adobe\Premiere Pro*"); Registry=@("HKCU:\Software\Adobe\Premiere Pro"); Warning=2; Note="Requires reactivation"}
    @{Name="Adobe Acrobat"; Category="Graphics"; AppData=@("Roaming\Adobe\Acrobat"); Registry=@("HKCU:\Software\Adobe\Acrobat Reader"); Warning=1; Note="May need license"}
    @{Name="GIMP"; Category="Graphics"; AppData=@("Roaming\GIMP"); Registry=@("HKCU:\Software\GIMP"); Warning=0; Note="Safe to migrate"}
    @{Name="Inkscape"; Category="Graphics"; AppData=@("Roaming\inkscape"); Registry=@("HKCU:\Software\Inkscape"); Warning=0; Note="Safe to migrate"}
    @{Name="Blender"; Category="Graphics"; AppData=@("Roaming\Blender Foundation"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Paint.NET"; Category="Graphics"; AppData=@("Local\paint.net"); Registry=@("HKCU:\Software\paint.net"); Warning=0; Note="Safe to migrate"}
    @{Name="CorelDRAW"; Category="Graphics"; AppData=@("Roaming\Corel"); Registry=@("HKCU:\Software\Corel"); Warning=2; Note="Requires reactivation"}
    @{Name="Figma"; Category="Graphics"; AppData=@("Roaming\Figma", "Local\Figma"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="AutoCAD"; Category="Graphics"; AppData=@("Roaming\Autodesk\AutoCAD*", "Local\Autodesk"); Registry=@("HKCU:\Software\Autodesk"); Warning=2; Note="Requires reactivation"}
    @{Name="SketchUp"; Category="Graphics"; AppData=@("Local\SketchUp"); Registry=@("HKCU:\Software\SketchUp"); Warning=1; Note="May need license"}
    
    # Media Players - Safe
    @{Name="VLC"; Category="Media"; AppData=@("Roaming\vlc"); Registry=@("HKCU:\Software\VideoLAN"); Warning=0; Note="Safe to migrate"}
    @{Name="Spotify"; Category="Media"; AppData=@("Roaming\Spotify", "Local\Spotify"); Registry=@("HKCU:\Software\Spotify"); Warning=0; Note="Re-login needed"}
    @{Name="iTunes"; Category="Media"; AppData=@("Roaming\Apple Computer\iTunes", "Local\Apple Computer"); Registry=@("HKCU:\Software\Apple Computer"); Warning=1; Note="Library paths may break"}
    @{Name="foobar2000"; Category="Media"; AppData=@("Roaming\foobar2000"); Registry=@("HKCU:\Software\foobar2000"); Warning=0; Note="Safe to migrate"}
    @{Name="Media Player Classic"; Category="Media"; AppData=@("Roaming\MPC-HC"); Registry=@("HKCU:\Software\MPC-HC"); Warning=0; Note="Safe to migrate"}
    @{Name="Plex"; Category="Media"; AppData=@("Local\Plex Media Server"); Registry=@("HKCU:\Software\Plex"); Warning=1; Note="Server needs reconfiguration"}
    @{Name="Kodi"; Category="Media"; AppData=@("Roaming\Kodi"); Registry=@(); Warning=0; Note="Safe to migrate"}
    
    # Gaming - Anti-cheat issues
    @{Name="Steam"; Category="Gaming"; AppData=@("Local\Steam"); Registry=@("HKCU:\Software\Valve\Steam"); Warning=1; Note="Game files need separate copy"}
    @{Name="Epic Games"; Category="Gaming"; AppData=@("Local\EpicGamesLauncher", "Local\Epic Games"); Registry=@("HKCU:\Software\Epic Games"); Warning=1; Note="Settings only"}
    @{Name="GOG Galaxy"; Category="Gaming"; AppData=@("Local\GOG.com"); Registry=@("HKCU:\Software\GOG.com"); Warning=0; Note="Safe to migrate"}
    @{Name="Origin/EA App"; Category="Gaming"; AppData=@("Roaming\Origin", "Local\Origin", "Local\Electronic Arts"); Registry=@("HKCU:\Software\Electronic Arts"); Warning=1; Note="May need re-login"}
    @{Name="Ubisoft Connect"; Category="Gaming"; AppData=@("Local\Ubisoft Game Launcher"); Registry=@("HKCU:\Software\Ubisoft"); Warning=1; Note="May need re-login"}
    @{Name="Battle.net"; Category="Gaming"; AppData=@("Local\Battle.net", "Local\Blizzard Entertainment"); Registry=@("HKCU:\Software\Blizzard Entertainment"); Warning=1; Note="May need re-login"}
    @{Name="Xbox"; Category="Gaming"; AppData=@("Local\Packages\Microsoft.XboxApp*"); Registry=@(); Warning=2; Note="UWP app - complex migration"}
    @{Name="NVIDIA GeForce"; Category="Gaming"; AppData=@("Local\NVIDIA Corporation"); Registry=@("HKCU:\Software\NVIDIA Corporation"); Warning=1; Note="Driver dependent"}
    @{Name="AMD Radeon"; Category="Gaming"; AppData=@("Local\AMD"); Registry=@("HKCU:\Software\AMD"); Warning=1; Note="Driver dependent"}
    
    # Utilities - Mostly safe
    @{Name="7-Zip"; Category="Utilities"; AppData=@(); Registry=@("HKCU:\Software\7-Zip"); Warning=0; Note="Safe to migrate"}
    @{Name="WinRAR"; Category="Utilities"; AppData=@("Roaming\WinRAR"); Registry=@("HKCU:\Software\WinRAR"); Warning=0; Note="Safe to migrate"}
    @{Name="Everything Search"; Category="Utilities"; AppData=@("Roaming\Everything"); Registry=@("HKCU:\Software\voidtools\Everything"); Warning=0; Note="Safe to migrate"}
    @{Name="ShareX"; Category="Utilities"; AppData=@("Roaming\ShareX"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Greenshot"; Category="Utilities"; AppData=@("Roaming\Greenshot"); Registry=@("HKCU:\Software\Greenshot"); Warning=0; Note="Safe to migrate"}
    @{Name="OBS Studio"; Category="Utilities"; AppData=@("Roaming\obs-studio"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Handbrake"; Category="Utilities"; AppData=@("Roaming\HandBrake"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="qBittorrent"; Category="Utilities"; AppData=@("Roaming\qBittorrent", "Local\qBittorrent"); Registry=@("HKCU:\Software\qBittorrent"); Warning=0; Note="Safe to migrate"}
    @{Name="TeamViewer"; Category="Utilities"; AppData=@("Roaming\TeamViewer"); Registry=@("HKCU:\Software\TeamViewer"); Warning=1; Note="ID changes on new hardware"}
    @{Name="AnyDesk"; Category="Utilities"; AppData=@("Roaming\AnyDesk"); Registry=@("HKCU:\Software\AnyDesk"); Warning=1; Note="ID changes on new hardware"}
    @{Name="KeePass"; Category="Utilities"; AppData=@("Roaming\KeePass"); Registry=@("HKCU:\Software\KeePass"); Warning=0; Note="Safe to migrate"}
    @{Name="Bitwarden"; Category="Utilities"; AppData=@("Roaming\Bitwarden"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="1Password"; Category="Utilities"; AppData=@("Local\1Password"); Registry=@("HKCU:\Software\1Password"); Warning=1; Note="May need re-login"}
    @{Name="PowerToys"; Category="Utilities"; AppData=@("Local\Microsoft\PowerToys"); Registry=@(); Warning=0; Note="Safe to migrate"}
    @{Name="Snagit"; Category="Utilities"; AppData=@("Local\TechSmith\Snagit"); Registry=@("HKCU:\Software\TechSmith\Snagit"); Warning=1; Note="May need license"}
    @{Name="Camtasia"; Category="Utilities"; AppData=@("Local\TechSmith\Camtasia*"); Registry=@("HKCU:\Software\TechSmith\Camtasia"); Warning=1; Note="May need license"}
    
    # Cloud Storage - Re-sync needed
    @{Name="OneDrive"; Category="Cloud"; AppData=@("Local\Microsoft\OneDrive"); Registry=@("HKCU:\Software\Microsoft\OneDrive"); Warning=1; Note="Needs re-sync"}
    @{Name="Google Drive"; Category="Cloud"; AppData=@("Local\Google\DriveFS"); Registry=@("HKCU:\Software\Google\DriveFS"); Warning=1; Note="Needs re-sync"}
    @{Name="Dropbox"; Category="Cloud"; AppData=@("Roaming\Dropbox", "Local\Dropbox"); Registry=@("HKCU:\Software\Dropbox"); Warning=1; Note="Needs re-sync"}
    @{Name="Box"; Category="Cloud"; AppData=@("Local\Box"); Registry=@("HKCU:\Software\Box"); Warning=1; Note="Needs re-sync"}
    @{Name="iCloud"; Category="Cloud"; AppData=@("Local\Apple Inc\iCloud"); Registry=@("HKCU:\Software\Apple Inc.\iCloud"); Warning=1; Note="Needs re-sync"}
    
    # Business/Accounting - License sensitive
    @{Name="QuickBooks"; Category="Business"; AppData=@("Local\Intuit\QuickBooks*"); Registry=@("HKCU:\Software\Intuit\QuickBooks"); Warning=2; Note="Requires reactivation"}
    @{Name="Sage"; Category="Business"; AppData=@("Local\Sage", "Roaming\Sage"); Registry=@("HKCU:\Software\Sage"); Warning=2; Note="Requires reactivation"}
    
    # Security - DO NOT MIGRATE
    @{Name="Malwarebytes"; Category="Security"; AppData=@("Local\Malwarebytes"); Registry=@("HKCU:\Software\Malwarebytes"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="Norton"; Category="Security"; AppData=@("Local\Norton"); Registry=@("HKCU:\Software\Norton"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="Kaspersky"; Category="Security"; AppData=@("Local\Kaspersky Lab"); Registry=@("HKCU:\Software\KasperskyLab"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="Bitdefender"; Category="Security"; AppData=@("Roaming\Bitdefender"); Registry=@("HKCU:\Software\Bitdefender"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="ESET"; Category="Security"; AppData=@("Roaming\ESET"); Registry=@("HKCU:\Software\ESET"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="Avast"; Category="Security"; AppData=@("Local\AVAST Software"); Registry=@("HKCU:\Software\AVAST Software"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="AVG"; Category="Security"; AppData=@("Local\AVG"); Registry=@("HKCU:\Software\AVG"); Warning=3; Note="DO NOT MIGRATE - reinstall"}
    @{Name="Windows Defender"; Category="Security"; AppData=@(); Registry=@(); Warning=3; Note="System component - DO NOT MIGRATE"}
    
    # VPN - Usually OK
    @{Name="NordVPN"; Category="VPN"; AppData=@("Local\NordVPN"); Registry=@("HKCU:\Software\NordVPN"); Warning=0; Note="Re-login needed"}
    @{Name="ExpressVPN"; Category="VPN"; AppData=@("Local\ExpressVPN"); Registry=@(); Warning=0; Note="Re-login needed"}
    @{Name="ProtonVPN"; Category="VPN"; AppData=@("Local\ProtonVPN"); Registry=@(); Warning=0; Note="Re-login needed"}
    @{Name="OpenVPN"; Category="VPN"; AppData=@("Roaming\OpenVPN Connect"); Registry=@("HKCU:\Software\OpenVPN"); Warning=0; Note="Profiles may need update"}
    
    # Remote Desktop
    @{Name="Remote Desktop"; Category="Remote"; AppData=@("Local\Microsoft\Terminal Server Client"); Registry=@("HKCU:\Software\Microsoft\Terminal Server Client"); Warning=0; Note="Safe to migrate"}
    @{Name="mRemoteNG"; Category="Remote"; AppData=@("Roaming\mRemoteNG"); Registry=@(); Warning=0; Note="Safe to migrate"}
)

# Selected apps for migration (updated by GUI)
$Script:SelectedApps = @{}

function Export-AppSettings {
    param(
        [Parameter(Mandatory=$true)]
        [string]$BackupPath,
        
        [string]$Username = $null,
        
        [switch]$AllUsers,
        
        [array]$SelectedUsers = $null,
        
        [hashtable]$SelectedApps = $null
    )
    
    Write-Host "  [AppSettings] ================================================" -ForegroundColor Cyan
    Write-Host "  [AppSettings] Exporting Application Settings" -ForegroundColor Cyan
    Write-Host "  [AppSettings] ================================================" -ForegroundColor Cyan
    
    $appSettingsPath = Join-Path $BackupPath "AppSettings"
    if (!(Test-Path $appSettingsPath)) {
        New-Item -Path $appSettingsPath -ItemType Directory -Force | Out-Null
    }
    
    # Get users to process
    $usersToProcess = @()
    if ($SelectedUsers -and $SelectedUsers.Count -gt 0) {
        # Use the specific list of selected users
        $usersToProcess = $SelectedUsers
        Write-Host "  [AppSettings] Processing $($usersToProcess.Count) selected users" -ForegroundColor Yellow
    } elseif ($AllUsers) {
        $userFolders = Get-ChildItem "C:\Users" -Directory | Where-Object {
            $_.Name -notin @("Public", "Default", "Default User", "All Users")
        }
        $usersToProcess = $userFolders.Name
        Write-Host "  [AppSettings] Processing ALL users ($($usersToProcess.Count) profiles)" -ForegroundColor Yellow
    } elseif ($Username) {
        $usersToProcess = @($Username)
    } else {
        $usersToProcess = @($env:USERNAME)
    }
    
    # Filter apps based on selection
    $appsToExport = $Script:CommonApps
    if ($SelectedApps -and $SelectedApps.Count -gt 0) {
        $appsToExport = $Script:CommonApps | Where-Object { 
            $SelectedApps.ContainsKey($_.Name) -and $SelectedApps[$_.Name] -eq $true
        }
        Write-Host "  [AppSettings] Exporting $($appsToExport.Count) selected apps" -ForegroundColor Cyan
    } else {
        # Default: exclude Warning level 3 (security software)
        $appsToExport = $Script:CommonApps | Where-Object { $_.Warning -lt 3 }
        Write-Host "  [AppSettings] Exporting all safe apps ($($appsToExport.Count) apps)" -ForegroundColor Cyan
    }
    
    $exportedApps = @()
    $totalApps = 0
    
    foreach ($user in $usersToProcess) {
        Write-Host "  [AppSettings] Processing user: $user" -ForegroundColor Yellow
        
        $userBackupPath = Join-Path $appSettingsPath $user
        if (!(Test-Path $userBackupPath)) {
            New-Item -Path $userBackupPath -ItemType Directory -Force | Out-Null
        }
        
        $userAppDataPath = "C:\Users\$user\AppData"
        
        foreach ($app in $appsToExport) {
            $appFound = $false
            $appBackupPath = Join-Path $userBackupPath ($app.Name -replace '[\\/:*?"<>|]', '_')
            
            # Check and backup AppData folders
            foreach ($appDataFolder in $app.AppData) {
                $sourcePath = Join-Path $userAppDataPath $appDataFolder
                
                # Handle wildcards
                if ($appDataFolder -match '\*') {
                    $parentPath = Split-Path (Join-Path $userAppDataPath $appDataFolder) -Parent
                    $pattern = Split-Path $appDataFolder -Leaf
                    if (Test-Path $parentPath) {
                        $matchingFolders = Get-ChildItem $parentPath -Directory -Filter $pattern -ErrorAction SilentlyContinue
                        foreach ($folder in $matchingFolders) {
                            $appFound = $true
                            $destPath = Join-Path $appBackupPath "AppData\$($folder.Name)"
                            if (!(Test-Path (Split-Path $destPath -Parent))) {
                                New-Item -Path (Split-Path $destPath -Parent) -ItemType Directory -Force | Out-Null
                            }
                            try {
                                Copy-Item $folder.FullName $destPath -Recurse -Force -ErrorAction SilentlyContinue
                            } catch {}
                        }
                    }
                } elseif (Test-Path $sourcePath) {
                    $appFound = $true
                    $destPath = Join-Path $appBackupPath "AppData\$appDataFolder"
                    if (!(Test-Path (Split-Path $destPath -Parent))) {
                        New-Item -Path (Split-Path $destPath -Parent) -ItemType Directory -Force | Out-Null
                    }
                    try {
                        Copy-Item $sourcePath $destPath -Recurse -Force -ErrorAction SilentlyContinue
                    } catch {}
                }
            }
            
            # Check and backup Registry keys
            foreach ($regPath in $app.Registry) {
                # Convert HKCU to user-specific path
                $userRegPath = $regPath -replace 'HKCU:', "Registry::HKEY_USERS\$user"
                
                # For current user, use HKCU directly
                if ($user -eq $env:USERNAME) {
                    $userRegPath = $regPath
                }
                
                # Handle wildcards in registry
                if ($regPath -match '\*') {
                    $parentPath = Split-Path $regPath -Parent
                    if (Test-Path $parentPath -ErrorAction SilentlyContinue) {
                        $appFound = $true
                        $regBackupFile = Join-Path $appBackupPath "Registry\$(($regPath -replace '[:*]', '_') -replace '\\', '-').reg"
                        if (!(Test-Path (Split-Path $regBackupFile -Parent))) {
                            New-Item -Path (Split-Path $regBackupFile -Parent) -ItemType Directory -Force | Out-Null
                        }
                        try {
                            $regExportPath = $regPath -replace 'HKCU:\\', 'HKEY_CURRENT_USER\'
                            reg export $regExportPath $regBackupFile /y 2>&1 | Out-Null
                        } catch {}
                    }
                } elseif ($regPath -and (Test-Path $regPath -ErrorAction SilentlyContinue)) {
                    $appFound = $true
                    $regBackupFile = Join-Path $appBackupPath "Registry\$(($regPath -replace '[:*]', '_') -replace '\\', '-').reg"
                    if (!(Test-Path (Split-Path $regBackupFile -Parent))) {
                        New-Item -Path (Split-Path $regBackupFile -Parent) -ItemType Directory -Force | Out-Null
                    }
                    try {
                        $regExportPath = $regPath -replace 'HKCU:\\', 'HKEY_CURRENT_USER\'
                        reg export $regExportPath $regBackupFile /y 2>&1 | Out-Null
                    } catch {}
                }
            }
            
            if ($appFound) {
                Write-Host "  [AppSettings]   Found: $($app.Name)" -ForegroundColor Green
                $exportedApps += @{User=$user; App=$app.Name}
                $totalApps++
            }
        }
    }
    
    # Save manifest
    $manifest = @{
        ExportDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        Users = $usersToProcess
        AppsFound = $exportedApps
        TotalApps = $totalApps
    }
    $manifest | ConvertTo-Json -Depth 3 | Out-File "$appSettingsPath\manifest.json" -Encoding UTF8
    
    Write-Host "  [AppSettings] Export complete - $totalApps app settings backed up" -ForegroundColor Green
    return @{Success=$true; Count=$totalApps; Apps=$exportedApps}
}

function Import-AppSettings {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourcePath,
        
        [switch]$Overwrite,
        
        [array]$SelectedUsers = $null
    )
    
    Write-Host "  [AppSettings] ================================================" -ForegroundColor Cyan
    Write-Host "  [AppSettings] Importing Application Settings" -ForegroundColor Cyan
    Write-Host "  [AppSettings] ================================================" -ForegroundColor Cyan
    
    $appSettingsPath = Join-Path $SourcePath "AppSettings"
    
    if (!(Test-Path $appSettingsPath)) {
        Write-Host "  [AppSettings] No app settings backup found" -ForegroundColor Yellow
        return @{Success=$false; Count=0}
    }
    
    $manifestFile = Join-Path $appSettingsPath "manifest.json"
    if (Test-Path $manifestFile) {
        $manifest = Get-Content $manifestFile -Raw | ConvertFrom-Json
        Write-Host "  [AppSettings] Backup from: $($manifest.ExportDate)" -ForegroundColor Gray
    }
    
    $importedCount = 0
    $skippedCount = 0
    
    # Get user folders in backup
    $userBackups = Get-ChildItem $appSettingsPath -Directory -ErrorAction SilentlyContinue
    
    # Filter to selected users if specified
    if ($SelectedUsers -and $SelectedUsers.Count -gt 0) {
        $userBackups = $userBackups | Where-Object { $_.Name -in $SelectedUsers }
        Write-Host "  [AppSettings] Importing for $($userBackups.Count) selected users" -ForegroundColor Yellow
    } else {
        Write-Host "  [AppSettings] Importing for all $($userBackups.Count) backed-up users" -ForegroundColor Yellow
    }
    
    foreach ($userBackup in $userBackups) {
        $username = $userBackup.Name
        Write-Host "  [AppSettings] Restoring for user: $username" -ForegroundColor Yellow
        
        $targetAppData = "C:\Users\$username\AppData"
        if (!(Test-Path $targetAppData)) {
            Write-Host "  [AppSettings]   User profile not found, skipping" -ForegroundColor Gray
            continue
        }
        
        # Get app backups for this user
        $appBackups = Get-ChildItem $userBackup.FullName -Directory -ErrorAction SilentlyContinue
        
        foreach ($appBackup in $appBackups) {
            # Restore AppData
            $appDataBackup = Join-Path $appBackup.FullName "AppData"
            if (Test-Path $appDataBackup) {
                $folders = Get-ChildItem $appDataBackup -Directory -Recurse -ErrorAction SilentlyContinue | Where-Object { $_.PSIsContainer }
                
                foreach ($folder in (Get-ChildItem $appDataBackup -Directory -ErrorAction SilentlyContinue)) {
                    $destPath = Join-Path $targetAppData $folder.Name
                    
                    if ((Test-Path $destPath) -and -not $Overwrite) {
                        Write-Host "  [AppSettings]   Skipped (exists): $($appBackup.Name)" -ForegroundColor Gray
                        $skippedCount++
                        continue
                    }
                    
                    try {
                        Copy-Item $folder.FullName $destPath -Recurse -Force -ErrorAction SilentlyContinue
                        Write-Host "  [AppSettings]   Restored: $($appBackup.Name)" -ForegroundColor Green
                        $importedCount++
                    } catch {
                        Write-Host "  [AppSettings]   Failed: $($appBackup.Name)" -ForegroundColor Red
                    }
                }
            }
            
            # Restore Registry
            $regBackup = Join-Path $appBackup.FullName "Registry"
            if (Test-Path $regBackup) {
                $regFiles = Get-ChildItem $regBackup -Filter "*.reg" -ErrorAction SilentlyContinue
                foreach ($regFile in $regFiles) {
                    try {
                        # Import silently
                        $result = Start-Process "reg.exe" -ArgumentList "import `"$($regFile.FullName)`"" -Wait -PassThru -NoNewWindow -WindowStyle Hidden
                        if ($result.ExitCode -eq 0) {
                            Write-Host "  [AppSettings]   Registry restored: $($regFile.BaseName)" -ForegroundColor Green
                        }
                    } catch {}
                }
            }
        }
    }
    
    Write-Host "  [AppSettings] Import complete - $importedCount restored, $skippedCount skipped" -ForegroundColor Green
    return @{Success=$true; Imported=$importedCount; Skipped=$skippedCount}
}

function Get-DetectedApps {
    <#
    .SYNOPSIS
        Returns a list of detected applications on the system
    #>
    $detected = @()
    
    foreach ($app in $Script:CommonApps) {
        $found = $false
        
        # Check AppData
        foreach ($appDataFolder in $app.AppData) {
            $checkPath = "C:\Users\$env:USERNAME\AppData\$appDataFolder"
            if ($checkPath -match '\*') {
                $parentPath = Split-Path $checkPath -Parent
                $pattern = Split-Path $checkPath -Leaf
                if ((Test-Path $parentPath) -and (Get-ChildItem $parentPath -Directory -Filter $pattern -ErrorAction SilentlyContinue)) {
                    $found = $true
                    break
                }
            } elseif (Test-Path $checkPath) {
                $found = $true
                break
            }
        }
        
        # Check Registry
        if (-not $found) {
            foreach ($regPath in $app.Registry) {
                if ($regPath -and (Test-Path $regPath -ErrorAction SilentlyContinue)) {
                    $found = $true
                    break
                }
            }
        }
        
        if ($found) {
            $detected += $app.Name
        }
    }
    
    return $detected
}
