<#
.SYNOPSIS
    ABC&D Systems - Authentication & Licensing Module
.DESCRIPTION
    Provides authentication, license validation, and API integration
.VERSION
    4.0
#>

#region Configuration

$Script:API_BASE = "https://abcdtoolset.com/api"

$Script:AuthConfig = @{
    RegistryPath = "HKLM:\SOFTWARE\ABC&D Systems\Migration Tool"
    Tiers = @{
        "trial" = @{ Days = 10; Features = @("basic_migration") }
        "single_monthly" = @{ Days = 30; Features = @("basic_migration", "scheduled_backup") }
        "unlimited_monthly" = @{ Days = 30; Features = @("all") }
    }
}

#endregion

#region Machine ID

function Get-MachineIdentifier {
    $parts = @()
    try {
        $cs = Get-CimInstance -ClassName Win32_ComputerSystemProduct -ErrorAction SilentlyContinue
        if ($cs.UUID) { $parts += $cs.UUID }
    } catch {}
    try {
        $bios = Get-CimInstance -ClassName Win32_BIOS -ErrorAction SilentlyContinue
        if ($bios.SerialNumber) { $parts += $bios.SerialNumber }
    } catch {}
    $parts += $env:COMPUTERNAME
    $combined = $parts -join "|"
    $bytes = [System.Text.Encoding]::UTF8.GetBytes($combined)
    $hash = [System.Security.Cryptography.SHA256]::Create().ComputeHash($bytes)
    return [Convert]::ToBase64String($hash).Substring(0, 32)
}

#endregion

#region Registry Functions

function Get-LicenseFromRegistry {
    $regPath = $Script:AuthConfig.RegistryPath
    if (!(Test-Path $regPath)) { return $null }
    
    try {
        return @{
            LicenseKey = (Get-ItemProperty -Path $regPath -Name "LicenseKey" -ErrorAction SilentlyContinue).LicenseKey
            Token = (Get-ItemProperty -Path $regPath -Name "LicenseToken" -ErrorAction SilentlyContinue).LicenseToken
            Expiry = (Get-ItemProperty -Path $regPath -Name "LicenseExpiry" -ErrorAction SilentlyContinue).LicenseExpiry
            Tier = (Get-ItemProperty -Path $regPath -Name "LicenseTier" -ErrorAction SilentlyContinue).LicenseTier
            UserEmail = (Get-ItemProperty -Path $regPath -Name "UserEmail" -ErrorAction SilentlyContinue).UserEmail
            UserName = (Get-ItemProperty -Path $regPath -Name "UserName" -ErrorAction SilentlyContinue).UserName
            Status = (Get-ItemProperty -Path $regPath -Name "Status" -ErrorAction SilentlyContinue).Status
            MachineId = (Get-ItemProperty -Path $regPath -Name "MachineId" -ErrorAction SilentlyContinue).MachineId
        }
    } catch {
        return $null
    }
}

function Save-LicenseToRegistry {
    param(
        [string]$LicenseKey,
        [string]$Token,
        [string]$Expiry,
        [string]$Tier,
        [string]$UserEmail,
        [string]$UserName,
        [string]$Status
    )
    
    $regPath = $Script:AuthConfig.RegistryPath
    
    if (!(Test-Path $regPath)) {
        New-Item -Path $regPath -Force | Out-Null
    }
    
    Set-ItemProperty -Path $regPath -Name "LicenseKey" -Value $LicenseKey
    Set-ItemProperty -Path $regPath -Name "LicenseToken" -Value $Token
    Set-ItemProperty -Path $regPath -Name "LicenseExpiry" -Value $Expiry
    Set-ItemProperty -Path $regPath -Name "LicenseTier" -Value $Tier
    Set-ItemProperty -Path $regPath -Name "UserEmail" -Value $UserEmail
    Set-ItemProperty -Path $regPath -Name "UserName" -Value $UserName
    Set-ItemProperty -Path $regPath -Name "Status" -Value $Status
    Set-ItemProperty -Path $regPath -Name "MachineId" -Value (Get-MachineIdentifier)
    Set-ItemProperty -Path $regPath -Name "ActivatedAt" -Value (Get-Date -Format "yyyy-MM-dd HH:mm:ss")
}

function Clear-LicenseFromRegistry {
    $regPath = $Script:AuthConfig.RegistryPath
    if (Test-Path $regPath) {
        Remove-Item -Path $regPath -Recurse -Force -ErrorAction SilentlyContinue
    }
}

#endregion

#region API Functions

function Invoke-LicenseAPI {
    param(
        [string]$Endpoint,
        [hashtable]$Body
    )
    
    try {
        $json = $Body | ConvertTo-Json -Depth 10
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/$Endpoint" -Method POST -Body $json -ContentType "application/json" -TimeoutSec 30
        return $response
    } catch {
        Write-Host "API Error: $_" -ForegroundColor Red
        return @{ success = $false; error = $_.Exception.Message }
    }
}

function Test-LicenseOnline {
    param([string]$LicenseKey, [string]$Token)
    
    $body = @{
        licenseKey = $LicenseKey
        token = $Token
        machineId = Get-MachineIdentifier
    }
    
    return Invoke-LicenseAPI -Endpoint "auth.php?action=validate" -Body $body
}

function Invoke-Login {
    param([string]$Email, [string]$Password)
    
    $body = @{
        email = $Email
        password = $Password
        machineId = Get-MachineIdentifier
    }
    
    return Invoke-LicenseAPI -Endpoint "auth.php?action=login" -Body $body
}

function Invoke-LicenseKeyActivation {
    param([string]$LicenseKey)
    
    $body = @{
        licenseKey = $LicenseKey
        machineId = Get-MachineIdentifier
    }
    
    return Invoke-LicenseAPI -Endpoint "auth.php?action=validate" -Body $body
}

#endregion

#region License Validation

function Test-LicenseValid {
    $license = Get-LicenseFromRegistry
    
    if (!$license -or !$license.LicenseKey) {
        return @{ Valid = $false; Reason = "No license found"; DaysRemaining = 0 }
    }
    
    # Check expiry locally first
    if ($license.Expiry) {
        $expiry = [DateTime]::Parse($license.Expiry)
        $daysLeft = ($expiry - (Get-Date)).Days
        
        if ($daysLeft -le 0) {
            return @{ Valid = $false; Reason = "License expired"; DaysRemaining = 0; ExpiryDate = $license.Expiry }
        }
        
        return @{
            Valid = $true
            Reason = "Active"
            DaysRemaining = $daysLeft
            ExpiryDate = $license.Expiry
            Tier = $license.Tier
            ShowWarning = ($daysLeft -le 7)
        }
    }
    
    return @{ Valid = $false; Reason = "Invalid license data"; DaysRemaining = 0 }
}

#endregion

#region Login Dialog

function Show-LoginDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Sign In - ABC&D Systems"
    $form.Size = New-Object System.Drawing.Size(400, 320)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(26, 26, 46)
    
    # Header
    $lblHeader = New-Object System.Windows.Forms.Label
    $lblHeader.Text = "Sign In to Your Account"
    $lblHeader.Location = New-Object System.Drawing.Point(20, 15)
    $lblHeader.Size = New-Object System.Drawing.Size(350, 30)
    $lblHeader.Font = New-Object System.Drawing.Font("Segoe UI", 14, [System.Drawing.FontStyle]::Bold)
    $lblHeader.ForeColor = [System.Drawing.Color]::Gold
    $form.Controls.Add($lblHeader)
    
    # Email
    $lblEmail = New-Object System.Windows.Forms.Label
    $lblEmail.Text = "Email:"
    $lblEmail.Location = New-Object System.Drawing.Point(20, 60)
    $lblEmail.Size = New-Object System.Drawing.Size(100, 20)
    $lblEmail.ForeColor = [System.Drawing.Color]::White
    $form.Controls.Add($lblEmail)
    
    $txtEmail = New-Object System.Windows.Forms.TextBox
    $txtEmail.Location = New-Object System.Drawing.Point(20, 82)
    $txtEmail.Size = New-Object System.Drawing.Size(340, 25)
    $form.Controls.Add($txtEmail)
    
    # Password
    $lblPass = New-Object System.Windows.Forms.Label
    $lblPass.Text = "Password:"
    $lblPass.Location = New-Object System.Drawing.Point(20, 115)
    $lblPass.Size = New-Object System.Drawing.Size(100, 20)
    $lblPass.ForeColor = [System.Drawing.Color]::White
    $form.Controls.Add($lblPass)
    
    $txtPass = New-Object System.Windows.Forms.TextBox
    $txtPass.Location = New-Object System.Drawing.Point(20, 137)
    $txtPass.Size = New-Object System.Drawing.Size(340, 25)
    $txtPass.UseSystemPasswordChar = $true
    $form.Controls.Add($txtPass)
    
    # Status label
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Location = New-Object System.Drawing.Point(20, 175)
    $lblStatus.Size = New-Object System.Drawing.Size(340, 20)
    $lblStatus.ForeColor = [System.Drawing.Color]::Red
    $form.Controls.Add($lblStatus)
    
    # Sign In button
    $btnLogin = New-Object System.Windows.Forms.Button
    $btnLogin.Text = "Sign In"
    $btnLogin.Location = New-Object System.Drawing.Point(20, 200)
    $btnLogin.Size = New-Object System.Drawing.Size(340, 40)
    $btnLogin.BackColor = [System.Drawing.Color]::Gold
    $btnLogin.ForeColor = [System.Drawing.Color]::Black
    $btnLogin.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnLogin.Font = New-Object System.Drawing.Font("Segoe UI", 10, [System.Drawing.FontStyle]::Bold)
    $form.Controls.Add($btnLogin)
    
    # Register link
    $lblRegister = New-Object System.Windows.Forms.LinkLabel
    $lblRegister.Text = "Don't have an account? Register here"
    $lblRegister.Location = New-Object System.Drawing.Point(80, 250)
    $lblRegister.Size = New-Object System.Drawing.Size(250, 20)
    $lblRegister.LinkColor = [System.Drawing.Color]::Gold
    $lblRegister.Add_LinkClicked({ Start-Process "https://abcdtoolset.com/" })
    $form.Controls.Add($lblRegister)
    
    $script:LoginResult = $null
    
    $btnLogin.Add_Click({
        $email = $txtEmail.Text.Trim()
        $password = $txtPass.Text
        
        if ([string]::IsNullOrEmpty($email) -or [string]::IsNullOrEmpty($password)) {
            $lblStatus.Text = "Please enter email and password"
            return
        }
        
        $lblStatus.ForeColor = [System.Drawing.Color]::Yellow
        $lblStatus.Text = "Signing in..."
        $form.Refresh()
        
        $result = Invoke-Login -Email $email -Password $password
        
        if ($result.success) {
            # Save to registry
            $expiry = if ($result.license.trialEndsAt) { $result.license.trialEndsAt } else { $result.license.expiresAt }
            
            Save-LicenseToRegistry `
                -LicenseKey $result.licenseKey `
                -Token $result.token `
                -Expiry $expiry `
                -Tier $result.license.plan `
                -UserEmail $result.user.email `
                -UserName $result.user.name `
                -Status $result.license.status
            
            $script:LoginResult = $result
            $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
            $form.Close()
        } else {
            $lblStatus.ForeColor = [System.Drawing.Color]::Red
            $lblStatus.Text = $result.error
        }
    })
    
    $form.AcceptButton = $btnLogin
    
    if ($form.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
        return $script:LoginResult
    }
    return $null
}

#endregion

#region License Key Dialog

function Show-LicenseKeyDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Activate License - ABC&D Systems"
    $form.Size = New-Object System.Drawing.Size(450, 220)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(26, 26, 46)
    
    $lblHeader = New-Object System.Windows.Forms.Label
    $lblHeader.Text = "Enter Your License Key"
    $lblHeader.Location = New-Object System.Drawing.Point(20, 15)
    $lblHeader.Size = New-Object System.Drawing.Size(400, 30)
    $lblHeader.Font = New-Object System.Drawing.Font("Segoe UI", 14, [System.Drawing.FontStyle]::Bold)
    $lblHeader.ForeColor = [System.Drawing.Color]::Gold
    $form.Controls.Add($lblHeader)
    
    $lblKey = New-Object System.Windows.Forms.Label
    $lblKey.Text = "License Key (e.g., MIG-XXXX-XXXX-XXXX-XXXX):"
    $lblKey.Location = New-Object System.Drawing.Point(20, 55)
    $lblKey.Size = New-Object System.Drawing.Size(400, 20)
    $lblKey.ForeColor = [System.Drawing.Color]::White
    $form.Controls.Add($lblKey)
    
    $txtKey = New-Object System.Windows.Forms.TextBox
    $txtKey.Location = New-Object System.Drawing.Point(20, 77)
    $txtKey.Size = New-Object System.Drawing.Size(390, 25)
    $txtKey.Font = New-Object System.Drawing.Font("Consolas", 11)
    $txtKey.CharacterCasing = [System.Windows.Forms.CharacterCasing]::Upper
    $form.Controls.Add($txtKey)
    
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Location = New-Object System.Drawing.Point(20, 110)
    $lblStatus.Size = New-Object System.Drawing.Size(390, 20)
    $lblStatus.ForeColor = [System.Drawing.Color]::Red
    $form.Controls.Add($lblStatus)
    
    $btnActivate = New-Object System.Windows.Forms.Button
    $btnActivate.Text = "Activate"
    $btnActivate.Location = New-Object System.Drawing.Point(20, 135)
    $btnActivate.Size = New-Object System.Drawing.Size(180, 35)
    $btnActivate.BackColor = [System.Drawing.Color]::Gold
    $btnActivate.ForeColor = [System.Drawing.Color]::Black
    $btnActivate.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $form.Controls.Add($btnActivate)
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = New-Object System.Drawing.Point(230, 135)
    $btnCancel.Size = New-Object System.Drawing.Size(180, 35)
    $btnCancel.BackColor = [System.Drawing.Color]::Gray
    $btnCancel.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnCancel.Add_Click({ $form.Close() })
    $form.Controls.Add($btnCancel)
    
    $script:ActivateResult = $null
    
    $btnActivate.Add_Click({
        $key = $txtKey.Text.Trim()
        
        if ([string]::IsNullOrEmpty($key)) {
            $lblStatus.Text = "Please enter a license key"
            return
        }
        
        $lblStatus.ForeColor = [System.Drawing.Color]::Yellow
        $lblStatus.Text = "Validating..."
        $form.Refresh()
        
        $result = Invoke-LicenseKeyActivation -LicenseKey $key
        
        if ($result.success -and $result.valid) {
            $expiry = if ($result.license.status -eq 'trial') { 
                (Get-Date).AddDays($result.license.daysRemaining).ToString("yyyy-MM-dd") 
            } else { 
                $result.license.expiresAt 
            }
            
            Save-LicenseToRegistry `
                -LicenseKey $result.license.key `
                -Token "" `
                -Expiry $expiry `
                -Tier $result.license.plan `
                -UserEmail $result.user.email `
                -UserName $result.user.name `
                -Status $result.license.status
            
            $script:ActivateResult = $result
            $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
            $form.Close()
        } else {
            $lblStatus.ForeColor = [System.Drawing.Color]::Red
            $lblStatus.Text = if ($result.reason) { $result.reason } else { $result.error }
        }
    })
    
    if ($form.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
        return $script:ActivateResult
    }
    return $null
}

#endregion

#region License Status Dialog

function Show-LicenseStatusDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $status = Test-LicenseValid
    $license = Get-LicenseFromRegistry
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "License Status - ABC&D Systems"
    $form.Size = New-Object System.Drawing.Size(450, 400)
    $form.StartPosition = "CenterParent"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::White
    
    # Header
    $lblHeader = New-Object System.Windows.Forms.Label
    $lblHeader.Text = "ABCD Systems License"
    $lblHeader.Location = New-Object System.Drawing.Point(20, 15)
    $lblHeader.Size = New-Object System.Drawing.Size(400, 30)
    $lblHeader.Font = New-Object System.Drawing.Font("Segoe UI", 14, [System.Drawing.FontStyle]::Bold)
    $form.Controls.Add($lblHeader)
    
    # Status indicator
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Location = New-Object System.Drawing.Point(20, 55)
    $lblStatus.Size = New-Object System.Drawing.Size(400, 25)
    $lblStatus.Font = New-Object System.Drawing.Font("Segoe UI", 11)
    
    if ($status.Valid) {
        $lblStatus.Text = [char]0x2713 + " License Active"
        $lblStatus.ForeColor = [System.Drawing.Color]::Green
    } else {
        $lblStatus.Text = [char]0x2717 + " " + $status.Reason
        $lblStatus.ForeColor = [System.Drawing.Color]::Red
    }
    $form.Controls.Add($lblStatus)
    
    # Details group
    $grpDetails = New-Object System.Windows.Forms.GroupBox
    $grpDetails.Text = "License Details"
    $grpDetails.Location = New-Object System.Drawing.Point(20, 90)
    $grpDetails.Size = New-Object System.Drawing.Size(395, 150)
    $form.Controls.Add($grpDetails)
    
    $yPos = 25
    $details = @(
        @{ Label = "Email:"; Value = if ($license -and $license.UserEmail) { $license.UserEmail } else { "Not registered" } },
        @{ Label = "License Key:"; Value = if ($license -and $license.LicenseKey) { $license.LicenseKey } else { "None" } },
        @{ Label = "Tier:"; Value = if ($license -and $license.Tier) { $license.Tier.ToUpper() } else { "None" } },
        @{ Label = "Expires:"; Value = if ($status.Valid) { $status.ExpiryDate } else { "N/A" } },
        @{ Label = "Days Left:"; Value = if ($status.Valid) { "$($status.DaysRemaining)" } else { "0" } }
    )
    
    foreach ($detail in $details) {
        $lbl = New-Object System.Windows.Forms.Label
        $lbl.Text = $detail.Label
        $lbl.Location = New-Object System.Drawing.Point(15, $yPos)
        $lbl.Size = New-Object System.Drawing.Size(100, 20)
        $lbl.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
        $grpDetails.Controls.Add($lbl)
        
        $val = New-Object System.Windows.Forms.Label
        $val.Text = $detail.Value
        $val.Location = New-Object System.Drawing.Point(120, $yPos)
        $val.Size = New-Object System.Drawing.Size(260, 20)
        $val.Font = New-Object System.Drawing.Font("Segoe UI", 9)
        $grpDetails.Controls.Add($val)
        
        $yPos += 25
    }
    
    # Buttons
    $btnY = 255
    
    # Sign In button
    $btnLogin = New-Object System.Windows.Forms.Button
    $btnLogin.Text = "Sign In"
    $btnLogin.Location = New-Object System.Drawing.Point(20, $btnY)
    $btnLogin.Size = New-Object System.Drawing.Size(115, 35)
    $btnLogin.BackColor = [System.Drawing.Color]::FromArgb(0, 120, 215)
    $btnLogin.ForeColor = [System.Drawing.Color]::White
    $btnLogin.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnLogin.Add_Click({
        $form.Hide()
        $result = Show-LoginDialog
        if ($result) {
            [System.Windows.Forms.MessageBox]::Show("License activated successfully!`n`nLicense: $($result.licenseKey)`nPlan: $($result.license.plan)`nDays Left: $($result.license.daysRemaining)", "Success", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
        }
        $form.Close()
    })
    $form.Controls.Add($btnLogin)
    
    # Enter Key button
    $btnKey = New-Object System.Windows.Forms.Button
    $btnKey.Text = "Enter Key"
    $btnKey.Location = New-Object System.Drawing.Point(145, $btnY)
    $btnKey.Size = New-Object System.Drawing.Size(115, 35)
    $btnKey.BackColor = [System.Drawing.Color]::DarkGreen
    $btnKey.ForeColor = [System.Drawing.Color]::White
    $btnKey.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnKey.Add_Click({
        $form.Hide()
        $result = Show-LicenseKeyDialog
        if ($result) {
            [System.Windows.Forms.MessageBox]::Show("License activated successfully!", "Success", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
        }
        $form.Close()
    })
    $form.Controls.Add($btnKey)
    
    # Subscribe button
    $btnSubscribe = New-Object System.Windows.Forms.Button
    $btnSubscribe.Text = "Subscribe"
    $btnSubscribe.Location = New-Object System.Drawing.Point(270, $btnY)
    $btnSubscribe.Size = New-Object System.Drawing.Size(115, 35)
    $btnSubscribe.BackColor = [System.Drawing.Color]::Gold
    $btnSubscribe.ForeColor = [System.Drawing.Color]::Black
    $btnSubscribe.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnSubscribe.Add_Click({ Start-Process "https://abcdtoolset.com/" })
    $form.Controls.Add($btnSubscribe)
    
    # Close button
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(270, 305)
    $btnClose.Size = New-Object System.Drawing.Size(115, 35)
    $btnClose.BackColor = [System.Drawing.Color]::LightGray
    $btnClose.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnClose.Add_Click({ $form.Close() })
    $form.Controls.Add($btnClose)
    
    # Logout button (if logged in)
    if ($license -and $license.LicenseKey) {
        $btnLogout = New-Object System.Windows.Forms.Button
        $btnLogout.Text = "Logout"
        $btnLogout.Location = New-Object System.Drawing.Point(20, 305)
        $btnLogout.Size = New-Object System.Drawing.Size(115, 35)
        $btnLogout.BackColor = [System.Drawing.Color]::IndianRed
        $btnLogout.ForeColor = [System.Drawing.Color]::White
        $btnLogout.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnLogout.Add_Click({
            Clear-LicenseFromRegistry
            [System.Windows.Forms.MessageBox]::Show("Logged out successfully.", "Logged Out", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
            $form.Close()
        })
        $form.Controls.Add($btnLogout)
    }
    
    $form.ShowDialog() | Out-Null
}

#endregion

