<#
.SYNOPSIS
    ABC&D Systems - Backup Settings Dialog
.DESCRIPTION
    Comprehensive backup configuration with independent schedules for:
    - VM Export (Hyper-V)
    - File Backup
    - Each with configurable retention
.VERSION
    4.0
#>

# =============================================================================
# BACKUP CONFIGURATION STORAGE
# =============================================================================

$Script:BackupConfigPath = "$env:ProgramData\ABCD-Migration\BackupConfig.json"

function Get-BackupConfig {
    if (Test-Path $Script:BackupConfigPath) {
        try {
            return Get-Content $Script:BackupConfigPath -Raw | ConvertFrom-Json
        } catch {
            return $null
        }
    }
    return $null
}

function Save-BackupConfig {
    param($Config)
    
    $configDir = Split-Path $Script:BackupConfigPath -Parent
    if (!(Test-Path $configDir)) {
        New-Item -Path $configDir -ItemType Directory -Force | Out-Null
    }
    
    $Config | ConvertTo-Json -Depth 5 | Out-File $Script:BackupConfigPath -Encoding UTF8
}

# =============================================================================
# VM EXPORT FUNCTIONS
# =============================================================================

function Get-AvailableVMs {
    <#
    .SYNOPSIS
        Gets list of Hyper-V VMs on this machine
    #>
    try {
        $vms = Get-VM -ErrorAction SilentlyContinue | Select-Object Name, State, Path
        return $vms
    } catch {
        return @()
    }
}

function Get-AvailableBackupPaths {
    <#
    .SYNOPSIS
        Gets available drives/paths for backup destination
    #>
    $paths = @()
    
    # Get all fixed drives except C:
    $drives = Get-CimInstance -ClassName Win32_LogicalDisk -ErrorAction SilentlyContinue | 
        Where-Object { $_.DriveType -eq 3 -and $_.DeviceID -ne "C:" }
    
    foreach ($drive in $drives) {
        $paths += @{
            Path = "$($drive.DeviceID)\Backup"
            Display = "$($drive.DeviceID)\Backup ($([math]::Round($drive.FreeSpace/1GB, 1)) GB free)"
            FreeSpace = $drive.FreeSpace
        }
    }
    
    # Add network paths if configured
    $paths += @{
        Path = "\\SERVER\Backup"
        Display = "\\SERVER\Backup (Network - Configure)"
        FreeSpace = 0
    }
    
    return $paths
}

function Export-VMBackup {
    <#
    .SYNOPSIS
        Exports Hyper-V VMs to backup location
    #>
    param(
        [string]$BackupPath,
        [string[]]$VMNames,  # Empty or "*" means all
        [switch]$RemoveSnapshots
    )
    
    $folderName = (Get-Date).ToString("yyyy-MM-dd_HHmm")
    $exportPath = Join-Path $BackupPath $folderName
    
    # Create backup folder
    if (!(Test-Path $exportPath)) {
        New-Item -ItemType Directory -Path $exportPath -Force | Out-Null
    }
    
    Write-Host "[VM Export] Exporting to: $exportPath" -ForegroundColor Cyan
    
    # Get VMs to export
    if (!$VMNames -or $VMNames -contains "*") {
        $vmsToExport = Get-VM
    } else {
        $vmsToExport = Get-VM | Where-Object { $VMNames -contains $_.Name }
    }
    
    foreach ($vm in $vmsToExport) {
        $vmName = $vm.Name
        
        # Per-VM snapshot cleanup BEFORE export (v4.0 fix)
        if ($RemoveSnapshots) {
            $snapshots = Get-VMSnapshot -VMName $vmName -ErrorAction SilentlyContinue
            if ($snapshots) {
                Write-Host "[VM Export] Removing snapshots for $vmName..." -ForegroundColor Yellow
                Remove-VMSnapshot -VMName $vmName -IncludeAllChildSnapshots -ErrorAction SilentlyContinue
                # Wait for merge to complete
                $mergeWait = 0
                do {
                    Start-Sleep -Seconds 5
                    $mergeWait += 5
                    $vmStatus = (Get-VM -Name $vmName -ErrorAction SilentlyContinue).Status
                } while ($vmStatus -and $vmStatus -match 'Merging' -and $mergeWait -lt 600)
                Write-Host "[VM Export]   Snapshots cleaned ($mergeWait sec)" -ForegroundColor Gray
            }
        }
        
        Write-Host "[VM Export] Exporting: $vmName..." -ForegroundColor Yellow
        try {
            Export-VM -Name $vmName -Path $exportPath -ErrorAction Stop
            Write-Host "[VM Export]   Success" -ForegroundColor Green
        } catch {
            Write-Host "[VM Export]   FAILED: $_" -ForegroundColor Red
        }
    }
    
    return $exportPath
}

function Remove-OldBackups {
    <#
    .SYNOPSIS
        Removes backups older than retention days
    #>
    param(
        [string]$BackupPath,
        [int]$RetentionDays
    )
    
    if (!(Test-Path $BackupPath)) { return }
    
    $cutoffDate = (Get-Date).AddDays(-$RetentionDays)
    
    $oldFolders = Get-ChildItem -Path $BackupPath -Directory -ErrorAction SilentlyContinue | 
        Where-Object { $_.CreationTime -lt $cutoffDate }
    
    foreach ($folder in $oldFolders) {
        Write-Host "[Cleanup] Removing old backup: $($folder.Name)" -ForegroundColor Yellow
        Remove-Item -Path $folder.FullName -Recurse -Force -ErrorAction SilentlyContinue
    }
}

# =============================================================================
# SCHEDULED TASK FUNCTIONS
# =============================================================================

function Register-VMExportTask {
    <#
    .SYNOPSIS
        Creates scheduled task for VM Export
    #>
    param(
        [string]$BackupPath,
        [string[]]$VMNames,
        [string]$Time,          # "21:00"
        [int]$RetentionDays,
        [bool]$RemoveSnapshots,
        [string]$Frequency = "Daily"  # Daily, Weekly
    )
    
    $taskName = "ABCD-VMExport"
    
    # Create the export script
    $scriptPath = "$env:ProgramData\ABCD-Migration\Scripts\VMExport.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    if (!(Test-Path $scriptDir)) {
        New-Item -Path $scriptDir -ItemType Directory -Force | Out-Null
    }
    
    $vmList = if ($VMNames -and $VMNames.Count -gt 0 -and $VMNames[0] -ne "*") {
        "@('" + ($VMNames -join "','") + "')"
    } else {
        "@('*')"
    }
    
    $scriptContent = @"
# ABCD VM Export Script - Auto-generated
`$ErrorActionPreference = 'Continue'
`$BackupPath = '$BackupPath'
`$VMNames = $vmList
`$RetentionDays = $RetentionDays
`$RemoveSnapshots = `$$RemoveSnapshots

# Create dated folder
`$folderName = (Get-Date).ToString("yyyy-MM-dd_HHmm")
`$exportPath = Join-Path `$BackupPath `$folderName

if (!(Test-Path `$exportPath)) {
    New-Item -ItemType Directory -Path `$exportPath -Force | Out-Null
}

# Log file
`$logFile = Join-Path `$BackupPath "VMExport_`$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
Start-Transcript -Path `$logFile

Write-Host "[VM Export] Starting backup to: `$exportPath"

# Get VMs to export
if (`$VMNames -contains '*') {
    `$vmsToExport = Get-VM
} else {
    `$vmsToExport = Get-VM | Where-Object { `$VMNames -contains `$_.Name }
}

foreach (`$vm in `$vmsToExport) {
    Write-Host "[VM Export] Exporting: `$(`$vm.Name)..."
    
    # Per-VM snapshot cleanup BEFORE export (v4.0)
    if (`$RemoveSnapshots) {
        `$snaps = Get-VMSnapshot -VMName `$vm.Name -ErrorAction SilentlyContinue
        if (`$snaps) {
            Write-Host "[VM Export]   Removing snapshots for `$(`$vm.Name)..."
            Remove-VMSnapshot -VMName `$vm.Name -IncludeAllChildSnapshots -ErrorAction SilentlyContinue
            `$mw = 0
            do { Start-Sleep -Seconds 5; `$mw += 5; `$st = (Get-VM -Name `$vm.Name -EA SilentlyContinue).Status } while (`$st -match 'Merging' -and `$mw -lt 600)
            Write-Host "[VM Export]   Snapshots merged (`$mw sec)"
        }
    }
    
    try {
        Export-VM -Name `$vm.Name -Path `$exportPath -ErrorAction Stop
        Write-Host "[VM Export]   Success"
    } catch {
        Write-Host "[VM Export]   FAILED: `$_"
    }
}

# Cleanup old backups
Write-Host "[VM Export] Cleaning up backups older than `$RetentionDays days..."
`$cutoffDate = (Get-Date).AddDays(-`$RetentionDays)
Get-ChildItem -Path `$BackupPath -Directory -ErrorAction SilentlyContinue | 
    Where-Object { `$_.CreationTime -lt `$cutoffDate } |
    ForEach-Object {
        Write-Host "[Cleanup] Removing: `$(`$_.Name)"
        Remove-Item -Path `$_.FullName -Recurse -Force -ErrorAction SilentlyContinue
    }

Write-Host "[VM Export] Backup complete"
Stop-Transcript
"@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8
    
    # Remove existing task
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    # Parse time
    $timeParts = $Time.Split(':')
    $hour = [int]$timeParts[0]
    $minute = [int]$timeParts[1]
    
    # Create trigger
    $trigger = New-ScheduledTaskTrigger -Daily -At "$($hour):$($minute)"
    
    # Create action
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    
    # Create principal (run as SYSTEM with highest privileges)
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    
    # Settings
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -WakeToRun -ExecutionTimeLimit (New-TimeSpan -Hours 4)
    
    # Register task
    Register-ScheduledTask -TaskName $taskName -Trigger $trigger -Action $action -Principal $principal -Settings $settings -Description "ABCD Systems - VM Export Backup"
    
    Write-Host "[Task] Created scheduled task: $taskName at $Time daily" -ForegroundColor Green
    
    return $true
}

function Remove-VMExportTask {
    Unregister-ScheduledTask -TaskName "ABCD-VMExport" -Confirm:$false -ErrorAction SilentlyContinue
}

function Get-VMExportTaskStatus {
    $task = Get-ScheduledTask -TaskName "ABCD-VMExport" -ErrorAction SilentlyContinue
    if ($task) {
        $info = Get-ScheduledTaskInfo -TaskName "ABCD-VMExport" -ErrorAction SilentlyContinue
        return @{
            Exists = $true
            State = $task.State
            LastRun = $info.LastRunTime
            NextRun = $info.NextRunTime
        }
    }
    return @{ Exists = $false }
}

function Register-StaggeredBackupTask {
    <#
    .SYNOPSIS
        Creates scheduled task for staggered VM backup (local stage -> network transfer)
    #>
    param(
        [string[]]$VMNames,
        [string]$LocalStagingPath,
        [string]$NetworkTargetPath,
        [string]$Time,
        [int]$RetentionDays,
        [bool]$RemoveSnapshots,
        [bool]$VerifyAfterTransfer = $true,
        [bool]$DeleteLocalAfterTransfer = $true
    )
    
    $taskName = "ABCD-VMExport-Staggered"
    
    $scriptPath = "$env:ProgramData\ABCD-Migration\Scripts\VMExportStaggered.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    if (!(Test-Path $scriptDir)) {
        New-Item -Path $scriptDir -ItemType Directory -Force | Out-Null
    }
    
    $vmList = if ($VMNames -and $VMNames.Count -gt 0 -and $VMNames[0] -ne "*") {
        "@('" + ($VMNames -join "','") + "')"
    } else {
        "@('*')"
    }
    
    $scriptContent = @"
# ABCD Staggered VM Export Script - Auto-generated $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')
# Exports each VM locally, cleans snapshots, transfers to network, verifies, deletes local
`$ErrorActionPreference = 'Continue'
`$VMNames = $vmList
`$LocalStagingPath = '$LocalStagingPath'
`$NetworkTargetPath = '$NetworkTargetPath'
`$RetentionDays = $RetentionDays
`$RemoveSnapshots = `$$RemoveSnapshots
`$VerifyAfterTransfer = `$$VerifyAfterTransfer
`$DeleteLocalAfterTransfer = `$$DeleteLocalAfterTransfer

# Log file
`$logDir = "$env:ProgramData\ABCD-Migration\Logs"
if (!(Test-Path `$logDir)) { New-Item -ItemType Directory -Path `$logDir -Force | Out-Null }
`$logFile = Join-Path `$logDir "VMExportStaggered_`$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
Start-Transcript -Path `$logFile

Write-Host "[Staggered] Starting staggered VM backup"
Write-Host "[Staggered] Local staging: `$LocalStagingPath"
Write-Host "[Staggered] Network target: `$NetworkTargetPath"

# Source backup management module
`$Script:InstallPath = "C:\ABCD-Migration"
`$Script:LogFile = `$logFile
`$moduleFile = "`$Script:InstallPath\Modules\Backup-Management.ps1"
if (Test-Path `$moduleFile) {
    . `$moduleFile
} else {
    Write-Host "[Staggered] ERROR: Cannot find Backup-Management module at `$moduleFile"
    Stop-Transcript
    exit 1
}

try {
    Import-Module Hyper-V -ErrorAction Stop
    
    # Get VMs
    if (`$VMNames -contains '*') {
        `$vmsToBackup = Get-VM
    } else {
        `$vmsToBackup = Get-VM | Where-Object { `$VMNames -contains `$_.Name }
    }
    
    if (-not `$vmsToBackup -or `$vmsToBackup.Count -eq 0) {
        Write-Host "[Staggered] No VMs found matching selection"
        Stop-Transcript
        exit 0
    }
    
    Write-Host "[Staggered] VMs to process: `$(`$vmsToBackup.Name -join ', ')"
    
    # Run staggered backup
    `$result = Invoke-StaggeredVMBackup ``
        -VMs `$vmsToBackup ``
        -LocalStagingPath `$LocalStagingPath ``
        -NetworkTargetPath `$NetworkTargetPath ``
        -RetentionDays `$RetentionDays ``
        -VerifyAfterTransfer `$VerifyAfterTransfer ``
        -DeleteLocalAfterTransfer `$DeleteLocalAfterTransfer ``
        -CleanSnapshots `$RemoveSnapshots
    
    if (`$result) {
        Write-Host "[Staggered] All VMs backed up successfully"
    } else {
        Write-Host "[Staggered] Backup completed with errors"
    }
}
catch {
    Write-Host "[Staggered] FATAL ERROR: `$(`$_.Exception.Message)"
}
finally {
    Stop-Transcript
}
"@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8
    
    # Remove existing task
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    # Parse time
    $timeParts = $Time.Split(':')
    $hour = [int]$timeParts[0]
    $minute = [int]$timeParts[1]
    
    # Create trigger
    $trigger = New-ScheduledTaskTrigger -Daily -At "$($hour):$($minute)"
    
    # Create action
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    
    # Create principal (run as SYSTEM with highest privileges)
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    
    # Settings - allow 12 hours for large VMs
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -WakeToRun -ExecutionTimeLimit (New-TimeSpan -Hours 12)
    
    # Register task
    Register-ScheduledTask -TaskName $taskName -Trigger $trigger -Action $action -Principal $principal -Settings $settings -Description "ABCD Systems - Staggered VM Export (Local Stage to Network)"
    
    Write-Host "[Task] Created scheduled task: $taskName at $Time daily" -ForegroundColor Green
    
    return $true
}

function Get-StaggeredTaskStatus {
    $task = Get-ScheduledTask -TaskName "ABCD-VMExport-Staggered" -ErrorAction SilentlyContinue
    if ($task) {
        $info = Get-ScheduledTaskInfo -TaskName "ABCD-VMExport-Staggered" -ErrorAction SilentlyContinue
        return @{
            Exists = $true
            State = $task.State
            LastRun = $info.LastRunTime
            NextRun = $info.NextRunTime
        }
    }
    return @{ Exists = $false }
}

# =============================================================================
# BACKUP SETTINGS DIALOG
# =============================================================================

function Show-BackupSettingsDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    # Load existing config
    $config = Get-BackupConfig
    if (!$config) {
        $config = @{
            VMExport = @{
                Enabled = $false
                BackupPath = "E:\Backup"
                VMNames = @("*")
                Time = "21:00"
                RetentionDays = 7
                RemoveSnapshots = $true
            }
            NetworkStaging = @{
                Enabled = $false
                LocalStagingPath = "C:\Temp_ABCD_VMExport"
                NetworkTargetPath = ""
                VerifyAfterTransfer = $true
                DeleteLocalAfterTransfer = $true
                ScheduleEnabled = $false
                ScheduleTime = "23:00"
            }
            FileBackup = @{
                Enabled = $false
                SourcePaths = @()
                DestinationPath = ""
                Time = "22:00"
                RetentionDays = 30
            }
        }
    }
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "VHD Backup Settings v4.0 - ABCD Systems"
    $form.Size = New-Object System.Drawing.Size(550, 960)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.AutoScroll = $true
    
    # Ensure NetworkStaging exists in config
    if (-not $config.NetworkStaging) {
        $config | Add-Member -NotePropertyName "NetworkStaging" -NotePropertyValue @{
            Enabled = $false
            LocalStagingPath = "C:\Temp_ABCD_VMExport"
            NetworkTargetPath = ""
            VerifyAfterTransfer = $true
            DeleteLocalAfterTransfer = $true
            ScheduleEnabled = $false
            ScheduleTime = "23:00"
        } -Force
    }
    
    # =========================================================================
    # VM EXPORT SECTION
    # =========================================================================
    
    $grpVMExport = New-Object System.Windows.Forms.GroupBox
    $grpVMExport.Text = "VM Export Backup (Hyper-V)"
    $grpVMExport.Location = New-Object System.Drawing.Point(10, 10)
    $grpVMExport.Size = New-Object System.Drawing.Size(510, 220)
    $form.Controls.Add($grpVMExport)
    
    # Enable checkbox
    $chkVMEnabled = New-Object System.Windows.Forms.CheckBox
    $chkVMEnabled.Text = "Enable VM Export Backup"
    $chkVMEnabled.Location = New-Object System.Drawing.Point(15, 22)
    $chkVMEnabled.Size = New-Object System.Drawing.Size(200, 20)
    $chkVMEnabled.Checked = $config.VMExport.Enabled
    $grpVMExport.Controls.Add($chkVMEnabled)
    
    # Backup Path
    $lblVMPath = New-Object System.Windows.Forms.Label
    $lblVMPath.Text = "Backup Path:"
    $lblVMPath.Location = New-Object System.Drawing.Point(15, 50)
    $lblVMPath.Size = New-Object System.Drawing.Size(80, 20)
    $grpVMExport.Controls.Add($lblVMPath)
    
    $cboVMPath = New-Object System.Windows.Forms.ComboBox
    $cboVMPath.Location = New-Object System.Drawing.Point(100, 47)
    $cboVMPath.Size = New-Object System.Drawing.Size(300, 22)
    $cboVMPath.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDown
    
    # Populate paths
    $paths = Get-AvailableBackupPaths
    foreach ($p in $paths) {
        $cboVMPath.Items.Add($p.Path)
    }
    $cboVMPath.Text = $config.VMExport.BackupPath
    $grpVMExport.Controls.Add($cboVMPath)
    
    $btnBrowseVM = New-Object System.Windows.Forms.Button
    $btnBrowseVM.Text = "..."
    $btnBrowseVM.Location = New-Object System.Drawing.Point(405, 46)
    $btnBrowseVM.Size = New-Object System.Drawing.Size(30, 23)
    $btnBrowseVM.Add_Click({
        $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
        $fbd.Description = "Select VM Export Backup Location"
        if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $cboVMPath.Text = $fbd.SelectedPath
        }
    })
    $grpVMExport.Controls.Add($btnBrowseVM)
    
    # VM Selection
    $lblVMs = New-Object System.Windows.Forms.Label
    $lblVMs.Text = "VMs to Export:"
    $lblVMs.Location = New-Object System.Drawing.Point(15, 78)
    $lblVMs.Size = New-Object System.Drawing.Size(80, 20)
    $grpVMExport.Controls.Add($lblVMs)
    
    $cboVMs = New-Object System.Windows.Forms.ComboBox
    $cboVMs.Location = New-Object System.Drawing.Point(100, 75)
    $cboVMs.Size = New-Object System.Drawing.Size(200, 22)
    $cboVMs.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    $cboVMs.Items.Add("* (All VMs)")
    
    # Add available VMs
    $vms = Get-AvailableVMs
    foreach ($vm in $vms) {
        $cboVMs.Items.Add($vm.Name)
    }
    
    if ($config.VMExport.VMNames -contains "*") {
        $cboVMs.SelectedIndex = 0
    } else {
        $idx = $cboVMs.Items.IndexOf($config.VMExport.VMNames[0])
        if ($idx -ge 0) { $cboVMs.SelectedIndex = $idx } else { $cboVMs.SelectedIndex = 0 }
    }
    $grpVMExport.Controls.Add($cboVMs)
    
    $btnSelectVMs = New-Object System.Windows.Forms.Button
    $btnSelectVMs.Text = "Select Multiple..."
    $btnSelectVMs.Location = New-Object System.Drawing.Point(310, 74)
    $btnSelectVMs.Size = New-Object System.Drawing.Size(100, 23)
    $grpVMExport.Controls.Add($btnSelectVMs)
    
    # Schedule Time
    $lblVMTime = New-Object System.Windows.Forms.Label
    $lblVMTime.Text = "Schedule Time:"
    $lblVMTime.Location = New-Object System.Drawing.Point(15, 108)
    $lblVMTime.Size = New-Object System.Drawing.Size(85, 20)
    $grpVMExport.Controls.Add($lblVMTime)
    
    $timeParts = $config.VMExport.Time.Split(':')
    
    $cboVMHour = New-Object System.Windows.Forms.ComboBox
    $cboVMHour.Location = New-Object System.Drawing.Point(100, 105)
    $cboVMHour.Size = New-Object System.Drawing.Size(50, 22)
    $cboVMHour.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    0..23 | ForEach-Object { $cboVMHour.Items.Add($_.ToString("00")) }
    $cboVMHour.SelectedItem = $timeParts[0]
    $grpVMExport.Controls.Add($cboVMHour)
    
    $lblColon1 = New-Object System.Windows.Forms.Label
    $lblColon1.Text = ":"
    $lblColon1.Location = New-Object System.Drawing.Point(152, 108)
    $lblColon1.Size = New-Object System.Drawing.Size(10, 20)
    $grpVMExport.Controls.Add($lblColon1)
    
    $cboVMMinute = New-Object System.Windows.Forms.ComboBox
    $cboVMMinute.Location = New-Object System.Drawing.Point(162, 105)
    $cboVMMinute.Size = New-Object System.Drawing.Size(50, 22)
    $cboVMMinute.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    @("00", "10", "15", "20", "30", "45") | ForEach-Object { $cboVMMinute.Items.Add($_) }
    $cboVMMinute.SelectedItem = $timeParts[1]
    $grpVMExport.Controls.Add($cboVMMinute)
    
    $lblVMDaily = New-Object System.Windows.Forms.Label
    $lblVMDaily.Text = "(Daily)"
    $lblVMDaily.Location = New-Object System.Drawing.Point(220, 108)
    $lblVMDaily.Size = New-Object System.Drawing.Size(50, 20)
    $lblVMDaily.ForeColor = [System.Drawing.Color]::Gray
    $grpVMExport.Controls.Add($lblVMDaily)
    
    # Retention
    $lblVMRetention = New-Object System.Windows.Forms.Label
    $lblVMRetention.Text = "Keep backups for:"
    $lblVMRetention.Location = New-Object System.Drawing.Point(15, 138)
    $lblVMRetention.Size = New-Object System.Drawing.Size(100, 20)
    $grpVMExport.Controls.Add($lblVMRetention)
    
    $numVMRetention = New-Object System.Windows.Forms.NumericUpDown
    $numVMRetention.Location = New-Object System.Drawing.Point(120, 135)
    $numVMRetention.Size = New-Object System.Drawing.Size(60, 22)
    $numVMRetention.Minimum = 1
    $numVMRetention.Maximum = 30
    $numVMRetention.Value = $config.VMExport.RetentionDays
    $grpVMExport.Controls.Add($numVMRetention)
    
    $lblVMDays = New-Object System.Windows.Forms.Label
    $lblVMDays.Text = "days (older deleted automatically)"
    $lblVMDays.Location = New-Object System.Drawing.Point(185, 138)
    $lblVMDays.Size = New-Object System.Drawing.Size(200, 20)
    $grpVMExport.Controls.Add($lblVMDays)
    
    # Remove Snapshots
    $chkRemoveSnapshots = New-Object System.Windows.Forms.CheckBox
    $chkRemoveSnapshots.Text = "Remove VM snapshots after export"
    $chkRemoveSnapshots.Location = New-Object System.Drawing.Point(15, 165)
    $chkRemoveSnapshots.Size = New-Object System.Drawing.Size(250, 20)
    $chkRemoveSnapshots.Checked = $config.VMExport.RemoveSnapshots
    $grpVMExport.Controls.Add($chkRemoveSnapshots)
    
    # Task Status
    $taskStatus = Get-VMExportTaskStatus
    $lblVMStatus = New-Object System.Windows.Forms.Label
    if ($taskStatus.Exists) {
        $lblVMStatus.Text = "Task Status: $($taskStatus.State) | Next: $($taskStatus.NextRun)"
        $lblVMStatus.ForeColor = [System.Drawing.Color]::Green
    } else {
        $lblVMStatus.Text = "Task Status: Not configured"
        $lblVMStatus.ForeColor = [System.Drawing.Color]::Gray
    }
    $lblVMStatus.Location = New-Object System.Drawing.Point(15, 190)
    $lblVMStatus.Size = New-Object System.Drawing.Size(480, 20)
    $grpVMExport.Controls.Add($lblVMStatus)
    
    # =========================================================================
    # NETWORK STAGING SECTION (VHD to Mapped Drive - Staggered Backup)
    # =========================================================================
    
    $grpNetStaging = New-Object System.Windows.Forms.GroupBox
    $grpNetStaging.Text = "Network Staging - VHD to Mapped Drive (Staggered Backup)"
    $grpNetStaging.Location = New-Object System.Drawing.Point(10, 240)
    $grpNetStaging.Size = New-Object System.Drawing.Size(510, 210)
    $form.Controls.Add($grpNetStaging)
    
    # Enable checkbox
    $chkNetEnabled = New-Object System.Windows.Forms.CheckBox
    $chkNetEnabled.Text = "Enable staggered backup to network/mapped drive"
    $chkNetEnabled.Location = New-Object System.Drawing.Point(15, 22)
    $chkNetEnabled.Size = New-Object System.Drawing.Size(350, 20)
    $chkNetEnabled.Checked = [bool]$config.NetworkStaging.Enabled
    $grpNetStaging.Controls.Add($chkNetEnabled)
    
    # Info label
    $lblNetInfo = New-Object System.Windows.Forms.Label
    $lblNetInfo.Text = "Exports each VM locally first, cleans snapshots, transfers to network, verifies, then deletes local copy before starting the next VM."
    $lblNetInfo.Location = New-Object System.Drawing.Point(15, 44)
    $lblNetInfo.Size = New-Object System.Drawing.Size(480, 28)
    $lblNetInfo.ForeColor = [System.Drawing.Color]::Gray
    $lblNetInfo.Font = New-Object System.Drawing.Font("Segoe UI", 7.5)
    $grpNetStaging.Controls.Add($lblNetInfo)
    
    # Local Staging Path
    $lblStagingPath = New-Object System.Windows.Forms.Label
    $lblStagingPath.Text = "Local Staging:"
    $lblStagingPath.Location = New-Object System.Drawing.Point(15, 78)
    $lblStagingPath.Size = New-Object System.Drawing.Size(85, 20)
    $grpNetStaging.Controls.Add($lblStagingPath)
    
    $txtStagingPath = New-Object System.Windows.Forms.TextBox
    $txtStagingPath.Location = New-Object System.Drawing.Point(105, 75)
    $txtStagingPath.Size = New-Object System.Drawing.Size(310, 22)
    $txtStagingPath.Text = if ($config.NetworkStaging.LocalStagingPath) { $config.NetworkStaging.LocalStagingPath } else { "C:\Temp_ABCD_VMExport" }
    $grpNetStaging.Controls.Add($txtStagingPath)
    
    $btnBrowseStaging = New-Object System.Windows.Forms.Button
    $btnBrowseStaging.Text = "..."
    $btnBrowseStaging.Location = New-Object System.Drawing.Point(420, 74)
    $btnBrowseStaging.Size = New-Object System.Drawing.Size(30, 23)
    $btnBrowseStaging.Add_Click({
        $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
        $fbd.Description = "Select local staging folder (fast local drive)"
        if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $txtStagingPath.Text = $fbd.SelectedPath
        }
    })
    $grpNetStaging.Controls.Add($btnBrowseStaging)
    
    # Network Target Path
    $lblNetTarget = New-Object System.Windows.Forms.Label
    $lblNetTarget.Text = "Network Target:"
    $lblNetTarget.Location = New-Object System.Drawing.Point(15, 105)
    $lblNetTarget.Size = New-Object System.Drawing.Size(90, 20)
    $grpNetStaging.Controls.Add($lblNetTarget)
    
    $txtNetTarget = New-Object System.Windows.Forms.TextBox
    $txtNetTarget.Location = New-Object System.Drawing.Point(105, 102)
    $txtNetTarget.Size = New-Object System.Drawing.Size(310, 22)
    $txtNetTarget.Text = if ($config.NetworkStaging.NetworkTargetPath) { $config.NetworkStaging.NetworkTargetPath } else { "" }
    $grpNetStaging.Controls.Add($txtNetTarget)
    
    $btnBrowseNetTarget = New-Object System.Windows.Forms.Button
    $btnBrowseNetTarget.Text = "..."
    $btnBrowseNetTarget.Location = New-Object System.Drawing.Point(420, 101)
    $btnBrowseNetTarget.Size = New-Object System.Drawing.Size(30, 23)
    $btnBrowseNetTarget.Add_Click({
        $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
        $fbd.Description = "Select network/mapped drive target (Z:\VM_Backups, \\server\share, etc.)"
        if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $txtNetTarget.Text = $fbd.SelectedPath
        }
    })
    $grpNetStaging.Controls.Add($btnBrowseNetTarget)
    
    $lblNetHint = New-Object System.Windows.Forms.Label
    $lblNetHint.Text = "Mapped drive (Z:\Backups) or UNC path (\\server\share)"
    $lblNetHint.Location = New-Object System.Drawing.Point(105, 125)
    $lblNetHint.Size = New-Object System.Drawing.Size(350, 15)
    $lblNetHint.ForeColor = [System.Drawing.Color]::Gray
    $lblNetHint.Font = New-Object System.Drawing.Font("Segoe UI", 7.5)
    $grpNetStaging.Controls.Add($lblNetHint)
    
    # Options row
    $chkVerify = New-Object System.Windows.Forms.CheckBox
    $chkVerify.Text = "Verify after transfer"
    $chkVerify.Location = New-Object System.Drawing.Point(15, 145)
    $chkVerify.Size = New-Object System.Drawing.Size(155, 20)
    $chkVerify.Checked = if ($null -ne $config.NetworkStaging.VerifyAfterTransfer) { [bool]$config.NetworkStaging.VerifyAfterTransfer } else { $true }
    $grpNetStaging.Controls.Add($chkVerify)
    
    $chkDeleteLocal = New-Object System.Windows.Forms.CheckBox
    $chkDeleteLocal.Text = "Delete local after transfer"
    $chkDeleteLocal.Location = New-Object System.Drawing.Point(175, 145)
    $chkDeleteLocal.Size = New-Object System.Drawing.Size(185, 20)
    $chkDeleteLocal.Checked = if ($null -ne $config.NetworkStaging.DeleteLocalAfterTransfer) { [bool]$config.NetworkStaging.DeleteLocalAfterTransfer } else { $true }
    $grpNetStaging.Controls.Add($chkDeleteLocal)
    
    $chkNetSnapshots = New-Object System.Windows.Forms.CheckBox
    $chkNetSnapshots.Text = "Clean snapshots"
    $chkNetSnapshots.Location = New-Object System.Drawing.Point(365, 145)
    $chkNetSnapshots.Size = New-Object System.Drawing.Size(130, 20)
    $chkNetSnapshots.Checked = if ($null -ne $config.NetworkStaging.RemoveSnapshots) { [bool]$config.NetworkStaging.RemoveSnapshots } else { $true }
    $grpNetStaging.Controls.Add($chkNetSnapshots)
    
    # Staggered retention (independent from VM Export)
    $lblNetRetention = New-Object System.Windows.Forms.Label
    $lblNetRetention.Text = "Keep on network:"
    $lblNetRetention.Location = New-Object System.Drawing.Point(215, 176)
    $lblNetRetention.Size = New-Object System.Drawing.Size(100, 20)
    $lblNetRetention.ForeColor = [System.Drawing.Color]::Gray
    $grpNetStaging.Controls.Add($lblNetRetention)
    
    $numNetRetention = New-Object System.Windows.Forms.NumericUpDown
    $numNetRetention.Location = New-Object System.Drawing.Point(315, 173)
    $numNetRetention.Size = New-Object System.Drawing.Size(50, 22)
    $numNetRetention.Minimum = 1
    $numNetRetention.Maximum = 365
    $numNetRetention.Value = if ($config.NetworkStaging.RetentionDays) { [Math]::Max(1, $config.NetworkStaging.RetentionDays) } else { [int]$numVMRetention.Value }
    $grpNetStaging.Controls.Add($numNetRetention)
    
    $lblNetDays = New-Object System.Windows.Forms.Label
    $lblNetDays.Text = "days"
    $lblNetDays.Location = New-Object System.Drawing.Point(368, 176)
    $lblNetDays.Size = New-Object System.Drawing.Size(30, 20)
    $lblNetDays.ForeColor = [System.Drawing.Color]::Gray
    $grpNetStaging.Controls.Add($lblNetDays)
    
    # Schedule for staggered
    $chkNetSchedule = New-Object System.Windows.Forms.CheckBox
    $chkNetSchedule.Text = "Schedule:"
    $chkNetSchedule.Location = New-Object System.Drawing.Point(15, 175)
    $chkNetSchedule.Size = New-Object System.Drawing.Size(75, 20)
    $chkNetSchedule.Checked = [bool]$config.NetworkStaging.ScheduleEnabled
    $grpNetStaging.Controls.Add($chkNetSchedule)
    
    $netTimeParts = if ($config.NetworkStaging.ScheduleTime) { $config.NetworkStaging.ScheduleTime.Split(':') } else { @("23","00") }
    
    $cboNetHour = New-Object System.Windows.Forms.ComboBox
    $cboNetHour.Location = New-Object System.Drawing.Point(95, 173)
    $cboNetHour.Size = New-Object System.Drawing.Size(50, 22)
    $cboNetHour.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    0..23 | ForEach-Object { $cboNetHour.Items.Add($_.ToString("00")) }
    $cboNetHour.SelectedItem = $netTimeParts[0]
    $grpNetStaging.Controls.Add($cboNetHour)
    
    $lblNetColon = New-Object System.Windows.Forms.Label
    $lblNetColon.Text = ":"
    $lblNetColon.Location = New-Object System.Drawing.Point(147, 176)
    $lblNetColon.Size = New-Object System.Drawing.Size(10, 20)
    $grpNetStaging.Controls.Add($lblNetColon)
    
    $cboNetMinute = New-Object System.Windows.Forms.ComboBox
    $cboNetMinute.Location = New-Object System.Drawing.Point(157, 173)
    $cboNetMinute.Size = New-Object System.Drawing.Size(50, 22)
    $cboNetMinute.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    @("00", "10", "15", "20", "30", "45") | ForEach-Object { $cboNetMinute.Items.Add($_) }
    $cboNetMinute.SelectedItem = if ($netTimeParts.Count -gt 1) { $netTimeParts[1] } else { "00" }
    $grpNetStaging.Controls.Add($cboNetMinute)
    
    $lblNetDaily = New-Object System.Windows.Forms.Label
    $lblNetDaily.Text = "(Daily - runs after local VM export)"
    $lblNetDaily.Location = New-Object System.Drawing.Point(215, 176)
    $lblNetDaily.Size = New-Object System.Drawing.Size(200, 20)
    $lblNetDaily.ForeColor = [System.Drawing.Color]::Gray
    $grpNetStaging.Controls.Add($lblNetDaily)
    
    # Test network button
    $btnTestNet = New-Object System.Windows.Forms.Button
    $btnTestNet.Text = "Test Network"
    $btnTestNet.Location = New-Object System.Drawing.Point(420, 170)
    $btnTestNet.Size = New-Object System.Drawing.Size(80, 25)
    $btnTestNet.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnTestNet.Add_Click({
        if ([string]::IsNullOrEmpty($txtNetTarget.Text)) {
            [System.Windows.Forms.MessageBox]::Show("Please enter a network target path first.", "No Path")
            return
        }
        $form.Cursor = [System.Windows.Forms.Cursors]::WaitCursor
        try {
            $testOK = Test-Path $txtNetTarget.Text
            if ($testOK) {
                $drive = Get-CimInstance -ClassName Win32_LogicalDisk -ErrorAction SilentlyContinue | 
                    Where-Object { $txtNetTarget.Text.StartsWith($_.DeviceID) } | Select-Object -First 1
                $freeInfo = if ($drive) { "`nFree space: $([math]::Round($drive.FreeSpace/1GB, 1)) GB" } else { "" }
                [System.Windows.Forms.MessageBox]::Show(
                    "Network path is accessible!$freeInfo",
                    "Connection OK",
                    [System.Windows.Forms.MessageBoxButtons]::OK,
                    [System.Windows.Forms.MessageBoxIcon]::Information
                )
            } else {
                [System.Windows.Forms.MessageBox]::Show(
                    "Cannot access: $($txtNetTarget.Text)`n`nMake sure the path exists and you have permissions.",
                    "Connection Failed",
                    [System.Windows.Forms.MessageBoxButtons]::OK,
                    [System.Windows.Forms.MessageBoxIcon]::Error
                )
            }
        } catch {
            [System.Windows.Forms.MessageBox]::Show("Error: $_", "Test Failed")
        }
        $form.Cursor = [System.Windows.Forms.Cursors]::Default
    })
    $grpNetStaging.Controls.Add($btnTestNet)
    
    # =========================================================================
    # FILE BACKUP SECTION (placeholder for future)
    # =========================================================================
    # FILE / FOLDER BACKUP
    # =========================================================================
    
    $grpFileBackup = New-Object System.Windows.Forms.GroupBox
    $grpFileBackup.Text = "File / Folder Backup"
    $grpFileBackup.Location = New-Object System.Drawing.Point(10, 460)
    $grpFileBackup.Size = New-Object System.Drawing.Size(510, 200)
    $form.Controls.Add($grpFileBackup)
    
    $chkFileEnabled = New-Object System.Windows.Forms.CheckBox
    $chkFileEnabled.Text = "Enable folder backup"
    $chkFileEnabled.Location = New-Object System.Drawing.Point(15, 22)
    $chkFileEnabled.Size = New-Object System.Drawing.Size(200, 20)
    $chkFileEnabled.Checked = $(if ($config.FileBackup) { $config.FileBackup.Enabled } else { $false })
    $grpFileBackup.Controls.Add($chkFileEnabled)
    
    $lblFolders = New-Object System.Windows.Forms.Label
    $lblFolders.Text = "Folders to back up:"
    $lblFolders.Location = New-Object System.Drawing.Point(15, 48)
    $lblFolders.Size = New-Object System.Drawing.Size(120, 18)
    $grpFileBackup.Controls.Add($lblFolders)
    
    $lstFolders = New-Object System.Windows.Forms.ListBox
    $lstFolders.Location = New-Object System.Drawing.Point(15, 68)
    $lstFolders.Size = New-Object System.Drawing.Size(380, 60)
    $lstFolders.BackColor = [System.Drawing.Color]::White
    # Load saved folders
    if ($config.FileBackup -and $config.FileBackup.SourcePaths) {
        foreach ($p in $config.FileBackup.SourcePaths) { $lstFolders.Items.Add($p) }
    }
    $grpFileBackup.Controls.Add($lstFolders)
    
    $btnAddFolder = New-Object System.Windows.Forms.Button
    $btnAddFolder.Text = "Add..."
    $btnAddFolder.Location = New-Object System.Drawing.Point(400, 68)
    $btnAddFolder.Size = New-Object System.Drawing.Size(100, 26)
    $btnAddFolder.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnAddFolder.Add_Click({
        $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
        $fbd.Description = "Select a folder to back up"
        $fbd.ShowNewFolderButton = $false
        if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            if (-not $lstFolders.Items.Contains($fbd.SelectedPath)) {
                $lstFolders.Items.Add($fbd.SelectedPath)
            }
        }
    })
    $grpFileBackup.Controls.Add($btnAddFolder)
    
    $btnRemoveFolder = New-Object System.Windows.Forms.Button
    $btnRemoveFolder.Text = "Remove"
    $btnRemoveFolder.Location = New-Object System.Drawing.Point(400, 100)
    $btnRemoveFolder.Size = New-Object System.Drawing.Size(100, 26)
    $btnRemoveFolder.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnRemoveFolder.Add_Click({
        if ($lstFolders.SelectedIndex -ge 0) { $lstFolders.Items.RemoveAt($lstFolders.SelectedIndex) }
    })
    $grpFileBackup.Controls.Add($btnRemoveFolder)
    
    $lblFileDest = New-Object System.Windows.Forms.Label
    $lblFileDest.Text = "Backup to:"
    $lblFileDest.Location = New-Object System.Drawing.Point(15, 138)
    $lblFileDest.Size = New-Object System.Drawing.Size(70, 18)
    $grpFileBackup.Controls.Add($lblFileDest)
    
    $txtFileDest = New-Object System.Windows.Forms.TextBox
    $txtFileDest.Location = New-Object System.Drawing.Point(90, 135)
    $txtFileDest.Size = New-Object System.Drawing.Size(305, 22)
    $txtFileDest.Text = $(if ($config.FileBackup -and $config.FileBackup.DestinationPath) { $config.FileBackup.DestinationPath } else { "$Script:InstallPath\Backup\Folders" })
    $grpFileBackup.Controls.Add($txtFileDest)
    
    $btnFileDestBrowse = New-Object System.Windows.Forms.Button
    $btnFileDestBrowse.Text = "Browse..."
    $btnFileDestBrowse.Location = New-Object System.Drawing.Point(400, 134)
    $btnFileDestBrowse.Size = New-Object System.Drawing.Size(100, 24)
    $btnFileDestBrowse.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnFileDestBrowse.Add_Click({
        $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
        $fbd.Description = "Select backup destination"
        $fbd.SelectedPath = $txtFileDest.Text
        $fbd.ShowNewFolderButton = $true
        if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $txtFileDest.Text = $fbd.SelectedPath
        }
    })
    $grpFileBackup.Controls.Add($btnFileDestBrowse)
    
    $lblFileRetention = New-Object System.Windows.Forms.Label
    $lblFileRetention.Text = "Keep copies:"
    $lblFileRetention.Location = New-Object System.Drawing.Point(15, 170)
    $lblFileRetention.Size = New-Object System.Drawing.Size(80, 18)
    $grpFileBackup.Controls.Add($lblFileRetention)
    
    $numFileKeep = New-Object System.Windows.Forms.NumericUpDown
    $numFileKeep.Location = New-Object System.Drawing.Point(100, 167)
    $numFileKeep.Size = New-Object System.Drawing.Size(55, 22)
    $numFileKeep.Minimum = 1
    $numFileKeep.Maximum = 365
    $numFileKeep.Value = $(if ($config.FileBackup -and $config.FileBackup.RetentionDays) { [Math]::Max(1, $config.FileBackup.RetentionDays) } else { 7 })
    $grpFileBackup.Controls.Add($numFileKeep)
    
    $lblFileKeepHint = New-Object System.Windows.Forms.Label
    $lblFileKeepHint.Text = "versions per folder (oldest auto-deleted)"
    $lblFileKeepHint.Location = New-Object System.Drawing.Point(160, 170)
    $lblFileKeepHint.Size = New-Object System.Drawing.Size(300, 18)
    $lblFileKeepHint.ForeColor = [System.Drawing.Color]::Gray
    $grpFileBackup.Controls.Add($lblFileKeepHint)
    
    # =========================================================================
    # GOOGLE DRIVE SECTION
    # =========================================================================
    
    $grpGDrive = New-Object System.Windows.Forms.GroupBox
    $grpGDrive.Text = "Google Drive Cloud Backup"
    $grpGDrive.Location = New-Object System.Drawing.Point(10, 670)
    $grpGDrive.Size = New-Object System.Drawing.Size(510, 50)
    $grpGDrive.Enabled = $true
    $form.Controls.Add($grpGDrive)
    
    $lblGDriveInfo = New-Object System.Windows.Forms.Label
    $lblGDriveInfo.Text = "Google Drive backup is configured via the G-Drive button on the main toolbar."
    $lblGDriveInfo.Location = New-Object System.Drawing.Point(15, 22)
    $lblGDriveInfo.Size = New-Object System.Drawing.Size(480, 20)
    $lblGDriveInfo.ForeColor = [System.Drawing.Color]::FromArgb(100, 200, 100)
    $grpGDrive.Controls.Add($lblGDriveInfo)
    
    # =========================================================================
    # LOG PANEL (v4.0 - real-time backup status)
    # =========================================================================
    
    $grpLog = New-Object System.Windows.Forms.GroupBox
    $grpLog.Text = "Backup Log"
    $grpLog.Location = New-Object System.Drawing.Point(10, 730)
    $grpLog.Size = New-Object System.Drawing.Size(510, 100)
    $form.Controls.Add($grpLog)
    
    $txtLog = New-Object System.Windows.Forms.TextBox
    $txtLog.Location = New-Object System.Drawing.Point(10, 18)
    $txtLog.Size = New-Object System.Drawing.Size(490, 75)
    $txtLog.Multiline = $true
    $txtLog.ScrollBars = [System.Windows.Forms.ScrollBars]::Vertical
    $txtLog.ReadOnly = $true
    $txtLog.BackColor = [System.Drawing.Color]::FromArgb(30, 30, 30)
    $txtLog.ForeColor = [System.Drawing.Color]::LightGreen
    $txtLog.Font = New-Object System.Drawing.Font("Consolas", 8)
    $txtLog.Text = "[$(Get-Date -Format 'HH:mm:ss')] VHD Backup v4.0 ready"
    $grpLog.Controls.Add($txtLog)
    
    # Helper to append log
    $Script:LogTextBox = $txtLog
    function Write-DialogLog {
        param([string]$Message)
        if ($Script:LogTextBox) {
            $Script:LogTextBox.AppendText("`r`n[$([DateTime]::Now.ToString('HH:mm:ss'))] $Message")
            $Script:LogTextBox.SelectionStart = $Script:LogTextBox.TextLength
            $Script:LogTextBox.ScrollToCaret()
            [System.Windows.Forms.Application]::DoEvents()
        }
    }
    
    # Timer for polling background jobs
    $Script:BackupJob = $null
    $jobTimer = New-Object System.Windows.Forms.Timer
    $jobTimer.Interval = 1000
    $jobTimer.Add_Tick({
        if ($Script:BackupJob -and $Script:BackupJob.State -ne 'Running') {
            $output = Receive-Job -Job $Script:BackupJob -ErrorAction SilentlyContinue
            if ($output) { $output | ForEach-Object { Write-DialogLog $_ } }
            Remove-Job -Job $Script:BackupJob -Force -ErrorAction SilentlyContinue
            $Script:BackupJob = $null
            $btnRunNow.Enabled = $true
            $btnRunStaggered.Enabled = $true
            Write-DialogLog "Backup operation complete"
            $jobTimer.Stop()
        }
        elseif ($Script:BackupJob) {
            $output = Receive-Job -Job $Script:BackupJob -ErrorAction SilentlyContinue
            if ($output) { $output | ForEach-Object { Write-DialogLog $_ } }
        }
    })
    
    # =========================================================================
    # BUTTONS
    # =========================================================================
    
    $btnSave = New-Object System.Windows.Forms.Button
    $btnSave.Text = "Save && Apply"
    $btnSave.Location = New-Object System.Drawing.Point(280, 880)
    $btnSave.Size = New-Object System.Drawing.Size(110, 30)
    $btnSave.BackColor = [System.Drawing.Color]::LightGreen
    $btnSave.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $form.Controls.Add($btnSave)
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = New-Object System.Drawing.Point(400, 880)
    $btnCancel.Size = New-Object System.Drawing.Size(80, 30)
    $btnCancel.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnCancel.Add_Click({ $form.Close() })
    $form.Controls.Add($btnCancel)
    
    $btnRunNow = New-Object System.Windows.Forms.Button
    $btnRunNow.Text = "Run VM Export Now"
    $btnRunNow.Location = New-Object System.Drawing.Point(10, 840)
    $btnRunNow.Size = New-Object System.Drawing.Size(130, 30)
    $btnRunNow.BackColor = [System.Drawing.Color]::LightSteelBlue
    $btnRunNow.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $form.Controls.Add($btnRunNow)
    
    $btnRunStaggered = New-Object System.Windows.Forms.Button
    $btnRunStaggered.Text = "Run Staggered Now"
    $btnRunStaggered.Location = New-Object System.Drawing.Point(150, 840)
    $btnRunStaggered.Size = New-Object System.Drawing.Size(130, 30)
    $btnRunStaggered.BackColor = [System.Drawing.Color]::FromArgb(180, 210, 255)
    $btnRunStaggered.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $form.Controls.Add($btnRunStaggered)
    
    $btnRemoveTask = New-Object System.Windows.Forms.Button
    $btnRemoveTask.Text = "Remove Schedule"
    $btnRemoveTask.Location = New-Object System.Drawing.Point(10, 880)
    $btnRemoveTask.Size = New-Object System.Drawing.Size(110, 30)
    $btnRemoveTask.BackColor = [System.Drawing.Color]::LightCoral
    $btnRemoveTask.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $form.Controls.Add($btnRemoveTask)
    
    # Button handlers
    $btnSave.Add_Click({
        # Validate network staging config before save
        if ($chkNetEnabled.Checked) {
            if ([string]::IsNullOrWhiteSpace($txtStagingPath.Text)) {
                [System.Windows.Forms.MessageBox]::Show(
                    "Please enter a local staging path for staggered backup.",
                    "Configuration Required",
                    [System.Windows.Forms.MessageBoxButtons]::OK,
                    [System.Windows.Forms.MessageBoxIcon]::Warning
                )
                return
            }
            if ([string]::IsNullOrWhiteSpace($txtNetTarget.Text)) {
                [System.Windows.Forms.MessageBox]::Show(
                    "Please enter a network target path (mapped drive or UNC) for staggered backup.",
                    "Configuration Required",
                    [System.Windows.Forms.MessageBoxButtons]::OK,
                    [System.Windows.Forms.MessageBoxIcon]::Warning
                )
                return
            }
        }
        
        # Build config
        $newConfig = @{
            VMExport = @{
                Enabled = $chkVMEnabled.Checked
                BackupPath = $cboVMPath.Text
                VMNames = if ($cboVMs.SelectedItem -eq "* (All VMs)") { @("*") } else { @($cboVMs.SelectedItem) }
                Time = "$($cboVMHour.SelectedItem):$($cboVMMinute.SelectedItem)"
                RetentionDays = [int]$numVMRetention.Value
                RemoveSnapshots = $chkRemoveSnapshots.Checked
            }
            NetworkStaging = @{
                Enabled = $chkNetEnabled.Checked
                LocalStagingPath = $txtStagingPath.Text
                NetworkTargetPath = $txtNetTarget.Text
                VerifyAfterTransfer = $chkVerify.Checked
                DeleteLocalAfterTransfer = $chkDeleteLocal.Checked
                RemoveSnapshots = $chkNetSnapshots.Checked
                RetentionDays = [int]$numNetRetention.Value
                ScheduleEnabled = $chkNetSchedule.Checked
                ScheduleTime = "$($cboNetHour.SelectedItem):$($cboNetMinute.SelectedItem)"
            }
            FileBackup = @{
                Enabled = $chkFileEnabled.Checked
                SourcePaths = @($lstFolders.Items | ForEach-Object { $_.ToString() })
                DestinationPath = $txtFileDest.Text
                Time = "22:00"
                RetentionDays = [int]$numFileKeep.Value
            }
        }
        
        # Save config
        Save-BackupConfig -Config $newConfig
        
        # Also update master backup config for Backup-Management integration
        try {
            $masterCfg = $null
            $masterCfgFile = "$env:ProgramData\ABCD-Migration\Settings\backup-master.json"
            if (Test-Path $masterCfgFile) {
                $masterCfg = Get-Content $masterCfgFile -Raw | ConvertFrom-Json
            }
            if ($masterCfg -and $masterCfg.LocalSettings -and $masterCfg.LocalSettings.VMBackup) {
                $masterCfg.LocalSettings.VMBackup | Add-Member -NotePropertyName "NetworkStagingEnabled" -NotePropertyValue $chkNetEnabled.Checked -Force
                $masterCfg.LocalSettings.VMBackup | Add-Member -NotePropertyName "LocalStagingPath" -NotePropertyValue $txtStagingPath.Text -Force
                $masterCfg.LocalSettings.VMBackup | Add-Member -NotePropertyName "NetworkTargetPath" -NotePropertyValue $txtNetTarget.Text -Force
                $masterCfg.LocalSettings.VMBackup | Add-Member -NotePropertyName "VerifyAfterTransfer" -NotePropertyValue $chkVerify.Checked -Force
                $masterCfg.LocalSettings.VMBackup | Add-Member -NotePropertyName "DeleteLocalAfterTransfer" -NotePropertyValue $chkDeleteLocal.Checked -Force
                $masterCfg | ConvertTo-Json -Depth 10 | Out-File $masterCfgFile -Encoding UTF8 -Force
            }
        }
        catch { <# Silent - master config sync is best-effort #> }
        
        # Create or remove VM Export scheduled task
        if ($chkVMEnabled.Checked) {
            try {
                Register-VMExportTask `
                    -BackupPath $newConfig.VMExport.BackupPath `
                    -VMNames $newConfig.VMExport.VMNames `
                    -Time $newConfig.VMExport.Time `
                    -RetentionDays $newConfig.VMExport.RetentionDays `
                    -RemoveSnapshots $newConfig.VMExport.RemoveSnapshots
            } catch { }
        } else {
            Remove-VMExportTask
        }
        
        # Create or remove Staggered Network scheduled task
        if ($chkNetEnabled.Checked -and $chkNetSchedule.Checked) {
            try {
                Register-StaggeredBackupTask `
                    -VMNames $newConfig.VMExport.VMNames `
                    -LocalStagingPath $newConfig.NetworkStaging.LocalStagingPath `
                    -NetworkTargetPath $newConfig.NetworkStaging.NetworkTargetPath `
                    -Time $newConfig.NetworkStaging.ScheduleTime `
                    -RetentionDays $newConfig.NetworkStaging.RetentionDays `
                    -RemoveSnapshots $newConfig.NetworkStaging.RemoveSnapshots `
                    -VerifyAfterTransfer $newConfig.NetworkStaging.VerifyAfterTransfer `
                    -DeleteLocalAfterTransfer $newConfig.NetworkStaging.DeleteLocalAfterTransfer
            } catch { }
        } else {
            Unregister-ScheduledTask -TaskName "ABCD-VMExport-Staggered" -Confirm:$false -ErrorAction SilentlyContinue
        }
        
        # Summary message
        $summary = "Backup settings saved!`n"
        if ($chkVMEnabled.Checked) {
            $summary += "`nVM Export: Scheduled for $($newConfig.VMExport.Time) daily"
        }
        if ($chkNetEnabled.Checked) {
            $summary += "`nStaggered Network: $(if ($chkNetSchedule.Checked) { "Scheduled for $($newConfig.NetworkStaging.ScheduleTime) daily" } else { "Enabled (manual run)" })"
            $summary += "`n  Staging: $($txtStagingPath.Text)"
            $summary += "`n  Target:  $($txtNetTarget.Text)"
        }
        $summary += "`n`nRetention: $($newConfig.VMExport.RetentionDays) days"
        
        [System.Windows.Forms.MessageBox]::Show(
            $summary,
            "Settings Saved",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        
        $form.Close()
    })
    
    $btnRunNow.Add_Click({
        if ([string]::IsNullOrEmpty($cboVMPath.Text)) {
            [System.Windows.Forms.MessageBox]::Show("Please select a backup path first.", "Error")
            return
        }
        if ($Script:BackupJob) {
            [System.Windows.Forms.MessageBox]::Show("A backup is already running. Check the log panel.", "Busy")
            return
        }
        
        $result = [System.Windows.Forms.MessageBox]::Show(
            "Run VM Export backup now?`n`nPath: $($cboVMPath.Text)`nVMs: $($cboVMs.SelectedItem)",
            "Confirm",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            $vmNames = if ($cboVMs.SelectedItem -eq "* (All VMs)") { @("*") } else { @($cboVMs.SelectedItem) }
            $bkPath = $cboVMPath.Text
            $rmSnap = $chkRemoveSnapshots.Checked
            $retDays = [int]$numVMRetention.Value
            
            Write-DialogLog "Starting VM Export to $bkPath..."
            $btnRunNow.Enabled = $false
            $btnRunStaggered.Enabled = $false
            
            $Script:BackupJob = Start-Job -ScriptBlock {
                param($BackupPath, $VMNames, $RemoveSnapshots, $RetentionDays)
                Import-Module Hyper-V -ErrorAction SilentlyContinue
                $folderName = (Get-Date).ToString("yyyy-MM-dd_HHmm")
                $exportPath = Join-Path $BackupPath $folderName
                if (!(Test-Path $exportPath)) { New-Item -ItemType Directory -Path $exportPath -Force | Out-Null }
                
                if ($VMNames -contains "*") { $vms = Get-VM } else { $vms = Get-VM | Where-Object { $VMNames -contains $_.Name } }
                
                foreach ($vm in $vms) {
                    if ($RemoveSnapshots) {
                        $snaps = Get-VMSnapshot -VMName $vm.Name -ErrorAction SilentlyContinue
                        if ($snaps) {
                            Write-Output "Removing snapshots for $($vm.Name)..."
                            Remove-VMSnapshot -VMName $vm.Name -IncludeAllChildSnapshots -EA SilentlyContinue
                            $mw = 0; do { Start-Sleep 5; $mw += 5; $st = (Get-VM $vm.Name -EA SilentlyContinue).Status } while ($st -match 'Merging' -and $mw -lt 600)
                            Write-Output "  Merged ($mw sec)"
                        }
                    }
                    Write-Output "Exporting $($vm.Name)..."
                    try { Export-VM -Name $vm.Name -Path $exportPath -ErrorAction Stop; Write-Output "  $($vm.Name) OK" } catch { Write-Output "  FAILED: $_" }
                }
                
                # Cleanup
                $cutoff = (Get-Date).AddDays(-$RetentionDays)
                Get-ChildItem $BackupPath -Directory -EA SilentlyContinue | Where-Object { $_.CreationTime -lt $cutoff } | ForEach-Object {
                    Write-Output "Cleanup: $($_.Name)"; Remove-Item $_.FullName -Recurse -Force -EA SilentlyContinue
                }
                Write-Output "VM Export complete → $exportPath"
            } -ArgumentList $bkPath, $vmNames, $rmSnap, $retDays
            
            $jobTimer.Start()
        }
    })
    
    $btnRunStaggered.Add_Click({
        if (-not $chkNetEnabled.Checked) {
            [System.Windows.Forms.MessageBox]::Show("Enable network staging first, then save settings before running.", "Not Configured")
            return
        }
        if ([string]::IsNullOrWhiteSpace($txtStagingPath.Text) -or [string]::IsNullOrWhiteSpace($txtNetTarget.Text)) {
            [System.Windows.Forms.MessageBox]::Show("Please configure both local staging path and network target.", "Configuration Required")
            return
        }
        if ($Script:BackupJob) {
            [System.Windows.Forms.MessageBox]::Show("A backup is already running. Check the log panel.", "Busy")
            return
        }
        
        $vmDisplay = $cboVMs.SelectedItem
        $result = [System.Windows.Forms.MessageBox]::Show(
            "Run staggered VM backup now?`n`nVMs: $vmDisplay`nStaging: $($txtStagingPath.Text)`nTarget: $($txtNetTarget.Text)`n`nThis runs in the background. Watch the log panel for progress.",
            "Confirm Staggered Backup",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            $vmNames = if ($cboVMs.SelectedItem -eq "* (All VMs)") { @("*") } else { @($cboVMs.SelectedItem) }
            $stagePath = $txtStagingPath.Text
            $netTarget = $txtNetTarget.Text
            $retDays = [int]$numNetRetention.Value
            $doVerify = $chkVerify.Checked
            $doDelete = $chkDeleteLocal.Checked
            $doSnap = $chkNetSnapshots.Checked
            
            Write-DialogLog "Starting staggered backup: $stagePath → $netTarget"
            $btnRunNow.Enabled = $false
            $btnRunStaggered.Enabled = $false
            
            $modulePath = if (Test-Path "$env:ProgramData\ABCD-Migration\Modules\Backup-Management.ps1") {
                "$env:ProgramData\ABCD-Migration\Modules\Backup-Management.ps1"
            } else { "C:\ABCD-Migration\Modules\Backup-Management.ps1" }
            
            $Script:BackupJob = Start-Job -ScriptBlock {
                param($VMNames, $StagePath, $NetTarget, $RetDays, $Verify, $Delete, $Snap, $ModulePath)
                Import-Module Hyper-V -ErrorAction SilentlyContinue
                if (Test-Path $ModulePath) { . $ModulePath }
                
                if ($VMNames -contains "*") { $vms = Get-VM } else { $vms = Get-VM | Where-Object { $VMNames -contains $_.Name } }
                
                if (-not $vms -or @($vms).Count -eq 0) { Write-Output "No VMs found"; return }
                Write-Output "Processing $(@($vms).Count) VM(s): $($vms.Name -join ', ')"
                
                $result = Invoke-StaggeredVMBackup `
                    -VMs $vms `
                    -LocalStagingPath $StagePath `
                    -NetworkTargetPath $NetTarget `
                    -RetentionDays $RetDays `
                    -VerifyAfterTransfer $Verify `
                    -DeleteLocalAfterTransfer $Delete `
                    -CleanSnapshots $Snap
                
                if ($result) { Write-Output "Staggered backup COMPLETE - all VMs OK" }
                else { Write-Output "Staggered backup done WITH ERRORS - check logs" }
            } -ArgumentList $vmNames, $stagePath, $netTarget, $retDays, $doVerify, $doDelete, $doSnap, $modulePath
            
            $jobTimer.Start()
        }
    })
    
    $btnRemoveTask.Add_Click({
        $result = [System.Windows.Forms.MessageBox]::Show(
            "Remove all backup scheduled tasks?`n`n- ABCD-VMExport (local)`n- ABCD-VMExport-Staggered (network)",
            "Confirm",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            Remove-VMExportTask
            Unregister-ScheduledTask -TaskName "ABCD-VMExport-Staggered" -Confirm:$false -ErrorAction SilentlyContinue
            $lblVMStatus.Text = "Task Status: Removed"
            $lblVMStatus.ForeColor = [System.Drawing.Color]::Gray
            $chkVMEnabled.Checked = $false
            $chkNetSchedule.Checked = $false
        }
    })
    
    $form.ShowDialog() | Out-Null
}
