#Requires -RunAsAdministrator

# =============================================================================
# MULTI-SOURCE BACKUP MANAGEMENT MODULE v2.0
# =============================================================================
# Features:
# - Multiple SOURCE machines on DESTINATION (each with own config/tab/JSON)
# - VM backup with local AND/OR remote destinations
# - Folder backup with versioning
# - Space-based retention (auto-delete oldest when space low)
# - Connection testing for all paths
# - Flexible scheduling
# =============================================================================

$Script:BackupConfigPath = "$Script:InstallPath\Settings"
$Script:MasterConfigFile = "$Script:InstallPath\Settings\backup-master.json"

#region Default Configuration Templates

$Script:DefaultMasterConfig = @{
    # List of configured sources (for DESTINATION mode)
    Sources = @()
    
    # Local machine settings (for SOURCE mode or DESTINATION local operations)
    LocalSettings = @{
        VMBackup = @{
            Enabled = $false
            VMNames = @("*")  # * = all, or comma-separated names
            
            # Local backup options
            LocalEnabled = $false
            LocalDrive = "E:"
            LocalPath = "Backup\VMs"
            LocalRetentionDays = 7
            LocalRetentionBySpace = $true
            LocalMinFreeGB = 50
            
            # Remote/Network backup options
            RemoteEnabled = $false
            RemotePath = ""  # UNC path like \\server\share
            RemoteUsername = ""
            RemoteRetentionDays = 7
            RemoteRetentionBySpace = $true
            RemoteMinFreeGB = 50
            
            # Schedule
            ScheduleEnabled = $false
            ScheduleTime = "22:00"
            
            # Remote schedule (separate)
            RemoteScheduleEnabled = $false
            RemoteScheduleTime = "23:00"
            
            # Network Staging (Staggered Backup)
            # When enabled, VMs export locally first then transfer to network
            # one at a time: Export → Snapshot Cleanup → Transfer → Verify → Delete Local → Next VM
            NetworkStagingEnabled = $false
            LocalStagingPath = "C:\Temp_ABCD_VMExport"  # Fast local drive for staging
            NetworkTargetPath = ""   # Mapped drive (Z:\Backups) or UNC (\\server\share)
            VerifyAfterTransfer = $true   # Compare size before deleting local
            DeleteLocalAfterTransfer = $true  # Remove staging copy after verified transfer
            
            # Options
            CleanSnapshots = $true  # ALWAYS true - snapshots always cleaned
            ExportRunningVMs = $true
        }
        
        FolderBackup = @{
            Enabled = $false
            Folders = @()  # Array of folder configs
            
            # Local backup
            LocalEnabled = $false
            LocalDrive = "E:"
            LocalPath = "Backup\Folders"
            LocalRetentionDays = 7
            LocalRetentionBySpace = $true
            LocalMinFreeGB = 20
            
            # Remote backup
            RemoteEnabled = $false
            RemotePath = ""
            RemoteUsername = ""
            RemoteRetentionDays = 7
            RemoteRetentionBySpace = $true
            RemoteMinFreeGB = 20
            
            # Schedule
            ScheduleEnabled = $false
            ScheduleTime = "23:00"
        }
        
        Cleanup = @{
            Enabled = $true
            ScheduleEnabled = $false
            ScheduleTime = "04:00"
            RetentionDays = 7
            Paths = @()
        }
        
        Startup = @{
            RunOnStartup = $false
            StartMinimized = $false
        }
    }
    
    LastUpdated = ""
    ComputerRole = ""  # "SOURCE" or "DESTINATION"
}

# Template for a source machine configuration
$Script:DefaultSourceConfig = @{
    # Identification
    Name = ""
    Description = ""
    Enabled = $true
    
    # Connection settings
    ComputerName = ""
    IPAddress = ""
    UseStaticIP = $false
    Username = ""
    # Password stored in Credential Manager
    
    # Share configuration
    ShareType = "Default"  # "Default" = ABCD-Migration$, "Custom" = custom path
    CustomSharePath = ""
    
    # What to sync from this source
    SyncMigrationData = $true
    MigrationDataLocalPath = ""
    
    SyncVMBackups = $false
    VMBackupSourcePath = ""
    VMBackupLocalDrive = "E:"
    VMBackupLocalPath = "SourceBackups"
    VMBackupRetentionDays = 7
    VMBackupRetentionBySpace = $true
    VMBackupMinFreeGB = 100
    
    SyncFolderBackups = $false
    FolderBackupSourcePath = ""
    FolderBackupLocalDrive = "E:"
    FolderBackupLocalPath = "SourceFolders"
    FolderBackupRetentionDays = 7
    FolderBackupRetentionBySpace = $true
    FolderBackupMinFreeGB = 50
    
    # Schedule
    ScheduleEnabled = $false
    ScheduleTime = "02:00"
    
    # Status
    LastSync = ""
    LastSyncStatus = ""
    LastSyncMessage = ""
}

#endregion

#region Configuration Management

function Get-MasterBackupConfig {
    <#
    .SYNOPSIS
        Loads the master backup configuration
    #>
    if (Test-Path $Script:MasterConfigFile) {
        try {
            $json = Get-Content $Script:MasterConfigFile -Raw -ErrorAction Stop | ConvertFrom-Json
            # Convert to hashtable for easier manipulation
            return $json
        }
        catch {
            Write-Host "  [Config] Error loading config: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    }
    
    # Return default
    return $Script:DefaultMasterConfig.Clone()
}

function Save-MasterBackupConfig {
    <#
    .SYNOPSIS
        Saves the master backup configuration
    #>
    param([Parameter(Mandatory=$true)]$Config)
    
    try {
        $configPath = Split-Path $Script:MasterConfigFile -Parent
        if (-not (Test-Path $configPath)) {
            New-Item -ItemType Directory -Path $configPath -Force | Out-Null
        }
        
        if ($Config -is [hashtable]) {
            $Config.LastUpdated = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        }
        else {
            $Config | Add-Member -NotePropertyName "LastUpdated" -NotePropertyValue (Get-Date).ToString("yyyy-MM-dd HH:mm:ss") -Force
        }
        
        $Config | ConvertTo-Json -Depth 10 | Out-File $Script:MasterConfigFile -Encoding UTF8 -Force
        Write-Host "  [Config] Master configuration saved" -ForegroundColor Green
        return $true
    }
    catch {
        Write-Host "  [Config] Error saving: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Get-SourceConfig {
    <#
    .SYNOPSIS
        Gets configuration for a specific source machine
    #>
    param([string]$SourceName)
    
    $configFile = "$Script:BackupConfigPath\source-$SourceName.json"
    
    if (Test-Path $configFile) {
        try {
            return Get-Content $configFile -Raw | ConvertFrom-Json
        }
        catch {
            Write-Host "  [Config] Error loading source '$SourceName': $($_.Exception.Message)" -ForegroundColor Yellow
        }
    }
    
    # Return new config with name set
    $newConfig = $Script:DefaultSourceConfig.Clone()
    $newConfig.Name = $SourceName
    return $newConfig
}

function Save-SourceConfig {
    <#
    .SYNOPSIS
        Saves configuration for a specific source machine
    #>
    param(
        [Parameter(Mandatory=$true)][string]$SourceName,
        [Parameter(Mandatory=$true)]$Config
    )
    
    try {
        $configFile = "$Script:BackupConfigPath\source-$SourceName.json"
        
        if (-not (Test-Path $Script:BackupConfigPath)) {
            New-Item -ItemType Directory -Path $Script:BackupConfigPath -Force | Out-Null
        }
        
        $Config | ConvertTo-Json -Depth 10 | Out-File $configFile -Encoding UTF8 -Force
        Write-Host "  [Config] Source '$SourceName' saved" -ForegroundColor Green
        return $true
    }
    catch {
        Write-Host "  [Config] Error: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Remove-SourceConfig {
    <#
    .SYNOPSIS
        Removes a source configuration completely
    #>
    param([string]$SourceName)
    
    # Remove config file
    $configFile = "$Script:BackupConfigPath\source-$SourceName.json"
    if (Test-Path $configFile) {
        Remove-Item $configFile -Force
    }
    
    # Remove from master config
    $master = Get-MasterBackupConfig
    if ($master.Sources) {
        $master.Sources = @($master.Sources | Where-Object { $_.Name -ne $SourceName })
        Save-MasterBackupConfig -Config $master
    }
    
    # Remove scheduled task
    $taskName = "ABCD-Migration-Sync-$SourceName"
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    Write-Host "  [Config] Source '$SourceName' removed" -ForegroundColor Yellow
}

function Get-AllSourceNames {
    <#
    .SYNOPSIS
        Gets list of all configured source names
    #>
    $names = @()
    
    Get-ChildItem -Path $Script:BackupConfigPath -Filter "source-*.json" -ErrorAction SilentlyContinue | ForEach-Object {
        $name = $_.BaseName -replace '^source-', ''
        $names += $name
    }
    
    return $names
}

#endregion

#region Drive and Space Management

function Get-AvailableDrives {
    <#
    .SYNOPSIS
        Gets list of available drives for backup destinations
    #>
    $drives = @()
    
    Get-CimInstance -ClassName Win32_LogicalDisk -Filter "DriveType=3" -ErrorAction SilentlyContinue | ForEach-Object {
        $drives += @{
            Letter = $_.DeviceID
            Label = $_.VolumeName
            TotalGB = [math]::Round($_.Size / 1GB, 2)
            FreeGB = [math]::Round($_.FreeSpace / 1GB, 2)
            UsedGB = [math]::Round(($_.Size - $_.FreeSpace) / 1GB, 2)
            PercentFree = if ($_.Size -gt 0) { [math]::Round(($_.FreeSpace / $_.Size) * 100, 1) } else { 0 }
        }
    }
    
    return $drives | Sort-Object Letter
}

function Get-PathSpace {
    <#
    .SYNOPSIS
        Gets space information for any path (local or network)
    #>
    param([string]$Path)
    
    $result = @{
        Success = $false
        TotalGB = 0
        FreeGB = 0
        UsedGB = 0
        PercentFree = 0
        Message = ""
    }
    
    if (-not $Path) {
        $result.Message = "Path not specified"
        return $result
    }
    
    try {
        if ($Path -match '^\\\\') {
            # Network path
            # Try to access and get space
            if (Test-Path $Path) {
                $root = [System.IO.Path]::GetPathRoot($Path)
                # Use WMI for network drives
                $drive = Get-CimInstance -ClassName Win32_LogicalDisk | Where-Object { $_.ProviderName -like "*$($root.TrimEnd('\'))*" } | Select-Object -First 1
                
                if ($drive) {
                    $result.TotalGB = [math]::Round($drive.Size / 1GB, 2)
                    $result.FreeGB = [math]::Round($drive.FreeSpace / 1GB, 2)
                }
                else {
                    # Fallback - assume large space available
                    $result.FreeGB = 9999
                    $result.TotalGB = 9999
                }
                $result.Success = $true
            }
            else {
                $result.Message = "Cannot access network path"
            }
        }
        else {
            # Local path
            $driveLetter = [System.IO.Path]::GetPathRoot($Path).TrimEnd('\')
            $disk = Get-CimInstance -ClassName Win32_LogicalDisk -Filter "DeviceID='$driveLetter'" -ErrorAction Stop
            
            if ($disk) {
                $result.TotalGB = [math]::Round($disk.Size / 1GB, 2)
                $result.FreeGB = [math]::Round($disk.FreeSpace / 1GB, 2)
                $result.UsedGB = $result.TotalGB - $result.FreeGB
                $result.PercentFree = [math]::Round(($disk.FreeSpace / $disk.Size) * 100, 1)
                $result.Success = $true
            }
            else {
                $result.Message = "Drive not found"
            }
        }
    }
    catch {
        $result.Message = "Error: $($_.Exception.Message)"
    }
    
    return $result
}

function Test-SpaceAndCleanup {
    <#
    .SYNOPSIS
        Checks space and performs cleanup if needed
    .RETURNS
        $true if sufficient space available (or was freed), $false otherwise
    #>
    param(
        [Parameter(Mandatory=$true)][string]$Path,
        [int]$RequiredGB = 10,
        [int]$MinFreeGB = 50,
        [bool]$AutoCleanup = $true,
        [int]$RetentionDays = 7
    )
    
    Write-Host "  [Space] Checking: $Path" -ForegroundColor Gray
    
    $space = Get-PathSpace -Path $Path
    
    if (-not $space.Success) {
        Write-Host "  [Space] Cannot check space: $($space.Message)" -ForegroundColor Yellow
        return $false
    }
    
    $neededGB = $RequiredGB + $MinFreeGB
    Write-Host "  [Space] Free: $($space.FreeGB) GB | Need: $neededGB GB (Required: $RequiredGB + Reserve: $MinFreeGB)" -ForegroundColor Gray
    
    if ($space.FreeGB -ge $neededGB) {
        Write-Host "  [Space] OK - Sufficient space" -ForegroundColor Green
        return $true
    }
    
    if (-not $AutoCleanup) {
        Write-Host "  [Space] FAILED - Insufficient space (auto-cleanup disabled)" -ForegroundColor Red
        return $false
    }
    
    Write-Host "  [Space] Insufficient - Attempting cleanup..." -ForegroundColor Yellow
    
    # Progressive cleanup - reduce retention until we have space
    $currentRetention = $RetentionDays
    
    while ($currentRetention -ge 0 -and $space.FreeGB -lt $neededGB) {
        Write-Host "  [Space] Cleaning files older than $currentRetention days..." -ForegroundColor Gray
        
        Invoke-BackupCleanup -Path $Path -RetentionDays $currentRetention -Silent
        
        $space = Get-PathSpace -Path $Path
        Write-Host "  [Space] After cleanup: $($space.FreeGB) GB free" -ForegroundColor Gray
        
        $currentRetention--
    }
    
    if ($space.FreeGB -ge $neededGB) {
        Write-Host "  [Space] OK - Space freed by cleanup" -ForegroundColor Green
        return $true
    }
    
    Write-Host "  [Space] FAILED - Could not free enough space" -ForegroundColor Red
    return $false
}

function Invoke-BackupCleanup {
    <#
    .SYNOPSIS
        Cleans up old backups from a path
    #>
    param(
        [Parameter(Mandatory=$true)][string]$Path,
        [int]$RetentionDays = 7,
        [switch]$Silent
    )
    
    if (-not (Test-Path $Path)) { return }
    
    $cutoffDate = (Get-Date).AddDays(-$RetentionDays)
    $removedFiles = 0
    $removedFolders = 0
    $freedBytes = 0
    
    if (-not $Silent) {
        Write-Host "  [Cleanup] Path: $Path | Retention: $RetentionDays days" -ForegroundColor Cyan
    }
    
    try {
        # Remove dated folders first (most space savings)
        Get-ChildItem -Path $Path -Directory -ErrorAction SilentlyContinue | ForEach-Object {
            $folderDate = $null
            $formats = @("dd-MM-yyyy", "yyyy-MM-dd", "yyyyMMdd", "yyyy-MM-dd_HHmmss", "yyyyMMdd_HHmmss")
            
            foreach ($format in $formats) {
                if ([DateTime]::TryParseExact($_.Name, $format, $null, [System.Globalization.DateTimeStyles]::None, [ref]$folderDate)) {
                    break
                }
            }
            
            if ($folderDate -and $folderDate -lt $cutoffDate) {
                try {
                    $size = (Get-ChildItem -Path $_.FullName -Recurse -File -ErrorAction SilentlyContinue | 
                        Measure-Object -Property Length -Sum).Sum
                    Remove-Item -Path $_.FullName -Recurse -Force -ErrorAction Stop
                    $removedFolders++
                    $freedBytes += $size
                    if (-not $Silent) {
                        Write-Host "  [Cleanup] Removed: $($_.Name)" -ForegroundColor Gray
                    }
                }
                catch {}
            }
        }
        
        # Remove old files
        Get-ChildItem -Path $Path -File -Recurse -ErrorAction SilentlyContinue |
            Where-Object { $_.LastWriteTime -lt $cutoffDate } |
            ForEach-Object {
                try {
                    $freedBytes += $_.Length
                    Remove-Item -Path $_.FullName -Force -ErrorAction Stop
                    $removedFiles++
                }
                catch {}
            }
        
        # Remove empty folders (bottom-up)
        Get-ChildItem -Path $Path -Directory -Recurse -ErrorAction SilentlyContinue |
            Sort-Object { $_.FullName.Length } -Descending |
            ForEach-Object {
                if ((Get-ChildItem -Path $_.FullName -Force -ErrorAction SilentlyContinue).Count -eq 0) {
                    Remove-Item -Path $_.FullName -Force -ErrorAction SilentlyContinue
                }
            }
        
        if (-not $Silent) {
            $freedGB = [math]::Round($freedBytes / 1GB, 2)
            Write-Host "  [Cleanup] Done: $removedFolders folders, $removedFiles files, ${freedGB}GB freed" -ForegroundColor Green
        }
    }
    catch {
        if (-not $Silent) {
            Write-Host "  [Cleanup] Error: $($_.Exception.Message)" -ForegroundColor Red
        }
    }
}

#endregion

#region Connection Testing

function Test-BackupPath {
    <#
    .SYNOPSIS
        Tests if a backup path is accessible and writable
    #>
    param(
        [Parameter(Mandatory=$true)][string]$Path,
        [string]$Username,
        [string]$Password,
        [switch]$CreateIfMissing
    )
    
    $result = @{
        Success = $false
        Accessible = $false
        Writable = $false
        SpaceGB = 0
        Message = ""
    }
    
    try {
        # For network paths, try to authenticate first
        if ($Path -match '^\\\\' -and $Username -and $Password) {
            $server = ($Path -split '\\')[2]
            net use "\\$server\IPC$" /user:$Username $Password 2>$null | Out-Null
        }
        
        # Check if accessible
        if (Test-Path $Path) {
            $result.Accessible = $true
        }
        elseif ($CreateIfMissing) {
            New-Item -ItemType Directory -Path $Path -Force | Out-Null
            $result.Accessible = $true
            $result.Message = "Path created"
        }
        else {
            $result.Message = "Path does not exist"
            return $result
        }
        
        # Test write access
        $testFile = Join-Path $Path "write_test_$(Get-Random).tmp"
        try {
            [System.IO.File]::WriteAllText($testFile, "test")
            Remove-Item $testFile -Force
            $result.Writable = $true
        }
        catch {
            $result.Message = "Cannot write to path"
            return $result
        }
        
        # Get space
        $space = Get-PathSpace -Path $Path
        $result.SpaceGB = $space.FreeGB
        
        $result.Success = $true
        $result.Message = "OK - $($space.FreeGB) GB free"
    }
    catch {
        $result.Message = "Error: $($_.Exception.Message)"
    }
    
    return $result
}

function Test-SourceConnection {
    <#
    .SYNOPSIS
        Tests connection to a SOURCE machine
    #>
    param(
        [Parameter(Mandatory=$true)]$SourceConfig
    )
    
    $result = @{
        Success = $false
        PingOK = $false
        ShareOK = $false
        CredentialsOK = $false
        Message = ""
        Details = @()
    }
    
    $target = if ($SourceConfig.UseStaticIP -and $SourceConfig.IPAddress) { 
        $SourceConfig.IPAddress 
    } else { 
        $SourceConfig.ComputerName 
    }
    
    if (-not $target) {
        $result.Message = "No computer name or IP specified"
        return $result
    }
    
    Write-Host "  [Test] Testing: $target" -ForegroundColor Yellow
    
    # Ping test
    try {
        if (Test-Connection -ComputerName $target -Count 2 -Quiet -ErrorAction SilentlyContinue) {
            $result.PingOK = $true
            $result.Details += "Ping: OK"
            Write-Host "  [Test]   Ping: OK" -ForegroundColor Green
        }
        else {
            $result.Details += "Ping: FAILED"
            $result.Message = "Cannot ping $target"
            Write-Host "  [Test]   Ping: FAILED" -ForegroundColor Red
            return $result
        }
    }
    catch {
        $result.Details += "Ping: ERROR"
        $result.Message = "Ping error: $($_.Exception.Message)"
        return $result
    }
    
    # Determine share path
    $sharePath = if ($SourceConfig.ShareType -eq "Custom" -and $SourceConfig.CustomSharePath) {
        $SourceConfig.CustomSharePath
    }
    else {
        "\\$target\ABCD-Migration$"
    }
    
    # Try authentication
    if ($SourceConfig.Username) {
        try {
            # Get password from credential manager or config
            $password = $SourceConfig.Password  # In production, use Credential Manager
            if ($password) {
                net use "\\$target\IPC$" /user:$($SourceConfig.Username) $password 2>$null | Out-Null
                $result.CredentialsOK = $true
                $result.Details += "Credentials: OK"
                Write-Host "  [Test]   Credentials: OK" -ForegroundColor Green
            }
        }
        catch {
            $result.Details += "Credentials: FAILED"
            Write-Host "  [Test]   Credentials: FAILED" -ForegroundColor Yellow
        }
    }
    
    # Test share access
    try {
        if (Test-Path $sharePath) {
            $result.ShareOK = $true
            $result.Details += "Share ($sharePath): OK"
            Write-Host "  [Test]   Share: OK" -ForegroundColor Green
        }
        else {
            $result.Details += "Share ($sharePath): NOT ACCESSIBLE"
            $result.Message = "Cannot access $sharePath"
            Write-Host "  [Test]   Share: FAILED" -ForegroundColor Red
            return $result
        }
    }
    catch {
        $result.Details += "Share: ERROR"
        $result.Message = "Share error: $($_.Exception.Message)"
        return $result
    }
    
    $result.Success = $result.PingOK -and $result.ShareOK
    $result.Message = if ($result.Success) { "Connection successful" } else { "Connection incomplete" }
    
    return $result
}

#endregion

#region VM Backup Operations

function Invoke-VMBackup {
    <#
    .SYNOPSIS
        Performs VM backup with configurable local and remote destinations
    #>
    param(
        [array]$VMNames = @("*"),
        
        # Local backup settings
        [bool]$LocalEnabled = $false,
        [string]$LocalDrive = "E:",
        [string]$LocalPath = "Backup\VMs",
        [int]$LocalRetentionDays = 7,
        [bool]$LocalRetentionBySpace = $true,
        [int]$LocalMinFreeGB = 50,
        
        # Remote backup settings
        [bool]$RemoteEnabled = $false,
        [string]$RemotePath = "",
        [string]$RemoteUsername = "",
        [string]$RemotePassword = "",
        [int]$RemoteRetentionDays = 7,
        [bool]$RemoteRetentionBySpace = $true,
        [int]$RemoteMinFreeGB = 50,
        
        # Options
        [bool]$CleanSnapshots = $true
    )
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    Write-Host "  VM BACKUP" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    
    # Check Hyper-V
    try {
        Import-Module Hyper-V -ErrorAction Stop
    }
    catch {
        Write-Host "  [VMBackup] ERROR: Hyper-V not available" -ForegroundColor Red
        return $false
    }
    
    # Get VMs
    if ($VMNames.Count -eq 0 -or $VMNames -contains "*") {
        $vms = Get-VM -ErrorAction SilentlyContinue
    }
    else {
        $vms = Get-VM -ErrorAction SilentlyContinue | Where-Object { $VMNames -contains $_.Name }
    }
    
    if (-not $vms -or $vms.Count -eq 0) {
        Write-Host "  [VMBackup] No VMs found" -ForegroundColor Yellow
        return $false
    }
    
    Write-Host "  [VMBackup] VMs: $($vms.Name -join ', ')" -ForegroundColor Gray
    
    # Estimate space needed (rough: total VHD size * 0.6 for compression)
    $estimatedGB = 0
    foreach ($vm in $vms) {
        $vm.HardDrives | ForEach-Object {
            if (Test-Path $_.Path) {
                $estimatedGB += [math]::Round((Get-Item $_.Path).Length / 1GB * 0.6, 0)
            }
        }
    }
    Write-Host "  [VMBackup] Estimated size: ~${estimatedGB} GB" -ForegroundColor Gray
    
    $dateFolder = (Get-Date).ToString("dd-MM-yyyy")
    $success = $true
    
    # LOCAL BACKUP
    if ($LocalEnabled) {
        $localFullPath = Join-Path "$LocalDrive\" $LocalPath
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [VMBackup] LOCAL: $localFullPath" -ForegroundColor Yellow
        
        # Check/free space
        $spaceOK = Test-SpaceAndCleanup -Path $localFullPath -RequiredGB $estimatedGB `
            -MinFreeGB $LocalMinFreeGB -AutoCleanup $LocalRetentionBySpace -RetentionDays $LocalRetentionDays
        
        if ($spaceOK) {
            $exportPath = Join-Path $localFullPath $dateFolder
            if (-not (Test-Path $exportPath)) {
                New-Item -ItemType Directory -Path $exportPath -Force | Out-Null
            }
            
            foreach ($vm in $vms) {
                Write-Host "  [VMBackup]   Exporting: $($vm.Name)..." -ForegroundColor Gray
                try {
                    Export-VM -Name $vm.Name -Path $exportPath -ErrorAction Stop
                    Write-Host "  [VMBackup]   OK" -ForegroundColor Green
                }
                catch {
                    Write-Host "  [VMBackup]   FAILED: $($_.Exception.Message)" -ForegroundColor Red
                    $success = $false
                }
            }
        }
        else {
            Write-Host "  [VMBackup] LOCAL SKIPPED - Insufficient space" -ForegroundColor Red
            $success = $false
        }
    }
    
    # REMOTE BACKUP (Staggered or Direct)
    if ($RemoteEnabled -and $RemotePath) {
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [VMBackup] REMOTE: $RemotePath" -ForegroundColor Yellow
        
        # Load staging config from master config
        $masterCfg = Get-MasterBackupConfig
        $stagingEnabled = $false
        $stagingPath = "C:\Temp_ABCD_VMExport"
        $verifyTransfer = $true
        $deleteAfterTransfer = $true
        
        if ($masterCfg -and $masterCfg.LocalSettings -and $masterCfg.LocalSettings.VMBackup) {
            $vmCfg = $masterCfg.LocalSettings.VMBackup
            $stagingEnabled = [bool]$vmCfg.NetworkStagingEnabled
            if ($vmCfg.LocalStagingPath) { $stagingPath = $vmCfg.LocalStagingPath }
            $verifyTransfer = if ($null -ne $vmCfg.VerifyAfterTransfer) { [bool]$vmCfg.VerifyAfterTransfer } else { $true }
            $deleteAfterTransfer = if ($null -ne $vmCfg.DeleteLocalAfterTransfer) { [bool]$vmCfg.DeleteLocalAfterTransfer } else { $true }
        }
        
        if ($stagingEnabled) {
            # === STAGGERED NETWORK BACKUP ===
            Write-Host "  [VMBackup] MODE: Staggered (Local Stage → Network Transfer)" -ForegroundColor Cyan
            Write-Host "  [VMBackup] Staging: $stagingPath" -ForegroundColor Gray
            Write-Host "  [VMBackup] Target:  $RemotePath" -ForegroundColor Gray
            
            $staggerResult = Invoke-StaggeredVMBackup `
                -VMs $vms `
                -LocalStagingPath $stagingPath `
                -NetworkTargetPath $RemotePath `
                -NetworkUsername $RemoteUsername `
                -NetworkPassword $RemotePassword `
                -RetentionDays $RemoteRetentionDays `
                -RetentionBySpace $RemoteRetentionBySpace `
                -MinFreeGB $RemoteMinFreeGB `
                -VerifyAfterTransfer $verifyTransfer `
                -DeleteLocalAfterTransfer $deleteAfterTransfer `
                -CleanSnapshots $CleanSnapshots
            
            if (-not $staggerResult) { $success = $false }
        }
        else {
            # === DIRECT NETWORK BACKUP (original behavior) ===
            Write-Host "  [VMBackup] MODE: Direct Export to Network" -ForegroundColor Cyan
            
            # Test connection
            $testResult = Test-BackupPath -Path $RemotePath -Username $RemoteUsername -Password $RemotePassword -CreateIfMissing
            
            if (-not $testResult.Success) {
                Write-Host "  [VMBackup] REMOTE SKIPPED - Cannot access: $($testResult.Message)" -ForegroundColor Red
                $success = $false
            }
            else {
                # Check/free space
                $spaceOK = Test-SpaceAndCleanup -Path $RemotePath -RequiredGB $estimatedGB `
                    -MinFreeGB $RemoteMinFreeGB -AutoCleanup $RemoteRetentionBySpace -RetentionDays $RemoteRetentionDays
                
                if ($spaceOK) {
                    $exportPath = Join-Path $RemotePath $dateFolder
                    if (-not (Test-Path $exportPath)) {
                        New-Item -ItemType Directory -Path $exportPath -Force | Out-Null
                    }
                    
                    foreach ($vm in $vms) {
                        Write-Host "  [VMBackup]   Exporting: $($vm.Name)..." -ForegroundColor Gray
                        try {
                            Export-VM -Name $vm.Name -Path $exportPath -ErrorAction Stop
                            Write-Host "  [VMBackup]   OK" -ForegroundColor Green
                        }
                        catch {
                            Write-Host "  [VMBackup]   FAILED: $($_.Exception.Message)" -ForegroundColor Red
                            $success = $false
                        }
                    }
                }
                else {
                    Write-Host "  [VMBackup] REMOTE SKIPPED - Insufficient space" -ForegroundColor Red
                    $success = $false
                }
            }
        }
    }
    
    # Clean snapshots - ALWAYS (not optional)
    # VM snapshots should always be cleaned after export to ensure consistent backups
    Write-Host "" -ForegroundColor Gray
    Write-Host "  [VMBackup] Cleaning VM snapshots (always after export)..." -ForegroundColor Gray
    Get-VM | Get-VMSnapshot | Remove-VMSnapshot -ErrorAction SilentlyContinue
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    Write-Host "  VM BACKUP COMPLETE (Success: $success)" -ForegroundColor $(if ($success) { "Green" } else { "Yellow" })
    Write-Host "  ========================================" -ForegroundColor Cyan
    
    return $success
}

#endregion

#region Staggered VM Backup (Local Stage → Network Transfer)

function Invoke-StaggeredVMBackup {
    <#
    .SYNOPSIS
        Staggered VM backup: exports each VM locally, cleans snapshots, transfers
        to network target, verifies, then deletes the local staging copy before
        moving to the next VM. Prevents Hyper-V SYSTEM account issues with direct
        network exports and keeps network traffic sequential.
    .DESCRIPTION
        For each VM in order:
          1. Export-VM to local staging path
          2. Wait for export to complete
          3. Remove snapshots for that VM
          4. Robocopy/Move export folder to network target
          5. Verify: compare folder size and file count
          6. Delete local staging copy (if verified)
          7. Proceed to next VM
    #>
    param(
        [Parameter(Mandatory=$true)]
        [array]$VMs,
        
        [Parameter(Mandatory=$true)]
        [string]$LocalStagingPath,
        
        [Parameter(Mandatory=$true)]
        [string]$NetworkTargetPath,
        
        [string]$NetworkUsername = "",
        [string]$NetworkPassword = "",
        
        [int]$RetentionDays = 7,
        [bool]$RetentionBySpace = $true,
        [int]$MinFreeGB = 50,
        
        [bool]$VerifyAfterTransfer = $true,
        [bool]$DeleteLocalAfterTransfer = $true,
        [bool]$CleanSnapshots = $true
    )
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  =============================================" -ForegroundColor Cyan
    Write-Host "  STAGGERED VM BACKUP - Local Stage → Network" -ForegroundColor Cyan
    Write-Host "  =============================================" -ForegroundColor Cyan
    Write-Host "  [Stagger] VMs to process: $($VMs.Name -join ', ')" -ForegroundColor Gray
    Write-Host "  [Stagger] Local staging:  $LocalStagingPath" -ForegroundColor Gray
    Write-Host "  [Stagger] Network target: $NetworkTargetPath" -ForegroundColor Gray
    Write-Host "  [Stagger] Verify: $VerifyAfterTransfer | Delete local: $DeleteLocalAfterTransfer" -ForegroundColor Gray
    Write-Host ""
    
    # Ensure local staging directory exists
    if (-not (Test-Path $LocalStagingPath)) {
        New-Item -ItemType Directory -Path $LocalStagingPath -Force | Out-Null
        Write-Host "  [Stagger] Created staging directory" -ForegroundColor Gray
    }
    
    # Authenticate to network path if credentials provided
    if ($NetworkTargetPath -match '^\\\\' -and $NetworkUsername -and $NetworkPassword) {
        try {
            $server = ($NetworkTargetPath -split '\\')[2]
            net use "\\$server\IPC$" /user:$NetworkUsername $NetworkPassword 2>$null | Out-Null
            Write-Host "  [Stagger] Network authentication OK" -ForegroundColor Green
        }
        catch {
            Write-Host "  [Stagger] WARNING: Network auth failed: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    }
    
    # Test network path accessibility
    $netTest = Test-BackupPath -Path $NetworkTargetPath -CreateIfMissing
    if (-not $netTest.Success) {
        Write-Host "  [Stagger] ERROR: Cannot access network target: $($netTest.Message)" -ForegroundColor Red
        return $false
    }
    Write-Host "  [Stagger] Network path OK ($($netTest.SpaceGB) GB free)" -ForegroundColor Green
    
    # Check/free space on network target
    $estimatedPerVM = 0
    foreach ($vm in $VMs) {
        $vm.HardDrives | ForEach-Object {
            if (Test-Path $_.Path) {
                $estimatedPerVM = [math]::Max($estimatedPerVM, [math]::Round((Get-Item $_.Path).Length / 1GB * 0.6, 0))
            }
        }
    }
    
    $spaceOK = Test-SpaceAndCleanup -Path $NetworkTargetPath -RequiredGB ($estimatedPerVM * $VMs.Count) `
        -MinFreeGB $MinFreeGB -AutoCleanup $RetentionBySpace -RetentionDays $RetentionDays
    
    if (-not $spaceOK) {
        Write-Host "  [Stagger] ERROR: Insufficient space on network target" -ForegroundColor Red
        return $false
    }
    
    # Create dated folder on network target
    $dateFolder = (Get-Date).ToString("dd-MM-yyyy")
    $networkDatePath = Join-Path $NetworkTargetPath $dateFolder
    if (-not (Test-Path $networkDatePath)) {
        New-Item -ItemType Directory -Path $networkDatePath -Force | Out-Null
    }
    
    $totalVMs = $VMs.Count
    $currentVM = 0
    $successCount = 0
    $errorCount = 0
    $failedVMs = @()
    $overallStart = Get-Date
    
    # ===== PROCESS EACH VM SEQUENTIALLY =====
    foreach ($vm in $VMs) {
        $currentVM++
        $vmName = $vm.Name
        $vmStart = Get-Date
        
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [Stagger] ----------------------------------------" -ForegroundColor Yellow
        Write-Host "  [Stagger] VM $currentVM of $totalVMs : $vmName" -ForegroundColor Yellow
        Write-Host "  [Stagger] State: $($vm.State)" -ForegroundColor Gray
        Write-Host "  [Stagger] ----------------------------------------" -ForegroundColor Yellow
        
        $vmStagingPath = $LocalStagingPath  # Export-VM creates a subfolder named after the VM
        $vmStagedFolder = Join-Path $LocalStagingPath $vmName
        
        # --- Step 1: Local Export ---
        Write-Host "  [Stagger] Step 1/5: Exporting '$vmName' to local staging..." -ForegroundColor Cyan
        
        # Clean any previous failed staging for this VM
        if (Test-Path $vmStagedFolder) {
            Write-Host "  [Stagger]   Cleaning stale staging folder..." -ForegroundColor Gray
            Remove-Item -Path $vmStagedFolder -Recurse -Force -ErrorAction SilentlyContinue
        }
        
        try {
            Export-VM -Name $vmName -Path $vmStagingPath -ErrorAction Stop
            Write-Host "  [Stagger]   Local export complete" -ForegroundColor Green
        }
        catch {
            Write-Host "  [Stagger]   EXPORT FAILED: $($_.Exception.Message)" -ForegroundColor Red
            Write-Host "  [Stagger]   Skipping $vmName, moving to next VM" -ForegroundColor Yellow
            $errorCount++; $failedVMs += $vm
            continue
        }
        
        # Verify the staged folder exists
        if (-not (Test-Path $vmStagedFolder)) {
            Write-Host "  [Stagger]   ERROR: Export folder not found at $vmStagedFolder" -ForegroundColor Red
            $errorCount++; $failedVMs += $vm
            continue
        }
        
        # Get local staging size
        $localSize = Get-FolderSizeBytes -FolderPath $vmStagedFolder
        $localSizeGB = [math]::Round($localSize / 1GB, 2)
        $localFileCount = (Get-ChildItem -Path $vmStagedFolder -Recurse -File -ErrorAction SilentlyContinue).Count
        Write-Host "  [Stagger]   Staged: $localSizeGB GB ($localFileCount files)" -ForegroundColor Gray
        
        # --- Step 2: Clean Snapshots (per-VM, wait for merge) ---
        if ($CleanSnapshots) {
            Write-Host "  [Stagger] Step 2/5: Cleaning snapshots for '$vmName'..." -ForegroundColor Cyan
            try {
                $snapshots = Get-VMSnapshot -VMName $vmName -ErrorAction SilentlyContinue
                if ($snapshots) {
                    $snapCount = @($snapshots).Count
                    Write-Host "  [Stagger]   Found $snapCount snapshot(s), removing with child snapshots..." -ForegroundColor Gray
                    Remove-VMSnapshot -VMName $vmName -IncludeAllChildSnapshots -ErrorAction Stop
                    
                    # Wait for merge to complete (critical - export fails during merge)
                    $mergeWait = 0
                    $maxWait = 600  # 10 minutes max
                    do {
                        Start-Sleep -Seconds 5
                        $mergeWait += 5
                        $vmStatus = (Get-VM -Name $vmName -ErrorAction SilentlyContinue).Status
                        if ($vmStatus -and $vmStatus -ne 'Operating normally' -and $vmStatus -ne 'Running') {
                            Write-Host "  [Stagger]   Waiting for merge... ($mergeWait sec, status: $vmStatus)" -ForegroundColor Gray
                        }
                    } while ($vmStatus -and $vmStatus -match 'Merging' -and $mergeWait -lt $maxWait)
                    
                    if ($mergeWait -ge $maxWait) {
                        Write-Host "  [Stagger]   WARNING: Merge wait timeout after $maxWait seconds" -ForegroundColor Yellow
                    }
                    Write-Host "  [Stagger]   Snapshots cleaned and merged" -ForegroundColor Green
                }
                else {
                    Write-Host "  [Stagger]   No snapshots to clean" -ForegroundColor Gray
                }
            }
            catch {
                Write-Host "  [Stagger]   WARNING: Snapshot cleanup issue: $($_.Exception.Message)" -ForegroundColor Yellow
            }
        }
        else {
            Write-Host "  [Stagger] Step 2/5: Snapshot cleanup skipped (disabled)" -ForegroundColor Gray
        }
        
        # --- Step 3: Transfer to Network ---
        Write-Host "  [Stagger] Step 3/5: Transferring '$vmName' to network target..." -ForegroundColor Cyan
        $networkVMPath = Join-Path $networkDatePath $vmName
        
        $transferSuccess = Copy-VMToNetwork -SourcePath $vmStagedFolder -DestinationPath $networkVMPath
        
        if (-not $transferSuccess) {
            Write-Host "  [Stagger]   TRANSFER FAILED for $vmName" -ForegroundColor Red
            Write-Host "  [Stagger]   Local staging copy preserved at: $vmStagedFolder" -ForegroundColor Yellow
            $errorCount++; $failedVMs += $vm
            continue
        }
        
        Write-Host "  [Stagger]   Transfer complete" -ForegroundColor Green
        
        # --- Step 4: Verify ---
        $verified = $true
        if ($VerifyAfterTransfer) {
            Write-Host "  [Stagger] Step 4/5: Verifying transfer..." -ForegroundColor Cyan
            $verified = Confirm-VMTransfer -LocalPath $vmStagedFolder -NetworkPath $networkVMPath
            
            if ($verified) {
                Write-Host "  [Stagger]   Verification PASSED" -ForegroundColor Green
            }
            else {
                Write-Host "  [Stagger]   Verification FAILED - local copy preserved" -ForegroundColor Red
                $errorCount++; $failedVMs += $vm
                continue
            }
        }
        else {
            Write-Host "  [Stagger] Step 4/5: Verification skipped (disabled)" -ForegroundColor Gray
        }
        
        # --- Step 5: Delete Local Staging ---
        if ($DeleteLocalAfterTransfer -and $verified) {
            Write-Host "  [Stagger] Step 5/5: Removing local staging copy..." -ForegroundColor Cyan
            try {
                Remove-Item -Path $vmStagedFolder -Recurse -Force -ErrorAction Stop
                Write-Host "  [Stagger]   Local staging cleaned" -ForegroundColor Green
            }
            catch {
                Write-Host "  [Stagger]   WARNING: Could not remove staging: $($_.Exception.Message)" -ForegroundColor Yellow
            }
        }
        else {
            Write-Host "  [Stagger] Step 5/5: Local staging preserved ($(if (-not $verified) { 'verification failed' } else { 'delete disabled' }))" -ForegroundColor Gray
        }
        
        $vmDuration = (Get-Date) - $vmStart
        $successCount++
        Write-Host "  [Stagger] '$vmName' complete in $([math]::Round($vmDuration.TotalMinutes, 1)) minutes" -ForegroundColor Green
    }
    
    # ===== RETRY FAILED VMs (one attempt) =====
    if ($failedVMs.Count -gt 0) {
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [Stagger] Retrying $($failedVMs.Count) failed VM(s)..." -ForegroundColor Yellow
        foreach ($retryVM in $failedVMs) {
            $vmName = $retryVM.Name
            Write-Host "  [Stagger] RETRY: $vmName" -ForegroundColor Yellow
            $vmStagedFolder = Join-Path $LocalStagingPath $vmName
            
            # Clean stale staging
            if (Test-Path $vmStagedFolder) {
                Remove-Item -Path $vmStagedFolder -Recurse -Force -ErrorAction SilentlyContinue
            }
            
            try {
                Export-VM -Name $vmName -Path $LocalStagingPath -ErrorAction Stop
                $networkVMPath = Join-Path $networkDatePath $vmName
                $transferOK = Copy-VMToNetwork -SourcePath $vmStagedFolder -DestinationPath $networkVMPath
                if ($transferOK) {
                    if ($DeleteLocalAfterTransfer) {
                        Remove-Item -Path $vmStagedFolder -Recurse -Force -ErrorAction SilentlyContinue
                    }
                    $successCount++
                    $errorCount--
                    Write-Host "  [Stagger] RETRY SUCCESS: $vmName" -ForegroundColor Green
                } else {
                    Write-Host "  [Stagger] RETRY FAILED: $vmName transfer failed" -ForegroundColor Red
                }
            }
            catch {
                Write-Host "  [Stagger] RETRY FAILED: $vmName - $($_.Exception.Message)" -ForegroundColor Red
            }
        }
    }
    
    # ===== SUMMARY =====
    $overallDuration = (Get-Date) - $overallStart
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  =============================================" -ForegroundColor Cyan
    Write-Host "  STAGGERED BACKUP COMPLETE" -ForegroundColor Cyan
    Write-Host "  =============================================" -ForegroundColor Cyan
    Write-Host "  [Stagger] Success: $successCount | Failed: $errorCount | Total: $totalVMs" -ForegroundColor $(if ($errorCount -eq 0) { "Green" } else { "Yellow" })
    Write-Host "  [Stagger] Duration: $([math]::Round($overallDuration.TotalMinutes, 1)) minutes" -ForegroundColor Gray
    Write-Host "  [Stagger] Network target: $networkDatePath" -ForegroundColor Gray
    
    # Clean up old exports on network target
    Write-Host "  [Stagger] Cleaning exports older than $RetentionDays days on network..." -ForegroundColor Gray
    Invoke-BackupCleanup -Path $NetworkTargetPath -RetentionDays $RetentionDays -Silent
    
    # Log summary
    if ($Script:LogFile) {
        $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
        "[$timestamp] [Stagger] Complete: $successCount/$totalVMs VMs backed up to $networkDatePath ($([math]::Round($overallDuration.TotalMinutes, 1)) min)" | 
            Out-File -FilePath $Script:LogFile -Append -Encoding UTF8
    }
    
    return ($errorCount -eq 0)
}

function Get-FolderSizeBytes {
    <#
    .SYNOPSIS
        Gets total size of all files in a folder recursively
    #>
    param([string]$FolderPath)
    
    if (-not (Test-Path $FolderPath)) { return 0 }
    
    $totalSize = (Get-ChildItem -Path $FolderPath -Recurse -File -ErrorAction SilentlyContinue | 
        Measure-Object -Property Length -Sum -ErrorAction SilentlyContinue).Sum
    
    return [long]($totalSize)
}

function Copy-VMToNetwork {
    <#
    .SYNOPSIS
        Copies a VM export folder to network target using Robocopy for reliability
    .DESCRIPTION
        Uses Robocopy with mirror mode for large file transfers. Falls back to
        Move-Item if Robocopy is not available. Returns $true on success.
    #>
    param(
        [Parameter(Mandatory=$true)][string]$SourcePath,
        [Parameter(Mandatory=$true)][string]$DestinationPath
    )
    
    # Ensure destination parent exists
    $destParent = Split-Path $DestinationPath -Parent
    if (-not (Test-Path $destParent)) {
        New-Item -ItemType Directory -Path $destParent -Force | Out-Null
    }
    
    try {
        # Prefer Robocopy for large transfers (retry logic, bandwidth control)
        $robocopy = Get-Command "robocopy.exe" -ErrorAction SilentlyContinue
        
        if ($robocopy) {
            Write-Host "  [Transfer]   Using Robocopy for reliable transfer..." -ForegroundColor Gray
            
            # Robocopy args:
            # /E = include subdirectories including empty
            # /R:3 = retry 3 times on failed copies
            # /W:10 = wait 10 seconds between retries
            # /MT:4 = 4 threads (conservative for network)
            # /NP = no progress percentage (cleaner logs)
            # /NDL = no directory list
            # /NFL = no file list (reduce noise)
            # /NJH = no job header
            # /NJS = no job summary (we do our own)
            $robocopyArgs = @(
                "`"$SourcePath`"",
                "`"$DestinationPath`"",
                "/E", "/R:3", "/W:10", "/MT:4",
                "/NP", "/NDL"
            )
            
            $process = Start-Process -FilePath "robocopy.exe" `
                -ArgumentList ($robocopyArgs -join " ") `
                -Wait -PassThru -NoNewWindow
            
            # Robocopy exit codes: 0-7 are success/info, 8+ are errors
            if ($process.ExitCode -le 7) {
                Write-Host "  [Transfer]   Robocopy complete (exit: $($process.ExitCode))" -ForegroundColor Green
                return $true
            }
            else {
                Write-Host "  [Transfer]   Robocopy failed (exit: $($process.ExitCode))" -ForegroundColor Red
                return $false
            }
        }
        else {
            # Fallback: Copy-Item (slower, no retry)
            Write-Host "  [Transfer]   Using Copy-Item (Robocopy not found)..." -ForegroundColor Yellow
            
            Copy-Item -Path $SourcePath -Destination $DestinationPath -Recurse -Force -ErrorAction Stop
            Write-Host "  [Transfer]   Copy complete" -ForegroundColor Green
            return $true
        }
    }
    catch {
        Write-Host "  [Transfer]   ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Confirm-VMTransfer {
    <#
    .SYNOPSIS
        Verifies a VM transfer by comparing folder name, file count, and total size
    .DESCRIPTION
        Compares the local staging copy to the network copy. Checks:
        1. Folder name matches
        2. File count matches
        3. Total size matches (within 1% tolerance for filesystem overhead)
    .RETURNS
        $true if verification passes, $false otherwise
    #>
    param(
        [Parameter(Mandatory=$true)][string]$LocalPath,
        [Parameter(Mandatory=$true)][string]$NetworkPath
    )
    
    try {
        # Check both paths exist
        if (-not (Test-Path $LocalPath)) {
            Write-Host "  [Verify]   Local path missing: $LocalPath" -ForegroundColor Red
            return $false
        }
        if (-not (Test-Path $NetworkPath)) {
            Write-Host "  [Verify]   Network path missing: $NetworkPath" -ForegroundColor Red
            return $false
        }
        
        # Compare folder names
        $localName = Split-Path $LocalPath -Leaf
        $networkName = Split-Path $NetworkPath -Leaf
        
        if ($localName -ne $networkName) {
            Write-Host "  [Verify]   Name mismatch: Local='$localName' vs Network='$networkName'" -ForegroundColor Red
            return $false
        }
        Write-Host "  [Verify]   Name: OK ($localName)" -ForegroundColor Gray
        
        # Compare file counts
        $localFiles = @(Get-ChildItem -Path $LocalPath -Recurse -File -ErrorAction SilentlyContinue)
        $networkFiles = @(Get-ChildItem -Path $NetworkPath -Recurse -File -ErrorAction SilentlyContinue)
        
        if ($localFiles.Count -ne $networkFiles.Count) {
            Write-Host "  [Verify]   File count mismatch: Local=$($localFiles.Count) vs Network=$($networkFiles.Count)" -ForegroundColor Red
            return $false
        }
        Write-Host "  [Verify]   File count: OK ($($localFiles.Count) files)" -ForegroundColor Gray
        
        # Compare total sizes (1% tolerance for filesystem overhead differences)
        $localSize = Get-FolderSizeBytes -FolderPath $LocalPath
        $networkSize = Get-FolderSizeBytes -FolderPath $NetworkPath
        
        $sizeDiffPercent = if ($localSize -gt 0) {
            [math]::Abs(($networkSize - $localSize) / $localSize) * 100
        } else { 0 }
        
        $localSizeGB = [math]::Round($localSize / 1GB, 3)
        $networkSizeGB = [math]::Round($networkSize / 1GB, 3)
        
        if ($sizeDiffPercent -gt 1.0) {
            Write-Host "  [Verify]   Size mismatch: Local=${localSizeGB}GB vs Network=${networkSizeGB}GB (diff: $([math]::Round($sizeDiffPercent, 2))%)" -ForegroundColor Red
            return $false
        }
        Write-Host "  [Verify]   Size: OK (Local=${localSizeGB}GB, Network=${networkSizeGB}GB, diff=$([math]::Round($sizeDiffPercent, 2))%)" -ForegroundColor Gray
        
        return $true
    }
    catch {
        Write-Host "  [Verify]   ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

#endregion

#region Folder Backup Operations

function Invoke-FolderBackup {
    <#
    .SYNOPSIS
        Backs up configured folders with versioning
    #>
    param(
        [array]$Folders,
        
        # Local backup
        [bool]$LocalEnabled = $false,
        [string]$LocalDrive = "E:",
        [string]$LocalPath = "Backup\Folders",
        [int]$LocalRetentionDays = 7,
        [bool]$LocalRetentionBySpace = $true,
        [int]$LocalMinFreeGB = 20,
        
        # Remote backup
        [bool]$RemoteEnabled = $false,
        [string]$RemotePath = "",
        [string]$RemoteUsername = "",
        [string]$RemotePassword = "",
        [int]$RemoteRetentionDays = 7,
        [bool]$RemoteRetentionBySpace = $true,
        [int]$RemoteMinFreeGB = 20
    )
    
    if (-not $Folders -or $Folders.Count -eq 0) {
        Write-Host "  [FolderBackup] No folders configured" -ForegroundColor Yellow
        return $false
    }
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    Write-Host "  FOLDER BACKUP ($($Folders.Count) folders)" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    
    $dateStamp = Get-Date -Format "yyyy-MM-dd_HHmmss"
    $success = $true
    
    foreach ($folder in $Folders) {
        $sourcePath = $folder.Path
        $folderName = if ($folder.Name) { $folder.Name } else { Split-Path $sourcePath -Leaf }
        $keepCopies = if ($folder.KeepCopies) { $folder.KeepCopies } else { 7 }
        
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [FolderBackup] $folderName" -ForegroundColor Yellow
        Write-Host "  [FolderBackup]   Source: $sourcePath" -ForegroundColor Gray
        
        if (-not (Test-Path $sourcePath)) {
            Write-Host "  [FolderBackup]   SKIPPED - Source not found" -ForegroundColor Red
            $success = $false
            continue
        }
        
        # Get source size
        $sourceSize = (Get-ChildItem -Path $sourcePath -Recurse -File -ErrorAction SilentlyContinue | 
            Measure-Object -Property Length -Sum).Sum
        $sourceSizeGB = [math]::Round($sourceSize / 1GB, 2)
        Write-Host "  [FolderBackup]   Size: $sourceSizeGB GB" -ForegroundColor Gray
        
        # LOCAL BACKUP
        if ($LocalEnabled -and ($folder.LocalEnabled -ne $false)) {
            $localFullPath = Join-Path "$LocalDrive\" $LocalPath
            $destFolder = Join-Path $localFullPath $folderName
            $destPath = Join-Path $destFolder $dateStamp
            
            Write-Host "  [FolderBackup]   Local: $destFolder" -ForegroundColor Gray
            
            $spaceOK = Test-SpaceAndCleanup -Path $localFullPath -RequiredGB ($sourceSizeGB + 1) `
                -MinFreeGB $LocalMinFreeGB -AutoCleanup $LocalRetentionBySpace -RetentionDays $LocalRetentionDays
            
            if ($spaceOK) {
                if (-not (Test-Path $destPath)) {
                    New-Item -ItemType Directory -Path $destPath -Force | Out-Null
                }
                
                # Copy with robocopy - capture locked files
                $lockedFilesLog = "$Script:InstallPath\Logs\LockedFiles_$(Get-Date -Format 'yyyyMMdd').log"
                $robocopyLog = "$destPath\robocopy.log"
                $robocopyArgs = "`"$sourcePath`" `"$destPath`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /LOG:`"$robocopyLog`""
                $exitCode = (Start-Process -FilePath "robocopy.exe" -ArgumentList $robocopyArgs -Wait -PassThru -NoNewWindow).ExitCode
                
                # Check for locked/skipped files (exit codes 1-7 are success with varying conditions)
                if ($exitCode -ge 8) {
                    # Parse log for skipped files
                    if (Test-Path $robocopyLog) {
                        $skipped = Select-String -Path $robocopyLog -Pattern "ERROR|FAILED|Access is denied" -SimpleMatch
                        if ($skipped) {
                            "[$([DateTime]::Now.ToString('yyyy-MM-dd HH:mm:ss'))] Folder: $sourcePath" | Out-File $lockedFilesLog -Append
                            $skipped | ForEach-Object { "  $($_.Line)" | Out-File $lockedFilesLog -Append }
                        }
                    }
                    Write-Host "  [FolderBackup]   Local backup: PARTIAL (some files skipped, see log)" -ForegroundColor Yellow
                }
                else {
                    Write-Host "  [FolderBackup]   Local backup: OK" -ForegroundColor Green
                }
                
                # Cleanup log file
                Remove-Item $robocopyLog -Force -ErrorAction SilentlyContinue
                
                # Cleanup old versions
                $existingVersions = Get-ChildItem -Path $destFolder -Directory -ErrorAction SilentlyContinue | 
                    Sort-Object Name -Descending
                
                if ($existingVersions.Count -gt $keepCopies) {
                    $toRemove = $existingVersions | Select-Object -Skip $keepCopies
                    foreach ($old in $toRemove) {
                        Remove-Item -Path $old.FullName -Recurse -Force -ErrorAction SilentlyContinue
                        Write-Host "  [FolderBackup]   Removed old: $($old.Name)" -ForegroundColor Gray
                    }
                }
            }
            else {
                Write-Host "  [FolderBackup]   Local SKIPPED - No space" -ForegroundColor Red
                $success = $false
            }
        }
        
        # REMOTE BACKUP
        if ($RemoteEnabled -and $RemotePath -and ($folder.RemoteEnabled -ne $false)) {
            $destFolder = Join-Path $RemotePath $folderName
            $destPath = Join-Path $destFolder $dateStamp
            
            Write-Host "  [FolderBackup]   Remote: $destFolder" -ForegroundColor Gray
            
            $testResult = Test-BackupPath -Path $RemotePath -Username $RemoteUsername -Password $RemotePassword -CreateIfMissing
            
            if ($testResult.Success) {
                $spaceOK = Test-SpaceAndCleanup -Path $RemotePath -RequiredGB ($sourceSizeGB + 1) `
                    -MinFreeGB $RemoteMinFreeGB -AutoCleanup $RemoteRetentionBySpace -RetentionDays $RemoteRetentionDays
                
                if ($spaceOK) {
                    if (-not (Test-Path $destPath)) {
                        New-Item -ItemType Directory -Path $destPath -Force | Out-Null
                    }
                    
                    # Copy with robocopy - capture locked files
                    $lockedFilesLog = "$Script:InstallPath\Logs\LockedFiles_$(Get-Date -Format 'yyyyMMdd').log"
                    $robocopyLog = "$destPath\robocopy.log"
                    $robocopyArgs = "`"$sourcePath`" `"$destPath`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /LOG:`"$robocopyLog`""
                    $exitCode = (Start-Process -FilePath "robocopy.exe" -ArgumentList $robocopyArgs -Wait -PassThru -NoNewWindow).ExitCode
                    
                    # Check for locked/skipped files
                    if ($exitCode -ge 8) {
                        if (Test-Path $robocopyLog) {
                            $skipped = Select-String -Path $robocopyLog -Pattern "ERROR|FAILED|Access is denied" -SimpleMatch
                            if ($skipped) {
                                "[$([DateTime]::Now.ToString('yyyy-MM-dd HH:mm:ss'))] Remote Folder: $sourcePath" | Out-File $lockedFilesLog -Append
                                $skipped | ForEach-Object { "  $($_.Line)" | Out-File $lockedFilesLog -Append }
                            }
                        }
                        Write-Host "  [FolderBackup]   Remote backup: PARTIAL (some files skipped)" -ForegroundColor Yellow
                    }
                    else {
                        Write-Host "  [FolderBackup]   Remote backup: OK" -ForegroundColor Green
                    }
                    
                    # Cleanup log file
                    Remove-Item $robocopyLog -Force -ErrorAction SilentlyContinue
                }
                else {
                    Write-Host "  [FolderBackup]   Remote SKIPPED - No space" -ForegroundColor Red
                    $success = $false
                }
            }
            else {
                Write-Host "  [FolderBackup]   Remote SKIPPED - $($testResult.Message)" -ForegroundColor Red
                $success = $false
            }
        }
    }
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    Write-Host "  FOLDER BACKUP COMPLETE (Success: $success)" -ForegroundColor $(if ($success) { "Green" } else { "Yellow" })
    Write-Host "  ========================================" -ForegroundColor Cyan
    
    return $success
}

#endregion

#region Source Sync Operations

function Invoke-SourceSync {
    <#
    .SYNOPSIS
        Syncs data from a configured SOURCE to local DESTINATION
    #>
    param(
        [Parameter(Mandatory=$true)][string]$SourceName
    )
    
    $config = Get-SourceConfig -SourceName $SourceName
    
    if (-not $config) {
        Write-Host "  [Sync] Source '$SourceName' not found" -ForegroundColor Red
        return $false
    }
    
    if (-not $config.Enabled) {
        Write-Host "  [Sync] Source '$SourceName' is disabled" -ForegroundColor Yellow
        return $false
    }
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    Write-Host "  SYNC FROM: $SourceName" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    
    # Test connection
    $testResult = Test-SourceConnection -SourceConfig $config
    
    if (-not $testResult.Success) {
        Write-Host "  [Sync] Connection FAILED: $($testResult.Message)" -ForegroundColor Red
        
        $config.LastSync = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
        $config.LastSyncStatus = "Failed"
        $config.LastSyncMessage = $testResult.Message
        Save-SourceConfig -SourceName $SourceName -Config $config
        
        return $false
    }
    
    $target = if ($config.UseStaticIP) { $config.IPAddress } else { $config.ComputerName }
    $sharePath = if ($config.ShareType -eq "Custom") { $config.CustomSharePath } else { "\\$target\ABCD-Migration$" }
    
    $success = $true
    
    # Sync VM Backups
    if ($config.SyncVMBackups -and $config.VMBackupSourcePath) {
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [Sync] VM Backups from $($config.VMBackupSourcePath)" -ForegroundColor Yellow
        
        $localPath = Join-Path "$($config.VMBackupLocalDrive)\" $config.VMBackupLocalPath
        $localPath = Join-Path $localPath $SourceName
        
        $spaceOK = Test-SpaceAndCleanup -Path $localPath -RequiredGB 50 `
            -MinFreeGB $config.VMBackupMinFreeGB -AutoCleanup $config.VMBackupRetentionBySpace `
            -RetentionDays $config.VMBackupRetentionDays
        
        if ($spaceOK) {
            if (-not (Test-Path $localPath)) {
                New-Item -ItemType Directory -Path $localPath -Force | Out-Null
            }
            
            $robocopyArgs = "`"$($config.VMBackupSourcePath)`" `"$localPath`" /E /COPYALL /R:3 /W:5 /MT:8 /NP"
            Write-Host "  [Sync]   Copying..." -ForegroundColor Gray
            cmd /c "robocopy $robocopyArgs" 2>$null | Out-Null
            
            Write-Host "  [Sync]   VM Backups: OK" -ForegroundColor Green
        }
        else {
            Write-Host "  [Sync]   VM Backups SKIPPED - No space" -ForegroundColor Red
            $success = $false
        }
    }
    
    # Sync Folder Backups
    if ($config.SyncFolderBackups -and $config.FolderBackupSourcePath) {
        Write-Host "" -ForegroundColor Yellow
        Write-Host "  [Sync] Folder Backups from $($config.FolderBackupSourcePath)" -ForegroundColor Yellow
        
        $localPath = Join-Path "$($config.FolderBackupLocalDrive)\" $config.FolderBackupLocalPath
        $localPath = Join-Path $localPath $SourceName
        
        $spaceOK = Test-SpaceAndCleanup -Path $localPath -RequiredGB 20 `
            -MinFreeGB $config.FolderBackupMinFreeGB -AutoCleanup $config.FolderBackupRetentionBySpace `
            -RetentionDays $config.FolderBackupRetentionDays
        
        if ($spaceOK) {
            if (-not (Test-Path $localPath)) {
                New-Item -ItemType Directory -Path $localPath -Force | Out-Null
            }
            
            $robocopyArgs = "`"$($config.FolderBackupSourcePath)`" `"$localPath`" /E /COPYALL /R:3 /W:5 /MT:8 /NP"
            Write-Host "  [Sync]   Copying..." -ForegroundColor Gray
            cmd /c "robocopy $robocopyArgs" 2>$null | Out-Null
            
            Write-Host "  [Sync]   Folder Backups: OK" -ForegroundColor Green
        }
        else {
            Write-Host "  [Sync]   Folder Backups SKIPPED - No space" -ForegroundColor Red
            $success = $false
        }
    }
    
    # Update status
    $config.LastSync = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    $config.LastSyncStatus = if ($success) { "Success" } else { "Partial" }
    $config.LastSyncMessage = if ($success) { "All items synced" } else { "Some items failed" }
    Save-SourceConfig -SourceName $SourceName -Config $config
    
    Write-Host "" -ForegroundColor Cyan
    Write-Host "  ========================================" -ForegroundColor Cyan
    Write-Host "  SYNC COMPLETE (Success: $success)" -ForegroundColor $(if ($success) { "Green" } else { "Yellow" })
    Write-Host "  ========================================" -ForegroundColor Cyan
    
    return $success
}

#endregion

#region Scheduled Tasks

function New-BackupSchedule {
    <#
    .SYNOPSIS
        Creates scheduled task for backup operation
    #>
    param(
        [Parameter(Mandatory=$true)]
        [ValidateSet("VMBackup", "VMBackupLocal", "VMBackupRemote", "VMBackupStaggered", "FolderBackup", "Cleanup", "SourceSync")]
        [string]$TaskType,
        
        [string]$Time = "22:00",
        [string]$SourceName = ""  # For SourceSync
    )
    
    $taskName = if ($TaskType -eq "SourceSync" -and $SourceName) {
        "ABCD-Migration-Sync-$SourceName"
    }
    else {
        "ABCD-Migration-$TaskType"
    }
    
    $scriptPath = "$Script:InstallPath\Scripts\$taskName.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    
    if (-not (Test-Path $scriptDir)) {
        New-Item -ItemType Directory -Path $scriptDir -Force | Out-Null
    }
    
    # Generate script content
    $scriptContent = @"
#Requires -RunAsAdministrator
# Auto-generated: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')
`$ErrorActionPreference = 'Continue'
`$Script:InstallPath = "C:\ABCD-Migration"
`$Script:LogFile = "`$Script:InstallPath\Logs\${taskName}_`$(Get-Date -Format 'yyyyMMdd_HHmmss').log"

Start-Transcript -Path `$Script:LogFile -Append

try {
    . "`$Script:InstallPath\Modules\Backup-Management.ps1"
"@
    
    switch ($TaskType) {
        "VMBackup" {
            $scriptContent += @"

    `$config = Get-MasterBackupConfig
    `$vm = `$config.LocalSettings.VMBackup
    
    Invoke-VMBackup -VMNames `$vm.VMNames ``
        -LocalEnabled `$vm.LocalEnabled -LocalDrive `$vm.LocalDrive -LocalPath `$vm.LocalPath ``
        -LocalRetentionDays `$vm.LocalRetentionDays -LocalRetentionBySpace `$vm.LocalRetentionBySpace ``
        -LocalMinFreeGB `$vm.LocalMinFreeGB ``
        -RemoteEnabled `$vm.RemoteEnabled -RemotePath `$vm.RemotePath ``
        -RemoteRetentionDays `$vm.RemoteRetentionDays -RemoteRetentionBySpace `$vm.RemoteRetentionBySpace ``
        -RemoteMinFreeGB `$vm.RemoteMinFreeGB ``
        -CleanSnapshots `$true
"@
        }
        
        "VMBackupLocal" {
            $scriptContent += @"

    `$config = Get-MasterBackupConfig
    `$vm = `$config.LocalSettings.VMBackup
    
    # LOCAL ONLY backup - snapshots always cleaned
    Invoke-VMBackup -VMNames `$vm.VMNames ``
        -LocalEnabled `$true -LocalDrive `$vm.LocalDrive -LocalPath `$vm.LocalPath ``
        -LocalRetentionDays `$vm.LocalRetentionDays -LocalRetentionBySpace `$vm.LocalRetentionBySpace ``
        -LocalMinFreeGB `$vm.LocalMinFreeGB ``
        -RemoteEnabled `$false ``
        -CleanSnapshots `$true
"@
        }
        
        "VMBackupRemote" {
            $scriptContent += @"

    `$config = Get-MasterBackupConfig
    `$vm = `$config.LocalSettings.VMBackup
    
    # REMOTE ONLY backup - snapshots always cleaned
    Invoke-VMBackup -VMNames `$vm.VMNames ``
        -LocalEnabled `$false ``
        -RemoteEnabled `$true -RemotePath `$vm.RemotePath ``
        -RemoteRetentionDays `$vm.RemoteRetentionDays -RemoteRetentionBySpace `$vm.RemoteRetentionBySpace ``
        -RemoteMinFreeGB `$vm.RemoteMinFreeGB ``
        -CleanSnapshots `$true
"@
        }
        
        "VMBackupStaggered" {
            $scriptContent += @"

    `$config = Get-MasterBackupConfig
    `$vm = `$config.LocalSettings.VMBackup
    
    # STAGGERED NETWORK backup
    # Exports each VM locally first, cleans snapshots, transfers to network,
    # verifies size/name match, deletes local staging, then starts next VM
    Invoke-VMBackup -VMNames `$vm.VMNames ``
        -LocalEnabled `$false ``
        -RemoteEnabled `$true -RemotePath `$vm.NetworkTargetPath ``
        -RemoteRetentionDays `$vm.RemoteRetentionDays -RemoteRetentionBySpace `$vm.RemoteRetentionBySpace ``
        -RemoteMinFreeGB `$vm.RemoteMinFreeGB ``
        -CleanSnapshots `$true
"@
        }
        
        "FolderBackup" {
            $scriptContent += @"

    `$config = Get-MasterBackupConfig
    `$fb = `$config.LocalSettings.FolderBackup
    
    Invoke-FolderBackup -Folders `$fb.Folders ``
        -LocalEnabled `$fb.LocalEnabled -LocalDrive `$fb.LocalDrive -LocalPath `$fb.LocalPath ``
        -LocalRetentionDays `$fb.LocalRetentionDays -LocalRetentionBySpace `$fb.LocalRetentionBySpace ``
        -LocalMinFreeGB `$fb.LocalMinFreeGB ``
        -RemoteEnabled `$fb.RemoteEnabled -RemotePath `$fb.RemotePath ``
        -RemoteRetentionDays `$fb.RemoteRetentionDays -RemoteRetentionBySpace `$fb.RemoteRetentionBySpace ``
        -RemoteMinFreeGB `$fb.RemoteMinFreeGB
"@
        }
        
        "Cleanup" {
            $scriptContent += @"

    `$config = Get-MasterBackupConfig
    `$cleanup = `$config.LocalSettings.Cleanup
    
    foreach (`$path in `$cleanup.Paths) {
        Invoke-BackupCleanup -Path `$path -RetentionDays `$cleanup.RetentionDays
    }
"@
        }
        
        "SourceSync" {
            $scriptContent += @"

    Invoke-SourceSync -SourceName "$SourceName"
"@
        }
    }
    
    $scriptContent += @"

}
catch {
    Write-Host "ERROR: `$(`$_.Exception.Message)" -ForegroundColor Red
}
finally {
    Stop-Transcript
}
"@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8 -Force
    
    # Remove existing task
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    # Parse time
    $timeParts = $Time -split ':'
    $triggerTime = "$($timeParts[0]):$($timeParts[1].PadLeft(2,'0'))"
    
    # Create task
    $action = New-ScheduledTaskAction -Execute "powershell.exe" `
        -Argument "-NoProfile -ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    $trigger = New-ScheduledTaskTrigger -Daily -At $triggerTime
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries `
        -StartWhenAvailable -ExecutionTimeLimit (New-TimeSpan -Hours 8)
    
    try {
        Register-ScheduledTask -TaskName $taskName -Action $action -Trigger $trigger `
            -Principal $principal -Settings $settings -Description "ABCD Migration - $TaskType" -Force | Out-Null
        
        Write-Host "  [Schedule] Created: $taskName at $Time" -ForegroundColor Green
        return $true
    }
    catch {
        Write-Host "  [Schedule] ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Remove-BackupSchedule {
    param(
        [string]$TaskType,
        [string]$SourceName = ""
    )
    
    $taskName = if ($TaskType -eq "SourceSync" -and $SourceName) {
        "ABCD-Migration-Sync-$SourceName"
    }
    else {
        "ABCD-Migration-$TaskType"
    }
    
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    Write-Host "  [Schedule] Removed: $taskName" -ForegroundColor Yellow
}

#endregion

#region Autologon Configuration

function Set-AutoLogon {
    <#
    .SYNOPSIS
        Configures automatic login using Sysinternals Autologon64
    .DESCRIPTION
        Sets up automatic login for scheduled backup tasks to run
        even when no user is logged in. Uses Autologon64.exe.
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$Username,
        
        [Parameter(Mandatory=$true)]
        [string]$Password,
        
        [string]$Domain = "."  # "." for local, or domain name
    )
    
    $autologonPath = "$Script:InstallPath\Tools\Autologon64.exe"
    
    if (-not (Test-Path $autologonPath)) {
        Write-Host "  [Autologon] ERROR: Autologon64.exe not found at $autologonPath" -ForegroundColor Red
        return @{ Success = $false; Message = "Autologon64.exe not found" }
    }
    
    try {
        # Run Autologon64 silently with /accepteula
        $process = Start-Process -FilePath $autologonPath -ArgumentList @(
            "/accepteula",
            $Username,
            $Domain,
            $Password
        ) -Wait -PassThru -NoNewWindow
        
        if ($process.ExitCode -eq 0) {
            Write-Host "  [Autologon] Configured for user: $Username" -ForegroundColor Green
            return @{ Success = $true; Message = "Auto-login configured for $Username" }
        }
        else {
            Write-Host "  [Autologon] Failed with exit code: $($process.ExitCode)" -ForegroundColor Red
            return @{ Success = $false; Message = "Autologon failed with exit code $($process.ExitCode)" }
        }
    }
    catch {
        Write-Host "  [Autologon] ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return @{ Success = $false; Message = $_.Exception.Message }
    }
}

function Remove-AutoLogon {
    <#
    .SYNOPSIS
        Removes automatic login configuration
    #>
    
    $autologonPath = "$Script:InstallPath\Tools\Autologon64.exe"
    
    if (-not (Test-Path $autologonPath)) {
        # Try to clear registry manually
        try {
            $regPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
            Set-ItemProperty -Path $regPath -Name "AutoAdminLogon" -Value "0" -ErrorAction Stop
            Remove-ItemProperty -Path $regPath -Name "DefaultPassword" -ErrorAction SilentlyContinue
            Write-Host "  [Autologon] Removed via registry" -ForegroundColor Yellow
            return @{ Success = $true; Message = "Auto-login removed" }
        }
        catch {
            return @{ Success = $false; Message = "Failed to remove auto-login" }
        }
    }
    
    try {
        # Run Autologon64 with empty password to disable
        Start-Process -FilePath $autologonPath -ArgumentList "/accepteula" -Wait -NoNewWindow
        
        # Also clear registry
        $regPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
        Set-ItemProperty -Path $regPath -Name "AutoAdminLogon" -Value "0" -ErrorAction SilentlyContinue
        Remove-ItemProperty -Path $regPath -Name "DefaultPassword" -ErrorAction SilentlyContinue
        
        Write-Host "  [Autologon] Removed" -ForegroundColor Yellow
        return @{ Success = $true; Message = "Auto-login removed" }
    }
    catch {
        return @{ Success = $false; Message = $_.Exception.Message }
    }
}

function Get-AutoLogonStatus {
    <#
    .SYNOPSIS
        Checks if auto-login is currently configured
    #>
    
    try {
        $regPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
        $autoLogin = Get-ItemProperty -Path $regPath -Name "AutoAdminLogon" -ErrorAction SilentlyContinue
        $defaultUser = Get-ItemProperty -Path $regPath -Name "DefaultUserName" -ErrorAction SilentlyContinue
        
        if ($autoLogin.AutoAdminLogon -eq "1" -and $defaultUser.DefaultUserName) {
            return @{
                Enabled = $true
                Username = $defaultUser.DefaultUserName
                Message = "Auto-login enabled for: $($defaultUser.DefaultUserName)"
            }
        }
        else {
            return @{
                Enabled = $false
                Username = ""
                Message = "Auto-login not configured"
            }
        }
    }
    catch {
        return @{
            Enabled = $false
            Username = ""
            Message = "Unable to check auto-login status"
        }
    }
}

#endregion

