#Requires -RunAsAdministrator

# =============================================================================
# FILE MIGRATION MODULE - WITH CRITICAL SYSTEM EXCLUSIONS
# =============================================================================
# WARNING: Copying Program Files between different Windows versions can cause
# BSOD if system files are overwritten. This module excludes all known
# Windows/Microsoft system folders to prevent this.
# =============================================================================

# Track copy issues for end-of-migration summary (never fail, just report)
$Script:CopyIssues = @()

function Clear-CopyIssues {
    $Script:CopyIssues = @()
}

function Add-CopyIssue {
    param([string]$Issue)
    if (-not $Script:CopyIssues) { $Script:CopyIssues = @() }
    $Script:CopyIssues += $Issue
}

function Get-CopyIssues {
    return $Script:CopyIssues
}

function Show-CopyIssuesSummary {
    <#
    .SYNOPSIS
        Shows a popup summary of any files that couldn't be copied (locked, permission denied, etc.)
        Call this at the end of migration
    #>
    if ($Script:CopyIssues -and $Script:CopyIssues.Count -gt 0) {
        $msg = "Migration completed with some files that could not be copied:`n`n"
        $msg += ($Script:CopyIssues | Select-Object -First 20) -join "`n"
        
        if ($Script:CopyIssues.Count -gt 20) {
            $msg += "`n`n... and $($Script:CopyIssues.Count - 20) more issues (see log file)"
        }
        
        $msg += "`n`nThese files may have been locked or in use. Check the log for details."
        
        [System.Windows.Forms.MessageBox]::Show($msg, "Migration Notes", [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
        
        # Log all issues
        if ($Script:LogFile) {
            $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
            "[$timestamp] [Summary] Copy Issues:" | Out-File -FilePath $Script:LogFile -Append -Encoding UTF8
            foreach ($issue in $Script:CopyIssues) {
                "[$timestamp] [Summary]   $issue" | Out-File -FilePath $Script:LogFile -Append -Encoding UTF8
            }
        }
        
        return $true
    }
    return $false
}

# Folders to ALWAYS exclude from Program Files copy
$Script:ProgramFilesExclusions = @(
    # Windows Core Components - NEVER COPY THESE
    'Windows Defender'
    'Windows Defender Advanced Threat Protection'
    'Windows Mail'
    'Windows Media Player'
    'Windows Multimedia Platform'
    'Windows NT'
    'Windows Photo Viewer'
    'Windows Portable Devices'
    'Windows Security'
    'Windows Sidebar'
    'WindowsPowerShell'
    'WindowsApps'
    'ModifiableWindowsApps'
    'Internet Explorer'
    'Microsoft Update Health Tools'
    'Microsoft OneDrive'
    'OneDrive'
    
    # Microsoft System Components - NEVER COPY
    'Microsoft'
    'Microsoft.NET'
    'Microsoft SDKs'
    'Microsoft SQL Server'
    'Microsoft Visual Studio'
    'Microsoft Office'
    'Microsoft Office 15'
    'Microsoft Office 16'
    
    # Common Files - System components
    'Common Files'
    
    # Package Cache and Installers
    'Package Cache'
    
    # Server Roles - NEVER COPY
    'Windows Server'
    'Reference Assemblies'
    
    # Development/Runtime - Version specific
    'dotnet'
    'IIS'
    'IIS Express'
    'MSBuild'
    'PowerShell'
    
    # Virtualization
    'Hyper-V'
    'VMware'
    
    # Security software - version specific
    'Malwarebytes'
    'ESET'
    'Norton'
    'McAfee'
    'Symantec'
    'Kaspersky'
    'Avast'
    'AVG'
    'Bitdefender'
)

# Folders to ALWAYS exclude from ProgramData copy  
$Script:ProgramDataExclusions = @(
    # ===========================================
    # WINDOWS SYSTEM - NEVER COPY
    # ===========================================
    'Microsoft'
    'Package Cache'
    'Packages'
    'regid.1991-06.com.microsoft'
    'ssh'
    'Start Menu'
    'SystemData'
    'USOPrivate'
    'USOShared'
    'WindowsHolographicDevices'
    'Application Data'
    'Desktop'
    'Documents'
    'Favorites'
    
    # ===========================================
    # LICENSING FOLDERS ONLY - Surgical exclusion
    # Keep settings/templates, exclude license tokens
    # ===========================================
    
    # QuickBooks / Intuit - ONLY licensing folders
    'Intuit\Entitlement Client'
    'Intuit\Entitlement Client\v6'
    'Intuit\Entitlement Client\v8'
    'COMMON FILES\INTUIT\QBUpdate'
    
    # Adobe - ONLY licensing folders (keep preferences)
    'Adobe\SLStore'
    'Adobe\FlexNet'
    
    # Autodesk - ONLY licensing folders
    'Autodesk\AdLM'
    'Autodesk\CLM'
    'FLEXnet'
    
    # Sage - ONLY licensing
    'Sage\Sage 50\Activation'
    
    # Corel - ONLY licensing
    'Corel\Corel Protected'
)

# Folders to ALWAYS exclude from User profile copy
$Script:UserProfileExclusions = @(
    # ===========================================
    # WINDOWS SYSTEM - NEVER COPY
    # ===========================================
    'AppData\Local\Microsoft\Windows'
    'AppData\Local\Microsoft\Edge'
    'AppData\Local\Microsoft\WindowsApps'
    'AppData\Local\Packages'
    'AppData\Local\Temp'
    'AppData\Local\D3DSCache'
    'AppData\Local\CrashDumps'
    'AppData\Local\VirtualStore'
    'AppData\Local\ConnectedDevicesPlatform'
    'AppData\LocalLow\Microsoft'
    'AppData\Roaming\Microsoft\Windows'
    'AppData\Roaming\Microsoft\Internet Explorer'
    'AppData\Roaming\Microsoft\SystemCertificates'
    'AppData\Roaming\Microsoft\Credentials'
    'AppData\Roaming\Microsoft\Crypto'
    'AppData\Roaming\Microsoft\Protect'
    
    # ===========================================
    # LICENSING FOLDERS IN USER PROFILE
    # Keep app settings, exclude license tokens
    # ===========================================
    'AppData\Local\Adobe\OOBE'
    'AppData\Roaming\Adobe\OOBE'
)

# License FILES to exclude (used in robocopy /XF)
$Script:LicenseFileExclusions = @(
    '*.ecml'
    'qbregistration.dat'
    '*.entitlement'
    'host.dat'
    'licensing.dat'
    'FLEXnet*.dat'
    'adobe_id*'
)

function Export-RegistryComplete {
    param(
        [Parameter(Mandatory=$true)]
        [string]$BackupPath
    )
    
    Write-Host "  [Registry] Extracting license keys for reference..." -ForegroundColor Cyan
    Write-Host "  [Registry] NOTE: License KEYS will be saved, but licensing FILES will NOT be copied" -ForegroundColor Yellow
    Write-Host "  [Registry]       (Hardware-bound licenses must be re-entered on destination)" -ForegroundColor Yellow
    
    $regPath = Join-Path $BackupPath "Registry"
    if (!(Test-Path $regPath)) {
        New-Item -Path $regPath -ItemType Directory -Force | Out-Null
    }
    
    # License keys to save for user reference
    $licenseKeys = @{
        QuickBooks = @()
        Adobe = @()
        Autodesk = @()
        Sage = @()
        Office = @()
    }
    
    try {
        # ===========================================
        # QUICKBOOKS - Extract license key only
        # ===========================================
        if (Test-Path "HKLM:\SOFTWARE\Intuit") {
            Write-Host "  [Registry] Scanning for QuickBooks license keys..." -ForegroundColor Yellow
            
            # Try to find QB license key in various locations
            $qbPaths = @(
                "HKLM:\SOFTWARE\Intuit\QuickBooks\CurrentVersion",
                "HKLM:\SOFTWARE\WOW6432Node\Intuit\QuickBooks\CurrentVersion",
                "HKLM:\SOFTWARE\Intuit\QuickBooks Enterprise Solutions\CurrentVersion"
            )
            
            foreach ($qbPath in $qbPaths) {
                if (Test-Path $qbPath) {
                    $props = Get-ItemProperty -Path $qbPath -ErrorAction SilentlyContinue
                    if ($props.LicenseNumber) {
                        $licenseKeys.QuickBooks += "License: $($props.LicenseNumber)"
                    }
                    if ($props.ProductNumber) {
                        $licenseKeys.QuickBooks += "Product: $($props.ProductNumber)"
                    }
                }
            }
            
            # Also check qbregistration.dat for license info
            $qbRegFiles = Get-ChildItem "C:\ProgramData\Intuit" -Filter "qbregistration.dat" -Recurse -ErrorAction SilentlyContinue
            foreach ($file in $qbRegFiles) {
                $content = Get-Content $file.FullName -Raw -ErrorAction SilentlyContinue
                if ($content -match "LicenseNumber=([^\r\n]+)") {
                    $licenseKeys.QuickBooks += "License (from file): $($Matches[1])"
                }
            }
            
            if ($licenseKeys.QuickBooks.Count -gt 0) {
                Write-Host "  [Registry] QuickBooks license key(s) found!" -ForegroundColor Green
            } else {
                Write-Host "  [Registry] QuickBooks installed but license key not found in registry" -ForegroundColor Yellow
            }
            
            # DO NOT export Intuit registry - it contains hardware-bound tokens
            Write-Host "  [Registry] SKIPPED: Intuit registry (contains hardware-bound tokens)" -ForegroundColor Gray
        }
        
        # ===========================================
        # ADOBE - Note for user
        # ===========================================
        if (Test-Path "HKLM:\SOFTWARE\Adobe") {
            Write-Host "  [Registry] Adobe detected - requires sign-in on destination" -ForegroundColor Yellow
            $licenseKeys.Adobe += "Sign in with your Adobe ID on destination"
            # DO NOT export Adobe registry
            Write-Host "  [Registry] SKIPPED: Adobe registry (cloud-based licensing)" -ForegroundColor Gray
        }
        
        # ===========================================
        # AUTODESK - Note for user
        # ===========================================
        if (Test-Path "HKLM:\SOFTWARE\Autodesk") {
            Write-Host "  [Registry] Autodesk detected - requires reactivation on destination" -ForegroundColor Yellow
            $licenseKeys.Autodesk += "Reactivate with your Autodesk account or serial number"
            # DO NOT export Autodesk registry
            Write-Host "  [Registry] SKIPPED: Autodesk registry (hardware-bound licensing)" -ForegroundColor Gray
        }
        
        # ===========================================
        # OFFICE - Check for product key
        # ===========================================
        $officePath = "HKLM:\SOFTWARE\Microsoft\Office\ClickToRun\Configuration"
        if (Test-Path $officePath) {
            $officeProps = Get-ItemProperty -Path $officePath -ErrorAction SilentlyContinue
            if ($officeProps.ProductReleaseIds) {
                Write-Host "  [Registry] Microsoft Office detected: $($officeProps.ProductReleaseIds)" -ForegroundColor Yellow
                $licenseKeys.Office += "Product: $($officeProps.ProductReleaseIds)"
                $licenseKeys.Office += "Sign in with your Microsoft account on destination"
            }
            Write-Host "  [Registry] SKIPPED: Office registry (sign-in required)" -ForegroundColor Gray
        }
        
        # ===========================================
        # ABCD Systems - Safe to export (no hardware binding)
        # ===========================================
        if (Test-Path "HKLM:\SOFTWARE\ABCD Systems") {
            Write-Host "  [Registry] Backing up ABCD Systems..." -ForegroundColor Yellow
            reg export "HKLM\SOFTWARE\ABCD Systems" "$regPath\ABCD_Systems.reg" /y 2>$null
            Write-Host "  [Registry] ABCD Systems backed up" -ForegroundColor Green
        }
        
        # ===========================================
        # Save license keys to file for user reference
        # ===========================================
        $keyFile = Join-Path $regPath "LICENSE_KEYS_REFERENCE.txt"
        $keyContent = @"
============================================================
LICENSE KEYS REFERENCE - Generated $(Get-Date)
============================================================

These keys were extracted from the SOURCE server.
You will need to re-enter them on the DESTINATION server.

Hardware-bound licenses CANNOT be migrated by file copy.
You must reactivate each application with these keys.

============================================================

"@
        
        foreach ($app in $licenseKeys.Keys) {
            if ($licenseKeys[$app].Count -gt 0) {
                $keyContent += "[$app]`r`n"
                foreach ($key in $licenseKeys[$app]) {
                    $keyContent += "  $key`r`n"
                }
                $keyContent += "`r`n"
            }
        }
        
        $keyContent += @"
============================================================
IMPORTANT NOTES:
============================================================

QuickBooks:
  1. Install QuickBooks on destination (if not already)
  2. Open QuickBooks
  3. Enter your license key when prompted
  4. Complete activation

Microsoft Office 365:
  1. Open any Office app
  2. Sign in with your Microsoft account
  3. License activates automatically

Adobe Creative Cloud:
  1. Open Creative Cloud app
  2. Sign in with your Adobe ID
  3. License activates automatically

If activation fails, contact the software vendor.
============================================================
"@
        
        $keyContent | Out-File -FilePath $keyFile -Encoding UTF8
        Write-Host "  [Registry] License keys saved to: LICENSE_KEYS_REFERENCE.txt" -ForegroundColor Green
        
        Write-Host "  [Registry] Registry export complete!" -ForegroundColor Green
        Write-Host "  [Registry] Location: $regPath" -ForegroundColor Gray
        
        return $true
    }
    catch {
        Write-Host "  [Registry] ERROR during registry backup: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Import-RegistryComplete {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourcePath
    )
    
    Write-Host "  [Registry] Starting registry import..." -ForegroundColor Cyan
    
    $regPath = Join-Path $SourcePath "Registry"
    
    if (!(Test-Path $regPath)) {
        Write-Host "  [Registry] WARNING: No registry backup found at $regPath" -ForegroundColor Yellow
        return $false
    }
    
    try {
        $regFiles = Get-ChildItem -Path $regPath -Filter "*.reg" -ErrorAction SilentlyContinue
        
        if ($regFiles.Count -eq 0) {
            Write-Host "  [Registry] No registry files found to import" -ForegroundColor Yellow
            return $false
        }
        
        foreach ($regFile in $regFiles) {
            Write-Host "  [Registry] Importing: $($regFile.Name)" -ForegroundColor Yellow
            $result = reg import $regFile.FullName 2>&1
            if ($LASTEXITCODE -eq 0) {
                Write-Host "  [Registry] Imported: $($regFile.Name)" -ForegroundColor Green
            } else {
                Write-Host "  [Registry] WARNING: Issues importing $($regFile.Name)" -ForegroundColor Yellow
            }
        }
        
        Write-Host "  [Registry] Registry import complete!" -ForegroundColor Green
        return $true
    }
    catch {
        Write-Host "  [Registry] ERROR during registry import: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Copy-ProgramFilesWithACL {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourceShare,
        
        [string]$DestinationPath = "C:\Program Files",
        
        [int]$Threads = 8,
        
        [bool]$Overwrite = $false,
        
        [bool]$NewerOnly = $false
    )
    
    Write-Host "  [Files] ================================================" -ForegroundColor Cyan
    Write-Host "  [Files] Copying Program Files WITH SYSTEM EXCLUSIONS" -ForegroundColor Cyan
    Write-Host "  [Files] ================================================" -ForegroundColor Cyan
    Write-Host "  [Files] Source: $SourceShare" -ForegroundColor Gray
    Write-Host "  [Files] Destination: $DestinationPath" -ForegroundColor Gray
    Write-Host "  [Files] Overwrite=$Overwrite, NewerOnly=$NewerOnly" -ForegroundColor Gray
    Write-Host "  [Files] Excluding $($Script:ProgramFilesExclusions.Count) system/Windows folders" -ForegroundColor Yellow
    
    if (!(Test-Path $SourceShare)) {
        Write-Host "  [Files] WARNING: Source path not accessible: $SourceShare" -ForegroundColor Yellow
        Add-CopyIssue "Program Files: Source not accessible ($SourceShare)"
        return $true  # Never fail
    }
    
    # Build exclusion string for robocopy /XD parameter
    $excludeArgs = ""
    foreach ($exclude in $Script:ProgramFilesExclusions) {
        $excludeArgs += " `"$exclude`""
    }
    
    # Determine copy flags
    $copyFlags = ""
    if ($Overwrite -eq $true) {
        if ($NewerOnly -eq $true) {
            $copyFlags = "/XO"
            Write-Host "  [Files] Mode: COPY if source is NEWER" -ForegroundColor Yellow
        } else {
            $copyFlags = "/IS /IT"
            Write-Host "  [Files] Mode: FORCE OVERWRITE ALL" -ForegroundColor Green
        }
    } else {
        $copyFlags = "/XC /XN /XO"
        Write-Host "  [Files] Mode: SKIP existing files" -ForegroundColor Yellow
    }
    
    Write-Host "  [Files] Robocopy flags: $copyFlags" -ForegroundColor Cyan
    
    try {
        $roboLog = "$Script:InstallPath\Logs\robocopy_ProgramFiles_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
        Write-Host "  [Files] Log: $roboLog" -ForegroundColor DarkGray
        
        # Run via cmd.exe to avoid escaping issues
        $cmdArgs = "/c robocopy ""$SourceShare"" ""$DestinationPath"" /E /COPY:DAT /MT:$Threads /XJ /R:1 /W:1 /NP $copyFlags /XD$excludeArgs /XF *.log *.tmp desktop.ini thumbs.db /LOG:""$roboLog"""
        
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList $cmdArgs -Wait -PassThru -NoNewWindow
        
        # Parse results
        $failedCount = 0
        if (Test-Path $roboLog) {
            $logLines = Get-Content $roboLog -Tail 15 -ErrorAction SilentlyContinue
            foreach ($line in $logLines) {
                if ($line -match "^\s*Files\s*:\s*(\d+)\s+(\d+)\s+(\d+)\s+\d+\s+(\d+)") {
                    $failedCount = [int]$Matches[4]
                    Write-Host "  [Files] Files - Total: $($Matches[1]), Copied: $($Matches[2]), Skipped: $($Matches[3]), Failed: $failedCount" -ForegroundColor Cyan
                }
            }
        }
        
        Write-Host "  [Files] Exit code: $($process.ExitCode)" -ForegroundColor Gray
        
        if ($failedCount -gt 0 -or $process.ExitCode -ge 8) {
            Write-Host "  [Files] Some files could not be copied (locked or in use)" -ForegroundColor Yellow
            Add-CopyIssue "Program Files: $failedCount file(s) could not be copied (see robocopy log)"
        }
        
        Write-Host "  [Files] Program Files copy complete!" -ForegroundColor Green
        return $true  # Never fail
    }
    catch {
        Write-Host "  [Files] WARNING: $($_.Exception.Message)" -ForegroundColor Yellow
        Add-CopyIssue "Program Files: Error - $($_.Exception.Message)"
        return $true  # Never fail
    }
}

function Copy-ProgramFilesX86WithACL {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourceShare,
        
        [string]$DestinationPath = "C:\Program Files (x86)",
        
        [int]$Threads = 8,
        
        [bool]$Overwrite = $false,
        
        [bool]$NewerOnly = $false
    )
    
    Write-Host "  [Files] ================================================" -ForegroundColor Cyan
    Write-Host "  [Files] Copying Program Files (x86) WITH SYSTEM EXCLUSIONS" -ForegroundColor Cyan
    Write-Host "  [Files] ================================================" -ForegroundColor Cyan
    Write-Host "  [Files] Source: $SourceShare" -ForegroundColor Gray
    Write-Host "  [Files] Destination: $DestinationPath" -ForegroundColor Gray
    Write-Host "  [Files] Overwrite=$Overwrite, NewerOnly=$NewerOnly" -ForegroundColor Gray
    
    if (!(Test-Path $SourceShare)) {
        Write-Host "  [Files] WARNING: Source path not accessible: $SourceShare" -ForegroundColor Yellow
        Add-CopyIssue "Program Files (x86): Source not accessible ($SourceShare)"
        return $true  # Never fail
    }
    
    # Build exclusion string
    $excludeArgs = ""
    foreach ($exclude in $Script:ProgramFilesExclusions) {
        $excludeArgs += " `"$exclude`""
    }
    
    # Determine copy flags
    $copyFlags = ""
    if ($Overwrite -eq $true) {
        if ($NewerOnly -eq $true) {
            $copyFlags = "/XO"
            Write-Host "  [Files] Mode: COPY if source is NEWER" -ForegroundColor Yellow
        } else {
            $copyFlags = "/IS /IT"
            Write-Host "  [Files] Mode: FORCE OVERWRITE ALL" -ForegroundColor Green
        }
    } else {
        $copyFlags = "/XC /XN /XO"
        Write-Host "  [Files] Mode: SKIP existing files" -ForegroundColor Yellow
    }
    
    Write-Host "  [Files] Robocopy flags: $copyFlags" -ForegroundColor Cyan
    
    try {
        $roboLog = "$Script:InstallPath\Logs\robocopy_ProgramFilesX86_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
        Write-Host "  [Files] Log: $roboLog" -ForegroundColor DarkGray
        
        # Run via cmd.exe
        $cmdArgs = "/c robocopy ""$SourceShare"" ""$DestinationPath"" /E /COPY:DAT /MT:$Threads /XJ /R:1 /W:1 /NP $copyFlags /XD$excludeArgs /XF *.log *.tmp desktop.ini thumbs.db /LOG:""$roboLog"""
        
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList $cmdArgs -Wait -PassThru -NoNewWindow
        
        # Parse results
        $failedCount = 0
        if (Test-Path $roboLog) {
            $logLines = Get-Content $roboLog -Tail 15 -ErrorAction SilentlyContinue
            foreach ($line in $logLines) {
                if ($line -match "^\s*Files\s*:\s*(\d+)\s+(\d+)\s+(\d+)\s+\d+\s+(\d+)") {
                    $failedCount = [int]$Matches[4]
                    Write-Host "  [Files] Files - Total: $($Matches[1]), Copied: $($Matches[2]), Skipped: $($Matches[3]), Failed: $failedCount" -ForegroundColor Cyan
                }
            }
        }
        
        Write-Host "  [Files] Exit code: $($process.ExitCode)" -ForegroundColor Gray
        
        if ($failedCount -gt 0 -or $process.ExitCode -ge 8) {
            Write-Host "  [Files] Some files could not be copied (locked or in use)" -ForegroundColor Yellow
            Add-CopyIssue "Program Files (x86): $failedCount file(s) could not be copied"
        }
        
        Write-Host "  [Files] Program Files (x86) copy complete!" -ForegroundColor Green
        return $true  # Never fail
    }
    catch {
        Write-Host "  [Files] WARNING: $($_.Exception.Message)" -ForegroundColor Yellow
        Add-CopyIssue "Program Files (x86): Error - $($_.Exception.Message)"
        return $true  # Never fail
    }
}

function Copy-ProgramDataWithACL {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourceShare,
        
        [int]$Threads = 8,
        
        [bool]$Overwrite = $false,
        
        [bool]$NewerOnly = $false
    )
    
    Write-Host "  [Files] ================================================" -ForegroundColor Cyan
    Write-Host "  [Files] Copying ProgramData (Settings & Templates)" -ForegroundColor Cyan
    Write-Host "  [Files] ================================================" -ForegroundColor Cyan
    Write-Host "  [Files] Source: $SourceShare" -ForegroundColor Gray
    Write-Host "  [Files] Overwrite=$Overwrite, NewerOnly=$NewerOnly" -ForegroundColor Gray
    Write-Host "  [Files] COPYING: App settings, templates, preferences" -ForegroundColor Green
    Write-Host "  [Files] EXCLUDING: License files, Windows system folders" -ForegroundColor Yellow
    
    if (!(Test-Path $SourceShare)) {
        Write-Host "  [Files] WARNING: Source path not accessible: $SourceShare" -ForegroundColor Yellow
        Add-CopyIssue "ProgramData: Source not accessible ($SourceShare)"
        return $true  # Never fail
    }
    
    $destPath = "C:\ProgramData"
    
    # Build exclusion string
    $excludeArgs = ""
    foreach ($exclude in $Script:ProgramDataExclusions) {
        $excludeArgs += " `"$exclude`""
    }
    
    # Determine copy flags
    $copyFlags = ""
    if ($Overwrite -eq $true) {
        if ($NewerOnly -eq $true) {
            $copyFlags = "/XO"
            Write-Host "  [Files] Mode: COPY if source is NEWER" -ForegroundColor Yellow
        } else {
            $copyFlags = "/IS /IT"
            Write-Host "  [Files] Mode: FORCE OVERWRITE ALL" -ForegroundColor Green
        }
    } else {
        $copyFlags = "/XC /XN /XO"
        Write-Host "  [Files] Mode: SKIP existing files" -ForegroundColor Yellow
    }
    
    Write-Host "  [Files] Robocopy flags: $copyFlags" -ForegroundColor Cyan
    
    try {
        $roboLog = "$Script:InstallPath\Logs\robocopy_ProgramData_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
        Write-Host "  [Files] Log: $roboLog" -ForegroundColor DarkGray
        
        # Run via cmd.exe
        $cmdArgs = "/c robocopy ""$SourceShare"" ""$destPath"" /E /COPY:DAT /MT:$Threads /XJ /R:1 /W:1 /NP $copyFlags /XD$excludeArgs /XF *.log *.tmp NTUSER* ntuser* UsrClass.dat* *.ecml qbregistration.dat *.entitlement host.dat cache.dat licensing.dat /LOG:""$roboLog"""
        
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList $cmdArgs -Wait -PassThru -NoNewWindow
        
        # Parse results
        $failedCount = 0
        if (Test-Path $roboLog) {
            $logLines = Get-Content $roboLog -Tail 15 -ErrorAction SilentlyContinue
            foreach ($line in $logLines) {
                if ($line -match "^\s*Files\s*:\s*(\d+)\s+(\d+)\s+(\d+)\s+\d+\s+(\d+)") {
                    $failedCount = [int]$Matches[4]
                    Write-Host "  [Files] Files - Total: $($Matches[1]), Copied: $($Matches[2]), Skipped: $($Matches[3]), Failed: $failedCount" -ForegroundColor Cyan
                }
            }
        }
        
        Write-Host "  [Files] Exit code: $($process.ExitCode)" -ForegroundColor Gray
        
        if ($failedCount -gt 0 -or $process.ExitCode -ge 8) {
            Write-Host "  [Files] Some files could not be copied (locked or in use)" -ForegroundColor Yellow
            Add-CopyIssue "ProgramData: $failedCount file(s) could not be copied"
        }
        
        Write-Host "  [Files] ProgramData copy complete!" -ForegroundColor Green
        return $true  # Never fail
    }
    catch {
        Write-Host "  [Files] WARNING: $($_.Exception.Message)" -ForegroundColor Yellow
        Add-CopyIssue "ProgramData: Error - $($_.Exception.Message)"
        return $true  # Never fail
    }
}

function Copy-UserDataWithACL {
    <#
    .SYNOPSIS
        Copies user profile data with optional AppData inclusion
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourceShare,
        
        [Parameter(Mandatory=$true)]
        [string]$Username,
        
        [int]$Threads = 4,
        
        [bool]$Overwrite = $false,
        
        [bool]$NewerOnly = $false,
        
        [bool]$IncludeAppData = $false
    )
    
    Write-Host "  [Files] Copying user data: $Username..." -ForegroundColor Cyan
    Write-Host "  [Files] Overwrite=$Overwrite, NewerOnly=$NewerOnly" -ForegroundColor Gray
    
    $sourcePath = Join-Path $SourceShare $Username
    $destPath = "C:\Users\$Username"
    
    if (!(Test-Path $sourcePath)) {
        Write-Host "  [Files] SKIPPED: No source profile for $Username" -ForegroundColor Yellow
        return $true  # Not a failure, just skip
    }
    
    if (!(Test-Path $destPath)) {
        Write-Host "  [Files] SKIPPED: Destination profile not created yet for $Username" -ForegroundColor Yellow
        Write-Host "  [Files]          User needs to log in first to create profile" -ForegroundColor Gray
        return $true  # Not a failure, just skip
    }
    
    # Build exclusion string
    $excludeArgs = ""
    foreach ($exclude in $Script:UserProfileExclusions) {
        $excludeArgs += " `"$exclude`""
    }
    
    # Exclude entire AppData if not included
    if (-not $IncludeAppData) {
        $excludeArgs += " `"AppData`""
        Write-Host "  [Files] AppData: EXCLUDED (not selected)" -ForegroundColor Yellow
    } else {
        Write-Host "  [Files] AppData: INCLUDED (with warnings)" -ForegroundColor Red
    }
    
    # Determine copy flags
    $copyFlags = ""
    if ($Overwrite -eq $true) {
        if ($NewerOnly -eq $true) {
            $copyFlags = "/XO"
            Write-Host "  [Files] Mode: COPY if source is NEWER" -ForegroundColor Yellow
        } else {
            $copyFlags = "/IS /IT"
            Write-Host "  [Files] Mode: FORCE OVERWRITE ALL" -ForegroundColor Green
        }
    } else {
        $copyFlags = "/XC /XN /XO"
        Write-Host "  [Files] Mode: SKIP existing files" -ForegroundColor Yellow
    }
    
    Write-Host "  [Files] Robocopy flags: $copyFlags" -ForegroundColor Cyan
    
    try {
        $roboLog = "$Script:InstallPath\Logs\robocopy_User_${Username}_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
        
        # Run via cmd.exe
        $cmdArgs = "/c robocopy ""$sourcePath"" ""$destPath"" /E /COPY:DAT /MT:$Threads /XJ /R:1 /W:1 /NP $copyFlags /XD$excludeArgs /XF NTUSER* ntuser* UsrClass.dat* *.tmp *.log desktop.ini thumbs.db *.ecml qbregistration.dat *.entitlement /LOG:""$roboLog"""
        
        $process = Start-Process -FilePath "cmd.exe" -ArgumentList $cmdArgs -Wait -PassThru -NoNewWindow
        
        # Parse results
        $failedCount = 0
        if (Test-Path $roboLog) {
            $logLines = Get-Content $roboLog -Tail 15 -ErrorAction SilentlyContinue
            foreach ($line in $logLines) {
                if ($line -match "^\s*Files\s*:\s*(\d+)\s+(\d+)\s+(\d+)\s+\d+\s+(\d+)") {
                    $failedCount = [int]$Matches[4]
                    Write-Host "  [Files] $Username - Total: $($Matches[1]), Copied: $($Matches[2]), Skipped: $($Matches[3]), Failed: $failedCount" -ForegroundColor Cyan
                }
            }
        }
        
        Write-Host "  [Files] Exit code: $($process.ExitCode)" -ForegroundColor Gray
        
        if ($failedCount -gt 0 -or $process.ExitCode -ge 8) {
            Write-Host "  [Files] Some files could not be copied for $Username (locked or in use)" -ForegroundColor Yellow
            Add-CopyIssue "User $Username : $failedCount file(s) could not be copied"
        }
        
        Write-Host "  [Files] User data copied: $Username" -ForegroundColor Green
        return $true  # Never fail
    }
    catch {
        Write-Host "  [Files] WARNING copying $Username data: $($_.Exception.Message)" -ForegroundColor Yellow
        Add-CopyIssue "User $Username : Error - $($_.Exception.Message)"
        return $true  # Never fail
    }
}

function Copy-AdditionalDrive {
    <#
    .SYNOPSIS
        Copies an additional data drive (D:, E:, etc.) from source to destination
        Never fails - logs issues for summary at end
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourceShare,
        
        [Parameter(Mandatory=$true)]
        [string]$DestinationDrive,
        
        [int]$Threads = 8,
        
        [bool]$Overwrite = $false,
        
        [bool]$NewerOnly = $false
    )
    
    # Helper to write to both console and log file
    function Write-DriveLog {
        param([string]$Message, [string]$Color = "White")
        Write-Host "  [Files] $Message" -ForegroundColor $Color
        if ($Script:LogFile) {
            $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
            "[$timestamp] [DrivesCopy] $Message" | Out-File -FilePath $Script:LogFile -Append -Encoding UTF8
        }
    }
    
    Write-DriveLog "================================================" "Cyan"
    Write-DriveLog "Copying Additional Drive: $DestinationDrive" "Cyan"
    Write-DriveLog "================================================" "Cyan"
    Write-DriveLog "Source: $SourceShare"
    Write-DriveLog "Destination: $DestinationDrive\"
    Write-DriveLog "Overwrite=$Overwrite, NewerOnly=$NewerOnly"
    
    # Check source accessibility - don't fail, just warn
    if (!(Test-Path $SourceShare)) {
        Write-DriveLog "WARNING: Source share not accessible: $SourceShare" "Yellow"
        if (-not $Script:CopyIssues) { $Script:CopyIssues = @() }
        $Script:CopyIssues += "Drive $DestinationDrive : Source share not accessible ($SourceShare)"
        return $true  # Don't fail, just continue
    }
    
    # Show what's in the source (exclude system folders from display)
    Write-DriveLog "--- Source Contents ---" "Cyan"
    $sourceItems = Get-ChildItem -Path $SourceShare -ErrorAction SilentlyContinue | 
        Where-Object { $_.Name -notin @('$RECYCLE.BIN', 'System Volume Information') } |
        Select-Object -First 10
    if ($sourceItems) {
        foreach ($item in $sourceItems) {
            $type = if ($item.PSIsContainer) { "[DIR]" } else { "[FILE]" }
            Write-DriveLog "  $type $($item.Name)"
        }
        $totalItems = (Get-ChildItem -Path $SourceShare -ErrorAction SilentlyContinue | 
            Where-Object { $_.Name -notin @('$RECYCLE.BIN', 'System Volume Information') }).Count
        if ($totalItems -gt 10) {
            Write-DriveLog "  ... and $($totalItems - 10) more items"
        }
    } else {
        Write-DriveLog "  (empty or no user data)" "Yellow"
    }
    
    $destPath = "$DestinationDrive\"
    
    if (!(Test-Path $destPath)) {
        Write-DriveLog "WARNING: Destination drive not found: $destPath" "Yellow"
        if (-not $Script:CopyIssues) { $Script:CopyIssues = @() }
        $Script:CopyIssues += "Drive $DestinationDrive : Destination drive not found"
        return $true  # Don't fail
    }
    
    # Determine copy flags
    $copyFlags = ""
    if ($Overwrite -eq $true) {
        if ($NewerOnly -eq $true) {
            $copyFlags = "/XO"
            Write-DriveLog "Mode: COPY if source is NEWER" "Yellow"
        } else {
            $copyFlags = "/IS /IT"
            Write-DriveLog "Mode: FORCE OVERWRITE ALL" "Green"
        }
    } else {
        $copyFlags = "/XC /XN /XO"
        Write-DriveLog "Mode: SKIP existing files" "Yellow"
    }
    
    Write-DriveLog "Robocopy flags: $copyFlags" "Cyan"
    Write-DriveLog "ALWAYS excluding: `$RECYCLE.BIN, System Volume Information" "Gray"
    
    try {
        Write-DriveLog "Starting robocopy..." "Yellow"
        
        # Save robocopy log
        $roboLogFile = "$Script:InstallPath\Logs\robocopy_$(Get-Date -Format 'yyyyMMdd_HHmmss')_$($DestinationDrive.TrimEnd(':\')).log"
        
        # Build robocopy arguments - quote paths that may contain spaces
        $roboArgs = @(
            "`"$SourceShare`""
            "`"$destPath`""
            '/E'
            '/COPY:DAT'
            "/MT:$Threads"
            '/XJ'
            '/R:1'
            '/W:1'
            '/NP'
        )
        
        # Add copy flags (split by space if multiple)
        if ($copyFlags) {
            $copyFlags.Split(' ') | Where-Object { $_ } | ForEach-Object { $roboArgs += $_ }
        }
        
        # Add exclusions - these folders should NEVER be copied (quote paths with spaces)
        $roboArgs += '/XD'
        $roboArgs += '$RECYCLE.BIN'
        $roboArgs += '"System Volume Information"'
        
        # Add file exclusions
        $roboArgs += '/XF'
        $roboArgs += '*.tmp'
        $roboArgs += 'thumbs.db'
        $roboArgs += 'desktop.ini'
        
        # Add log file (quote path in case of spaces)
        $roboArgs += "/LOG:`"$roboLogFile`""
        
        Write-DriveLog "Robocopy log: $roboLogFile"
        
        # Join args into single string and run robocopy
        $argString = $roboArgs -join ' '
        Write-DriveLog "Command: robocopy $argString" "Gray"
        
        $process = Start-Process -FilePath "robocopy.exe" -ArgumentList $argString -Wait -PassThru -NoNewWindow
        
        $exitCode = $process.ExitCode
        
        # Parse log for summary and failed files
        $failedFiles = @()
        $copiedCount = 0
        $skippedCount = 0
        $failedCount = 0
        
        if (Test-Path $roboLogFile) {
            $logContent = Get-Content $roboLogFile -ErrorAction SilentlyContinue
            
            # Parse summary line
            foreach ($line in $logContent) {
                if ($line -match "^\s*Files\s*:\s*(\d+)\s+(\d+)\s+(\d+)\s+\d+\s+(\d+)") {
                    $copiedCount = [int]$Matches[2]
                    $skippedCount = [int]$Matches[3]
                    $failedCount = [int]$Matches[4]
                }
                # Capture failed files
                if ($line -match "^\s+\*FAILED\*\s+(.+)$") {
                    $failedFiles += $Matches[1].Trim()
                }
            }
            
            Write-DriveLog "--- Robocopy Summary ---" "Cyan"
            Write-DriveLog "  Copied: $copiedCount, Skipped: $skippedCount, Failed: $failedCount"
            
            # Show last few lines for context
            $tailLines = Get-Content $roboLogFile -Tail 12 -ErrorAction SilentlyContinue
            foreach ($line in $tailLines) {
                if ($line -match "^\s*(Dirs|Files|Bytes|Ended|Speed)") {
                    Write-DriveLog "  $($line.Trim())"
                }
            }
        } else {
            Write-DriveLog "WARNING: Robocopy log file not created" "Yellow"
        }
        
        Write-DriveLog "Exit code: $exitCode"
        
        # Track any failures for end-of-migration summary
        if ($failedCount -gt 0 -or $exitCode -ge 8) {
            if (-not $Script:CopyIssues) { $Script:CopyIssues = @() }
            $Script:CopyIssues += "Drive $DestinationDrive : $failedCount file(s) could not be copied (locked or permission denied)"
            if ($failedFiles.Count -gt 0) {
                foreach ($f in ($failedFiles | Select-Object -First 5)) {
                    $Script:CopyIssues += "  - $f"
                }
                if ($failedFiles.Count -gt 5) {
                    $Script:CopyIssues += "  - ... and $($failedFiles.Count - 5) more (see robocopy log)"
                }
            }
        }
        
        switch ($exitCode) {
            0 { Write-DriveLog "Result: Already in sync (no changes needed)" "Green" }
            1 { Write-DriveLog "Result: Files copied successfully!" "Green" }
            2 { Write-DriveLog "Result: Extra files in destination" "Green" }
            3 { Write-DriveLog "Result: Files copied + extras in destination" "Green" }
            {$_ -ge 4 -and $_ -lt 8} { Write-DriveLog "Result: Some files skipped (code: $exitCode)" "Yellow" }
            {$_ -ge 8} { Write-DriveLog "Result: Some files could not be copied - see log" "Yellow" }
            default { Write-DriveLog "Result: Completed (code: $exitCode)" "Yellow" }
        }
        
        Write-DriveLog "Drive copy completed" "Green"
        return $true  # Always return true - never fail the migration
    }
    catch {
        Write-DriveLog "WARNING: $($_.Exception.Message)" "Yellow"
        if (-not $Script:CopyIssues) { $Script:CopyIssues = @() }
        $Script:CopyIssues += "Drive $DestinationDrive : Error during copy - $($_.Exception.Message)"
        return $true  # Don't fail
    }
}
