<#
.SYNOPSIS
    ABC&D Migration Tool - Google Drive Cloud Backup Module (rclone Edition)
.DESCRIPTION
    Enables backup/restore to Google Drive using rclone.
    First run: pops up a browser for Google login + 2FA.
    All subsequent runs: uses saved token silently.
    No Google Cloud Console needed - rclone has a pre-approved OAuth Client ID.
.VERSION
    4.0.1
#>

$Script:API_BASE = "https://abcdtoolset.com/api"

# ============================================================================
# RCLONE CONFIGURATION
# ============================================================================

$Script:ToolsRoot = Join-Path (Split-Path $PSScriptRoot -Parent) "Tools"
$Script:RclonePath = Join-Path $Script:ToolsRoot "rclone\rclone.exe"
$Script:RcloneConfig = Join-Path $env:APPDATA "ABCD-Migration\rclone.conf"
$Script:RemoteName = "abcd-gdrive"
$Script:DefaultBackupFolder = "ABCD-Migration-Backups"
$Script:LogDir = Join-Path $env:APPDATA "ABCD-Migration\logs"

# ============================================================================
# RCLONE HELPERS
# ============================================================================

function Write-GDriveLog {
    param([string]$Message, [string]$Level = "INFO")
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    if (-not (Test-Path $Script:LogDir)) { New-Item -Path $Script:LogDir -ItemType Directory -Force | Out-Null }
    "[$timestamp] [$Level] $Message" | Add-Content -Path (Join-Path $Script:LogDir "gdrive.log") -Encoding UTF8
}

function Test-RcloneAvailable {
    if (Test-Path $Script:RclonePath) { return $Script:RclonePath }
    $sys = Get-Command rclone -ErrorAction SilentlyContinue
    if ($sys) { return $sys.Source }
    return $null
}

function Invoke-Rclone {
    param([string[]]$Arguments, [switch]$ShowWindow, [int]$TimeoutSeconds = 300)
    
    $rclone = Test-RcloneAvailable
    if (-not $rclone) { throw "rclone not found. Click 'Install rclone' first." }
    
    $configDir = Split-Path $Script:RcloneConfig -Parent
    if (-not (Test-Path $configDir)) { New-Item -Path $configDir -ItemType Directory -Force | Out-Null }
    
    $allArgs = @("--config", $Script:RcloneConfig) + $Arguments
    Write-GDriveLog "Running: rclone $($allArgs -join ' ')"
    
    $psi = New-Object System.Diagnostics.ProcessStartInfo
    $psi.FileName = $rclone
    $psi.Arguments = $allArgs -join ' '
    $psi.UseShellExecute = $false
    $psi.RedirectStandardOutput = $true
    $psi.RedirectStandardError = $true
    $psi.CreateNoWindow = (-not $ShowWindow)
    
    $process = [System.Diagnostics.Process]::Start($psi)
    $stdout = $process.StandardOutput.ReadToEnd()
    $stderr = $process.StandardError.ReadToEnd()
    $process.WaitForExit($TimeoutSeconds * 1000) | Out-Null
    
    return [PSCustomObject]@{
        ExitCode = $process.ExitCode
        Output   = $stdout.Trim()
        Error    = $stderr.Trim()
        Success  = ($process.ExitCode -eq 0)
    }
}

function Install-Rclone {
    $rcloneDir = Join-Path $Script:ToolsRoot "rclone"
    if (-not (Test-Path $rcloneDir)) { New-Item -Path $rcloneDir -ItemType Directory -Force | Out-Null }
    
    Write-GDriveLog "Downloading rclone"
    $zipPath = Join-Path $env:TEMP "rclone-download.zip"
    $extractPath = Join-Path $env:TEMP "rclone-extract"
    
    [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
    
    try {
        (New-Object System.Net.WebClient).DownloadFile(
            "https://downloads.rclone.org/rclone-current-windows-amd64.zip", $zipPath)
    } catch {
        Write-GDriveLog "Primary download failed: $($_.Exception.Message)"
        # Fallback to GitHub releases
        try {
            Write-GDriveLog "Trying GitHub mirror..."
            (New-Object System.Net.WebClient).DownloadFile(
                "https://github.com/rclone/rclone/releases/latest/download/rclone-current-windows-amd64.zip", $zipPath)
        } catch {
            Write-GDriveLog "Download failed: $($_.Exception.Message)"
            throw "Could not download rclone. Check internet connection and try again."
        }
    }
    
    if (Test-Path $extractPath) { Remove-Item $extractPath -Recurse -Force }
    Expand-Archive -Path $zipPath -DestinationPath $extractPath -Force
    
    $exe = Get-ChildItem -Path $extractPath -Filter "rclone.exe" -Recurse | Select-Object -First 1
    if (-not $exe) { throw "rclone.exe not found in download" }
    
    Copy-Item $exe.FullName -Destination (Join-Path $rcloneDir "rclone.exe") -Force
    Remove-Item $zipPath -Force -ErrorAction SilentlyContinue
    Remove-Item $extractPath -Recurse -Force -ErrorAction SilentlyContinue
    
    Write-GDriveLog "rclone installed to $rcloneDir"
    return $true
}

# ============================================================================
# CORE GOOGLE DRIVE FUNCTIONS
# ============================================================================

function Get-GDriveToken {
    # Legacy compat - returns license creds for server API calls
    if (Get-Command Get-SavedCredentials -ErrorAction SilentlyContinue) {
        $creds = Get-SavedCredentials
        if ($creds -and $creds.Token) { return $creds.Token }
    }
    return $null
}

function Test-GDriveConnection {
    if (-not (Test-RcloneAvailable)) { return @{ Connected = $false; Error = "rclone not installed" } }
    if (-not (Test-Path $Script:RcloneConfig)) { return @{ Connected = $false; Error = "Not configured" } }
    
    $configContent = Get-Content $Script:RcloneConfig -Raw -ErrorAction SilentlyContinue
    if (-not $configContent -or $configContent -notmatch "\[$Script:RemoteName\]") {
        return @{ Connected = $false; Error = "Not configured" }
    }
    
    $result = Invoke-Rclone -Arguments @("about", "${Script:RemoteName}:", "--json") -TimeoutSeconds 15
    
    if ($result.Success) {
        $email = "Connected"
        try {
            if ($configContent -match '"access_token":"([^"]+)"') {
                $headers = @{ Authorization = "Bearer $($Matches[1])" }
                $userInfo = Invoke-RestMethod -Uri "https://www.googleapis.com/drive/v3/about?fields=user" -Headers $headers -ErrorAction SilentlyContinue
                if ($userInfo.user.emailAddress) { $email = $userInfo.user.emailAddress }
            }
        } catch { }
        
        return @{ Connected = $true; Email = $email; FolderId = $Script:DefaultBackupFolder }
    }
    return @{ Connected = $false; Error = "Token expired or invalid" }
}

function Connect-GDrive {
    if (-not (Test-RcloneAvailable)) {
        Write-Host "Installing rclone..." -ForegroundColor Yellow
        Install-Rclone
    }
    
    # This triggers the browser popup for Google login + 2FA
    $result = Invoke-Rclone -Arguments @(
        "config", "create", $Script:RemoteName, "drive",
        "--drive-scope", "drive.file",
        "--auto-confirm"
    ) -ShowWindow -TimeoutSeconds 180
    
    Start-Sleep -Seconds 2
    $status = Test-GDriveConnection
    
    if ($status.Connected) {
        Invoke-Rclone -Arguments @("mkdir", "${Script:RemoteName}:$Script:DefaultBackupFolder") -TimeoutSeconds 30
        Report-GDriveStatus -Connected $true -Email $status.Email
        Push-GDriveTokenToAccount -Email $status.Email
        Write-GDriveLog "Google Drive connected: $($status.Email)"
        return $true
    }
    return $false
}

function Disconnect-GDrive {
    Invoke-Rclone -Arguments @("config", "delete", $Script:RemoteName) | Out-Null
    Report-GDriveStatus -Connected $false
    Write-GDriveLog "Google Drive disconnected"
    Write-Host "Google Drive disconnected" -ForegroundColor Yellow
    return $true
}

function Report-GDriveStatus {
    param([bool]$Connected, [string]$Email = "")
    try {
        $licenseKey = ""; $machineId = ""
        if (Get-Command Get-SavedCredentials -ErrorAction SilentlyContinue) {
            $creds = Get-SavedCredentials
            if ($creds) { $licenseKey = $creds.LicenseKey; $machineId = $creds.MachineId }
        }
        if ($licenseKey) {
            $body = @{ licenseKey=$licenseKey; machineId=$machineId; connected=$Connected; email=$Email; method="rclone" } | ConvertTo-Json
            Invoke-RestMethod -Uri "$Script:API_BASE/gdrive.php?action=report-status" `
                -Method POST -Body $body -ContentType "application/json" -TimeoutSec 10 -ErrorAction SilentlyContinue
        }
    } catch { Write-GDriveLog "Failed to report status: $_" "WARN" }
}

# ============================================================================
# Push rclone credentials to account (server-side) after auth
# ============================================================================
function Push-GDriveTokenToAccount {
    param([string]$Email = "", [string]$FolderName = "")
    
    try {
        if (-not (Test-Path $Script:RcloneConfig)) {
            Write-GDriveLog "No rclone config to push" "WARN"
            return $false
        }
        
        $rcloneContent = Get-Content $Script:RcloneConfig -Raw
        if (-not $rcloneContent -or $rcloneContent.Length -lt 20) {
            Write-GDriveLog "Rclone config empty" "WARN"
            return $false
        }
        
        $licenseKey = ""; $machineId = ""
        if (Get-Command Get-SavedCredentials -ErrorAction SilentlyContinue) {
            $creds = Get-SavedCredentials
            if ($creds) { $licenseKey = $creds.LicenseKey; $machineId = $creds.MachineId }
        }
        
        if (-not $licenseKey) {
            Write-GDriveLog "No license key - cannot save credentials to account" "WARN"
            return $false
        }
        
        $gdConfig = Get-GDriveConfig
        if (-not $FolderName -and $gdConfig -and $gdConfig.FolderName) { $FolderName = $gdConfig.FolderName }
        if (-not $FolderName) { $FolderName = $Script:DefaultBackupFolder }
        
        $body = @{
            licenseKey = $licenseKey
            machineId = $machineId
            rclone_config = $rcloneContent
            gdrive_email = $Email
            folder_name = $FolderName
        } | ConvertTo-Json
        
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/gdrive.php?action=store-token" `
            -Method POST -Body $body -ContentType "application/json" -TimeoutSec 15 -ErrorAction Stop
        
        if ($response.success) {
            Write-GDriveLog "Google Drive credentials saved to your account ($Email)"
            return $true
        } else {
            Write-GDriveLog "Failed to save credentials: $($response.error)" "WARN"
            return $false
        }
    } catch {
        Write-GDriveLog "Error pushing credentials to account: $($_.Exception.Message)" "WARN"
        return $false
    }
}

# ============================================================================
# Pull rclone credentials from account (for DESTINATION restore)
# ============================================================================
function Pull-GDriveTokenFromAccount {
    try {
        $licenseKey = ""; $machineId = ""
        if (Get-Command Get-SavedCredentials -ErrorAction SilentlyContinue) {
            $creds = Get-SavedCredentials
            if ($creds) { $licenseKey = $creds.LicenseKey; $machineId = $creds.MachineId }
        }
        
        if (-not $licenseKey) {
            return @{ HasToken = $false; Error = "No license key" }
        }
        
        $body = @{ licenseKey = $licenseKey; machineId = $machineId } | ConvertTo-Json
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/gdrive.php?action=get-token" `
            -Method POST -Body $body -ContentType "application/json" -TimeoutSec 15 -ErrorAction Stop
        
        if ($response.success -and $response.has_token) {
            return @{
                HasToken = $true
                RcloneConfig = $response.rclone_config
                Email = $response.gdrive_email
                FolderName = $response.folder_name
            }
        }
        return @{ HasToken = $false; Error = if ($response.error) { $response.error } else { "No credentials stored" } }
    } catch {
        return @{ HasToken = $false; Error = $_.Exception.Message }
    }
}

# ============================================================================
# Inject pulled rclone credentials into local config
# ============================================================================
function Restore-GDriveFromAccount {
    $pulled = Pull-GDriveTokenFromAccount
    if (-not $pulled.HasToken) {
        return @{ Success = $false; Error = $pulled.Error }
    }
    
    # Ensure rclone is installed
    if (-not (Test-RcloneAvailable)) {
        Write-GDriveLog "Installing rclone for credential restore..."
        Install-Rclone
    }
    
    # Write the rclone config
    $configDir = Split-Path $Script:RcloneConfig -Parent
    if (-not (Test-Path $configDir)) {
        New-Item -ItemType Directory -Path $configDir -Force | Out-Null
    }
    $pulled.RcloneConfig | Out-File $Script:RcloneConfig -Encoding UTF8 -Force
    
    # Test connection
    $status = Test-GDriveConnection
    if ($status.Connected) {
        Write-GDriveLog "Google Drive credentials restored from account ($($pulled.Email))"
        Report-GDriveStatus -Connected $true -Email $pulled.Email
        return @{ Success = $true; Email = $pulled.Email; FolderName = $pulled.FolderName }
    }
    
    return @{ Success = $false; Error = "Token restored but connection test failed - may need re-auth" }
}

function New-GDriveBackupFolder {
    param([string]$Name = $Script:DefaultBackupFolder)
    $result = Invoke-Rclone -Arguments @("mkdir", "${Script:RemoteName}:$Name") -TimeoutSeconds 30
    if ($result.Success) { return @{ Success = $true; FolderId = $Name; Name = $Name } }
    return @{ Success = $false; Error = $result.Error }
}

function Get-GDriveBackups {
    $result = Invoke-Rclone -Arguments @("lsjson", "${Script:RemoteName}:$Script:DefaultBackupFolder", "--dirs-only") -TimeoutSeconds 30
    if ($result.Success -and $result.Output) {
        $items = $result.Output | ConvertFrom-Json
        return $items | ForEach-Object {
            [PSCustomObject]@{ Id = $_.Path; Name = $_.Name; Size = "Folder"; Modified = $_.ModTime }
        }
    }
    return @()
}

function Get-GDriveAccessToken {
    # Legacy compat stub - rclone handles auth directly
    $status = Test-GDriveConnection
    if ($status.Connected) {
        return @{ AccessToken = "rclone-managed"; FolderId = $Script:DefaultBackupFolder }
    }
    return $null
}

# ============================================================================
# SIMPLE DIALOG (backward compat for Show-GDriveDialog without mode)
# ============================================================================

function Show-GDriveDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $status = Test-GDriveConnection
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Google Drive Cloud Backup"
    $form.Size = New-Object System.Drawing.Size(450, 340)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(26, 26, 46)
    $form.ForeColor = [System.Drawing.Color]::White
    $form.Font = New-Object System.Drawing.Font("Segoe UI", 9)
    
    $title = New-Object System.Windows.Forms.Label
    $title.Text = [char]0x2601 + " Google Drive Cloud Backup"
    $title.Font = New-Object System.Drawing.Font("Segoe UI", 14, [System.Drawing.FontStyle]::Bold)
    $title.ForeColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
    $title.Location = New-Object System.Drawing.Point(20, 15)
    $title.AutoSize = $true
    $form.Controls.Add($title)
    
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Location = New-Object System.Drawing.Point(20, 55)
    $lblStatus.Size = New-Object System.Drawing.Size(390, 50)
    $lblStatus.Font = New-Object System.Drawing.Font("Segoe UI", 10)
    
    if ($status.Connected) {
        $lblStatus.Text = "Connected to: $($status.Email)`n`nBackups use rclone - no browser needed for transfers."
        $lblStatus.ForeColor = [System.Drawing.Color]::LightGreen
    } else {
        $rcloneReady = Test-RcloneAvailable
        if ($rcloneReady) {
            $lblStatus.Text = "Not connected. Click 'Connect' to sign in.`nA browser opens for Google login + 2FA (one time only)."
        } else {
            $lblStatus.Text = "rclone not installed. Click 'Install rclone' first,`nthen 'Connect' to sign in to Google Drive."
        }
        $lblStatus.ForeColor = [System.Drawing.Color]::Orange
    }
    $form.Controls.Add($lblStatus)
    
    # Install rclone button (only if missing)
    if (-not (Test-RcloneAvailable)) {
        $btnInstall = New-Object System.Windows.Forms.Button
        $btnInstall.Text = "Install rclone"
        $btnInstall.Location = New-Object System.Drawing.Point(20, 120)
        $btnInstall.Size = New-Object System.Drawing.Size(130, 45)
        $btnInstall.BackColor = [System.Drawing.Color]::FromArgb(255, 165, 0)
        $btnInstall.ForeColor = [System.Drawing.Color]::Black
        $btnInstall.FlatStyle = "Flat"
        $btnInstall.Font = New-Object System.Drawing.Font("Segoe UI", 10, [System.Drawing.FontStyle]::Bold)
        $btnInstall.Add_Click({
            $btnInstall.Text = "Installing..."
            $btnInstall.Enabled = $false
            $form.Refresh()
            try {
                Install-Rclone
                [System.Windows.Forms.MessageBox]::Show("rclone installed! Now click 'Connect Google Drive'.", "Installed", "OK", "Information")
                $btnInstall.Visible = $false
                $lblStatus.Text = "rclone ready. Click 'Connect Google Drive' to sign in."
            } catch {
                [System.Windows.Forms.MessageBox]::Show("Install failed: $_", "Error", "OK", "Error")
                $btnInstall.Text = "Install rclone"
                $btnInstall.Enabled = $true
            }
        })
        $form.Controls.Add($btnInstall)
    }
    
    # Connect / Disconnect
    $btnConnect = New-Object System.Windows.Forms.Button
    $yPos = if (-not (Test-RcloneAvailable)) { 175 } else { 120 }
    $btnConnect.Location = New-Object System.Drawing.Point(20, $yPos)
    $btnConnect.Size = New-Object System.Drawing.Size(180, 45)
    $btnConnect.FlatStyle = "Flat"
    $btnConnect.Font = New-Object System.Drawing.Font("Segoe UI", 10, [System.Drawing.FontStyle]::Bold)
    
    if ($status.Connected) {
        $btnConnect.Text = "Disconnect"
        $btnConnect.BackColor = [System.Drawing.Color]::FromArgb(180, 60, 60)
        $btnConnect.ForeColor = [System.Drawing.Color]::White
        $btnConnect.Add_Click({
            Disconnect-GDrive
            [System.Windows.Forms.MessageBox]::Show("Google Drive disconnected.`nLocal token removed. Drive files untouched.", "Disconnected", "OK", "Information")
            $form.Close()
        })
    } else {
        $btnConnect.Text = "Connect Google Drive"
        $btnConnect.BackColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
        $btnConnect.ForeColor = [System.Drawing.Color]::Black
        $btnConnect.Add_Click({
            $btnConnect.Text = "Connecting..."
            $btnConnect.Enabled = $false
            $form.Refresh()
            try {
                $success = Connect-GDrive
                if ($success) {
                    [System.Windows.Forms.MessageBox]::Show("Google Drive connected!`nBackups go to: $Script:DefaultBackupFolder", "Connected!", "OK", "Information")
                    $form.Close()
                } else {
                    [System.Windows.Forms.MessageBox]::Show("Connection failed. Try again.", "Error", "OK", "Error")
                    $btnConnect.Text = "Connect Google Drive"
                    $btnConnect.Enabled = $true
                }
            } catch {
                [System.Windows.Forms.MessageBox]::Show("Error: $_", "Error", "OK", "Error")
                $btnConnect.Text = "Connect Google Drive"
                $btnConnect.Enabled = $true
            }
        })
    }
    $form.Controls.Add($btnConnect)
    
    # Check Status
    $btnCheck = New-Object System.Windows.Forms.Button
    $btnCheck.Text = "Check Status"
    $btnCheck.Location = New-Object System.Drawing.Point(210, $yPos)
    $btnCheck.Size = New-Object System.Drawing.Size(130, 45)
    $btnCheck.BackColor = [System.Drawing.Color]::FromArgb(60, 60, 100)
    $btnCheck.ForeColor = [System.Drawing.Color]::White
    $btnCheck.FlatStyle = "Flat"
    $btnCheck.Add_Click({
        $newStatus = Test-GDriveConnection
        if ($newStatus.Connected) {
            [System.Windows.Forms.MessageBox]::Show("Connected to: $($newStatus.Email)", "Connected!", "OK", "Information")
        } else {
            [System.Windows.Forms.MessageBox]::Show("Not connected.`n$($newStatus.Error)", "Status", "OK", "Warning")
        }
    })
    $form.Controls.Add($btnCheck)
    
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(320, 260)
    $btnClose.Size = New-Object System.Drawing.Size(90, 32)
    $btnClose.BackColor = [System.Drawing.Color]::FromArgb(50, 50, 60)
    $btnClose.ForeColor = [System.Drawing.Color]::White
    $btnClose.FlatStyle = "Flat"
    $btnClose.Add_Click({ $form.Close() })
    $form.Controls.Add($btnClose)
    
    $form.ShowDialog() | Out-Null
}

# =============================================================================
# CONFIG PERSISTENCE
# =============================================================================

$Script:GDriveConfig = @{
    ManifestPath = "C:\ABCD-Migration\Settings\GDriveManifest.json"
    SourceFolder = ""
    ConfigPath   = "C:\ABCD-Migration\Settings\GDriveConfig.json"
}

function Get-GDriveConfig {
    if (Test-Path $Script:GDriveConfig.ConfigPath) {
        try { return Get-Content $Script:GDriveConfig.ConfigPath -Raw | ConvertFrom-Json } catch { return $null }
    }
    return $null
}

function Save-GDriveConfig {
    param($Config)
    $dir = Split-Path $Script:GDriveConfig.ConfigPath -Parent
    if (!(Test-Path $dir)) { New-Item -Path $dir -ItemType Directory -Force | Out-Null }
    $Config | ConvertTo-Json -Depth 5 | Out-File $Script:GDriveConfig.ConfigPath -Encoding UTF8
}

# =============================================================================
# SCHEDULED GOOGLE DRIVE SYNC
# =============================================================================

$Script:GDriveSyncTaskName = "ABCD-Migration-GDriveSync"
$Script:GDriveSyncConfigPath = "$Script:InstallPath\Settings\gdrive-sync-schedule.json"

function Get-GDriveSyncSchedule {
    if (Test-Path $Script:GDriveSyncConfigPath) {
        try { return Get-Content $Script:GDriveSyncConfigPath -Raw | ConvertFrom-Json } catch { return $null }
    }
    return $null
}

function Save-GDriveSyncSchedule {
    param($Config)
    $dir = Split-Path $Script:GDriveSyncConfigPath -Parent
    if (!(Test-Path $dir)) { New-Item -Path $dir -ItemType Directory -Force | Out-Null }
    $Config | ConvertTo-Json -Depth 5 | Out-File $Script:GDriveSyncConfigPath -Encoding UTF8
}

function Register-GDriveSyncTask {
    param(
        [int]$Hour = 2,
        [int]$Minute = 0
    )
    
    $scriptPath = "$Script:InstallPath\Scripts\Run-GDriveSync.ps1"
    if (-not (Test-Path $scriptPath)) {
        throw "Sync script not found: $scriptPath"
    }
    
    $triggerTime = "{0}:{1:00}" -f $Hour, $Minute
    
    $action = New-ScheduledTaskAction -Execute "powershell.exe" `
        -Argument "-NoProfile -ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    
    $trigger = New-ScheduledTaskTrigger -Daily -At $triggerTime
    
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    
    $settings = New-ScheduledTaskSettingsSet `
        -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries `
        -StartWhenAvailable -DontStopOnIdleEnd `
        -ExecutionTimeLimit (New-TimeSpan -Hours 5) `
        -MultipleInstances IgnoreNew
    
    try {
        Register-ScheduledTask -TaskName $Script:GDriveSyncTaskName `
            -Action $action -Trigger $trigger -Principal $principal -Settings $settings `
            -Description "ABCD Systems - Scheduled Google Drive Sync (runs daily at $triggerTime)" `
            -Force | Out-Null
        
        Write-GDriveLog "Scheduled task registered: $Script:GDriveSyncTaskName at $triggerTime"
        return $true
    } catch {
        Write-GDriveLog "Failed to register task: $($_.Exception.Message)"
        throw $_
    }
}

function Remove-GDriveSyncTask {
    try {
        Unregister-ScheduledTask -TaskName $Script:GDriveSyncTaskName -Confirm:$false -ErrorAction SilentlyContinue
        Write-GDriveLog "Scheduled task removed: $Script:GDriveSyncTaskName"
        return $true
    } catch {
        Write-GDriveLog "Failed to remove task: $($_.Exception.Message)"
        return $false
    }
}

function Test-GDriveSyncTaskExists {
    try {
        $task = Get-ScheduledTask -TaskName $Script:GDriveSyncTaskName -ErrorAction SilentlyContinue
        return ($null -ne $task)
    } catch { return $false }
}

# =============================================================================
# SOURCE/DESTINATION OPERATIONS - rclone powered
# =============================================================================

function Push-ToGoogleDrive {
    <#
    .SYNOPSIS
        SOURCE mode: Sync local folder to Google Drive (mirror)
        - Same size+name = skip
        - New/changed files = upload  
        - Deleted local files = delete from Drive
    #>
    param(
        [string]$LocalPath,
        [string]$FolderName,
        [switch]$DeltaOnly
    )
    
    if (-not (Test-GDriveConnection).Connected) {
        Write-Warning "Google Drive not connected. Run Connect-GDrive first."
        return @{ Success = $false; Error = "Not connected" }
    }
    
    $remotePath = "${Script:RemoteName}:$Script:DefaultBackupFolder/$FolderName"
    Write-Host "[GDrive SOURCE] Syncing to: $remotePath" -ForegroundColor Cyan
    Write-GDriveLog "Sync started: $LocalPath -> $remotePath"
    
    # sync = mirror (skips identical, uploads new, DELETES removed from source)
    # copy = additive only (skips identical, uploads new, KEEPS removed)
    $rcloneCmd = if ($DeltaOnly) { "sync" } else { "copy" }
    $rArgs = @($rcloneCmd, $LocalPath, $remotePath, "--progress", "--stats-one-line")
    $rArgs += "--transfers", "4"
    $rArgs += "--checkers", "8"
    $rArgs += "--retries", "3"
    $rArgs += "--contimeout", "60s"
    $rArgs += "--timeout", "300s"
    $rArgs += "--low-level-retries", "10"
    
    $result = Invoke-Rclone -Arguments $rArgs -ShowWindow -TimeoutSeconds 3600
    
    if ($result.Success) {
        Write-Host "[GDrive SOURCE] Sync complete!" -ForegroundColor Green
        Write-GDriveLog "Sync completed"
        
        try {
            if (Get-Command Get-SavedCredentials -ErrorAction SilentlyContinue) {
                $creds = Get-SavedCredentials
                if ($creds.LicenseKey) {
                    $body = @{ licenseKey=$creds.LicenseKey; machineId=$creds.MachineId; type="backup"; detail="Google Drive sync: $FolderName" } | ConvertTo-Json
                    Invoke-RestMethod -Uri "$Script:API_BASE/auth.php?action=record-usage" -Method POST -Body $body -ContentType "application/json" -ErrorAction SilentlyContinue
                }
            }
        } catch { }
        
        return @{ Success = $true; Uploaded = "rclone sync" }
    }
    
    Write-Host "[GDrive SOURCE] Sync failed: $($result.Error)" -ForegroundColor Red
    return @{ Success = $false; Error = $result.Error }
}

function Pull-FromGoogleDrive {
    <#
    .SYNOPSIS
        DESTINATION mode: Sync Google Drive folder to local (mirror)
        - Same size+name = skip
        - New/changed remote files = download
        - Deleted remote files = delete locally
    #>
    param(
        [string]$SourceFolderName,
        [string]$LocalDestination,
        [switch]$DeltaOnly
    )
    
    if (-not (Test-GDriveConnection).Connected) {
        Write-Warning "Google Drive not connected."
        return @{ Success = $false; Error = "Not connected" }
    }
    
    $remotePath = "${Script:RemoteName}:$Script:DefaultBackupFolder/$SourceFolderName"
    Write-Host "[GDrive DEST] Syncing from: $remotePath" -ForegroundColor Cyan
    Write-GDriveLog "Sync started: $remotePath -> $LocalDestination"
    
    if (-not (Test-Path $LocalDestination)) {
        New-Item -Path $LocalDestination -ItemType Directory -Force | Out-Null
    }
    
    # sync = mirror, copy = additive only
    $rcloneCmd = if ($DeltaOnly) { "sync" } else { "copy" }
    $rArgs = @($rcloneCmd, $remotePath, $LocalDestination, "--progress", "--stats-one-line")
    $rArgs += "--transfers", "4"
    $rArgs += "--retries", "3"
    
    $result = Invoke-Rclone -Arguments $rArgs -ShowWindow -TimeoutSeconds 3600
    
    if ($result.Success) {
        Write-Host "[GDrive DEST] Sync complete!" -ForegroundColor Green
        Write-GDriveLog "Sync completed"
        return @{ Success = $true; Downloaded = "rclone sync" }
    }
    
    Write-Host "[GDrive DEST] Sync failed: $($result.Error)" -ForegroundColor Red
    return @{ Success = $false; Error = $result.Error }
}

# =============================================================================
# FULL SOURCE/DESTINATION DIALOG
# =============================================================================

function Show-GoogleDriveDialog {
    param(
        [ValidateSet("SOURCE", "DESTINATION")]
        [string]$Mode = "SOURCE"
    )
    
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $config = Get-GDriveConfig
    if (!$config) {
        # Check for backup-manifest.json to find the most recent backup path
        $defaultLocalPath = "$Script:InstallPath\Backup"
        $manifestPath = "$Script:InstallPath\Backup\backup-manifest.json"
        if (Test-Path $manifestPath) {
            try {
                $manifest = Get-Content $manifestPath -Raw | ConvertFrom-Json
                if ($manifest.BackupPath -and (Test-Path $manifest.BackupPath)) {
                    $defaultLocalPath = $manifest.BackupPath
                }
            } catch { }
        }
        $config = @{ Connected=$false; Email=""; SourceComputer=$env:COMPUTERNAME; FolderName="ABCD-Migration-$env:COMPUTERNAME"; LocalPath=$defaultLocalPath; DeltaSync=$true }
    }
    
    # --- Form ---
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Google Drive - $Mode Mode - ABCD Systems"
    $form.Size = New-Object System.Drawing.Size(580, 620)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(20, 20, 20)
    $form.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.Font = New-Object System.Drawing.Font("Segoe UI", 9)
    
    $W = 540  # inner content width
    
    # =====================================================================
    # CONNECTION STATUS
    # =====================================================================
    $grpStatus = New-Object System.Windows.Forms.GroupBox
    $grpStatus.Text = "Connection Status"
    $grpStatus.Location = New-Object System.Drawing.Point(10, 10)
    $grpStatus.Size = New-Object System.Drawing.Size($W, 85)
    $grpStatus.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
    $grpStatus.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.Controls.Add($grpStatus)
    
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Text = "Checking..."
    $lblStatus.Location = New-Object System.Drawing.Point(15, 25)
    $lblStatus.Size = New-Object System.Drawing.Size(350, 20)
    $grpStatus.Controls.Add($lblStatus)
    
    $lblEmail = New-Object System.Windows.Forms.Label
    $lblEmail.Text = ""
    $lblEmail.Location = New-Object System.Drawing.Point(15, 47)
    $lblEmail.Size = New-Object System.Drawing.Size(350, 20)
    $lblEmail.ForeColor = [System.Drawing.Color]::FromArgb(180, 180, 180)
    $grpStatus.Controls.Add($lblEmail)
    
    $btnLogin = New-Object System.Windows.Forms.Button
    $btnLogin.Text = "Connect"
    $btnLogin.Location = New-Object System.Drawing.Point(420, 25)
    $btnLogin.Size = New-Object System.Drawing.Size(105, 28)
    $btnLogin.BackColor = [System.Drawing.Color]::FromArgb(66, 133, 244)
    $btnLogin.ForeColor = [System.Drawing.Color]::Black
    $btnLogin.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpStatus.Controls.Add($btnLogin)
    
    $btnDisconnect = New-Object System.Windows.Forms.Button
    $btnDisconnect.Text = "Disconnect"
    $btnDisconnect.Location = New-Object System.Drawing.Point(420, 55)
    $btnDisconnect.Size = New-Object System.Drawing.Size(105, 25)
    $btnDisconnect.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
    $btnDisconnect.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $btnDisconnect.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnDisconnect.Visible = $false
    $grpStatus.Controls.Add($btnDisconnect)
    
    # "Restore from Account" button (DESTINATION mode only)
    $btnRestoreAcct = New-Object System.Windows.Forms.Button
    $btnRestoreAcct.Text = "Use Account Credentials"
    $btnRestoreAcct.Location = New-Object System.Drawing.Point(15, 55)
    $btnRestoreAcct.Size = New-Object System.Drawing.Size(200, 25)
    $btnRestoreAcct.BackColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
    $btnRestoreAcct.ForeColor = [System.Drawing.Color]::Black
    $btnRestoreAcct.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnRestoreAcct.Font = New-Object System.Drawing.Font("Segoe UI", 8, [System.Drawing.FontStyle]::Bold)
    $btnRestoreAcct.Visible = $false
    $grpStatus.Controls.Add($btnRestoreAcct)
    
    $lblAcctInfo = New-Object System.Windows.Forms.Label
    $lblAcctInfo.Text = ""
    $lblAcctInfo.Location = New-Object System.Drawing.Point(220, 58)
    $lblAcctInfo.Size = New-Object System.Drawing.Size(195, 18)
    $lblAcctInfo.ForeColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
    $lblAcctInfo.Font = New-Object System.Drawing.Font("Segoe UI", 8)
    $grpStatus.Controls.Add($lblAcctInfo)
    
    # =====================================================================
    # SYNC SETTINGS
    # =====================================================================
    $grpSettings = New-Object System.Windows.Forms.GroupBox
    $grpSettings.Text = if ($Mode -eq "SOURCE") { "SOURCE Settings (Sync Local to Drive)" } else { "DESTINATION Settings (Sync Drive to Local)" }
    $grpSettings.Location = New-Object System.Drawing.Point(10, 105)
    $grpSettings.Size = New-Object System.Drawing.Size($W, 185)
    $grpSettings.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
    $grpSettings.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.Controls.Add($grpSettings)
    
    if ($Mode -eq "SOURCE") {
        # Info
        $lblInfo = New-Object System.Windows.Forms.Label
        $lblInfo.Text = "Mirrors a local folder to Google Drive. Same-size files are skipped, new or changed files are uploaded, and files deleted locally are also removed from Drive."
        $lblInfo.Location = New-Object System.Drawing.Point(15, 22)
        $lblInfo.Size = New-Object System.Drawing.Size(510, 32)
        $lblInfo.ForeColor = [System.Drawing.Color]::FromArgb(180, 180, 180)
        $grpSettings.Controls.Add($lblInfo)
        
        # Local path
        $lblLocal = New-Object System.Windows.Forms.Label
        $lblLocal.Text = "Local Path:"
        $lblLocal.Location = New-Object System.Drawing.Point(15, 64)
        $lblLocal.Size = New-Object System.Drawing.Size(85, 20)
        $grpSettings.Controls.Add($lblLocal)
        
        $txtLocalPath = New-Object System.Windows.Forms.TextBox
        $txtLocalPath.Location = New-Object System.Drawing.Point(105, 61)
        $txtLocalPath.Size = New-Object System.Drawing.Size(330, 22)
        $txtLocalPath.Text = "$Script:InstallPath\Backup"
        $txtLocalPath.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
        $txtLocalPath.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $grpSettings.Controls.Add($txtLocalPath)
        
        $btnBrowseLocal = New-Object System.Windows.Forms.Button
        $btnBrowseLocal.Text = "Browse..."
        $btnBrowseLocal.Location = New-Object System.Drawing.Point(445, 60)
        $btnBrowseLocal.Size = New-Object System.Drawing.Size(80, 24)
        $btnBrowseLocal.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
        $btnBrowseLocal.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $btnBrowseLocal.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnBrowseLocal.Add_Click({
            $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
            $fbd.Description = "Select local folder to sync to Google Drive"
            $fbd.SelectedPath = $txtLocalPath.Text
            $fbd.ShowNewFolderButton = $true
            if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
                $txtLocalPath.Text = $fbd.SelectedPath
            }
        })
        $grpSettings.Controls.Add($btnBrowseLocal)
        
        # Drive folder name
        $lblFolder = New-Object System.Windows.Forms.Label
        $lblFolder.Text = "Drive Folder:"
        $lblFolder.Location = New-Object System.Drawing.Point(15, 94)
        $lblFolder.Size = New-Object System.Drawing.Size(85, 20)
        $grpSettings.Controls.Add($lblFolder)
        
        $txtFolder = New-Object System.Windows.Forms.TextBox
        $txtFolder.Location = New-Object System.Drawing.Point(105, 91)
        $txtFolder.Size = New-Object System.Drawing.Size(230, 22)
        $txtFolder.Text = "ABCD-Migration-$env:COMPUTERNAME"
        $txtFolder.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
        $txtFolder.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $grpSettings.Controls.Add($txtFolder)
        
        $lblFolderHint = New-Object System.Windows.Forms.Label
        $lblFolderHint.Text = "(folder name on Google Drive)"
        $lblFolderHint.Location = New-Object System.Drawing.Point(345, 94)
        $lblFolderHint.Size = New-Object System.Drawing.Size(180, 18)
        $lblFolderHint.ForeColor = [System.Drawing.Color]::FromArgb(120, 120, 120)
        $lblFolderHint.Font = New-Object System.Drawing.Font("Segoe UI", 8)
        $grpSettings.Controls.Add($lblFolderHint)
        
    } else {
        # DESTINATION mode
        $lblInfo = New-Object System.Windows.Forms.Label
        $lblInfo.Text = "Mirrors a Google Drive folder to this PC. Same-size files are skipped, new or changed files are downloaded, and files removed from Drive are cleaned up locally."
        $lblInfo.Location = New-Object System.Drawing.Point(15, 22)
        $lblInfo.Size = New-Object System.Drawing.Size(510, 32)
        $lblInfo.ForeColor = [System.Drawing.Color]::FromArgb(180, 180, 180)
        $grpSettings.Controls.Add($lblInfo)
        
        # Drive folder (source)
        $lblSource = New-Object System.Windows.Forms.Label
        $lblSource.Text = "Drive Folder:"
        $lblSource.Location = New-Object System.Drawing.Point(15, 64)
        $lblSource.Size = New-Object System.Drawing.Size(85, 20)
        $grpSettings.Controls.Add($lblSource)
        
        $txtFolder = New-Object System.Windows.Forms.TextBox
        $txtFolder.Location = New-Object System.Drawing.Point(105, 61)
        $txtFolder.Size = New-Object System.Drawing.Size(330, 22)
        $txtFolder.Text = ""
        $txtFolder.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
        $txtFolder.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $grpSettings.Controls.Add($txtFolder)
        
        $btnBrowse = New-Object System.Windows.Forms.Button
        $btnBrowse.Text = "List..."
        $btnBrowse.Location = New-Object System.Drawing.Point(445, 60)
        $btnBrowse.Size = New-Object System.Drawing.Size(80, 24)
        $btnBrowse.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
        $btnBrowse.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $btnBrowse.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnBrowse.Add_Click({
            try {
                $backups = Get-GDriveBackups
                if ($backups.Count -gt 0) {
                    $list = ($backups | ForEach-Object { $_.Name }) -join "`n"
                    $txtFolder.Text = $backups[0].Name
                    [System.Windows.Forms.MessageBox]::Show("Available folders:`n`n$list", "Drive Folders")
                } else {
                    [System.Windows.Forms.MessageBox]::Show("No backup folders found on Drive.", "Empty")
                }
            } catch {
                [System.Windows.Forms.MessageBox]::Show("Error listing folders: $($_.Exception.Message)", "Error")
            }
        })
        $grpSettings.Controls.Add($btnBrowse)
        
        # Local destination
        $lblLocal = New-Object System.Windows.Forms.Label
        $lblLocal.Text = "Save To:"
        $lblLocal.Location = New-Object System.Drawing.Point(15, 94)
        $lblLocal.Size = New-Object System.Drawing.Size(85, 20)
        $grpSettings.Controls.Add($lblLocal)
        
        $txtLocalPath = New-Object System.Windows.Forms.TextBox
        $txtLocalPath.Location = New-Object System.Drawing.Point(105, 91)
        $txtLocalPath.Size = New-Object System.Drawing.Size(330, 22)
        $txtLocalPath.Text = "$Script:InstallPath\Backup"
        $txtLocalPath.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
        $txtLocalPath.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $grpSettings.Controls.Add($txtLocalPath)
        
        $btnBrowseLocal = New-Object System.Windows.Forms.Button
        $btnBrowseLocal.Text = "Browse..."
        $btnBrowseLocal.Location = New-Object System.Drawing.Point(445, 90)
        $btnBrowseLocal.Size = New-Object System.Drawing.Size(80, 24)
        $btnBrowseLocal.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
        $btnBrowseLocal.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
        $btnBrowseLocal.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
        $btnBrowseLocal.Add_Click({
            $fbd = New-Object System.Windows.Forms.FolderBrowserDialog
            $fbd.Description = "Select local folder to save files to"
            $fbd.SelectedPath = $txtLocalPath.Text
            $fbd.ShowNewFolderButton = $true
            if ($fbd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
                $txtLocalPath.Text = $fbd.SelectedPath
            }
        })
        $grpSettings.Controls.Add($btnBrowseLocal)
    }
    
    # Mirror mode checkbox
    $chkDelta = New-Object System.Windows.Forms.CheckBox
    $chkDelta.Text = "Mirror mode (sync + delete removed files)"
    $chkDelta.Location = New-Object System.Drawing.Point(15, 125)
    $chkDelta.Size = New-Object System.Drawing.Size(320, 20)
    $chkDelta.Checked = $true
    $chkDelta.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $grpSettings.Controls.Add($chkDelta)
    
    $lblDeltaOn = New-Object System.Windows.Forms.Label
    $lblDeltaOn.Text = "ON = identical files skipped, new/changed uploaded, deleted files removed from destination"
    $lblDeltaOn.Location = New-Object System.Drawing.Point(35, 147)
    $lblDeltaOn.Size = New-Object System.Drawing.Size(500, 15)
    $lblDeltaOn.ForeColor = [System.Drawing.Color]::FromArgb(140, 140, 140)
    $lblDeltaOn.Font = New-Object System.Drawing.Font("Segoe UI", 7.5)
    $grpSettings.Controls.Add($lblDeltaOn)
    
    $lblDeltaOff = New-Object System.Windows.Forms.Label
    $lblDeltaOff.Text = "OFF = additive only -- uploads new/changed files but never deletes anything"
    $lblDeltaOff.Location = New-Object System.Drawing.Point(35, 162)
    $lblDeltaOff.Size = New-Object System.Drawing.Size(500, 15)
    $lblDeltaOff.ForeColor = [System.Drawing.Color]::FromArgb(140, 140, 140)
    $lblDeltaOff.Font = New-Object System.Drawing.Font("Segoe UI", 7.5)
    $grpSettings.Controls.Add($lblDeltaOff)
    
    # =====================================================================
    # SCHEDULED SYNC
    # =====================================================================
    $grpSchedule = New-Object System.Windows.Forms.GroupBox
    $grpSchedule.Text = "Scheduled Sync"
    $grpSchedule.Location = New-Object System.Drawing.Point(10, 300)
    $grpSchedule.Size = New-Object System.Drawing.Size($W, 155)
    $grpSchedule.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
    $grpSchedule.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.Controls.Add($grpSchedule)
    
    # Enable checkbox
    $chkScheduleEnabled = New-Object System.Windows.Forms.CheckBox
    $chkScheduleEnabled.Text = "Enable daily scheduled sync"
    $chkScheduleEnabled.Location = New-Object System.Drawing.Point(15, 24)
    $chkScheduleEnabled.Size = New-Object System.Drawing.Size(220, 20)
    $chkScheduleEnabled.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $grpSchedule.Controls.Add($chkScheduleEnabled)
    
    # Time picker
    $lblSchedTime = New-Object System.Windows.Forms.Label
    $lblSchedTime.Text = "Run at:"
    $lblSchedTime.Location = New-Object System.Drawing.Point(280, 25)
    $lblSchedTime.Size = New-Object System.Drawing.Size(50, 20)
    $grpSchedule.Controls.Add($lblSchedTime)
    
    $cmbHour = New-Object System.Windows.Forms.ComboBox
    $cmbHour.Location = New-Object System.Drawing.Point(335, 22)
    $cmbHour.Size = New-Object System.Drawing.Size(50, 22)
    $cmbHour.DropDownStyle = "DropDownList"
    $cmbHour.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
    $cmbHour.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    for ($i = 0; $i -le 23; $i++) { $cmbHour.Items.Add($i.ToString("00")) | Out-Null }
    $cmbHour.SelectedIndex = 2
    $grpSchedule.Controls.Add($cmbHour)
    
    $lblColon = New-Object System.Windows.Forms.Label
    $lblColon.Text = ":"
    $lblColon.Location = New-Object System.Drawing.Point(388, 25)
    $lblColon.Size = New-Object System.Drawing.Size(10, 20)
    $grpSchedule.Controls.Add($lblColon)
    
    $cmbMinute = New-Object System.Windows.Forms.ComboBox
    $cmbMinute.Location = New-Object System.Drawing.Point(398, 22)
    $cmbMinute.Size = New-Object System.Drawing.Size(50, 22)
    $cmbMinute.DropDownStyle = "DropDownList"
    $cmbMinute.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
    $cmbMinute.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    for ($i = 0; $i -le 55; $i += 5) { $cmbMinute.Items.Add($i.ToString("00")) | Out-Null }
    $cmbMinute.SelectedIndex = 0
    $grpSchedule.Controls.Add($cmbMinute)
    
    $lblSchedTip = New-Object System.Windows.Forms.Label
    $lblSchedTip.Text = "Tip: Set this 3-4 hours after your local backup or Windows Update tasks finish"
    $lblSchedTip.Location = New-Object System.Drawing.Point(15, 50)
    $lblSchedTip.Size = New-Object System.Drawing.Size(510, 16)
    $lblSchedTip.ForeColor = [System.Drawing.Color]::FromArgb(120, 120, 120)
    $lblSchedTip.Font = New-Object System.Drawing.Font("Segoe UI", 8)
    $grpSchedule.Controls.Add($lblSchedTip)
    
    # Retention
    $chkRetention = New-Object System.Windows.Forms.CheckBox
    $chkRetention.Text = "Auto-delete files on Drive older than"
    $chkRetention.Location = New-Object System.Drawing.Point(15, 76)
    $chkRetention.Size = New-Object System.Drawing.Size(255, 20)
    $chkRetention.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $grpSchedule.Controls.Add($chkRetention)
    
    $nudRetDays = New-Object System.Windows.Forms.NumericUpDown
    $nudRetDays.Location = New-Object System.Drawing.Point(275, 74)
    $nudRetDays.Size = New-Object System.Drawing.Size(60, 22)
    $nudRetDays.Minimum = 1
    $nudRetDays.Maximum = 9999
    $nudRetDays.Value = 90
    $nudRetDays.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
    $nudRetDays.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $nudRetDays.Enabled = $false
    $grpSchedule.Controls.Add($nudRetDays)
    
    $lblRetDays = New-Object System.Windows.Forms.Label
    $lblRetDays.Text = "days"
    $lblRetDays.Location = New-Object System.Drawing.Point(340, 77)
    $lblRetDays.Size = New-Object System.Drawing.Size(35, 20)
    $grpSchedule.Controls.Add($lblRetDays)
    
    $chkRetention.Add_CheckedChanged({ $nudRetDays.Enabled = $chkRetention.Checked })
    
    $lblRetInfo = New-Object System.Windows.Forms.Label
    $lblRetInfo.Text = "Unchecked = keep all files on Drive forever (no automatic cleanup)"
    $lblRetInfo.Location = New-Object System.Drawing.Point(35, 98)
    $lblRetInfo.Size = New-Object System.Drawing.Size(490, 16)
    $lblRetInfo.ForeColor = [System.Drawing.Color]::FromArgb(120, 120, 120)
    $lblRetInfo.Font = New-Object System.Drawing.Font("Segoe UI", 8)
    $grpSchedule.Controls.Add($lblRetInfo)
    
    # Status
    $lblSchedStatus = New-Object System.Windows.Forms.Label
    $lblSchedStatus.Location = New-Object System.Drawing.Point(15, 122)
    $lblSchedStatus.Size = New-Object System.Drawing.Size(510, 20)
    $lblSchedStatus.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
    if (Test-GDriveSyncTaskExists) {
        $lblSchedStatus.Text = "Status: Scheduled task is ACTIVE"
        $lblSchedStatus.ForeColor = [System.Drawing.Color]::FromArgb(76, 175, 80)
    } else {
        $lblSchedStatus.Text = "Status: Not scheduled"
        $lblSchedStatus.ForeColor = [System.Drawing.Color]::FromArgb(150, 150, 150)
    }
    $grpSchedule.Controls.Add($lblSchedStatus)
    
    # Load saved schedule config
    $schedConfig = Get-GDriveSyncSchedule
    if ($schedConfig) {
        $chkScheduleEnabled.Checked = $true
        if ($schedConfig.Hour -ne $null) { $cmbHour.SelectedItem = ([int]$schedConfig.Hour).ToString("00") }
        if ($schedConfig.Minute -ne $null) {
            $minVal = ([int]$schedConfig.Minute).ToString("00")
            $idx = $cmbMinute.Items.IndexOf($minVal)
            if ($idx -ge 0) { $cmbMinute.SelectedIndex = $idx }
        }
        if ($schedConfig.RetentionEnabled) { $chkRetention.Checked = $true; $nudRetDays.Value = [Math]::Max(1, [int]$schedConfig.RetentionDays) }
    }
    
    # =====================================================================
    # BUTTONS
    # =====================================================================
    $btnY = 500
    
    $btnTest = New-Object System.Windows.Forms.Button
    $btnTest.Text = "Test Sync"
    $btnTest.Location = New-Object System.Drawing.Point(10, $btnY)
    $btnTest.Size = New-Object System.Drawing.Size(100, 35)
    $btnTest.BackColor = [System.Drawing.Color]::FromArgb(0, 120, 215)
    $btnTest.ForeColor = [System.Drawing.Color]::White
    $btnTest.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnTest.Add_Click({
        $s = Test-GDriveConnection
        if (-not $s.Connected) { [System.Windows.Forms.MessageBox]::Show("Connect to Google Drive first.", "Not Connected"); return }
        $target = "$Script:DefaultBackupFolder/$($txtFolder.Text)"
        if ($Mode -eq "SOURCE") {
            [System.Windows.Forms.MessageBox]::Show("Sync: Local -> Drive`n`nLocal: $($txtLocalPath.Text)`nDrive: $target`n`nSame files skipped`nNew/changed files uploaded`nDeleted files removed from Drive", "Sync Preview")
        } else {
            [System.Windows.Forms.MessageBox]::Show("Sync: Drive -> Local`n`nDrive: $target`nLocal: $($txtLocalPath.Text)`n`nSame files skipped`nNew/changed files downloaded`nRemoved files cleaned up locally", "Sync Preview")
        }
    })
    $form.Controls.Add($btnTest)
    
    $btnRunNow = New-Object System.Windows.Forms.Button
    $btnRunNow.Text = "Run Now"
    $btnRunNow.Location = New-Object System.Drawing.Point(120, $btnY)
    $btnRunNow.Size = New-Object System.Drawing.Size(110, 35)
    $btnRunNow.BackColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
    $btnRunNow.ForeColor = [System.Drawing.Color]::Black
    $btnRunNow.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnRunNow.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
    $btnRunNow.Add_Click({
        $s = Test-GDriveConnection
        if (-not $s.Connected) { [System.Windows.Forms.MessageBox]::Show("Connect to Google Drive first.", "Not Connected"); return }
        if ([string]::IsNullOrWhiteSpace($txtLocalPath.Text) -or [string]::IsNullOrWhiteSpace($txtFolder.Text)) {
            [System.Windows.Forms.MessageBox]::Show("Set Local Path and Drive Folder first.", "Missing Settings"); return
        }
        
        $modeDesc = if ($chkDelta.Checked) { "Mirror (sync + delete)" } else { "Additive (copy only)" }
        $confirm = [System.Windows.Forms.MessageBox]::Show(
            "Run sync now?`n`nLocal: $($txtLocalPath.Text)`nDrive: $Script:DefaultBackupFolder/$($txtFolder.Text)`nMode: $modeDesc",
            "Confirm Sync", "YesNo", "Question")
        
        if ($confirm -eq "Yes") {
            $btnRunNow.Text = "Syncing..."
            $btnRunNow.Enabled = $false
            $form.Refresh()
            try {
                if ($Mode -eq "SOURCE") {
                    $result = Push-ToGoogleDrive -LocalPath $txtLocalPath.Text -FolderName $txtFolder.Text -DeltaOnly:$chkDelta.Checked
                } else {
                    $result = Pull-FromGoogleDrive -SourceFolderName $txtFolder.Text -LocalDestination $txtLocalPath.Text -DeltaOnly:$chkDelta.Checked
                }
                if ($result.Success) {
                    if ($Mode -eq "DESTINATION") {
                        [System.Windows.Forms.MessageBox]::Show(
                            "Google Drive sync completed!`n`nFiles downloaded to:`n$($txtLocalPath.Text)`n`nTo restore this backup:`n1. Close this dialog`n2. In DESTINATION mode, enter this path in the SOURCE field:`n   $($txtLocalPath.Text)`n3. Select options and click RUN",
                            "Sync Complete - Ready to Restore", "OK", "Information")
                    } else {
                        [System.Windows.Forms.MessageBox]::Show("Sync completed successfully!", "Done", "OK", "Information")
                    }
                } else {
                    [System.Windows.Forms.MessageBox]::Show("Sync failed: $($result.Error)", "Error", "OK", "Error")
                }
            } catch {
                [System.Windows.Forms.MessageBox]::Show("Error: $($_.Exception.Message)", "Error", "OK", "Error")
            }
            $btnRunNow.Text = "Run Now"
            $btnRunNow.Enabled = $true
        }
    })
    $form.Controls.Add($btnRunNow)
    
    $btnSave = New-Object System.Windows.Forms.Button
    $btnSave.Text = "Save Settings"
    $btnSave.Location = New-Object System.Drawing.Point(320, $btnY)
    $btnSave.Size = New-Object System.Drawing.Size(120, 35)
    $btnSave.BackColor = [System.Drawing.Color]::FromArgb(76, 175, 80)
    $btnSave.ForeColor = [System.Drawing.Color]::Black
    $btnSave.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnSave.Font = New-Object System.Drawing.Font("Segoe UI", 9, [System.Drawing.FontStyle]::Bold)
    $btnSave.Add_Click({
        Save-GDriveConfig -Config @{ Mode=$Mode; FolderName=$txtFolder.Text; LocalPath=$txtLocalPath.Text; DeltaSync=$chkDelta.Checked; SavedAt=(Get-Date).ToString("o") }
        
        if ($chkScheduleEnabled.Checked) {
            $schedHour = [int]$cmbHour.SelectedItem
            $schedMinute = [int]$cmbMinute.SelectedItem
            
            $schedConf = @{
                Mode             = $Mode
                FolderName       = $txtFolder.Text
                LocalPath        = $txtLocalPath.Text
                MirrorMode       = $chkDelta.Checked
                Hour             = $schedHour
                Minute           = $schedMinute
                RetentionEnabled = $chkRetention.Checked
                RetentionDays    = [int]$nudRetDays.Value
                SavedAt          = (Get-Date).ToString("o")
            }
            Save-GDriveSyncSchedule -Config $schedConf
            
            try {
                Register-GDriveSyncTask -Hour $schedHour -Minute $schedMinute
                $lblSchedStatus.Text = "Status: Scheduled task is ACTIVE - daily at $($cmbHour.SelectedItem):$($cmbMinute.SelectedItem)"
                $lblSchedStatus.ForeColor = [System.Drawing.Color]::FromArgb(76, 175, 80)
                [System.Windows.Forms.MessageBox]::Show("Settings saved!`n`nSync scheduled daily at $($cmbHour.SelectedItem):$($cmbMinute.SelectedItem).", "Saved", "OK", "Information")
            } catch {
                [System.Windows.Forms.MessageBox]::Show("Settings saved but failed to create scheduled task:`n`n$($_.Exception.Message)", "Warning", "OK", "Warning")
            }
        } else {
            Remove-GDriveSyncTask
            if (Test-Path $Script:GDriveSyncConfigPath) { Remove-Item $Script:GDriveSyncConfigPath -Force -ErrorAction SilentlyContinue }
            $lblSchedStatus.Text = "Status: Not scheduled"
            $lblSchedStatus.ForeColor = [System.Drawing.Color]::FromArgb(150, 150, 150)
            [System.Windows.Forms.MessageBox]::Show("Settings saved. Scheduled sync disabled.", "Saved", "OK", "Information")
        }
    })
    $form.Controls.Add($btnSave)
    
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(450, $btnY)
    $btnClose.Size = New-Object System.Drawing.Size(100, 35)
    $btnClose.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
    $btnClose.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $btnClose.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnClose.Add_Click({ $form.Close() })
    $form.Controls.Add($btnClose)
    
    # =====================================================================
    # STATUS UPDATER + EVENTS
    # =====================================================================
    $updateStatus = {
        $s = Test-GDriveConnection
        if ($s.Connected) {
            $lblStatus.Text = "Connected (rclone)"
            $lblStatus.ForeColor = [System.Drawing.Color]::FromArgb(76, 175, 80)
            $lblEmail.Text = $s.Email
            $btnLogin.Visible = $false
            $btnDisconnect.Visible = $true
            $btnRestoreAcct.Visible = $false
            $lblAcctInfo.Text = ""
        } else {
            $lblStatus.Text = "Not Connected"
            $lblStatus.ForeColor = [System.Drawing.Color]::FromArgb(244, 67, 54)
            $lblEmail.Text = if ($s.Error) { $s.Error } else { "" }
            $btnLogin.Visible = $true
            $btnDisconnect.Visible = $false
            $btnRestoreAcct.Visible = $false
            $lblAcctInfo.Text = ""
            
            # DESTINATION mode: check if account has stored credentials
            if ($Mode -eq "DESTINATION") {
                try {
                    $acctCreds = Pull-GDriveTokenFromAccount
                    if ($acctCreds.HasToken) {
                        $btnRestoreAcct.Visible = $true
                        $acctEmail = if ($acctCreds.Email) { $acctCreds.Email } else { "credentials found" }
                        $lblAcctInfo.Text = $acctEmail
                        $lblEmail.Text = "Account has saved Drive credentials"
                        $lblEmail.ForeColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
                        # Store folder name for auto-fill
                        if ($acctCreds.FolderName -and $txtFolder.Text -eq "") {
                            $txtFolder.Text = $acctCreds.FolderName
                        }
                    }
                } catch { }
            }
        }
    }
    
    $btnRestoreAcct.Add_Click({
        $btnRestoreAcct.Text = "Restoring..."
        $btnRestoreAcct.Enabled = $false
        $form.Refresh()
        try {
            $result = Restore-GDriveFromAccount
            if ($result.Success) {
                [System.Windows.Forms.MessageBox]::Show(
                    "Google Drive credentials restored from your account!`n`nConnected as: $($result.Email)`nDrive folder: $($result.FolderName)`n`nYou can now sync or browse your backup files.",
                    "Credentials Restored", "OK", "Information")
                if ($result.FolderName -and $txtFolder) { $txtFolder.Text = $result.FolderName }
                & $updateStatus
            } else {
                $errMsg = if ($result.Error) { $result.Error } else { "Unknown error" }
                $doManual = [System.Windows.Forms.MessageBox]::Show(
                    "Could not restore credentials automatically:`n$errMsg`n`nWould you like to sign in manually with Google instead?",
                    "Restore Failed", "YesNo", "Warning")
                if ($doManual -eq "Yes") {
                    Connect-GDrive
                    Start-Sleep -Seconds 1
                    & $updateStatus
                }
            }
        } catch {
            [System.Windows.Forms.MessageBox]::Show("Error: $($_.Exception.Message)", "Error", "OK", "Error")
        }
        $btnRestoreAcct.Text = "Use Account Credentials"
        $btnRestoreAcct.Enabled = $true
    })
    
    $btnLogin.Add_Click({
        $btnLogin.Text = "Connecting..."
        $btnLogin.Enabled = $false
        $form.Refresh()
        try {
            Connect-GDrive
            Start-Sleep -Seconds 1
            & $updateStatus
        } catch {
            [System.Windows.Forms.MessageBox]::Show("Connection error: $($_.Exception.Message)", "Google Drive Error", "OK", "Error")
        }
        $btnLogin.Text = "Connect"
        $btnLogin.Enabled = $true
    })
    
    $btnDisconnect.Add_Click({ Disconnect-GDrive; & $updateStatus })
    
    # Apply saved config
    if ($config.FolderName) { $txtFolder.Text = $config.FolderName }
    if ($config.LocalPath) { $txtLocalPath.Text = $config.LocalPath }
    if ($null -ne $config.DeltaSync) { $chkDelta.Checked = $config.DeltaSync }
    
    $form.Add_Shown({ & $updateStatus })
    $form.ShowDialog() | Out-Null
}
