#Requires -RunAsAdministrator

# =============================================================================
# HELP SYSTEM MODULE - Comprehensive Documentation
# =============================================================================
# Provides help content for all features of the ABC&D Migration Tool
# =============================================================================

$Script:HelpContent = @{
    
    # Main Overview
    "Overview" = @"
ABCD SYSTEMS BACKUP AND MIGRATION TOOLS v4.0
=============================================

A comprehensive Windows server/workstation migration utility designed for 
IT professionals to migrate data, settings, and configurations between machines.

SUPPORTED PLATFORMS:
- Windows 10 / 11
- Windows Server 2016 / 2019 / 2022 / 2025

REQUIREMENTS:
- PowerShell 5.1 or higher
- Administrator privileges
- Network connectivity between SOURCE and DESTINATION
- Valid license key (10-day free trial available)

WORKFLOW:
1. Run as SOURCE on the old machine - exports data to a network share
2. Run as DESTINATION on the new machine - imports data from the share
3. Optional: Schedule automated migrations for off-hours execution
4. Optional: Back up to Google Drive for cloud-based transfers

KEY FEATURES:
- User accounts and passwords (with master password)
- Network shares and permissions
- Printer drivers and configurations
- Program Files / Program Data
- Application settings (100+ applications)
- Additional drives (D: through L:)
- VHD/VHDX disk imaging (Disk2VHD)
- Hyper-V VM export and snapshot management
- Service management for locked files
- File versioning and rollback
- Google Drive cloud backup (via rclone)
- QuickBooks management and service control
- Windows Update automation with reboot handling
- License key extraction (Windows, Office, Adobe, etc.)
- Password recovery (via LaZagne)
- Scheduled task automation

TOOLBAR BUTTONS (bottom row):
  Disk to VHDX   - Create disk images
  VM Export       - Export Hyper-V VMs
  Rollback        - Restore file versions
  VHD Backup      - Backup management settings
  QuickBooks      - QuickBooks service management
  Win Updates     - Windows Update automation
  G-Drive         - Google Drive cloud backup

RECOVERY TOOLS (right side):
  Keys            - Extract license keys from SOURCE
  Passwords       - Recover saved passwords (LaZagne)
  Report          - Open backup/report folder

WEBSITE & SUPPORT:
  Portal: https://abcdtoolset.com
  Admin:  https://abcdtoolset.com/admin/
"@

    # Mode Selection
    "Modes" = @"
OPERATING MODES
===============

SOURCE MODE (Export)
--------------------
Run this on the OLD machine you're migrating FROM.

Actions performed:
- Creates network share (ABCD-Migration$) for data transfer
- Exports user accounts and encrypted passwords
- Exports network share configurations
- Backs up printer drivers using printbrm.exe
- Creates registry backups
- Exports RDS/Terminal Services profiles
- Copies selected folders to the backup location

Important: SOURCE mode is READ-ONLY for your data. It creates exports
but does not modify your existing system configuration.


DESTINATION MODE (Import)
-------------------------
Run this on the NEW machine you're migrating TO.

Actions performed:
- Connects to SOURCE machine's network share
- Imports user accounts with master password
- Recreates network shares with original permissions
- Installs printer drivers and creates printers
- Restores registry settings
- Copies files from SOURCE to local drives
- Optionally creates rollback copies

Important: DESTINATION mode WRITES data to your system. Ensure you have
the correct SOURCE machine specified before starting.
"@

    # Connection Settings
    "Connection" = @"
CONNECTION SETTINGS
===================

SOURCE COMPUTER NAME
--------------------
Enter the hostname or IP address of the SOURCE machine.
Examples: SERVER01, 192.168.1.100, fileserver.domain.local

The tool will connect to: \\COMPUTERNAME\ABCD-Migration$

NETWORK CREDENTIALS
-------------------
Username: Domain\Username or COMPUTERNAME\Username format
Password: The password for the account

These credentials are used to:
- Access the SOURCE machine's network share
- Stop/start services on the SOURCE machine (if enabled)
- Create remote VHD backups

The credentials are stored securely for scheduled tasks using
Windows Credential Manager (encrypted, per-machine).

TESTING CONNECTION
------------------
Click "Test Connection" to verify:
- Network path is accessible
- Credentials are valid
- Required shares exist
"@

    # Migration Options
    "MigrationOptions" = @"
MIGRATION OPTIONS
=================

USERS
-----
Exports/imports local user accounts.
- Preserves user SIDs where possible
- Sets master password on import (for security)
- Users must change password at first login (recommended)

SHARES
------
Exports/imports network share configurations.
- Share names, paths, and descriptions
- Share permissions (ACLs)
- Hidden shares (ending with $)

PRINTERS
--------
Uses Windows printbrm.exe for reliable printer migration.
- Printer drivers
- Printer ports
- Printer configurations
- Print queue settings

PROGRAM FILES / PROGRAM FILES (X86)
-----------------------------------
Copies installed program files.
WARNING: Applications may need reinstallation for registry entries.
Best for portable applications or data folders within Program Files.

PROGRAM DATA
------------
Copies application data from C:\ProgramData.
Includes: Configuration files, databases, license files

USER DATA
---------
Copies all user profiles from C:\Users.
Includes: Desktop, Documents, Downloads, AppData, etc.
Excludes: NTUSER.DAT (registry hives - handled separately)

APP SETTINGS
------------
Migrates application-specific settings for 100+ applications.
See the "Select Apps..." button for the full list.
Categorized by risk level (Safe, Moderate, Advanced, Expert)

ADDITIONAL DRIVES (D: through L:)
---------------------------------
Copies entire drive contents.
Excludes: $RECYCLE.BIN, System Volume Information
Uses robocopy with multi-threading for fast transfers.
"@

    # Settings Options
    "Settings" = @"
SETTINGS
========

OVERWRITE EXISTING FILES
------------------------
When checked: Replaces files on DESTINATION even if they exist
When unchecked: Skips files that already exist (faster for re-runs)

ONLY REPLACE IF SOURCE IS NEWER
-------------------------------
Sub-option of Overwrite. When checked:
- Compares file timestamps
- Only overwrites if SOURCE file is newer
- Preserves newer files on DESTINATION

STOP SERVICES FOR LOCKED FILES
------------------------------
When checked, automatically stops services/processes that lock files:

DATABASE SERVICES:
- QuickBooks Database Server (all versions)
- SQL Server / SQL Server Agent
- MySQL / MariaDB / PostgreSQL
- Sage, MYOB, Pervasive PSQL

APPLICATIONS:
- Microsoft Access (locks .mdb/.accdb files)
- Steam, Epic Games, Origin, Ubisoft, Battle.net
- Adobe Creative Suite
- AutoCAD, SolidWorks, Revit
- Outlook (PST/OST files)
- Dropbox, OneDrive, Google Drive

Services are automatically RESTARTED after file copy completes.

ENABLE FILE VERSIONING (ROLLBACK)
---------------------------------
When checked, creates dated backup copies before overwriting:
- Stores up to 7 days of previous versions
- Location: C:\ABCD-Migration\Rollback\[date]\
- Allows recovery if migration causes issues

MASTER PASSWORD
---------------
(DESTINATION mode only)
Sets a temporary password for all imported user accounts.
Users must change this password at first login.
Requirements: 8+ characters recommended

COPY THREADS
------------
Number of parallel copy threads (1-32).
- Higher = faster, but more CPU/disk usage
- Recommended: 8-16 for local SSD, 4-8 for network
- Default: 8
"@

    # Scheduling
    "Scheduling" = @"
SCHEDULED TASKS
===============

Schedule migrations to run automatically (e.g., overnight).

DATE & TIME
-----------
Select when the migration should run.
Recommended: After business hours when files aren't in use.

SCHEDULE BUTTON
---------------
Creates a Windows Scheduled Task that:
- Runs with SYSTEM privileges
- Stores network credentials securely
- Executes the migration with current settings
- Logs all activity to the Logs folder

SAVE SETTINGS BUTTON (formerly "Run Now")
-----------------------------------------
Saves current settings to JSON without running migration.
Settings are automatically loaded when the tool starts.

REMOVE BUTTON
-------------
Deletes the scheduled task from Windows Task Scheduler.

Settings file location: C:\ABCD-Migration\Settings\migration-settings.json
"@

    # VHD Backup
    "VHDBackup" = @"
VHD BACKUP (DISK2VHD)
=====================

Creates full disk images for backup or P2V migration.

WHAT IS A VHD?
--------------
Virtual Hard Disk - a file that contains the complete contents
of a physical hard drive, including the boot sector, partitions,
and file system.

USE CASES:
- Full system backup before migration
- P2V (Physical to Virtual) conversion to Hyper-V
- Disaster recovery - boot from VHD if needed
- Testing - run the backup in a VM first

HOW TO USE:
1. Click "Create VHD Backup..." button
2. Select drives to include (C:, D:, etc.)
3. Choose output location (needs space = used disk space)
4. Select format:
   - VHDX: Modern format, supports >2TB, recommended
   - VHD: Legacy format, wider compatibility
5. Enable VSS Snapshot for consistent backup
6. Click Create VHD

VSS (Volume Shadow Copy):
- Creates a point-in-time snapshot
- Allows backup while system is running
- Ensures file consistency

AFTER CREATION:
- Double-click VHD/VHDX to mount as a drive letter
- Attach to Hyper-V VM for P2V conversion
- Convert to VMware/VirtualBox format if needed
"@

    # VM Export
    "VMExport" = @"
HYPER-V VM EXPORT
=================

Export running Hyper-V virtual machines with automatic snapshot cleanup.

REQUIREMENTS:
- Hyper-V role installed
- Administrator privileges
- Sufficient disk space for export

HOW IT WORKS:
1. Creates dated export folder (e.g., E:\Backup\14-12-2024)
2. Exports selected VMs to the folder
3. Cleans up old snapshots/checkpoints
4. Optionally restarts VMs after export

RETENTION:
- Keeps exports for 7 days by default
- Automatically removes older exports
- Configurable retention period

SCHEDULING:
- Can be scheduled for off-hours
- Exports are then copied to DESTINATION
- Useful for VM migration between hosts

SNAPSHOT CLEANUP:
After export, removes all VM snapshots/checkpoints to:
- Reduce disk space usage
- Improve VM performance
- Prepare for clean migration
"@

    # Rollback
    "Rollback" = @"
FILE VERSIONING & ROLLBACK
==========================

Automatically preserves previous file versions during migration.

HOW IT WORKS:
1. Before overwriting a file on DESTINATION
2. Copies the existing file to Rollback folder
3. Organizes by date: Rollback\2024-12-14\original\path\file.ext
4. Keeps versions for 7 days
5. Automatically cleans up older versions

WHEN TO USE:
- First-time migrations (safety net)
- Incremental updates (preserve working state)
- Testing migrations (easy undo)

ROLLBACK LOCATION:
C:\ABCD-Migration\Rollback\
  └── 2024-12-14\
      └── C\
          └── Users\
              └── John\
                  └── Documents\
                      └── important.docx

TO RESTORE:
1. Navigate to C:\ABCD-Migration\Rollback\[date]\
2. Find the file you need
3. Copy it back to its original location

AUTOMATIC CLEANUP:
- Files older than 7 days are automatically deleted
- Runs at each migration start
- Prevents rollback folder from growing indefinitely
"@

    # System Tray
    "SystemTray" = @"
SYSTEM TRAY OPERATION
=====================

The tool minimizes to the system tray for background operation.

MINIMIZING:
- Click the minimize button (or close if running scheduled)
- Icon appears in system tray (notification area)
- Double-click tray icon to restore window

TRAY MENU (Right-click):
- Open: Restore the main window
- Status: Show current operation status
- View Logs: Open the logs folder
- Exit: Completely close the application

BACKGROUND OPERATION:
- Scheduled tasks run even when minimized
- Progress continues in background
- Notifications appear for important events

WHY SYSTEM TRAY?
- Keeps tool running for scheduled tasks
- Monitors SOURCE/DESTINATION communication
- Allows long migrations without window clutter
"@

    # Permissions
    "Permissions" = @"
PERMISSION HANDLING
===================

The tool automatically handles permission issues during migration.

ACCESS DENIED ERRORS:
When a file cannot be copied due to permissions:
1. Takes ownership of the file (if possible)
2. Grants read permission to current user
3. Retries the copy operation
4. Restores original permissions after copy

COMMON PERMISSION ISSUES:
- SYSTEM-owned files
- Protected Windows directories
- Files with explicit DENY permissions
- Encrypted files (EFS)

WHAT'S PRESERVED:
- Original NTFS ACLs are copied with /COPYALL flag
- Owner information
- Audit settings (SACL)
- File attributes

WHAT MAY NEED MANUAL INTERVENTION:
- EFS encrypted files (need certificate)
- Files locked by running processes
- Files with corrupted security descriptors

BEST PRACTICE:
Run the tool as Administrator (required)
Use "Stop services for locked files" option
Run during off-hours when fewer files are in use
"@

    # Troubleshooting
    "Troubleshooting" = @"
TROUBLESHOOTING
===============

COMMON ISSUES:

"Cannot access SOURCE machine"
- Verify computer name is correct
- Check network connectivity (ping the machine)
- Ensure File and Printer Sharing is enabled on SOURCE
- Verify firewall allows SMB (port 445)
- Confirm credentials have access to admin shares

"Files failed to copy"
- Check the log for specific error codes
- Error 32: File in use - enable "Stop services" option
- Error 5: Access denied - verify admin rights
- Error 112: Disk full - free up space on DESTINATION

"QuickBooks files locked"
- Enable "Stop services for locked files"
- Or manually stop QuickBooks Database Server on SOURCE
- Run migration after business hours

"Migration is very slow"
- Increase copy threads (8-16 recommended)
- Check network speed between machines
- Avoid running during backup windows
- Consider scheduling for overnight

"Scheduled task doesn't run"
- Check Task Scheduler for errors
- Verify network credentials are correct
- Ensure SOURCE machine is powered on
- Check Windows Event Log for details

"VM Export fails"
- Ensure Hyper-V role is installed
- Check VM is not in saved state
- Verify sufficient disk space for export
- Run as Administrator

LOG FILES:
Location: C:\ABCD-Migration\Logs\
- Bootstrap_*.log: Startup issues
- ABCD-Migration_*.log: Migration details
- Backup_*.log: Backup operations
- Check robocopy logs for file-level errors

GETTING HELP:
1. Check this help documentation
2. Review the log files
3. Contact ABCD Systems support
"@

    # Backup Management
    "BackupManagement" = @"
BACKUP MANAGEMENT
=================

Access via "Backup Settings..." button for comprehensive backup configuration.

VM BACKUP (Hyper-V)
-------------------
Schedule automatic exports of Hyper-V virtual machines.

Settings:
- VM Names: Comma-separated list, or * for all VMs
- Backup Location: Where to store exports (e.g., E:\Backup)
- Schedule Time: Daily backup time (HH:mm format)
- Retention: Days to keep old backups
- Clean Snapshots: Remove checkpoints after export
- Copy to Destination: Transfer backups over network

Export creates dated folders:
  E:\Backup\14-12-2024\
    └── VMName\
        └── (exported VM files)


CUSTOM FOLDER BACKUP
--------------------
Back up up to 20 specific folders with versioning.

For each folder:
- Source Path: Folder to back up
- Display Name: Friendly name for organization
- Keep Copies: Number of versions to retain

Creates timestamped backups:
  E:\Backup\Folders\
    └── MyData\
        ├── 2024-12-14_220000\
        ├── 2024-12-13_220000\
        └── 2024-12-12_220000\


CLEANUP SETTINGS
----------------
Automatic removal of old backup files.

Settings:
- Retention Days: Delete files older than N days
- Cleanup Time: When to run cleanup
- Paths: Folders to clean (one per line)

Cleanup process:
1. Removes files older than retention period
2. Removes empty folders (bottom-up)
3. Removes dated folders (dd-MM-yyyy format) past retention


STARTUP OPTIONS
---------------
Configure tool behavior at Windows startup.

Options:
- Run on Startup: Launch tool when Windows starts
- Start Minimized: Open in system tray


SCHEDULED TASKS CREATED
-----------------------
When enabled, creates Windows Scheduled Tasks:
- ABCD-Migration-VMBackup: VM export schedule
- ABCD-Migration-FolderBackup: Folder backup schedule
- ABCD-Migration-Cleanup: Automatic cleanup
- ABCD-Migration-VMCopy: Network copy of VM backups
- ABCD-Migration-Startup: Run on Windows login
"@

    # Google Drive Cloud Backup
    "GoogleDrive" = @"
GOOGLE DRIVE CLOUD BACKUP
=========================

Back up migration data to Google Drive using rclone.

FIRST-TIME SETUP:
1. Click the G-Drive button on the main toolbar
2. Click 'Connect Google Drive'
3. A browser window opens - sign in to your Google account
4. Complete 2FA verification if enabled
5. Click 'Allow' to grant access
6. Browser closes - token saved locally

After the first connection, all operations are silent (no browser).

HOW IT WORKS:
- Uses rclone with a pre-approved OAuth Client ID
- OAuth tokens stay on YOUR PC (never sent to our servers)
- Our server only tracks whether Drive is connected (for admin dashboard)
- rclone auto-downloads (~20MB) on first use if not bundled

SOURCE MODE (Push):
- Backs up migration data to Google Drive
- Each source computer gets its own folder
- Default folder: ABCD-Migration-Backups/<ComputerName>
- Only changed files are uploaded (delta sync)

DESTINATION MODE (Pull):
- Downloads migration data from Google Drive
- Enter the SOURCE computer's folder name
- Click 'List...' to see available folders on Drive

FILE LOCATIONS:
- rclone binary: Tools\rclone\rclone.exe
- Token config: %APPDATA%\ABCD-Migration\rclone.conf
- Logs: %APPDATA%\ABCD-Migration\logs\gdrive.log

DISCONNECTING:
- Click Disconnect in the G-Drive dialog
- Removes local token only
- Files on Google Drive are NOT deleted
"@

    # QuickBooks Management
    "QuickBooks" = @"
QUICKBOOKS MANAGEMENT
=====================

Manages QuickBooks installations during migrations.

FEATURES:
- Detects all installed QuickBooks versions
- Stops QuickBooks services before file copy
- Restarts services after migration completes
- Manages QuickBooks Database Server
- Handles payroll and update files
- Credential security via Windows DPAPI

USAGE:
1. Click the 'QuickBooks...' button on the toolbar
2. Review detected installations
3. Configure which services to manage
4. Settings are applied during migration RUN

SERVICES MANAGED:
- QuickBooks Database Server Manager (all versions)
- QBCFMonitorService
- QuickBooksDB##
- Intuit.QuickBooks.FCS

IMPORTANT NOTES:
- QuickBooks must be properly licensed on DESTINATION
- Company files (.QBW) are migrated, not licenses
- Database Server Manager may need reconfiguration
- Users should verify data after migration
"@

    # Windows Update Management
    "WindowsUpdate" = @"
WINDOWS UPDATE MANAGEMENT
=========================

Automates Windows Update installation with reboot management.

FEATURES:
- Scans for available updates
- Downloads and installs updates
- Handles reboot-and-continue cycles
- Tracks update progress across reboots
- Optional auto-logon for unattended updates

USAGE:
1. Click 'Win Updates...' on the toolbar
2. Click 'Scan for Updates' to check
3. Review available updates
4. Click 'Install Updates' to begin
5. System will reboot as needed

SCHEDULED UPDATES:
- Can schedule updates for off-hours
- Auto-logon used for reboot continuity
- Progress tracked in state file
- Logs all update activities

STATE FILES:
- C:\ABCD-Migration\Settings\WUState.json
- Tracks which updates are pending/installed
- Survives reboots for continuation

NOTES:
- Requires administrator privileges
- Some updates require multiple reboots
- Feature updates (version upgrades) not included
- Driver updates included by default
"@

    # Recovery Tools (Keys, Passwords, Report)
    "RecoveryTools" = @"
RECOVERY TOOLS
==============

Three tools for extracting credentials and licenses from SOURCE machines.

KEYS BUTTON (License Key Finder):
- Extracts product keys from Windows Registry
- Finds: Windows, Office, Adobe, Autodesk, and more
- Extracts saved WiFi passwords
- Generates HTML report saved to Desktop
- Open report in browser and print to PDF
- SOURCE mode only

PASSWORDS BUTTON (LaZagne):
- Recovers saved passwords from browsers, email, etc.
- LaZagne is NOT bundled (antivirus false positives)
- You must download LaZagne.exe from GitHub
- Place in: C:\ABCD-Migration\Tools\LaZagne.exe
- Results saved to: Backup\passwords_lazagne.txt
- SOURCE mode only

REPORT BUTTON:
- Opens the Backup folder in Windows Explorer
- View all extracted keys, passwords, and reports
- Located at: C:\ABCD-Migration\Backup\

IMPORTANT:
- Run these tools BEFORE migrating
- Keys and passwords are from the SOURCE machine
- Store reports securely (they contain credentials)
- Delete reports after migration is complete
"@

    # Licensing
    "Licensing" = @"
LICENSING & AUTHENTICATION
==========================

The ABC&D Migration Tool requires a valid license to operate.

LICENSE TIERS:
- Single Server ($14.95/month): 1 server, 20 PC migrations
- Unlimited ($50/month): Site license, unlimited servers

LICENSE KEY FORMAT: MIG-XXXX-XXXX-XXXX-XXXX

ACTIVATION:
1. Click the 'License' button (top right)
2. Click 'Sign In' with your email/password
3. Or click 'Enter Key' to enter a license key directly
4. License validates online against abcdtoolset.com

OFFLINE MODE:
- License cached locally for 7 days
- Tool works without internet after initial activation
- Re-validates when connection is available

MACHINE TRACKING:
- Each machine activation counts toward your server limit
- Machine ID based on hardware fingerprint
- Deactivate machines from the admin portal

MANAGE YOUR LICENSE:
- Web portal: https://abcdtoolset.com/user/
- Admin panel: https://abcdtoolset.com/admin/
- View usage, manage machines, update payment

TRIAL:
- 10-day trial available for new accounts
- Full functionality during trial period
- Subscribe before trial expires to keep access
"@
}

function Show-HelpWindow {
    <#
    .SYNOPSIS
        Displays the comprehensive help window
    #>
    param(
        [string]$InitialTopic = "Overview"
    )
    
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $helpForm = New-Object System.Windows.Forms.Form
    $helpForm.Text = "ABC&D Migration Tool - Help"
    $helpForm.Size = New-Object System.Drawing.Size(900, 700)
    $helpForm.StartPosition = "CenterScreen"
    $helpForm.Icon = [System.Drawing.SystemIcons]::Information
    
    # Topic list (left panel)
    $lstTopics = New-Object System.Windows.Forms.ListBox
    $lstTopics.Location = New-Object System.Drawing.Point(10, 10)
    $lstTopics.Size = New-Object System.Drawing.Size(180, 640)
    $lstTopics.Font = New-Object System.Drawing.Font("Segoe UI", 10)
    
    # Add topics
    $topics = @(
        "Overview",
        "Modes",
        "Connection",
        "MigrationOptions",
        "Settings",
        "Scheduling",
        "VHDBackup",
        "VMExport",
        "BackupManagement",
        "Rollback",
        "GoogleDrive",
        "QuickBooks",
        "WindowsUpdate",
        "RecoveryTools",
        "Licensing",
        "SystemTray",
        "Permissions",
        "Troubleshooting"
    )
    
    foreach ($topic in $topics) {
        $displayName = switch ($topic) {
            "Overview" { "Overview" }
            "Modes" { "SOURCE / DESTINATION Modes" }
            "Connection" { "Connection Settings" }
            "MigrationOptions" { "Migration Options" }
            "Settings" { "Settings & Options" }
            "Scheduling" { "Scheduled Tasks" }
            "VHDBackup" { "VHD Backup (Disk2VHD)" }
            "VMExport" { "Hyper-V VM Export" }
            "BackupManagement" { "Backup Management" }
            "Rollback" { "File Versioning" }
            "GoogleDrive" { "Google Drive Backup" }
            "QuickBooks" { "QuickBooks Management" }
            "WindowsUpdate" { "Windows Updates" }
            "RecoveryTools" { "Keys / Passwords / Report" }
            "Licensing" { "Licensing & Auth" }
            "SystemTray" { "System Tray" }
            "Permissions" { "Permission Handling" }
            "Troubleshooting" { "Troubleshooting" }
            default { $topic }
        }
        $lstTopics.Items.Add($displayName) | Out-Null
    }
    
    $helpForm.Controls.Add($lstTopics)
    
    # Content panel (right side)
    $txtContent = New-Object System.Windows.Forms.RichTextBox
    $txtContent.Location = New-Object System.Drawing.Point(200, 10)
    $txtContent.Size = New-Object System.Drawing.Size(675, 600)
    $txtContent.Font = New-Object System.Drawing.Font("Consolas", 10)
    $txtContent.ReadOnly = $true
    $txtContent.BackColor = [System.Drawing.Color]::White
    $txtContent.BorderStyle = "FixedSingle"
    $txtContent.WordWrap = $true
    $helpForm.Controls.Add($txtContent)
    
    # Close button
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(780, 620)
    $btnClose.Size = New-Object System.Drawing.Size(95, 30)
    $btnClose.Add_Click({ $helpForm.Close() })
    $helpForm.Controls.Add($btnClose)
    
    # Search box
    $lblSearch = New-Object System.Windows.Forms.Label
    $lblSearch.Text = "Search:"
    $lblSearch.Location = New-Object System.Drawing.Point(200, 620)
    $lblSearch.Size = New-Object System.Drawing.Size(50, 25)
    $helpForm.Controls.Add($lblSearch)
    
    $txtSearch = New-Object System.Windows.Forms.TextBox
    $txtSearch.Location = New-Object System.Drawing.Point(255, 618)
    $txtSearch.Size = New-Object System.Drawing.Size(200, 25)
    $helpForm.Controls.Add($txtSearch)
    
    $btnSearch = New-Object System.Windows.Forms.Button
    $btnSearch.Text = "Find"
    $btnSearch.Location = New-Object System.Drawing.Point(460, 617)
    $btnSearch.Size = New-Object System.Drawing.Size(60, 27)
    $helpForm.Controls.Add($btnSearch)
    
    # Topic selection handler
    $lstTopics.Add_SelectedIndexChanged({
        $selectedDisplay = $lstTopics.SelectedItem
        $topicKey = switch ($selectedDisplay) {
            "Overview" { "Overview" }
            "SOURCE / DESTINATION Modes" { "Modes" }
            "Connection Settings" { "Connection" }
            "Migration Options" { "MigrationOptions" }
            "Settings & Options" { "Settings" }
            "Scheduled Tasks" { "Scheduling" }
            "VHD Backup (Disk2VHD)" { "VHDBackup" }
            "Hyper-V VM Export" { "VMExport" }
            "Backup Management" { "BackupManagement" }
            "File Versioning" { "Rollback" }
            "Google Drive Backup" { "GoogleDrive" }
            "QuickBooks Management" { "QuickBooks" }
            "Windows Updates" { "WindowsUpdate" }
            "Keys / Passwords / Report" { "RecoveryTools" }
            "Licensing & Auth" { "Licensing" }
            "System Tray" { "SystemTray" }
            "Permission Handling" { "Permissions" }
            "Troubleshooting" { "Troubleshooting" }
            default { "Overview" }
        }
        
        if ($Script:HelpContent.ContainsKey($topicKey)) {
            $txtContent.Text = $Script:HelpContent[$topicKey]
        }
    })
    
    # Search handler
    $btnSearch.Add_Click({
        $searchText = $txtSearch.Text
        if ($searchText) {
            $start = $txtContent.SelectionStart + $txtContent.SelectionLength
            $index = $txtContent.Text.IndexOf($searchText, $start, [System.StringComparison]::OrdinalIgnoreCase)
            
            if ($index -eq -1) {
                # Wrap around
                $index = $txtContent.Text.IndexOf($searchText, 0, [System.StringComparison]::OrdinalIgnoreCase)
            }
            
            if ($index -ge 0) {
                $txtContent.Select($index, $searchText.Length)
                $txtContent.ScrollToCaret()
                $txtContent.Focus()
            }
            else {
                [System.Windows.Forms.MessageBox]::Show("Text not found: $searchText", "Search", 
                    [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Information)
            }
        }
    })
    
    # Select initial topic
    $initialIndex = $topics.IndexOf($InitialTopic)
    if ($initialIndex -ge 0) {
        $lstTopics.SelectedIndex = $initialIndex
    }
    else {
        $lstTopics.SelectedIndex = 0
    }
    
    $helpForm.ShowDialog()
}

function Get-HelpText {
    <#
    .SYNOPSIS
        Returns help text for a specific topic
    #>
    param(
        [string]$Topic = "Overview"
    )
    
    if ($Script:HelpContent.ContainsKey($Topic)) {
        return $Script:HelpContent[$Topic]
    }
    return "Help topic not found: $Topic"
}

