#Requires -RunAsAdministrator

<#
.SYNOPSIS
    License Key Finder Module for ABCD-Migration Migration Toolkit
    
.DESCRIPTION
    Scans the system for installed software and extracts license/product keys
    from known registry locations. Exports results to HTML for printing to PDF.
    
    This is similar to EaseUS KeyFinder, ProduKey, etc. but built in PowerShell.
    
.NOTES
    - Windows product keys require decoding from DigitalProductId
    - Some software stores keys in plaintext, others are encoded
    - Cloud-based software (Adobe CC, Office 365) won't have local keys
#>

# ============================================================
# KEY FINDER FUNCTIONS
# ============================================================

function Get-WindowsProductKey {
    <#
    .SYNOPSIS
        Extracts Windows product key from registry
    #>
    
    try {
        # Method 1: Try SoftwareLicensingService (Windows 8+)
        $key = (Get-WmiObject -Query "SELECT OA3xOriginalProductKey FROM SoftwareLicensingService" -ErrorAction SilentlyContinue).OA3xOriginalProductKey
        if ($key) {
            return @{
                Product = "Windows (OEM Key)"
                Key = $key
                Source = "SoftwareLicensingService"
            }
        }
        
        # Method 2: Decode from DigitalProductId
        $regPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion"
        $digitalProductId = (Get-ItemProperty -Path $regPath -ErrorAction SilentlyContinue).DigitalProductId
        
        if ($digitalProductId) {
            $key = Decode-ProductKey -DigitalProductId $digitalProductId
            if ($key) {
                $productName = (Get-ItemProperty -Path $regPath).ProductName
                return @{
                    Product = "Windows - $productName"
                    Key = $key
                    Source = "Registry (DigitalProductId)"
                }
            }
        }
        
        # Method 3: Try backup key location
        $backupPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\SoftwareProtectionPlatform"
        $backupKey = (Get-ItemProperty -Path $backupPath -ErrorAction SilentlyContinue).BackupProductKeyDefault
        if ($backupKey) {
            return @{
                Product = "Windows (Backup Key)"
                Key = $backupKey
                Source = "Registry (BackupProductKeyDefault)"
            }
        }
        
        return $null
    }
    catch {
        return $null
    }
}

function Decode-ProductKey {
    param(
        [byte[]]$DigitalProductId
    )
    
    try {
        # Windows 8+ uses different offset
        $isWin8OrLater = [Environment]::OSVersion.Version.Major -ge 6 -and [Environment]::OSVersion.Version.Minor -ge 2
        
        if ($isWin8OrLater) {
            $keyOffset = 52
        } else {
            $keyOffset = 52
        }
        
        # Check if we have enough data
        if ($DigitalProductId.Length -lt ($keyOffset + 15)) {
            return $null
        }
        
        $chars = "BCDFGHJKMPQRTVWXY2346789"
        $keyOutput = ""
        $last = 0
        
        $keyPart = $DigitalProductId[$keyOffset..($keyOffset + 14)]
        
        for ($i = 24; $i -ge 0; $i--) {
            $current = 0
            for ($j = 14; $j -ge 0; $j--) {
                $current = $current * 256
                $current = $keyPart[$j] + $current
                $keyPart[$j] = [math]::Floor($current / 24)
                $current = $current % 24
                $last = $current
            }
            $keyOutput = $chars[$current] + $keyOutput
        }
        
        # Format with dashes
        $keyFormatted = ""
        for ($i = 0; $i -lt 25; $i++) {
            $keyFormatted += $keyOutput[$i]
            if (($i + 1) % 5 -eq 0 -and $i -lt 24) {
                $keyFormatted += "-"
            }
        }
        
        return $keyFormatted
    }
    catch {
        return $null
    }
}

function Get-OfficeProductKeys {
    <#
    .SYNOPSIS
        Extracts Microsoft Office product keys
    #>
    
    $keys = @()
    
    # Office registry paths to check
    $officePaths = @(
        @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\16.0\Registration"; Version = "Office 2016/2019/365" },
        @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\15.0\Registration"; Version = "Office 2013" },
        @{ Path = "HKLM:\SOFTWARE\Microsoft\Office\14.0\Registration"; Version = "Office 2010" },
        @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\16.0\Registration"; Version = "Office 2016/2019/365 (32-bit)" },
        @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\15.0\Registration"; Version = "Office 2013 (32-bit)" },
        @{ Path = "HKLM:\SOFTWARE\Wow6432Node\Microsoft\Office\14.0\Registration"; Version = "Office 2010 (32-bit)" }
    )
    
    foreach ($officePath in $officePaths) {
        if (Test-Path $officePath.Path) {
            $subKeys = Get-ChildItem -Path $officePath.Path -ErrorAction SilentlyContinue
            foreach ($subKey in $subKeys) {
                $props = Get-ItemProperty -Path $subKey.PSPath -ErrorAction SilentlyContinue
                if ($props.DigitalProductId) {
                    $key = Decode-ProductKey -DigitalProductId $props.DigitalProductId
                    if ($key) {
                        $productName = $props.ProductName
                        if (-not $productName) { $productName = $officePath.Version }
                        
                        $keys += @{
                            Product = "Microsoft Office - $productName"
                            Key = $key
                            Source = "Registry"
                        }
                    }
                }
                # Some Office versions store partial key
                if ($props.ProductKey) {
                    $keys += @{
                        Product = "Microsoft Office - $($officePath.Version)"
                        Key = $props.ProductKey
                        Source = "Registry (Partial Key)"
                    }
                }
            }
        }
    }
    
    # Check Click-to-Run Office
    $c2rPath = "HKLM:\SOFTWARE\Microsoft\Office\ClickToRun\Configuration"
    if (Test-Path $c2rPath) {
        $c2rProps = Get-ItemProperty -Path $c2rPath -ErrorAction SilentlyContinue
        if ($c2rProps.ProductReleaseIds) {
            $keys += @{
                Product = "Microsoft 365 / Office Click-to-Run"
                Key = "Cloud-based license - Sign in with Microsoft account"
                Source = "Product: $($c2rProps.ProductReleaseIds)"
            }
        }
    }
    
    return $keys
}

function Get-QuickBooksKeys {
    <#
    .SYNOPSIS
        Extracts QuickBooks product/license keys
    #>
    
    $keys = @()
    
    # QuickBooks registry paths
    $qbPaths = @(
        "HKLM:\SOFTWARE\Intuit\QuickBooks",
        "HKLM:\SOFTWARE\Wow6432Node\Intuit\QuickBooks",
        "HKLM:\SOFTWARE\Intuit\QuickBooks Enterprise Solutions",
        "HKLM:\SOFTWARE\Wow6432Node\Intuit\QuickBooks Enterprise Solutions"
    )
    
    foreach ($basePath in $qbPaths) {
        if (Test-Path $basePath) {
            # Check for version subkeys
            $versions = Get-ChildItem -Path $basePath -ErrorAction SilentlyContinue
            foreach ($version in $versions) {
                $props = Get-ItemProperty -Path $version.PSPath -ErrorAction SilentlyContinue
                
                $licenseNumber = $null
                $productNumber = $null
                
                if ($props.LicenseNumber) { $licenseNumber = $props.LicenseNumber }
                if ($props.ProductNumber) { $productNumber = $props.ProductNumber }
                if ($props.InstallLicenseKey) { $licenseNumber = $props.InstallLicenseKey }
                
                if ($licenseNumber -or $productNumber) {
                    $versionName = $version.PSChildName
                    $keys += @{
                        Product = "QuickBooks $versionName"
                        Key = if ($licenseNumber) { "License: $licenseNumber" } else { "" }
                        ProductNumber = if ($productNumber) { "Product #: $productNumber" } else { "" }
                        Source = "Registry"
                    }
                }
            }
        }
    }
    
    # Also check qbregistration.dat files
    $qbRegFiles = Get-ChildItem "C:\ProgramData\Intuit" -Filter "qbregistration.dat" -Recurse -ErrorAction SilentlyContinue
    foreach ($file in $qbRegFiles) {
        try {
            $content = Get-Content $file.FullName -Raw -ErrorAction SilentlyContinue
            if ($content -match "LicenseNumber=([^\r\n]+)") {
                $keys += @{
                    Product = "QuickBooks (from registration file)"
                    Key = "License: $($Matches[1])"
                    Source = $file.FullName
                }
            }
        }
        catch {}
    }
    
    return $keys
}

function Get-AdobeKeys {
    <#
    .SYNOPSIS
        Checks for Adobe products (most are cloud-based now)
    #>
    
    $keys = @()
    
    # Adobe CS6 and earlier might have keys
    $adobePaths = @(
        "HKLM:\SOFTWARE\Adobe",
        "HKLM:\SOFTWARE\Wow6432Node\Adobe"
    )
    
    foreach ($basePath in $adobePaths) {
        if (Test-Path $basePath) {
            # Check for serial numbers in various locations
            $products = Get-ChildItem -Path $basePath -Recurse -ErrorAction SilentlyContinue | 
                Where-Object { $_.Property -contains "Serial" -or $_.Property -contains "SerialNumber" }
            
            foreach ($product in $products) {
                $props = Get-ItemProperty -Path $product.PSPath -ErrorAction SilentlyContinue
                $serial = $props.Serial
                if (-not $serial) { $serial = $props.SerialNumber }
                
                if ($serial) {
                    $keys += @{
                        Product = "Adobe - $($product.PSChildName)"
                        Key = $serial
                        Source = "Registry"
                    }
                }
            }
        }
    }
    
    # Check for Creative Cloud
    $cc64 = Test-Path "C:\Program Files\Adobe\Adobe Creative Cloud"
    $cc32 = Test-Path "C:\Program Files (x86)\Adobe\Adobe Creative Cloud"
    if ($cc64 -or $cc32) {
        $keys += @{
            Product = "Adobe Creative Cloud"
            Key = "Cloud-based license - Sign in with Adobe ID"
            Source = "Installation detected"
        }
    }
    
    return $keys
}

function Get-AutodeskKeys {
    <#
    .SYNOPSIS
        Extracts Autodesk product keys
    #>
    
    $keys = @()
    
    $autodeskPath = "HKLM:\SOFTWARE\Autodesk"
    if (Test-Path $autodeskPath) {
        $products = Get-ChildItem -Path $autodeskPath -Recurse -ErrorAction SilentlyContinue |
            Where-Object { $_.Property -contains "SerialNumber" -or $_.Property -contains "ProductKey" -or $_.Property -contains "Serial" }
        
        foreach ($product in $products) {
            $props = Get-ItemProperty -Path $product.PSPath -ErrorAction SilentlyContinue
            
            $serial = $props.SerialNumber
            if (-not $serial) { $serial = $props.Serial }
            $productKey = $props.ProductKey
            
            if ($serial -or $productKey) {
                $keyInfo = @()
                if ($serial) { $keyInfo += "Serial: $serial" }
                if ($productKey) { $keyInfo += "Product Key: $productKey" }
                
                $keys += @{
                    Product = "Autodesk - $($product.PSChildName)"
                    Key = $keyInfo -join " | "
                    Source = "Registry"
                }
            }
        }
    }
    
    return $keys
}

function Get-GenericSoftwareKeys {
    <#
    .SYNOPSIS
        Scans for other common software with known key locations
    #>
    
    $keys = @()
    
    # Define software to scan with their registry paths
    $softwareList = @(
        @{ 
            Name = "VMware Workstation"
            Paths = @("HKLM:\SOFTWARE\VMware, Inc.\VMware Workstation", "HKLM:\SOFTWARE\Wow6432Node\VMware, Inc.\VMware Workstation")
            Properties = @("SerialNumber", "Serial", "License")
        },
        @{
            Name = "WinRAR"
            Paths = @("HKLM:\SOFTWARE\WinRAR", "HKLM:\SOFTWARE\Wow6432Node\WinRAR", "HKCU:\SOFTWARE\WinRAR")
            Properties = @("License", "LicenseKey")
        },
        @{
            Name = "Sublime Text"
            Paths = @("HKCU:\SOFTWARE\Sublime Text 3", "HKCU:\SOFTWARE\Sublime Text")
            Properties = @("license", "License")
        },
        @{
            Name = "Corel"
            Paths = @("HKLM:\SOFTWARE\Corel", "HKLM:\SOFTWARE\Wow6432Node\Corel")
            Properties = @("Serial", "SerialNumber")
        },
        @{
            Name = "Sage"
            Paths = @("HKLM:\SOFTWARE\Sage", "HKLM:\SOFTWARE\Wow6432Node\Sage")
            Properties = @("SerialNumber", "ActivationKey", "Serial")
        },
        @{
            Name = "SQL Server"
            Paths = @("HKLM:\SOFTWARE\Microsoft\Microsoft SQL Server")
            Properties = @("ProductKey", "DigitalProductId")
        }
    )
    
    foreach ($software in $softwareList) {
        foreach ($path in $software.Paths) {
            if (Test-Path $path) {
                # Check main path and subkeys
                $pathsToCheck = @($path)
                $pathsToCheck += Get-ChildItem -Path $path -Recurse -ErrorAction SilentlyContinue | Select-Object -ExpandProperty PSPath
                
                foreach ($checkPath in $pathsToCheck) {
                    try {
                        $props = Get-ItemProperty -Path $checkPath -ErrorAction SilentlyContinue
                        foreach ($prop in $software.Properties) {
                            $value = $props.$prop
                            if ($value -and $value -is [string] -and $value.Length -gt 5) {
                                $keys += @{
                                    Product = $software.Name
                                    Key = $value
                                    Source = "Registry"
                                }
                                break  # Found a key, move to next path
                            }
                        }
                    }
                    catch {}
                }
            }
        }
    }
    
    return $keys
}

function Get-WifiPasswords {
    <#
    .SYNOPSIS
        Extracts saved WiFi passwords (useful for migration)
    #>
    
    $wifiKeys = @()
    
    try {
        $profiles = netsh wlan show profiles 2>$null | Select-String "All User Profile" | ForEach-Object {
            ($_ -split ":")[1].Trim()
        }
        
        foreach ($profile in $profiles) {
            if ($profile) {
                $password = netsh wlan show profile name="$profile" key=clear 2>$null | 
                    Select-String "Key Content" | 
                    ForEach-Object { ($_ -split ":")[1].Trim() }
                
                if ($password) {
                    $wifiKeys += @{
                        Network = $profile
                        Password = $password
                    }
                }
            }
        }
    }
    catch {}
    
    return $wifiKeys
}

# ============================================================
# MAIN EXPORT FUNCTION
# ============================================================

function Export-AllLicenseKeys {
    <#
    .SYNOPSIS
        Main function to find all license keys and export to HTML/PDF
    #>
    param(
        [string]$OutputPath = "C:\ABCD-Migration\Backup",
        [switch]$IncludeWifi
    )
    
    Write-Host "  [KeyFinder] ================================================" -ForegroundColor Cyan
    Write-Host "  [KeyFinder] Scanning System for License Keys" -ForegroundColor Cyan
    Write-Host "  [KeyFinder] ================================================" -ForegroundColor Cyan
    
    if (!(Test-Path $OutputPath)) {
        New-Item -Path $OutputPath -ItemType Directory -Force | Out-Null
    }
    
    $allKeys = @()
    $scanDate = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $computerName = $env:COMPUTERNAME
    
    # Windows
    Write-Host "  [KeyFinder] Scanning Windows..." -ForegroundColor Yellow
    $windowsKey = Get-WindowsProductKey
    if ($windowsKey) {
        $allKeys += $windowsKey
        Write-Host "  [KeyFinder]   Found: Windows Product Key" -ForegroundColor Green
    }
    
    # Office
    Write-Host "  [KeyFinder] Scanning Microsoft Office..." -ForegroundColor Yellow
    $officeKeys = Get-OfficeProductKeys
    foreach ($key in $officeKeys) {
        $allKeys += $key
        Write-Host "  [KeyFinder]   Found: $($key.Product)" -ForegroundColor Green
    }
    
    # QuickBooks
    Write-Host "  [KeyFinder] Scanning QuickBooks..." -ForegroundColor Yellow
    $qbKeys = Get-QuickBooksKeys
    foreach ($key in $qbKeys) {
        $allKeys += $key
        Write-Host "  [KeyFinder]   Found: $($key.Product)" -ForegroundColor Green
    }
    
    # Adobe
    Write-Host "  [KeyFinder] Scanning Adobe..." -ForegroundColor Yellow
    $adobeKeys = Get-AdobeKeys
    foreach ($key in $adobeKeys) {
        $allKeys += $key
        Write-Host "  [KeyFinder]   Found: $($key.Product)" -ForegroundColor Green
    }
    
    # Autodesk
    Write-Host "  [KeyFinder] Scanning Autodesk..." -ForegroundColor Yellow
    $autodeskKeys = Get-AutodeskKeys
    foreach ($key in $autodeskKeys) {
        $allKeys += $key
        Write-Host "  [KeyFinder]   Found: $($key.Product)" -ForegroundColor Green
    }
    
    # Other Software
    Write-Host "  [KeyFinder] Scanning other software..." -ForegroundColor Yellow
    $otherKeys = Get-GenericSoftwareKeys
    foreach ($key in $otherKeys) {
        $allKeys += $key
        Write-Host "  [KeyFinder]   Found: $($key.Product)" -ForegroundColor Green
    }
    
    # WiFi Passwords (optional)
    $wifiPasswords = @()
    if ($IncludeWifi) {
        Write-Host "  [KeyFinder] Scanning WiFi passwords..." -ForegroundColor Yellow
        $wifiPasswords = Get-WifiPasswords
        foreach ($wifi in $wifiPasswords) {
            Write-Host "  [KeyFinder]   Found WiFi: $($wifi.Network)" -ForegroundColor Green
        }
    }
    
    Write-Host "  [KeyFinder] ------------------------------------------------" -ForegroundColor Cyan
    Write-Host "  [KeyFinder] Total keys found: $($allKeys.Count)" -ForegroundColor Green
    
    # Generate HTML Report
    $htmlPath = Join-Path $OutputPath "License_Keys_Report.html"
    $txtPath = Join-Path $OutputPath "License_Keys_Report.txt"
    
    # Build HTML
    $html = @"
<!DOCTYPE html>
<html>
<head>
    <title>License Keys Report - $computerName</title>
    <style>
        body { font-family: 'Segoe UI', Arial, sans-serif; margin: 40px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        h1 { color: #333; border-bottom: 3px solid #0078d4; padding-bottom: 10px; }
        h2 { color: #0078d4; margin-top: 30px; }
        .info { background: #e7f3ff; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .warning { background: #fff3cd; padding: 15px; border-radius: 5px; margin-bottom: 20px; border-left: 4px solid #ffc107; }
        table { width: 100%; border-collapse: collapse; margin-top: 10px; }
        th { background: #0078d4; color: white; padding: 12px; text-align: left; }
        td { padding: 10px; border-bottom: 1px solid #ddd; }
        tr:hover { background: #f5f5f5; }
        .key { font-family: 'Consolas', monospace; background: #f0f0f0; padding: 3px 8px; border-radius: 3px; }
        .footer { margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; color: #666; font-size: 12px; }
        @media print { 
            body { margin: 0; background: white; }
            .container { box-shadow: none; }
            .no-print { display: none; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>LICENSE KEYS REPORT</h1>
        
        <div class="info">
            <strong>Computer:</strong> $computerName<br>
            <strong>Generated:</strong> $scanDate<br>
            <strong>Keys Found:</strong> $($allKeys.Count)
        </div>
        
        <div class="warning">
            <strong>CONFIDENTIAL</strong><br>
            This document contains sensitive license information. Store securely and destroy after migration.
        </div>
        
        <h2>Software License Keys</h2>
        <table>
            <tr>
                <th>Product</th>
                <th>License Key / Information</th>
            </tr>
"@

    foreach ($key in $allKeys) {
        $keyDisplay = $key.Key
        if ($key.ProductNumber) {
            $keyDisplay += "<br><small>$($key.ProductNumber)</small>"
        }
        $html += @"
            <tr>
                <td><strong>$($key.Product)</strong><br><small style="color:#666">$($key.Source)</small></td>
                <td><span class="key">$keyDisplay</span></td>
            </tr>
"@
    }

    $html += @"
        </table>
"@

    # Add WiFi section if included
    if ($wifiPasswords.Count -gt 0) {
        $html += @"
        
        <h2>WiFi Network Passwords</h2>
        <table>
            <tr>
                <th>Network Name (SSID)</th>
                <th>Password</th>
            </tr>
"@
        foreach ($wifi in $wifiPasswords) {
            $html += @"
            <tr>
                <td><strong>$($wifi.Network)</strong></td>
                <td><span class="key">$($wifi.Password)</span></td>
            </tr>
"@
        }
        $html += "</table>"
    }

    $html += @"
        
        <div class="footer">
            <p>Generated by ABCD-Migration Migration Toolkit</p>
            <p class="no-print"><button onclick="window.print()">Print to PDF</button></p>
        </div>
    </div>
</body>
</html>
"@

    # Save HTML
    $html | Out-File -FilePath $htmlPath -Encoding UTF8
    Write-Host "  [KeyFinder] HTML Report: $htmlPath" -ForegroundColor Green
    
    # Also save plain text version
    $txt = @"
============================================================
LICENSE KEYS REPORT
============================================================
Computer: $computerName
Generated: $scanDate
Keys Found: $($allKeys.Count)
============================================================

*** CONFIDENTIAL - Store securely! ***

SOFTWARE LICENSE KEYS
------------------------------------------------------------

"@

    foreach ($key in $allKeys) {
        $txt += "Product: $($key.Product)`r`n"
        $txt += "Key: $($key.Key)`r`n"
        if ($key.ProductNumber) { $txt += "$($key.ProductNumber)`r`n" }
        $txt += "Source: $($key.Source)`r`n"
        $txt += "------------------------------------------------------------`r`n"
    }

    if ($wifiPasswords.Count -gt 0) {
        $txt += "`r`nWIFI PASSWORDS`r`n"
        $txt += "------------------------------------------------------------`r`n"
        foreach ($wifi in $wifiPasswords) {
            $txt += "Network: $($wifi.Network)`r`n"
            $txt += "Password: $($wifi.Password)`r`n"
            $txt += "------------------------------------------------------------`r`n"
        }
    }

    $txt | Out-File -FilePath $txtPath -Encoding UTF8
    Write-Host "  [KeyFinder] Text Report: $txtPath" -ForegroundColor Green
    
    # Copy to Admin Desktop
    $desktopPath = [Environment]::GetFolderPath("Desktop")
    $desktopHtml = Join-Path $desktopPath "License_Keys_Report.html"
    Copy-Item $htmlPath $desktopHtml -Force
    Write-Host "  [KeyFinder] Copied to Desktop: License_Keys_Report.html" -ForegroundColor Green
    
    # Try to open in browser for printing
    Write-Host "  [KeyFinder] Opening report in browser (print to PDF)..." -ForegroundColor Yellow
    Start-Process $desktopHtml
    
    Write-Host "  [KeyFinder] ================================================" -ForegroundColor Cyan
    Write-Host "  [KeyFinder] DONE! Print the report to PDF for safekeeping" -ForegroundColor Green
    Write-Host "  [KeyFinder] ================================================" -ForegroundColor Cyan
    
    return @{
        Keys = $allKeys
        WifiPasswords = $wifiPasswords
        HtmlPath = $htmlPath
        TextPath = $txtPath
    }
}

