<#
.SYNOPSIS
    ABC&D Migration Tool - Licensing Module
.DESCRIPTION
    Handles license validation, login, and usage tracking via web API.
.VERSION
    4.0
#>

$Script:API_BASE = "https://abcdtoolset.com/api"
$Script:SettingsPath = "$env:APPDATA\ABCD-Migration\settings.json"

function Get-SavedCredentials {
    if (Test-Path $Script:SettingsPath) {
        try {
            return Get-Content $Script:SettingsPath -Raw | ConvertFrom-Json
        } catch {
            return $null
        }
    }
    return $null
}

function Save-Credentials {
    param(
        [string]$Token,
        [string]$LicenseKey,
        [string]$Email
    )
    
    $dir = Split-Path $Script:SettingsPath -Parent
    if (!(Test-Path $dir)) {
        New-Item -ItemType Directory -Path $dir -Force | Out-Null
    }
    
    @{
        Token = $Token
        LicenseKey = $LicenseKey
        Email = $Email
        SavedAt = (Get-Date).ToString("o")
    } | ConvertTo-Json | Set-Content $Script:SettingsPath -Force
}

function Get-MachineId {
    try {
        return (Get-CimInstance Win32_ComputerSystemProduct).UUID
    } catch {
        return [System.Environment]::MachineName + "-" + [System.Environment]::UserName
    }
}

function Test-License {
    param(
        [string]$LicenseKey,
        [string]$Token
    )
    
    try {
        $body = @{ machineId = Get-MachineId }
        if ($LicenseKey) { $body.licenseKey = $LicenseKey }
        if ($Token) { $body.token = $Token }
        
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/auth.php?action=validate" `
            -Method POST -Body ($body | ConvertTo-Json) -ContentType "application/json" -TimeoutSec 30
        
        return $response
    }
    catch {
        Write-Warning "License validation error: $_"
        return @{ success = $false; valid = $false; error = $_.Exception.Message }
    }
}

function Invoke-Login {
    param(
        [Parameter(Mandatory)][string]$Email,
        [Parameter(Mandatory)][string]$Password
    )
    
    try {
        $body = @{
            email = $Email
            password = $Password
            machineId = Get-MachineId
        }
        
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/auth.php?action=login" `
            -Method POST -Body ($body | ConvertTo-Json) -ContentType "application/json" -TimeoutSec 30
        
        if ($response.success) {
            Save-Credentials -Token $response.token -LicenseKey $response.licenseKey -Email $Email
        }
        
        return $response
    }
    catch {
        return @{ success = $false; error = $_.Exception.Message }
    }
}

function Register-License {
    param(
        [Parameter(Mandatory)][string]$Name,
        [Parameter(Mandatory)][string]$Email,
        [Parameter(Mandatory)][string]$Password,
        [string]$Plan = "single_monthly"
    )
    
    try {
        $body = @{
            name = $Name
            email = $Email
            password = $Password
            plan = $Plan
        }
        
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/auth.php?action=register" `
            -Method POST -Body ($body | ConvertTo-Json) -ContentType "application/json" -TimeoutSec 30
        
        if ($response.success) {
            Save-Credentials -Token $response.token -LicenseKey $response.licenseKey -Email $Email
        }
        
        return $response
    }
    catch {
        return @{ success = $false; error = $_.Exception.Message }
    }
}

function Record-Usage {
    param(
        [Parameter(Mandatory)]
        [ValidateSet("migration", "backup", "server")]
        [string]$Type,
        
        [string]$Detail = ""
    )
    
    $creds = Get-SavedCredentials
    if (!$creds -or !$creds.Token) {
        return @{ success = $false; error = "No saved credentials" }
    }
    
    try {
        $body = @{
            token = $creds.Token
            type = $Type
            detail = $Detail
            machineId = Get-MachineId
            machineName = $env:COMPUTERNAME
        }
        
        $response = Invoke-RestMethod -Uri "$Script:API_BASE/auth.php?action=record-usage" `
            -Method POST -Body ($body | ConvertTo-Json) -ContentType "application/json" -TimeoutSec 30
        
        return $response
    }
    catch {
        return @{ success = $false; error = $_.Exception.Message }
    }
}

function Show-LoginDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "ABC&D Migration Tool - Sign In"
    $form.Size = New-Object System.Drawing.Size(420, 380)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(26, 26, 46)
    $form.ForeColor = [System.Drawing.Color]::White
    $form.Font = New-Object System.Drawing.Font("Segoe UI", 9)
    
    # Title
    $title = New-Object System.Windows.Forms.Label
    $title.Text = "ABC&D Migration Tool"
    $title.Font = New-Object System.Drawing.Font("Segoe UI", 16, [System.Drawing.FontStyle]::Bold)
    $title.ForeColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
    $title.Location = New-Object System.Drawing.Point(20, 15)
    $title.AutoSize = $true
    $form.Controls.Add($title)
    
    $subtitle = New-Object System.Windows.Forms.Label
    $subtitle.Text = "Sign in or enter your license key"
    $subtitle.ForeColor = [System.Drawing.Color]::Gray
    $subtitle.Location = New-Object System.Drawing.Point(20, 50)
    $subtitle.AutoSize = $true
    $form.Controls.Add($subtitle)
    
    # Email
    $lblEmail = New-Object System.Windows.Forms.Label
    $lblEmail.Text = "Email:"
    $lblEmail.Location = New-Object System.Drawing.Point(20, 85)
    $lblEmail.AutoSize = $true
    $form.Controls.Add($lblEmail)
    
    $txtEmail = New-Object System.Windows.Forms.TextBox
    $txtEmail.Location = New-Object System.Drawing.Point(20, 105)
    $txtEmail.Size = New-Object System.Drawing.Size(360, 28)
    $txtEmail.BackColor = [System.Drawing.Color]::FromArgb(15, 15, 25)
    $txtEmail.ForeColor = [System.Drawing.Color]::White
    $txtEmail.BorderStyle = "FixedSingle"
    $form.Controls.Add($txtEmail)
    
    # Password
    $lblPass = New-Object System.Windows.Forms.Label
    $lblPass.Text = "Password:"
    $lblPass.Location = New-Object System.Drawing.Point(20, 140)
    $lblPass.AutoSize = $true
    $form.Controls.Add($lblPass)
    
    $txtPass = New-Object System.Windows.Forms.TextBox
    $txtPass.Location = New-Object System.Drawing.Point(20, 160)
    $txtPass.Size = New-Object System.Drawing.Size(360, 28)
    $txtPass.BackColor = [System.Drawing.Color]::FromArgb(15, 15, 25)
    $txtPass.ForeColor = [System.Drawing.Color]::White
    $txtPass.BorderStyle = "FixedSingle"
    $txtPass.UseSystemPasswordChar = $true
    $form.Controls.Add($txtPass)
    
    # Separator
    $lblOr = New-Object System.Windows.Forms.Label
    $lblOr.Text = "---------- OR enter License Key ----------"
    $lblOr.Location = New-Object System.Drawing.Point(60, 200)
    $lblOr.ForeColor = [System.Drawing.Color]::DimGray
    $lblOr.AutoSize = $true
    $form.Controls.Add($lblOr)
    
    # License Key
    $txtLicense = New-Object System.Windows.Forms.TextBox
    $txtLicense.Location = New-Object System.Drawing.Point(20, 230)
    $txtLicense.Size = New-Object System.Drawing.Size(360, 28)
    $txtLicense.BackColor = [System.Drawing.Color]::FromArgb(15, 15, 25)
    $txtLicense.ForeColor = [System.Drawing.Color]::White
    $txtLicense.BorderStyle = "FixedSingle"
    $txtLicense.Text = "MIG-"
    $txtLicense.Font = New-Object System.Drawing.Font("Consolas", 10)
    $form.Controls.Add($txtLicense)
    
    # Status label
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Location = New-Object System.Drawing.Point(20, 265)
    $lblStatus.Size = New-Object System.Drawing.Size(360, 20)
    $lblStatus.ForeColor = [System.Drawing.Color]::Orange
    $form.Controls.Add($lblStatus)
    
    # Buttons
    $btnLogin = New-Object System.Windows.Forms.Button
    $btnLogin.Text = "Sign In"
    $btnLogin.Location = New-Object System.Drawing.Point(20, 295)
    $btnLogin.Size = New-Object System.Drawing.Size(170, 40)
    $btnLogin.BackColor = [System.Drawing.Color]::FromArgb(212, 175, 55)
    $btnLogin.ForeColor = [System.Drawing.Color]::Black
    $btnLogin.FlatStyle = "Flat"
    $btnLogin.Font = New-Object System.Drawing.Font("Segoe UI", 10, [System.Drawing.FontStyle]::Bold)
    $form.Controls.Add($btnLogin)
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Exit"
    $btnCancel.Location = New-Object System.Drawing.Point(210, 295)
    $btnCancel.Size = New-Object System.Drawing.Size(170, 40)
    $btnCancel.BackColor = [System.Drawing.Color]::FromArgb(60, 60, 70)
    $btnCancel.FlatStyle = "Flat"
    $form.Controls.Add($btnCancel)
    
    $result = @{ Success = $false }
    
    $btnLogin.Add_Click({
        $lblStatus.Text = "Validating..."
        $form.Refresh()
        
        # Check license key first
        if ($txtLicense.Text -and $txtLicense.Text -ne "MIG-" -and $txtLicense.Text.Length -gt 15) {
            $res = Test-License -LicenseKey $txtLicense.Text.Trim()
            if ($res.valid) {
                Save-Credentials -LicenseKey $txtLicense.Text.Trim() -Token "" -Email ""
                $result.Success = $true
                $result.License = $res.license
                $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
                $form.Close()
            } else {
                $lblStatus.Text = $(if ($res.error) { $res.error } elseif ($res.reason) { $res.reason } else { "Invalid license key" })
                $lblStatus.ForeColor = [System.Drawing.Color]::Red
            }
        }
        # Check email/password
        elseif ($txtEmail.Text -and $txtPass.Text) {
            $res = Invoke-Login -Email $txtEmail.Text.Trim() -Password $txtPass.Text
            if ($res.success) {
                $result.Success = $true
                $result.License = $res.license
                $result.Token = $res.token
                $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
                $form.Close()
            } else {
                $lblStatus.Text = $(if ($res.error) { $res.error } else { "Login failed" })
                $lblStatus.ForeColor = [System.Drawing.Color]::Red
            }
        }
        else {
            $lblStatus.Text = "Enter email/password OR license key"
            $lblStatus.ForeColor = [System.Drawing.Color]::Orange
        }
    })
    
    $btnCancel.Add_Click({
        $form.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
        $form.Close()
    })
    
    # Enter key support
    $txtPass.Add_KeyDown({
        if ($_.KeyCode -eq "Enter") { $btnLogin.PerformClick() }
    })
    $txtLicense.Add_KeyDown({
        if ($_.KeyCode -eq "Enter") { $btnLogin.PerformClick() }
    })
    
    $form.ShowDialog() | Out-Null
    return $result
}
