#Requires -RunAsAdministrator
<#
.SYNOPSIS
    Logging Module for ABCD Migration Tool
.DESCRIPTION
    Provides comprehensive logging with automatic cleanup of logs older than 7 days
#>

$Script:LogFile = $null
$Script:LogPath = $null

function Initialize-Logging {
    param(
        [Parameter(Mandatory=$true)]
        [string]$LogPath,
        
        [int]$RetentionDays = 7
    )
    
    $Script:LogPath = $LogPath
    
    if (!(Test-Path $LogPath)) {
        New-Item -Path $LogPath -ItemType Directory -Force | Out-Null
    }
    
    # Clean up old logs (older than RetentionDays)
    try {
        $cutoffDate = (Get-Date).AddDays(-$RetentionDays)
        Get-ChildItem -Path $LogPath -Filter "*.log" -ErrorAction SilentlyContinue | 
            Where-Object { $_.LastWriteTime -lt $cutoffDate } | 
            ForEach-Object {
                Remove-Item $_.FullName -Force -ErrorAction SilentlyContinue
            }
    }
    catch {
        # Ignore cleanup errors
    }
    
    $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
    $Script:LogFile = Join-Path $LogPath "ABCD-Migration_$timestamp.log"
    
    $header = "========================================`r`n"
    $header += "ABCD-Migration v2.2 Migration Log`r`n"
    $header += "========================================`r`n"
    $header += "Date: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')`r`n"
    $header += "Computer: $env:COMPUTERNAME`r`n"
    $header += "User: $env:USERNAME`r`n"
    $header += "PowerShell: $($PSVersionTable.PSVersion)`r`n"
    $header += "OS: $((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).Caption)`r`n"
    $header += "Log Retention: $RetentionDays days`r`n"
    $header += "========================================`r`n"
    
    $header | Out-File -FilePath $Script:LogFile -Encoding UTF8
    
    return $Script:LogFile
}

function Write-LogMessage {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Message,
        
        [ValidateSet('Info', 'Success', 'Warning', 'Error', 'Debug')]
        [string]$Level = 'Info'
    )
    
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "[$timestamp] [$Level] $Message"
    
    if ($Script:LogFile) {
        try {
            $logEntry | Out-File -FilePath $Script:LogFile -Append -Encoding UTF8
        }
        catch {
            # If log file write fails, output to console
            Write-Host "LOG WRITE FAILED: $logEntry" -ForegroundColor Magenta
        }
    }
    
    switch ($Level) {
        'Success' { Write-Host $Message -ForegroundColor Green }
        'Warning' { Write-Host $Message -ForegroundColor Yellow }
        'Error' { Write-Host $Message -ForegroundColor Red }
        'Debug' { Write-Host "[DEBUG] $Message" -ForegroundColor Cyan }
        default { Write-Host $Message -ForegroundColor White }
    }
}

function Write-LogSection {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Title
    )
    
    $separator = "=" * 50
    Write-LogMessage $separator -Level Info
    Write-LogMessage $Title -Level Info
    Write-LogMessage $separator -Level Info
}

function Write-LogError {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Message,
        
        [System.Management.Automation.ErrorRecord]$ErrorRecord
    )
    
    Write-LogMessage "ERROR: $Message" -Level Error
    
    if ($ErrorRecord) {
        Write-LogMessage "  Exception: $($ErrorRecord.Exception.Message)" -Level Error
        Write-LogMessage "  Line: $($ErrorRecord.InvocationInfo.ScriptLineNumber)" -Level Error
        Write-LogMessage "  Command: $($ErrorRecord.InvocationInfo.Line.Trim())" -Level Error
    }
}

function Get-LogFilePath {
    return $Script:LogFile
}
