#Requires -RunAsAdministrator

<#
.SYNOPSIS
    Printer Migration Module using printbrm.exe
    
.DESCRIPTION
    Uses Microsoft's Print BRM (Backup Recovery Migration) tool to properly
    export and import printers WITH their drivers.
    
    This is the ONLY reliable way to migrate printer drivers between servers.
    Simple file copy does NOT work because drivers need to be registered.
    
.NOTES
    printbrm.exe is built into Windows Server and Windows 10/11 Pro+
#>

# Built-in Windows printers to skip
$Script:BuiltInPrinters = @(
    'Microsoft XPS Document Writer'
    'Microsoft Print to PDF'
    'Fax'
    'OneNote'
    'OneNote (Desktop)'
    'OneNote for Windows 10'
    'Send To OneNote 2016'
)

function Export-PrinterConfiguration {
    param(
        [Parameter(Mandatory=$false)]
        [string]$BackupPath = "C:\ABCD-Migration\Backup"
    )
    
    Write-Host "  [Printers] ================================================" -ForegroundColor Cyan
    Write-Host "  [Printers] Exporting Printers WITH DRIVERS (printbrm)" -ForegroundColor Cyan
    Write-Host "  [Printers] ================================================" -ForegroundColor Cyan
    
    # Ensure backup path exists
    if (!(Test-Path $BackupPath)) {
        New-Item -Path $BackupPath -ItemType Directory -Force | Out-Null
    }
    
    $exportFile = Join-Path $BackupPath "printers.printerExport"
    
    # Check if printbrm exists
    $printbrmPath = "$env:SystemRoot\System32\spool\tools\printbrm.exe"
    if (!(Test-Path $printbrmPath)) {
        # Try alternate location
        $printbrmPath = "$env:SystemRoot\System32\printbrm.exe"
    }
    
    if (!(Test-Path $printbrmPath)) {
        Write-Host "  [Printers] ERROR: printbrm.exe not found!" -ForegroundColor Red
        Write-Host "  [Printers] This tool is required for printer driver migration" -ForegroundColor Red
        Write-Host "  [Printers] Falling back to configuration-only export..." -ForegroundColor Yellow
        return Export-PrinterConfigurationLegacy -BackupPath $BackupPath
    }
    
    Write-Host "  [Printers] Using: $printbrmPath" -ForegroundColor Gray
    
    # List printers for reference
    Write-Host "  [Printers] Scanning installed printers..." -ForegroundColor Yellow
    $printers = Get-Printer -ErrorAction SilentlyContinue
    $realPrinters = $printers | Where-Object { 
        $name = $_.Name
        -not ($Script:BuiltInPrinters | Where-Object { $name -like "*$_*" })
    }
    
    Write-Host "  [Printers] Found $($realPrinters.Count) non-built-in printers:" -ForegroundColor Green
    foreach ($printer in $realPrinters) {
        Write-Host "  [Printers]   - $($printer.Name) [$($printer.DriverName)]" -ForegroundColor Gray
    }
    
    # Run printbrm backup
    Write-Host "  [Printers] Creating printer backup with drivers..." -ForegroundColor Yellow
    Write-Host "  [Printers] (This may take a moment)" -ForegroundColor Gray
    
    try {
        # Remove old export if exists
        if (Test-Path $exportFile) {
            Remove-Item $exportFile -Force
        }
        
        # printbrm -b = backup, -s = server (local), -f = file
        $arguments = "-b -s \\$env:COMPUTERNAME -f `"$exportFile`""
        
        $process = Start-Process -FilePath $printbrmPath -ArgumentList $arguments -Wait -PassThru -NoNewWindow -RedirectStandardOutput "$BackupPath\printbrm_output.log" -RedirectStandardError "$BackupPath\printbrm_error.log"
        
        if ($process.ExitCode -eq 0 -and (Test-Path $exportFile)) {
            $fileSize = (Get-Item $exportFile).Length / 1MB
            Write-Host "  [Printers] Backup created successfully!" -ForegroundColor Green
            Write-Host "  [Printers] File: printers.printerExport" -ForegroundColor Gray
            Write-Host "  [Printers] Size: $([math]::Round($fileSize, 2)) MB (includes drivers)" -ForegroundColor Gray
            
            # Also save printer list as JSON for reference
            $printerList = @{
                ExportDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
                SourceServer = $env:COMPUTERNAME
                Method = "printbrm"
                DriversIncluded = $true
                Printers = @()
            }
            
            foreach ($printer in $realPrinters) {
                $printerList.Printers += @{
                    Name = $printer.Name
                    DriverName = $printer.DriverName
                    PortName = $printer.PortName
                    Shared = $printer.Shared
                    ShareName = $printer.ShareName
                }
            }
            
            $printerList | ConvertTo-Json -Depth 3 | Out-File "$BackupPath\printers_reference.json" -Encoding UTF8
            
            return @{
                Success = $true
                ExportFile = $exportFile
                PrinterCount = $realPrinters.Count
            }
        }
        else {
            Write-Host "  [Printers] ERROR: printbrm backup failed (exit code: $($process.ExitCode))" -ForegroundColor Red
            
            # Show error log if exists
            if (Test-Path "$BackupPath\printbrm_error.log") {
                $errorLog = Get-Content "$BackupPath\printbrm_error.log" -Raw
                if ($errorLog) {
                    Write-Host "  [Printers] Error: $errorLog" -ForegroundColor Red
                }
            }
            
            Write-Host "  [Printers] Falling back to legacy export..." -ForegroundColor Yellow
            return Export-PrinterConfigurationLegacy -BackupPath $BackupPath
        }
    }
    catch {
        Write-Host "  [Printers] ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return @{ Success = $false; PrinterCount = 0 }
    }
}

function Import-PrinterConfiguration {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourcePath
    )
    
    Write-Host "  [Printers] ================================================" -ForegroundColor Cyan
    Write-Host "  [Printers] Importing Printers" -ForegroundColor Cyan
    Write-Host "  [Printers] ================================================" -ForegroundColor Cyan
    
    $importFile = Join-Path $SourcePath "printers.printerExport"
    $legacyConfig = Join-Path $SourcePath "printers_reference.json"
    
    # Check if printbrm export file exists
    if (!(Test-Path $importFile)) {
        Write-Host "  [Printers] No printbrm export found, checking for legacy..." -ForegroundColor Yellow
        
        # Try legacy import
        if (Test-Path $legacyConfig) {
            Write-Host "  [Printers] Found legacy export - using enhanced import" -ForegroundColor Yellow
            return Import-PrinterConfigurationLegacy -SourcePath $SourcePath
        }
        
        Write-Host "  [Printers] No printer configuration found to import" -ForegroundColor Yellow
        return @{ Success = $false; Imported = 0; Failed = 0 }
    }
    
    # Check if printbrm exists on this machine
    $printbrmPath = "$env:SystemRoot\System32\spool\tools\printbrm.exe"
    if (!(Test-Path $printbrmPath)) {
        $printbrmPath = "$env:SystemRoot\System32\printbrm.exe"
    }
    
    if (!(Test-Path $printbrmPath)) {
        Write-Host "  [Printers] printbrm.exe not found (Windows Home?)" -ForegroundColor Yellow
        Write-Host "  [Printers] Falling back to legacy import..." -ForegroundColor Yellow
        
        if (Test-Path $legacyConfig) {
            return Import-PrinterConfigurationLegacy -SourcePath $SourcePath
        } else {
            Write-Host "  [Printers] ERROR: Cannot import - printbrm not available and no legacy config" -ForegroundColor Red
            return @{ Success = $false; Imported = 0; Failed = 0 }
        }
    }
    
    Write-Host "  [Printers] Using printbrm: $printbrmPath" -ForegroundColor Gray
    Write-Host "  [Printers] Import file: printers.printerExport" -ForegroundColor Gray
    
    $fileSize = (Get-Item $importFile).Length / 1MB
    Write-Host "  [Printers] File size: $([math]::Round($fileSize, 2)) MB" -ForegroundColor Gray
    
    # Show what we're about to import
    $refFile = Join-Path $SourcePath "printers_reference.json"
    if (Test-Path $refFile) {
        $config = Get-Content $refFile -Raw | ConvertFrom-Json
        Write-Host "  [Printers] Source server: $($config.SourceServer)" -ForegroundColor Gray
        Write-Host "  [Printers] Printers to import:" -ForegroundColor Green
        foreach ($printer in $config.Printers) {
            Write-Host "  [Printers]   - $($printer.Name)" -ForegroundColor Gray
        }
    }
    
    # Run printbrm restore
    Write-Host "  [Printers] Restoring printers and installing drivers..." -ForegroundColor Yellow
    Write-Host "  [Printers] (This may take a few minutes)" -ForegroundColor Gray
    
    $tempLogPath = "$env:TEMP\printbrm_import"
    if (!(Test-Path $tempLogPath)) {
        New-Item -Path $tempLogPath -ItemType Directory -Force | Out-Null
    }
    
    try {
        # printbrm -r = restore, -s = server (local), -f = file, -o = overwrite
        $arguments = "-r -s \\$env:COMPUTERNAME -f `"$importFile`" -o"
        
        $process = Start-Process -FilePath $printbrmPath -ArgumentList $arguments -Wait -PassThru -NoNewWindow -RedirectStandardOutput "$tempLogPath\output.log" -RedirectStandardError "$tempLogPath\error.log"
        
        if ($process.ExitCode -eq 0) {
            Write-Host "  [Printers] Printer restore completed successfully!" -ForegroundColor Green
            
            # Give spooler time to register
            Start-Sleep -Seconds 2
            
            # List imported printers
            $printers = Get-Printer -ErrorAction SilentlyContinue
            $realPrinters = $printers | Where-Object { 
                $name = $_.Name
                -not ($Script:BuiltInPrinters | Where-Object { $name -like "*$_*" })
            }
            
            Write-Host "  [Printers] Printers now available:" -ForegroundColor Green
            foreach ($printer in $realPrinters) {
                Write-Host "  [Printers]   + $($printer.Name)" -ForegroundColor Green
            }
            
            return @{ 
                Success = $true
                Imported = $realPrinters.Count
                Failed = 0 
            }
        }
        else {
            Write-Host "  [Printers] WARNING: printbrm had issues (exit code: $($process.ExitCode))" -ForegroundColor Yellow
            
            # Show error log if exists
            if (Test-Path "$tempLogPath\error.log") {
                $errorLog = Get-Content "$tempLogPath\error.log" -Raw
                if ($errorLog) {
                    Write-Host "  [Printers] Details: $errorLog" -ForegroundColor Yellow
                }
            }
            
            Write-Host "  [Printers] Some printers may have been imported." -ForegroundColor Yellow
            Write-Host "  [Printers] Check Devices & Printers in Control Panel." -ForegroundColor Yellow
            
            return @{ 
                Success = $false
                Imported = 0
                Failed = 1 
            }
        }
    }
    catch {
        Write-Host "  [Printers] ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return @{ 
            Success = $false
            Imported = 0
            Failed = 1 
        }
    }
}

# Legacy export function (fallback if printbrm not available or fails)
function Export-PrinterConfigurationLegacy {
    param(
        [Parameter(Mandatory=$false)]
        [string]$BackupPath = "C:\ABCD-Migration\Backup"
    )
    
    Write-Host "  [Printers] Using legacy export (configuration + driver info)" -ForegroundColor Yellow
    Write-Host "  [Printers] NOTE: Windows Home detected - printbrm.exe not available" -ForegroundColor Yellow
    
    try {
        $printers = Get-Printer -ErrorAction SilentlyContinue | Where-Object { $_.Type -eq "Local" -or $_.Type -eq "Connection" }
        
        if (-not $printers) {
            Write-Host "  [Printers] No printers found to export" -ForegroundColor Gray
            return @{ Success = $true; PrinterCount = 0; DriversIncluded = $false }
        }
        
        # Create driver backup folder
        $driverBackupPath = Join-Path $BackupPath "PrinterDrivers"
        if (!(Test-Path $driverBackupPath)) {
            New-Item -Path $driverBackupPath -ItemType Directory -Force | Out-Null
        }
        
        $exportData = @{
            ExportDate = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
            SourceServer = $env:COMPUTERNAME
            Method = "legacy-enhanced"
            DriversIncluded = $false
            DriverFilesExported = $false
            Note = "Legacy export with driver info. Use pnputil to install drivers on destination."
            Printers = @()
            DriverSources = @()
        }
        
        $driversProcessed = @{}
        
        foreach ($printer in $printers) {
            # Skip built-in printers
            $isBuiltIn = $false
            foreach ($builtIn in $Script:BuiltInPrinters) {
                if ($printer.Name -like "*$builtIn*") {
                    $isBuiltIn = $true
                    break
                }
            }
            
            if ($isBuiltIn) { continue }
            
            # Get printer port info
            $port = Get-PrinterPort -Name $printer.PortName -ErrorAction SilentlyContinue
            
            $printerData = @{
                Name = $printer.Name
                DriverName = $printer.DriverName
                PortName = $printer.PortName
                PortType = if ($port) { $port.Description } else { "Unknown" }
                PortAddress = if ($port.PrinterHostAddress) { $port.PrinterHostAddress } else { $printer.PortName }
                Shared = $printer.Shared
                ShareName = $printer.ShareName
                Location = $printer.Location
                Comment = $printer.Comment
            }
            
            $exportData.Printers += $printerData
            Write-Host "  [Printers] Exported: $($printer.Name)" -ForegroundColor Yellow
            
            # Try to find and export driver INF files (only once per driver)
            if (-not $driversProcessed.ContainsKey($printer.DriverName)) {
                $driversProcessed[$printer.DriverName] = $true
                
                try {
                    # Get driver info
                    $driver = Get-PrinterDriver -Name $printer.DriverName -ErrorAction SilentlyContinue
                    
                    if ($driver) {
                        $driverInfo = @{
                            Name = $driver.Name
                            InfPath = $driver.InfPath
                            ConfigFile = $driver.ConfigFile
                            DriverPath = $driver.DriverPath
                            Manufacturer = $driver.Manufacturer
                        }
                        
                        # Try to copy INF file if it exists in DriverStore
                        if ($driver.InfPath -and (Test-Path $driver.InfPath)) {
                            $infFileName = Split-Path $driver.InfPath -Leaf
                            $infFolder = Split-Path $driver.InfPath -Parent
                            $destFolder = Join-Path $driverBackupPath $driver.Name.Replace(' ', '_').Replace('/', '_')
                            
                            if (!(Test-Path $destFolder)) {
                                New-Item -Path $destFolder -ItemType Directory -Force | Out-Null
                            }
                            
                            # Copy INF and related files
                            Copy-Item "$infFolder\*" $destFolder -Recurse -Force -ErrorAction SilentlyContinue
                            
                            $driverInfo.BackupPath = $destFolder
                            $exportData.DriverFilesExported = $true
                            Write-Host "  [Printers]   Driver files backed up: $($driver.Name)" -ForegroundColor Green
                        }
                        
                        $exportData.DriverSources += $driverInfo
                    }
                }
                catch {
                    Write-Host "  [Printers]   Could not backup driver: $($printer.DriverName)" -ForegroundColor Gray
                }
            }
        }
        
        # Save to JSON file
        $exportData | ConvertTo-Json -Depth 4 | Out-File "$BackupPath\printers_reference.json" -Encoding UTF8
        
        $printerCount = $exportData.Printers.Count
        Write-Host "  [Printers] Legacy export complete - $printerCount printers" -ForegroundColor Yellow
        
        if ($exportData.DriverFilesExported) {
            Write-Host "  [Printers] Driver INF files backed up to: $driverBackupPath" -ForegroundColor Green
            Write-Host "  [Printers] On destination, run: pnputil /add-driver <path>\*.inf /install" -ForegroundColor Cyan
        } else {
            Write-Host "  [Printers] WARNING: Could not backup driver files" -ForegroundColor Yellow
            Write-Host "  [Printers] You may need to install drivers manually" -ForegroundColor Yellow
        }
        
        return @{
            Success = $true
            PrinterCount = $printerCount
            DriversIncluded = $false
            DriverFilesExported = $exportData.DriverFilesExported
        }
    }
    catch {
        Write-Host "  [Printers] ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return @{ Success = $false; PrinterCount = 0 }
    }
}

function Import-PrinterConfigurationLegacy {
    param(
        [Parameter(Mandatory=$true)]
        [string]$SourcePath
    )
    
    Write-Host "  [Printers] Importing from legacy export..." -ForegroundColor Yellow
    
    $configFile = Join-Path $SourcePath "printers_reference.json"
    
    if (!(Test-Path $configFile)) {
        Write-Host "  [Printers] No printer configuration found" -ForegroundColor Yellow
        return @{ Success = $false; Imported = 0 }
    }
    
    try {
        $config = Get-Content $configFile -Raw | ConvertFrom-Json
        $imported = 0
        $failed = 0
        
        # First, try to install drivers if INF files are available
        $driverPath = Join-Path $SourcePath "PrinterDrivers"
        if (Test-Path $driverPath) {
            Write-Host "  [Printers] Installing drivers from backup..." -ForegroundColor Yellow
            
            Get-ChildItem $driverPath -Directory | ForEach-Object {
                $infFiles = Get-ChildItem $_.FullName -Filter "*.inf" -ErrorAction SilentlyContinue
                foreach ($inf in $infFiles) {
                    Write-Host "  [Printers]   Installing: $($inf.Name)" -ForegroundColor Gray
                    $result = Start-Process "pnputil.exe" -ArgumentList "/add-driver `"$($inf.FullName)`" /install" -Wait -PassThru -NoNewWindow
                    if ($result.ExitCode -eq 0) {
                        Write-Host "  [Printers]   Driver installed successfully" -ForegroundColor Green
                    }
                }
            }
        }
        
        # Now create printers
        foreach ($printer in $config.Printers) {
            try {
                # Check if printer already exists
                $existing = Get-Printer -Name $printer.Name -ErrorAction SilentlyContinue
                if ($existing) {
                    Write-Host "  [Printers] Printer already exists: $($printer.Name)" -ForegroundColor Gray
                    continue
                }
                
                # Create port if needed
                $portExists = Get-PrinterPort -Name $printer.PortName -ErrorAction SilentlyContinue
                if (-not $portExists) {
                    if ($printer.PortAddress -and $printer.PortAddress -match '^\d+\.\d+\.\d+\.\d+') {
                        # Network printer - create TCP/IP port
                        Add-PrinterPort -Name $printer.PortName -PrinterHostAddress $printer.PortAddress -ErrorAction SilentlyContinue
                        Write-Host "  [Printers]   Created port: $($printer.PortName)" -ForegroundColor Gray
                    }
                }
                
                # Check if driver is available
                $driverAvailable = Get-PrinterDriver -Name $printer.DriverName -ErrorAction SilentlyContinue
                
                if ($driverAvailable) {
                    # Create the printer
                    Add-Printer -Name $printer.Name -DriverName $printer.DriverName -PortName $printer.PortName -ErrorAction Stop
                    
                    # Set sharing if enabled
                    if ($printer.Shared -and $printer.ShareName) {
                        Set-Printer -Name $printer.Name -Shared $true -ShareName $printer.ShareName -ErrorAction SilentlyContinue
                    }
                    
                    Write-Host "  [Printers] Created: $($printer.Name)" -ForegroundColor Green
                    $imported++
                } else {
                    Write-Host "  [Printers] SKIPPED: $($printer.Name) - driver not found: $($printer.DriverName)" -ForegroundColor Yellow
                    $failed++
                }
            }
            catch {
                Write-Host "  [Printers] FAILED: $($printer.Name) - $($_.Exception.Message)" -ForegroundColor Red
                $failed++
            }
        }
        
        Write-Host "  [Printers] Import complete - $imported created, $failed failed" -ForegroundColor $(if ($failed -eq 0) { "Green" } else { "Yellow" })
        
        return @{ Success = $true; Imported = $imported; Failed = $failed }
    }
    catch {
        Write-Host "  [Printers] ERROR: $($_.Exception.Message)" -ForegroundColor Red
        return @{ Success = $false; Imported = 0 }
    }
}
