<#
.SYNOPSIS
    ABC&D Systems - QuickBooks Management Module
.DESCRIPTION
    Comprehensive QuickBooks management including:
    - Desktop updates with reboot handling
    - Payroll updates
    - User logoff for updates
    - Autologon for reboot continuation
    - Scheduled update tasks
.VERSION
    4.0
#>

# =============================================================================
# CONFIGURATION
# =============================================================================

$Script:QBConfig = @{
    ToolsPath = "$PSScriptRoot\..\Tools"
    LogPath = "C:\ABCD-Migration\Logs"
    StatePath = "C:\ABCD-Migration\Settings\QBUpdateState.json"
    
    # QuickBooks processes to stop
    Processes = @(
        "QBW32", "QBW", "QBDBMgrN", "QBDBMgr", 
        "QBUpdate", "qbupdate", "AutoUpdate",
        "QBLaunch", "QBCFMonitorService"
    )
    
    # QuickBooks services to stop
    Services = @(
        "QuickBooksDB*",
        "QBCFMonitorService",
        "QBIDPService",
        "IntuitUpdateService"
    )
    
    # Update executable locations
    UpdatePaths = @(
        "C:\Program Files\Intuit\QuickBooks*\QBUpdate.exe",
        "C:\Program Files (x86)\Intuit\QuickBooks*\QBUpdate.exe",
        "C:\Program Files\Common Files\Intuit\QuickBooks\QBUpdate\QBUpdate.exe"
    )
}

# =============================================================================
# AUTOLOGON MANAGEMENT
# =============================================================================

function Set-AutoLogon {
    <#
    .SYNOPSIS
        Configures Windows auto-logon using Autologon64.exe
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$Username,
        
        [Parameter(Mandatory=$true)]
        [string]$Password,
        
        [string]$Domain = $env:COMPUTERNAME
    )
    
    $autologonPath = Join-Path $Script:QBConfig.ToolsPath "Autologon64.exe"
    
    if (!(Test-Path $autologonPath)) {
        Write-Host "  [AutoLogon] ERROR: Autologon64.exe not found" -ForegroundColor Red
        return $false
    }
    
    try {
        # Use Autologon64.exe to set auto-login
        # Syntax: Autologon64.exe <user> <domain> <password> /accepteula
        $process = Start-Process -FilePath $autologonPath -ArgumentList "`"$Username`" `"$Domain`" `"$Password`" /accepteula" -Wait -PassThru -NoNewWindow
        
        if ($process.ExitCode -eq 0) {
            Write-Host "  [AutoLogon] Configured for: $Domain\$Username" -ForegroundColor Green
            return $true
        } else {
            Write-Host "  [AutoLogon] Failed with exit code: $($process.ExitCode)" -ForegroundColor Red
            return $false
        }
    } catch {
        Write-Host "  [AutoLogon] ERROR: $_" -ForegroundColor Red
        return $false
    }
}

function Clear-AutoLogon {
    <#
    .SYNOPSIS
        Removes auto-logon configuration
    #>
    
    $autologonPath = Join-Path $Script:QBConfig.ToolsPath "Autologon64.exe"
    
    if (!(Test-Path $autologonPath)) {
        # Fall back to registry method
        try {
            $regPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
            Set-ItemProperty -Path $regPath -Name "AutoAdminLogon" -Value "0" -ErrorAction SilentlyContinue
            Remove-ItemProperty -Path $regPath -Name "DefaultPassword" -ErrorAction SilentlyContinue
            Write-Host "  [AutoLogon] Cleared via registry" -ForegroundColor Green
            return $true
        } catch {
            return $false
        }
    }
    
    try {
        # Autologon64.exe with /d flag disables
        $process = Start-Process -FilePath $autologonPath -ArgumentList "/d" -Wait -PassThru -NoNewWindow
        Write-Host "  [AutoLogon] Cleared" -ForegroundColor Green
        return $true
    } catch {
        Write-Host "  [AutoLogon] Clear failed: $_" -ForegroundColor Red
        return $false
    }
}

# =============================================================================
# QUICKBOOKS PROCESS/SERVICE MANAGEMENT
# =============================================================================

function Stop-AllQuickBooks {
    <#
    .SYNOPSIS
        Stops all QuickBooks processes and services
    #>
    param([switch]$Force)
    
    Write-Host "  [QB] Stopping QuickBooks processes and services..." -ForegroundColor Yellow
    
    $stopped = @{
        Services = @()
        Processes = @()
    }
    
    # Stop services first
    foreach ($pattern in $Script:QBConfig.Services) {
        try {
            $services = Get-Service -Name $pattern -ErrorAction SilentlyContinue | Where-Object { $_.Status -eq 'Running' }
            foreach ($svc in $services) {
                Write-Host "  [QB] Stopping service: $($svc.DisplayName)" -ForegroundColor Yellow
                Stop-Service -Name $svc.Name -Force -ErrorAction SilentlyContinue
                $stopped.Services += $svc.Name
            }
        } catch {}
    }
    
    # Stop processes
    foreach ($procName in $Script:QBConfig.Processes) {
        try {
            $procs = Get-Process -Name $procName -ErrorAction SilentlyContinue
            foreach ($p in $procs) {
                Write-Host "  [QB] Stopping process: $($p.ProcessName)" -ForegroundColor Yellow
                Stop-Process -Id $p.Id -Force -ErrorAction SilentlyContinue
                $stopped.Processes += $p.ProcessName
            }
        } catch {}
    }
    
    # Wait for processes to fully stop
    Start-Sleep -Seconds 3
    
    Write-Host "  [QB] Stopped $($stopped.Services.Count) services, $($stopped.Processes.Count) processes" -ForegroundColor Green
    return $stopped
}

function Start-QuickBooksServices {
    <#
    .SYNOPSIS
        Restarts QuickBooks services
    #>
    
    Write-Host "  [QB] Starting QuickBooks services..." -ForegroundColor Yellow
    
    foreach ($pattern in $Script:QBConfig.Services) {
        try {
            $services = Get-Service -Name $pattern -ErrorAction SilentlyContinue
            foreach ($svc in $services) {
                if ($svc.StartType -ne 'Disabled') {
                    Start-Service -Name $svc.Name -ErrorAction SilentlyContinue
                    Write-Host "  [QB] Started: $($svc.DisplayName)" -ForegroundColor Green
                }
            }
        } catch {}
    }
}

# =============================================================================
# USER SESSION MANAGEMENT
# =============================================================================

function Invoke-LogoffAllUsers {
    <#
    .SYNOPSIS
        Logs off all users except current session
    #>
    param(
        [int]$WarningMinutes = 5,
        [string]$Message = "System maintenance requires all users to log off. Please save your work."
    )
    
    Write-Host "  [Sessions] Logging off all users..." -ForegroundColor Yellow
    
    $currentSessionId = [System.Diagnostics.Process]::GetCurrentProcess().SessionId
    
    # Send warning message
    if ($WarningMinutes -gt 0) {
        try {
            msg * /TIME:$($WarningMinutes * 60) "$Message System will proceed in $WarningMinutes minutes."
        } catch {}
        
        Write-Host "  [Sessions] Warning sent. Waiting $WarningMinutes minutes..." -ForegroundColor Yellow
        Start-Sleep -Seconds ($WarningMinutes * 60)
    }
    
    # Get all sessions
    $loggedOff = @()
    try {
        $sessions = query session 2>$null | Select-Object -Skip 1
        foreach ($line in $sessions) {
            if ($line -match '^\s*>?(\S+)\s+(\S+)\s+(\d+)\s+(Active|Disc)') {
                $sessionId = [int]$matches[3]
                $userName = $matches[2]
                
                if ($sessionId -ne $currentSessionId -and $sessionId -ne 0 -and $userName -ne "services") {
                    Write-Host "  [Sessions] Logging off: $userName (Session $sessionId)" -ForegroundColor Yellow
                    logoff $sessionId 2>$null
                    $loggedOff += $userName
                }
            }
        }
    } catch {
        Write-Host "  [Sessions] Error: $_" -ForegroundColor Red
    }
    
    Write-Host "  [Sessions] Logged off $($loggedOff.Count) user(s)" -ForegroundColor Green
    return $loggedOff
}

# =============================================================================
# QUICKBOOKS UPDATE FUNCTIONS
# =============================================================================

function Get-QuickBooksInstallations {
    <#
    .SYNOPSIS
        Finds all QuickBooks installations
    #>
    
    $installations = @()
    
    $searchPaths = @(
        "C:\Program Files\Intuit",
        "C:\Program Files (x86)\Intuit"
    )
    
    foreach ($path in $searchPaths) {
        if (Test-Path $path) {
            $qbFolders = Get-ChildItem -Path $path -Directory -Filter "QuickBooks*" -ErrorAction SilentlyContinue
            foreach ($folder in $qbFolders) {
                $qbwPath = Join-Path $folder.FullName "QBW32.exe"
                $qbwPathAlt = Join-Path $folder.FullName "QBW.exe"
                
                if ((Test-Path $qbwPath) -or (Test-Path $qbwPathAlt)) {
                    $exePath = if (Test-Path $qbwPath) { $qbwPath } else { $qbwPathAlt }
                    $version = (Get-Item $exePath).VersionInfo.ProductVersion
                    
                    $installations += @{
                        Name = $folder.Name
                        Path = $folder.FullName
                        Executable = $exePath
                        Version = $version
                        UpdateExe = Join-Path $folder.FullName "QBUpdate.exe"
                    }
                }
            }
        }
    }
    
    return $installations
}

function Get-QuickBooksUpdatePath {
    <#
    .SYNOPSIS
        Finds QuickBooks update executable
    #>
    
    foreach ($pattern in $Script:QBConfig.UpdatePaths) {
        $paths = Get-Item -Path $pattern -ErrorAction SilentlyContinue
        if ($paths) {
            return $paths[0].FullName
        }
    }
    
    # Check each installation
    $installations = Get-QuickBooksInstallations
    foreach ($inst in $installations) {
        if (Test-Path $inst.UpdateExe) {
            return $inst.UpdateExe
        }
    }
    
    return $null
}

function Invoke-QuickBooksUpdate {
    <#
    .SYNOPSIS
        Runs QuickBooks update and handles reboot if needed
    #>
    param(
        [switch]$Silent,
        [switch]$AutoReboot,
        [string]$AdminUsername,
        [string]$AdminPassword
    )
    
    $updateExe = Get-QuickBooksUpdatePath
    
    if (!$updateExe) {
        Write-Host "  [QB Update] QuickBooks update utility not found" -ForegroundColor Red
        return @{ Success = $false; RebootRequired = $false; Error = "Update utility not found" }
    }
    
    Write-Host "  [QB Update] Starting QuickBooks update..." -ForegroundColor Cyan
    Write-Host "  [QB Update] Using: $updateExe" -ForegroundColor Gray
    
    # Stop QB first
    Stop-AllQuickBooks
    
    try {
        $args = if ($Silent) { "/silent" } else { "" }
        $process = Start-Process -FilePath $updateExe -ArgumentList $args -Wait -PassThru
        
        $exitCode = $process.ExitCode
        Write-Host "  [QB Update] Exit code: $exitCode" -ForegroundColor Gray
        
        # Check if reboot is required
        $rebootRequired = Test-RebootRequired
        
        if ($rebootRequired) {
            Write-Host "  [QB Update] Reboot required to complete update" -ForegroundColor Yellow
            
            if ($AutoReboot -and $AdminUsername -and $AdminPassword) {
                # Save state for continuation
                Save-UpdateState -Phase "PostReboot" -Task "QBDesktop"
                
                # Set autologon
                Set-AutoLogon -Username $AdminUsername -Password $AdminPassword
                
                # Create continuation task
                Register-UpdateContinuationTask
                
                Write-Host "  [QB Update] Rebooting in 30 seconds..." -ForegroundColor Yellow
                shutdown /r /t 30 /c "QuickBooks update requires restart"
                
                return @{ Success = $true; RebootRequired = $true; Rebooting = $true }
            }
            
            return @{ Success = $true; RebootRequired = $true; Rebooting = $false }
        }
        
        Write-Host "  [QB Update] Update completed successfully" -ForegroundColor Green
        return @{ Success = $true; RebootRequired = $false }
        
    } catch {
        Write-Host "  [QB Update] ERROR: $_" -ForegroundColor Red
        return @{ Success = $false; Error = $_.Exception.Message }
    }
}

function Invoke-PayrollUpdate {
    <#
    .SYNOPSIS
        Runs QuickBooks Payroll update
    #>
    param([switch]$Silent)
    
    Write-Host "  [Payroll] Starting Payroll update..." -ForegroundColor Cyan
    
    # Payroll updates typically require QB to be running
    $installations = Get-QuickBooksInstallations
    
    if ($installations.Count -eq 0) {
        Write-Host "  [Payroll] No QuickBooks installations found" -ForegroundColor Red
        return @{ Success = $false; Error = "QuickBooks not found" }
    }
    
    # Use the first installation
    $qbExe = $installations[0].Executable
    
    try {
        # Launch QB with payroll update switch
        Write-Host "  [Payroll] Launching QuickBooks for payroll update..." -ForegroundColor Yellow
        
        # Note: Actual payroll update command varies by version
        # This launches QB which should prompt for payroll updates
        $process = Start-Process -FilePath $qbExe -PassThru
        
        Write-Host "  [Payroll] QuickBooks launched. Payroll update will be prompted." -ForegroundColor Green
        Write-Host "  [Payroll] Note: Manual intervention may be required for payroll updates." -ForegroundColor Yellow
        
        return @{ Success = $true; ManualRequired = $true }
        
    } catch {
        Write-Host "  [Payroll] ERROR: $_" -ForegroundColor Red
        return @{ Success = $false; Error = $_.Exception.Message }
    }
}

# =============================================================================
# REBOOT HANDLING
# =============================================================================

function Test-RebootRequired {
    <#
    .SYNOPSIS
        Checks if system reboot is required
    #>
    
    $rebootKeys = @(
        "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\RebootRequired",
        "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\RebootPending",
        "HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager\PendingFileRenameOperations"
    )
    
    foreach ($key in $rebootKeys) {
        if (Test-Path $key) {
            return $true
        }
    }
    
    # Check for pending file rename operations
    try {
        $pendingOps = Get-ItemProperty "HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager" -Name PendingFileRenameOperations -ErrorAction SilentlyContinue
        if ($pendingOps.PendingFileRenameOperations) {
            return $true
        }
    } catch {}
    
    return $false
}

function Save-UpdateState {
    <#
    .SYNOPSIS
        Saves update state for continuation after reboot
    #>
    param(
        [string]$Phase,
        [string]$Task,
        [hashtable]$AdditionalData = @{}
    )
    
    $state = @{
        Phase = $Phase
        Task = $Task
        SavedAt = (Get-Date).ToString("o")
        Computer = $env:COMPUTERNAME
        Data = $AdditionalData
    }
    
    $stateDir = Split-Path $Script:QBConfig.StatePath -Parent
    if (!(Test-Path $stateDir)) {
        New-Item -Path $stateDir -ItemType Directory -Force | Out-Null
    }
    
    $state | ConvertTo-Json -Depth 5 | Out-File $Script:QBConfig.StatePath -Encoding UTF8
    Write-Host "  [State] Saved: Phase=$Phase, Task=$Task" -ForegroundColor Gray
}

function Get-UpdateState {
    if (Test-Path $Script:QBConfig.StatePath) {
        try {
            return Get-Content $Script:QBConfig.StatePath -Raw | ConvertFrom-Json
        } catch {
            return $null
        }
    }
    return $null
}

function Clear-UpdateState {
    if (Test-Path $Script:QBConfig.StatePath) {
        Remove-Item $Script:QBConfig.StatePath -Force -ErrorAction SilentlyContinue
    }
}

function Register-UpdateContinuationTask {
    <#
    .SYNOPSIS
        Creates a scheduled task to continue updates after reboot
    #>
    
    $taskName = "ABCD-UpdateContinuation"
    
    # Create continuation script
    $scriptPath = "C:\ABCD-Migration\Scripts\UpdateContinuation.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    if (!(Test-Path $scriptDir)) {
        New-Item -Path $scriptDir -ItemType Directory -Force | Out-Null
    }
    
    $scriptContent = @'
# ABCD Update Continuation Script
$ErrorActionPreference = 'Continue'
$logFile = "C:\ABCD-Migration\Logs\UpdateContinuation_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
Start-Transcript -Path $logFile

# Wait for system to fully start
Start-Sleep -Seconds 60

# Import module
$modulePath = "C:\ABCD-Migration\Modules\QuickBooks-Management.ps1"
if (Test-Path $modulePath) {
    . $modulePath
}

# Get saved state
$state = Get-UpdateState

if ($state) {
    Write-Host "Continuing update: Phase=$($state.Phase), Task=$($state.Task)"
    
    # Clear autologon immediately
    Clear-AutoLogon
    
    switch ($state.Phase) {
        "PostReboot" {
            # Continue with payroll update
            Write-Host "Running payroll update..."
            Invoke-PayrollUpdate
            
            # Restart QB services
            Start-QuickBooksServices
        }
    }
    
    # Clear state
    Clear-UpdateState
}

# Remove this task
Unregister-ScheduledTask -TaskName "ABCD-UpdateContinuation" -Confirm:$false -ErrorAction SilentlyContinue

Stop-Transcript
'@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8
    
    # Remove existing task
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    # Create task to run at logon
    $trigger = New-ScheduledTaskTrigger -AtLogOn
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries
    
    Register-ScheduledTask -TaskName $taskName -Trigger $trigger -Action $action -Principal $principal -Settings $settings -Description "ABCD Update Continuation"
    
    Write-Host "  [Task] Created continuation task: $taskName" -ForegroundColor Green
}

# =============================================================================
# FULL UPDATE WORKFLOW
# =============================================================================

function Invoke-FullQuickBooksUpdate {
    <#
    .SYNOPSIS
        Complete QuickBooks update workflow:
        1. Log off users
        2. Stop QB
        3. Run desktop update
        4. Reboot if needed (with autologon)
        5. Run payroll update
        6. Restart services
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$AdminUsername,
        
        [Parameter(Mandatory=$true)]
        [string]$AdminPassword,
        
        [int]$UserWarningMinutes = 5,
        [switch]$Force
    )
    
    $logFile = Join-Path $Script:QBConfig.LogPath "QBFullUpdate_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
    Start-Transcript -Path $logFile
    
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host "ABCD Systems - QuickBooks Full Update" -ForegroundColor Cyan
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host ""
    
    try {
        # Step 1: Log off users
        Write-Host "[Step 1] Logging off users..." -ForegroundColor Yellow
        if (-not $Force) {
            Invoke-LogoffAllUsers -WarningMinutes $UserWarningMinutes -Message "QuickBooks updates require all users to log off."
        } else {
            Invoke-LogoffAllUsers -WarningMinutes 0
        }
        
        # Step 2: Stop QuickBooks
        Write-Host "[Step 2] Stopping QuickBooks..." -ForegroundColor Yellow
        Stop-AllQuickBooks -Force
        
        # Step 3: Run QuickBooks Desktop Update
        Write-Host "[Step 3] Running QuickBooks Desktop Update..." -ForegroundColor Yellow
        $updateResult = Invoke-QuickBooksUpdate -Silent -AutoReboot -AdminUsername $AdminUsername -AdminPassword $AdminPassword
        
        if ($updateResult.Rebooting) {
            Write-Host "[Update] System is rebooting. Update will continue after restart." -ForegroundColor Yellow
            Stop-Transcript
            return @{ Success = $true; Status = "Rebooting" }
        }
        
        # Step 4: Run Payroll Update (if no reboot needed)
        Write-Host "[Step 4] Running Payroll Update..." -ForegroundColor Yellow
        $payrollResult = Invoke-PayrollUpdate
        
        # Step 5: Restart services
        Write-Host "[Step 5] Restarting QuickBooks services..." -ForegroundColor Yellow
        Start-QuickBooksServices
        
        Write-Host "" -ForegroundColor Cyan
        Write-Host "========================================" -ForegroundColor Green
        Write-Host "QuickBooks Update Complete!" -ForegroundColor Green
        Write-Host "========================================" -ForegroundColor Green
        
        Stop-Transcript
        return @{ Success = $true; Status = "Complete" }
        
    } catch {
        Write-Host "ERROR: $_" -ForegroundColor Red
        Stop-Transcript
        return @{ Success = $false; Error = $_.Exception.Message }
    }
}

# =============================================================================
# SCHEDULED TASK FOR QB UPDATES
# =============================================================================

function Register-QuickBooksUpdateTask {
    <#
    .SYNOPSIS
        Creates scheduled task for automated QB updates
    #>
    param(
        [string]$Time = "02:00",
        [string]$AdminUsername,
        [string]$AdminPassword,
        [ValidateSet("Daily", "Weekly", "Monthly")]
        [string]$Frequency = "Weekly",
        [int]$DayOfWeek = 0  # 0 = Sunday
    )
    
    $taskName = "ABCD-QuickBooksUpdate"
    
    # Create the update script
    $scriptPath = "C:\ABCD-Migration\Scripts\QBScheduledUpdate.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    if (!(Test-Path $scriptDir)) {
        New-Item -Path $scriptDir -ItemType Directory -Force | Out-Null
    }
    
    # Encrypt password for storage (DPAPI)
    $encPassword = $AdminPassword | ConvertTo-SecureString -AsPlainText -Force | ConvertFrom-SecureString
    
    $scriptContent = @"
# ABCD QuickBooks Scheduled Update
`$ErrorActionPreference = 'Continue'
`$logFile = "C:\ABCD-Migration\Logs\QBScheduled_`$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
Start-Transcript -Path `$logFile

# Import modules
. "C:\ABCD-Migration\Modules\QuickBooks-Management.ps1"
. "C:\ABCD-Migration\Modules\Service-Management.ps1"

# Decrypt password
`$encPassword = '$encPassword'
`$securePass = `$encPassword | ConvertTo-SecureString
`$cred = New-Object System.Management.Automation.PSCredential('$AdminUsername', `$securePass)
`$plainPassword = `$cred.GetNetworkCredential().Password

# Step 1: Stop ALL services (aggressive - like backup mode)
Write-Host "[QB Scheduled] Stopping all non-essential services..." -ForegroundColor Yellow
`$stoppedServices = Stop-AllServicesForBackup
Write-Host "[QB Scheduled] Stopped `$(`$stoppedServices.Services.Count) services" -ForegroundColor Green

# Step 2: Run full QuickBooks update
Write-Host "[QB Scheduled] Running QuickBooks updates..." -ForegroundColor Cyan
Invoke-FullQuickBooksUpdate -AdminUsername '$AdminUsername' -AdminPassword `$plainPassword -UserWarningMinutes 0 -Force

# Step 3: Restart all stopped services
Write-Host "[QB Scheduled] Restarting services..." -ForegroundColor Yellow
Start-ServicesAfterBackup -StoppedServices `$stoppedServices.Services
Write-Host "[QB Scheduled] Services restarted" -ForegroundColor Green

Stop-Transcript
"@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8
    
    # Remove existing task
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    # Create trigger based on frequency
    $timeParts = $Time.Split(':')
    $triggerTime = "$($timeParts[0]):$($timeParts[1])"
    
    switch ($Frequency) {
        "Daily" {
            $trigger = New-ScheduledTaskTrigger -Daily -At $triggerTime
        }
        "Weekly" {
            $daysOfWeek = @("Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday")
            $trigger = New-ScheduledTaskTrigger -Weekly -DaysOfWeek $daysOfWeek[$DayOfWeek] -At $triggerTime
        }
        "Monthly" {
            $trigger = New-ScheduledTaskTrigger -Weekly -DaysOfWeek Sunday -WeeksInterval 4 -At $triggerTime
        }
    }
    
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -WakeToRun -ExecutionTimeLimit (New-TimeSpan -Hours 4)
    
    Register-ScheduledTask -TaskName $taskName -Trigger $trigger -Action $action -Principal $principal -Settings $settings -Description "ABCD Systems - QuickBooks Automatic Updates"
    
    Write-Host "  [Task] Created: $taskName ($Frequency at $Time)" -ForegroundColor Green
    return $true
}

# =============================================================================
# GUI DIALOG
# =============================================================================

function Show-QuickBooksSettingsDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "QuickBooks Management - ABCD Systems"
    $form.Size = New-Object System.Drawing.Size(500, 450)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    
    # =========================================================================
    # INSTALLATIONS INFO
    # =========================================================================
    
    $grpInstall = New-Object System.Windows.Forms.GroupBox
    $grpInstall.Text = "QuickBooks Installations"
    $grpInstall.Location = New-Object System.Drawing.Point(10, 10)
    $grpInstall.Size = New-Object System.Drawing.Size(460, 80)
    $form.Controls.Add($grpInstall)
    
    $lstInstall = New-Object System.Windows.Forms.ListBox
    $lstInstall.Location = New-Object System.Drawing.Point(10, 20)
    $lstInstall.Size = New-Object System.Drawing.Size(440, 50)
    $grpInstall.Controls.Add($lstInstall)
    
    $installations = Get-QuickBooksInstallations
    if ($installations.Count -eq 0) {
        $lstInstall.Items.Add("No QuickBooks installations found")
    } else {
        foreach ($inst in $installations) {
            $lstInstall.Items.Add("$($inst.Name) - v$($inst.Version)")
        }
    }
    
    # =========================================================================
    # MANUAL UPDATE SECTION
    # =========================================================================
    
    $grpManual = New-Object System.Windows.Forms.GroupBox
    $grpManual.Text = "Manual Update"
    $grpManual.Location = New-Object System.Drawing.Point(10, 100)
    $grpManual.Size = New-Object System.Drawing.Size(460, 100)
    $form.Controls.Add($grpManual)
    
    $lblManualInfo = New-Object System.Windows.Forms.Label
    $lblManualInfo.Text = "Run updates now. This will log off all users and may require a reboot."
    $lblManualInfo.Location = New-Object System.Drawing.Point(10, 22)
    $lblManualInfo.Size = New-Object System.Drawing.Size(440, 20)
    $grpManual.Controls.Add($lblManualInfo)
    
    $btnRunUpdate = New-Object System.Windows.Forms.Button
    $btnRunUpdate.Text = "Run Full Update Now"
    $btnRunUpdate.Location = New-Object System.Drawing.Point(10, 50)
    $btnRunUpdate.Size = New-Object System.Drawing.Size(140, 30)
    $btnRunUpdate.BackColor = [System.Drawing.Color]::FromArgb(46, 139, 87)
    $btnRunUpdate.ForeColor = [System.Drawing.Color]::Black
    $btnRunUpdate.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpManual.Controls.Add($btnRunUpdate)
    
    $btnPayrollOnly = New-Object System.Windows.Forms.Button
    $btnPayrollOnly.Text = "Payroll Update Only"
    $btnPayrollOnly.Location = New-Object System.Drawing.Point(160, 50)
    $btnPayrollOnly.Size = New-Object System.Drawing.Size(120, 30)
    $btnPayrollOnly.BackColor = [System.Drawing.Color]::SteelBlue
    $btnPayrollOnly.ForeColor = [System.Drawing.Color]::Black
    $btnPayrollOnly.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpManual.Controls.Add($btnPayrollOnly)
    
    $btnStopQB = New-Object System.Windows.Forms.Button
    $btnStopQB.Text = "Stop All QB"
    $btnStopQB.Location = New-Object System.Drawing.Point(290, 50)
    $btnStopQB.Size = New-Object System.Drawing.Size(80, 30)
    $btnStopQB.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpManual.Controls.Add($btnStopQB)
    
    $btnStartQB = New-Object System.Windows.Forms.Button
    $btnStartQB.Text = "Start QB"
    $btnStartQB.Location = New-Object System.Drawing.Point(375, 50)
    $btnStartQB.Size = New-Object System.Drawing.Size(75, 30)
    $btnStartQB.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpManual.Controls.Add($btnStartQB)
    
    # =========================================================================
    # SCHEDULED UPDATE SECTION
    # =========================================================================
    
    $grpSchedule = New-Object System.Windows.Forms.GroupBox
    $grpSchedule.Text = "Scheduled Updates"
    $grpSchedule.Location = New-Object System.Drawing.Point(10, 210)
    $grpSchedule.Size = New-Object System.Drawing.Size(460, 150)
    $form.Controls.Add($grpSchedule)
    
    $chkEnableSchedule = New-Object System.Windows.Forms.CheckBox
    $chkEnableSchedule.Text = "Enable scheduled QuickBooks updates"
    $chkEnableSchedule.Location = New-Object System.Drawing.Point(10, 22)
    $chkEnableSchedule.Size = New-Object System.Drawing.Size(250, 20)
    $grpSchedule.Controls.Add($chkEnableSchedule)
    
    $lblFreq = New-Object System.Windows.Forms.Label
    $lblFreq.Text = "Frequency:"
    $lblFreq.Location = New-Object System.Drawing.Point(10, 50)
    $lblFreq.Size = New-Object System.Drawing.Size(70, 20)
    $grpSchedule.Controls.Add($lblFreq)
    
    $cboFrequency = New-Object System.Windows.Forms.ComboBox
    $cboFrequency.Location = New-Object System.Drawing.Point(80, 47)
    $cboFrequency.Size = New-Object System.Drawing.Size(100, 22)
    $cboFrequency.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    $cboFrequency.Items.AddRange(@("Weekly", "Daily", "Monthly"))
    $cboFrequency.SelectedIndex = 0
    $grpSchedule.Controls.Add($cboFrequency)
    
    $lblTime = New-Object System.Windows.Forms.Label
    $lblTime.Text = "Time:"
    $lblTime.Location = New-Object System.Drawing.Point(200, 50)
    $lblTime.Size = New-Object System.Drawing.Size(40, 20)
    $grpSchedule.Controls.Add($lblTime)
    
    $cboHour = New-Object System.Windows.Forms.ComboBox
    $cboHour.Location = New-Object System.Drawing.Point(240, 47)
    $cboHour.Size = New-Object System.Drawing.Size(50, 22)
    $cboHour.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    0..23 | ForEach-Object { $cboHour.Items.Add($_.ToString("00")) }
    $cboHour.SelectedItem = "02"
    $grpSchedule.Controls.Add($cboHour)
    
    $lblColon = New-Object System.Windows.Forms.Label
    $lblColon.Text = ":"
    $lblColon.Location = New-Object System.Drawing.Point(292, 50)
    $grpSchedule.Controls.Add($lblColon)
    
    $cboMinute = New-Object System.Windows.Forms.ComboBox
    $cboMinute.Location = New-Object System.Drawing.Point(300, 47)
    $cboMinute.Size = New-Object System.Drawing.Size(50, 22)
    $cboMinute.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    @("00", "15", "30", "45") | ForEach-Object { $cboMinute.Items.Add($_) }
    $cboMinute.SelectedItem = "00"
    $grpSchedule.Controls.Add($cboMinute)
    
    $lblAdmin = New-Object System.Windows.Forms.Label
    $lblAdmin.Text = "Admin User:"
    $lblAdmin.Location = New-Object System.Drawing.Point(10, 80)
    $lblAdmin.Size = New-Object System.Drawing.Size(70, 20)
    $grpSchedule.Controls.Add($lblAdmin)
    
    $txtAdminUser = New-Object System.Windows.Forms.TextBox
    $txtAdminUser.Location = New-Object System.Drawing.Point(80, 77)
    $txtAdminUser.Size = New-Object System.Drawing.Size(150, 22)
    $txtAdminUser.Text = $env:USERNAME
    $grpSchedule.Controls.Add($txtAdminUser)
    
    $lblAdminPass = New-Object System.Windows.Forms.Label
    $lblAdminPass.Text = "Password:"
    $lblAdminPass.Location = New-Object System.Drawing.Point(240, 80)
    $lblAdminPass.Size = New-Object System.Drawing.Size(60, 20)
    $grpSchedule.Controls.Add($lblAdminPass)
    
    $txtAdminPass = New-Object System.Windows.Forms.TextBox
    $txtAdminPass.Location = New-Object System.Drawing.Point(300, 77)
    $txtAdminPass.Size = New-Object System.Drawing.Size(150, 22)
    $txtAdminPass.UseSystemPasswordChar = $true
    $grpSchedule.Controls.Add($txtAdminPass)
    
    $lblNote = New-Object System.Windows.Forms.Label
    $lblNote.Text = "Note: Password is encrypted and stored locally for auto-login after reboot."
    $lblNote.Location = New-Object System.Drawing.Point(10, 105)
    $lblNote.Size = New-Object System.Drawing.Size(440, 20)
    $lblNote.ForeColor = [System.Drawing.Color]::Gray
    $lblNote.Font = New-Object System.Drawing.Font("Arial", 8)
    $grpSchedule.Controls.Add($lblNote)
    
    $btnSaveSchedule = New-Object System.Windows.Forms.Button
    $btnSaveSchedule.Text = "Save Schedule"
    $btnSaveSchedule.Location = New-Object System.Drawing.Point(350, 120)
    $btnSaveSchedule.Size = New-Object System.Drawing.Size(100, 25)
    $btnSaveSchedule.BackColor = [System.Drawing.Color]::LightGreen
    $btnSaveSchedule.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpSchedule.Controls.Add($btnSaveSchedule)
    
    # =========================================================================
    # BUTTONS
    # =========================================================================
    
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(380, 375)
    $btnClose.Size = New-Object System.Drawing.Size(90, 30)
    $btnClose.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnClose.Add_Click({ $form.Close() })
    $form.Controls.Add($btnClose)
    
    # =========================================================================
    # EVENT HANDLERS
    # =========================================================================
    
    $btnRunUpdate.Add_Click({
        $result = [System.Windows.Forms.MessageBox]::Show(
            "This will:`n`n1. Log off all users (5 min warning)`n2. Stop QuickBooks`n3. Run Desktop updates`n4. Reboot if needed`n5. Run Payroll updates`n`nContinue?",
            "Confirm Full Update",
            [System.Windows.Forms.MessageBoxButtons]::YesNo,
            [System.Windows.Forms.MessageBoxIcon]::Question
        )
        
        if ($result -eq [System.Windows.Forms.DialogResult]::Yes) {
            # Get credentials
            $credForm = New-Object System.Windows.Forms.Form
            $credForm.Text = "Admin Credentials"
            $credForm.Size = New-Object System.Drawing.Size(350, 180)
            $credForm.StartPosition = "CenterParent"
            $credForm.FormBorderStyle = "FixedDialog"
            
            $lblU = New-Object System.Windows.Forms.Label
            $lblU.Text = "Username:"
            $lblU.Location = New-Object System.Drawing.Point(20, 20)
            $credForm.Controls.Add($lblU)
            
            $txtU = New-Object System.Windows.Forms.TextBox
            $txtU.Location = New-Object System.Drawing.Point(100, 17)
            $txtU.Size = New-Object System.Drawing.Size(200, 22)
            $txtU.Text = $env:USERNAME
            $credForm.Controls.Add($txtU)
            
            $lblP = New-Object System.Windows.Forms.Label
            $lblP.Text = "Password:"
            $lblP.Location = New-Object System.Drawing.Point(20, 55)
            $credForm.Controls.Add($lblP)
            
            $txtP = New-Object System.Windows.Forms.TextBox
            $txtP.Location = New-Object System.Drawing.Point(100, 52)
            $txtP.Size = New-Object System.Drawing.Size(200, 22)
            $txtP.UseSystemPasswordChar = $true
            $credForm.Controls.Add($txtP)
            
            $btnOK = New-Object System.Windows.Forms.Button
            $btnOK.Text = "Start Update"
            $btnOK.Location = New-Object System.Drawing.Point(100, 95)
            $btnOK.Size = New-Object System.Drawing.Size(100, 30)
            $btnOK.DialogResult = [System.Windows.Forms.DialogResult]::OK
            $credForm.Controls.Add($btnOK)
            
            if ($credForm.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
                $form.Hide()
                Invoke-FullQuickBooksUpdate -AdminUsername $txtU.Text -AdminPassword $txtP.Text
                $form.Close()
            }
        }
    })
    
    $btnPayrollOnly.Add_Click({
        Stop-AllQuickBooks
        $result = Invoke-PayrollUpdate
        if ($result.Success) {
            [System.Windows.Forms.MessageBox]::Show("QuickBooks launched for payroll update.", "Payroll Update")
        }
    })
    
    $btnStopQB.Add_Click({
        Stop-AllQuickBooks -Force
        [System.Windows.Forms.MessageBox]::Show("QuickBooks stopped.", "Stopped")
    })
    
    $btnStartQB.Add_Click({
        Start-QuickBooksServices
        [System.Windows.Forms.MessageBox]::Show("QuickBooks services started.", "Started")
    })
    
    $btnSaveSchedule.Add_Click({
        if ($chkEnableSchedule.Checked) {
            if ([string]::IsNullOrEmpty($txtAdminPass.Text)) {
                [System.Windows.Forms.MessageBox]::Show("Please enter admin password for auto-login.", "Error")
                return
            }
            
            Register-QuickBooksUpdateTask `
                -Time "$($cboHour.SelectedItem):$($cboMinute.SelectedItem)" `
                -AdminUsername $txtAdminUser.Text `
                -AdminPassword $txtAdminPass.Text `
                -Frequency $cboFrequency.SelectedItem
            
            [System.Windows.Forms.MessageBox]::Show(
                "QuickBooks update scheduled!`n`nFrequency: $($cboFrequency.SelectedItem)`nTime: $($cboHour.SelectedItem):$($cboMinute.SelectedItem)",
                "Schedule Saved"
            )
        } else {
            Unregister-ScheduledTask -TaskName "ABCD-QuickBooksUpdate" -Confirm:$false -ErrorAction SilentlyContinue
            [System.Windows.Forms.MessageBox]::Show("QuickBooks update schedule removed.", "Schedule Removed")
        }
    })
    
    $form.ShowDialog() | Out-Null
}
