#Requires -RunAsAdministrator

# =============================================================================
# ROLLBACK MODULE - File Versioning and Recovery
# =============================================================================
# Creates dated backup copies of files before overwriting
# Enables recovery from migration issues
# =============================================================================

$Script:RollbackEnabled = $false
$Script:RollbackPath = "$Script:InstallPath\Rollback"
$Script:RetentionDays = 7
$Script:RollbackStats = @{
    FilesBackedUp = 0
    BytesBackedUp = 0
    Errors = 0
}

function Initialize-Rollback {
    <#
    .SYNOPSIS
        Initializes the rollback system
    #>
    param(
        [bool]$Enabled = $true,
        [int]$RetentionDays = 7,
        [string]$RollbackPath = "$Script:InstallPath\Rollback"
    )
    
    $Script:RollbackEnabled = $Enabled
    $Script:RetentionDays = $RetentionDays
    $Script:RollbackPath = $RollbackPath
    
    $Script:RollbackStats = @{
        FilesBackedUp = 0
        BytesBackedUp = 0
        Errors = 0
    }
    
    if ($Script:RollbackEnabled) {
        Write-Host "  [Rollback] Initializing file versioning system..." -ForegroundColor Cyan
        Write-Host "  [Rollback] Path: $Script:RollbackPath" -ForegroundColor Gray
        Write-Host "  [Rollback] Retention: $Script:RetentionDays days" -ForegroundColor Gray
        
        # Create rollback directory
        if (-not (Test-Path $Script:RollbackPath)) {
            New-Item -ItemType Directory -Path $Script:RollbackPath -Force | Out-Null
        }
        
        # Clean up old rollback files
        Clear-OldRollbacks
    }
}

function Clear-OldRollbacks {
    <#
    .SYNOPSIS
        Removes rollback files older than retention period
    #>
    param(
        [int]$Days = $Script:RetentionDays
    )
    
    if (-not (Test-Path $Script:RollbackPath)) {
        return
    }
    
    $cutoffDate = (Get-Date).AddDays(-$Days)
    $removedCount = 0
    $removedBytes = 0
    
    Write-Host "  [Rollback] Cleaning up files older than $Days days..." -ForegroundColor Gray
    
    try {
        # Get all date folders
        $dateFolders = Get-ChildItem -Path $Script:RollbackPath -Directory -ErrorAction SilentlyContinue
        
        foreach ($folder in $dateFolders) {
            # Check if folder name is a date
            $folderDate = $null
            if ([DateTime]::TryParseExact($folder.Name, "yyyy-MM-dd", $null, [System.Globalization.DateTimeStyles]::None, [ref]$folderDate)) {
                if ($folderDate -lt $cutoffDate) {
                    # Get size before removal
                    $folderSize = (Get-ChildItem -Path $folder.FullName -Recurse -ErrorAction SilentlyContinue | 
                        Measure-Object -Property Length -Sum).Sum
                    
                    Remove-Item -Path $folder.FullName -Recurse -Force -ErrorAction SilentlyContinue
                    $removedCount++
                    $removedBytes += $folderSize
                }
            }
        }
        
        if ($removedCount -gt 0) {
            $removedMB = [math]::Round($removedBytes / 1MB, 2)
            Write-Host "  [Rollback] Removed $removedCount old backup folder(s) (${removedMB}MB)" -ForegroundColor Green
        }
        else {
            Write-Host "  [Rollback] No old backups to clean up" -ForegroundColor Gray
        }
    }
    catch {
        Write-Host "  [Rollback] Error during cleanup: $($_.Exception.Message)" -ForegroundColor Yellow
    }
}

function Backup-FileForRollback {
    <#
    .SYNOPSIS
        Creates a backup copy of a file before overwriting
    .PARAMETER FilePath
        The full path to the file being overwritten
    .RETURNS
        $true if backup successful or not needed, $false on error
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$FilePath
    )
    
    if (-not $Script:RollbackEnabled) {
        return $true
    }
    
    if (-not (Test-Path $FilePath)) {
        return $true  # Nothing to backup
    }
    
    try {
        # Get file info
        $fileInfo = Get-Item $FilePath -ErrorAction Stop
        
        # Skip very large files (>1GB) - configurable
        if ($fileInfo.Length -gt 1GB) {
            Write-Host "  [Rollback] Skipping large file (>1GB): $FilePath" -ForegroundColor Yellow
            return $true
        }
        
        # Create today's rollback folder
        $dateFolder = Get-Date -Format "yyyy-MM-dd"
        $todayPath = Join-Path $Script:RollbackPath $dateFolder
        
        if (-not (Test-Path $todayPath)) {
            New-Item -ItemType Directory -Path $todayPath -Force | Out-Null
        }
        
        # Create relative path structure
        $driveLetter = $FilePath.Substring(0, 1)
        $relativePath = $FilePath.Substring(3)  # Remove "C:\"
        $backupDir = Join-Path $todayPath "$driveLetter\$(Split-Path $relativePath -Parent)"
        $backupFile = Join-Path $todayPath "$driveLetter\$relativePath"
        
        # Create directory structure
        if (-not (Test-Path $backupDir)) {
            New-Item -ItemType Directory -Path $backupDir -Force | Out-Null
        }
        
        # If backup already exists (multiple copies same day), add timestamp
        if (Test-Path $backupFile) {
            $timestamp = Get-Date -Format "HHmmss"
            $ext = $fileInfo.Extension
            $baseName = $fileInfo.BaseName
            $backupFile = Join-Path $backupDir "${baseName}_${timestamp}${ext}"
        }
        
        # Copy file to rollback location
        Copy-Item -Path $FilePath -Destination $backupFile -Force -ErrorAction Stop
        
        $Script:RollbackStats.FilesBackedUp++
        $Script:RollbackStats.BytesBackedUp += $fileInfo.Length
        
        return $true
    }
    catch {
        $Script:RollbackStats.Errors++
        Write-Host "  [Rollback] ERROR backing up $FilePath : $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Backup-DirectoryForRollback {
    <#
    .SYNOPSIS
        Backs up all files in a directory before migration
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$DirectoryPath,
        
        [string[]]$ExcludePatterns = @("*.tmp", "*.log", "thumbs.db", "desktop.ini")
    )
    
    if (-not $Script:RollbackEnabled) {
        return $true
    }
    
    if (-not (Test-Path $DirectoryPath)) {
        return $true
    }
    
    $success = $true
    
    try {
        $files = Get-ChildItem -Path $DirectoryPath -File -Recurse -ErrorAction SilentlyContinue |
            Where-Object {
                $file = $_
                $excluded = $false
                foreach ($pattern in $ExcludePatterns) {
                    if ($file.Name -like $pattern) {
                        $excluded = $true
                        break
                    }
                }
                -not $excluded
            }
        
        $totalFiles = $files.Count
        $backedUp = 0
        
        foreach ($file in $files) {
            $result = Backup-FileForRollback -FilePath $file.FullName
            if (-not $result) {
                $success = $false
            }
            else {
                $backedUp++
            }
            
            # Progress indicator every 100 files
            if ($backedUp % 100 -eq 0) {
                Write-Host "  [Rollback] Backed up $backedUp / $totalFiles files..." -ForegroundColor Gray
            }
        }
    }
    catch {
        Write-Host "  [Rollback] ERROR during directory backup: $($_.Exception.Message)" -ForegroundColor Red
        $success = $false
    }
    
    return $success
}

function Get-RollbackStats {
    <#
    .SYNOPSIS
        Returns statistics about rollback operations
    #>
    $stats = $Script:RollbackStats.Clone()
    $stats.BytesBackedUpMB = [math]::Round($stats.BytesBackedUp / 1MB, 2)
    return $stats
}

function Get-RollbackFolders {
    <#
    .SYNOPSIS
        Lists available rollback date folders
    #>
    if (-not (Test-Path $Script:RollbackPath)) {
        return @()
    }
    
    $folders = @()
    
    Get-ChildItem -Path $Script:RollbackPath -Directory | ForEach-Object {
        $folderDate = $null
        if ([DateTime]::TryParseExact($_.Name, "yyyy-MM-dd", $null, [System.Globalization.DateTimeStyles]::None, [ref]$folderDate)) {
            $size = (Get-ChildItem -Path $_.FullName -Recurse -ErrorAction SilentlyContinue | 
                Measure-Object -Property Length -Sum).Sum
            
            $folders += @{
                Date = $folderDate
                Name = $_.Name
                Path = $_.FullName
                SizeMB = [math]::Round($size / 1MB, 2)
                FileCount = (Get-ChildItem -Path $_.FullName -Recurse -File -ErrorAction SilentlyContinue).Count
            }
        }
    }
    
    return $folders | Sort-Object Date -Descending
}

function Restore-FromRollback {
    <#
    .SYNOPSIS
        Restores files from a rollback folder
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$RollbackDate,  # Format: yyyy-MM-dd
        
        [string]$TargetPath = $null,  # Optional: specific path to restore
        
        [switch]$WhatIf
    )
    
    $rollbackFolder = Join-Path $Script:RollbackPath $RollbackDate
    
    if (-not (Test-Path $rollbackFolder)) {
        Write-Host "  [Rollback] ERROR: Rollback folder not found: $RollbackDate" -ForegroundColor Red
        return $false
    }
    
    Write-Host "  [Rollback] Restoring from $RollbackDate..." -ForegroundColor Cyan
    
    $restoredCount = 0
    $errorCount = 0
    
    # Get all backed up files
    $files = Get-ChildItem -Path $rollbackFolder -File -Recurse -ErrorAction SilentlyContinue
    
    foreach ($file in $files) {
        # Reconstruct original path
        $relativePath = $file.FullName.Substring($rollbackFolder.Length + 1)
        $driveLetter = $relativePath.Substring(0, 1)
        $filePath = $relativePath.Substring(2)
        $originalPath = "${driveLetter}:\$filePath"
        
        # Filter by target path if specified
        if ($TargetPath -and -not $originalPath.StartsWith($TargetPath, [System.StringComparison]::OrdinalIgnoreCase)) {
            continue
        }
        
        if ($WhatIf) {
            Write-Host "  [Rollback] Would restore: $originalPath" -ForegroundColor Yellow
            $restoredCount++
            continue
        }
        
        try {
            # Create directory if needed
            $originalDir = Split-Path $originalPath -Parent
            if (-not (Test-Path $originalDir)) {
                New-Item -ItemType Directory -Path $originalDir -Force | Out-Null
            }
            
            Copy-Item -Path $file.FullName -Destination $originalPath -Force -ErrorAction Stop
            $restoredCount++
        }
        catch {
            Write-Host "  [Rollback] ERROR restoring $originalPath : $($_.Exception.Message)" -ForegroundColor Red
            $errorCount++
        }
    }
    
    if ($WhatIf) {
        Write-Host "  [Rollback] Would restore $restoredCount files" -ForegroundColor Yellow
    }
    else {
        Write-Host "  [Rollback] Restored $restoredCount files ($errorCount errors)" -ForegroundColor Green
    }
    
    return ($errorCount -eq 0)
}

function Show-RollbackDialog {
    <#
    .SYNOPSIS
        Shows a dialog to select and restore from rollback
    #>
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $folders = Get-RollbackFolders
    
    if ($folders.Count -eq 0) {
        [System.Windows.Forms.MessageBox]::Show(
            "No rollback folders found.`n`nRollback folders are created when 'Enable File Versioning' is checked during migration.",
            "No Rollbacks Available",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        return
    }
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Restore from Rollback"
    $form.Size = New-Object System.Drawing.Size(500, 400)
    $form.StartPosition = "CenterParent"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    
    $lblInfo = New-Object System.Windows.Forms.Label
    $lblInfo.Text = "Select a rollback date to restore files from:"
    $lblInfo.Location = New-Object System.Drawing.Point(20, 20)
    $lblInfo.Size = New-Object System.Drawing.Size(440, 20)
    $form.Controls.Add($lblInfo)
    
    $lstRollbacks = New-Object System.Windows.Forms.ListView
    $lstRollbacks.Location = New-Object System.Drawing.Point(20, 50)
    $lstRollbacks.Size = New-Object System.Drawing.Size(440, 200)
    $lstRollbacks.View = "Details"
    $lstRollbacks.FullRowSelect = $true
    $lstRollbacks.GridLines = $true
    
    $lstRollbacks.Columns.Add("Date", 100)
    $lstRollbacks.Columns.Add("Files", 80)
    $lstRollbacks.Columns.Add("Size (MB)", 80)
    $lstRollbacks.Columns.Add("Path", 160)
    
    foreach ($folder in $folders) {
        $item = New-Object System.Windows.Forms.ListViewItem($folder.Name)
        $item.SubItems.Add($folder.FileCount.ToString())
        $item.SubItems.Add($folder.SizeMB.ToString())
        $item.SubItems.Add($folder.Path)
        $item.Tag = $folder.Name
        $lstRollbacks.Items.Add($item)
    }
    
    $form.Controls.Add($lstRollbacks)
    
    $lblWarning = New-Object System.Windows.Forms.Label
    $lblWarning.Text = "WARNING: This will overwrite current files with backed up versions!"
    $lblWarning.Location = New-Object System.Drawing.Point(20, 260)
    $lblWarning.Size = New-Object System.Drawing.Size(440, 20)
    $lblWarning.ForeColor = [System.Drawing.Color]::Red
    $form.Controls.Add($lblWarning)
    
    $chkPreview = New-Object System.Windows.Forms.CheckBox
    $chkPreview.Text = "Preview only (don't restore)"
    $chkPreview.Location = New-Object System.Drawing.Point(20, 285)
    $chkPreview.Size = New-Object System.Drawing.Size(200, 25)
    $form.Controls.Add($chkPreview)
    
    $btnRestore = New-Object System.Windows.Forms.Button
    $btnRestore.Text = "Restore"
    $btnRestore.Location = New-Object System.Drawing.Point(280, 320)
    $btnRestore.Size = New-Object System.Drawing.Size(80, 30)
    $btnRestore.Add_Click({
        if ($lstRollbacks.SelectedItems.Count -eq 0) {
            [System.Windows.Forms.MessageBox]::Show("Please select a rollback date.", "Selection Required",
                [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        $selectedDate = $lstRollbacks.SelectedItems[0].Tag
        
        if (-not $chkPreview.Checked) {
            $confirm = [System.Windows.Forms.MessageBox]::Show(
                "Are you sure you want to restore files from $selectedDate?`n`nThis will overwrite current files.",
                "Confirm Restore",
                [System.Windows.Forms.MessageBoxButtons]::YesNo,
                [System.Windows.Forms.MessageBoxIcon]::Warning
            )
            
            if ($confirm -ne [System.Windows.Forms.DialogResult]::Yes) {
                return
            }
        }
        
        $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
        $form.Tag = @{ Date = $selectedDate; Preview = $chkPreview.Checked }
        $form.Close()
    })
    $form.Controls.Add($btnRestore)
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = New-Object System.Drawing.Point(370, 320)
    $btnCancel.Size = New-Object System.Drawing.Size(80, 30)
    $btnCancel.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
    $form.Controls.Add($btnCancel)
    
    $result = $form.ShowDialog()
    
    if ($result -eq [System.Windows.Forms.DialogResult]::OK -and $form.Tag) {
        Restore-FromRollback -RollbackDate $form.Tag.Date -WhatIf:$form.Tag.Preview
    }
}
