#Requires -RunAsAdministrator
<#
.SYNOPSIS
    Scheduled Task Management for ABCD Migration Tool
.DESCRIPTION
    Creates, updates, and removes scheduled tasks for automated migration
#>

$Script:TaskNameSource = "ABCD-Migration-SOURCE-Backup"
$Script:TaskNameDestination = "ABCD-Migration-DESTINATION-Import"
$Script:SettingsFile = "C:\ABCD-Migration\Settings\migration-settings.json"

function Save-MigrationSettings {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode,  # "SOURCE" or "DESTINATION"
        
        [hashtable]$Settings
    )
    
    $settingsPath = Split-Path $Script:SettingsFile -Parent
    if (!(Test-Path $settingsPath)) {
        New-Item -Path $settingsPath -ItemType Directory -Force | Out-Null
    }
    
    # Load existing settings or create new
    $allSettings = @{}
    if (Test-Path $Script:SettingsFile) {
        try {
            $jsonContent = Get-Content $Script:SettingsFile -Raw | ConvertFrom-Json
            # Convert PSCustomObject to hashtable (PS 5.1 compatible)
            foreach ($prop in $jsonContent.PSObject.Properties) {
                if ($prop.Value -is [PSCustomObject]) {
                    # Nested object - convert to hashtable
                    $nestedHash = @{}
                    foreach ($nestedProp in $prop.Value.PSObject.Properties) {
                        $nestedHash[$nestedProp.Name] = $nestedProp.Value
                    }
                    $allSettings[$prop.Name] = $nestedHash
                } else {
                    $allSettings[$prop.Name] = $prop.Value
                }
            }
        } catch {
            $allSettings = @{}
        }
    }
    
    # Update settings for this mode
    $allSettings[$Mode] = $Settings
    $allSettings["LastUpdated"] = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    
    # Save to file
    $allSettings | ConvertTo-Json -Depth 5 | Out-File $Script:SettingsFile -Encoding UTF8 -Force
    
    Write-Host "  [Settings] Saved $Mode settings to $Script:SettingsFile" -ForegroundColor Green
    return $true
}

function Load-MigrationSettings {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode  # "SOURCE" or "DESTINATION"
    )
    
    if (!(Test-Path $Script:SettingsFile)) {
        Write-Host "  [Settings] No saved settings found at $Script:SettingsFile" -ForegroundColor Yellow
        return $null
    }
    
    try {
        $jsonContent = Get-Content $Script:SettingsFile -Raw | ConvertFrom-Json
        
        # Check if the mode exists in saved settings
        if ($jsonContent.PSObject.Properties.Name -contains $Mode) {
            $modeSettings = $jsonContent.$Mode
            Write-Host "  [Settings] Loaded $Mode settings from $Script:SettingsFile" -ForegroundColor Green
            return $modeSettings
        } else {
            Write-Host "  [Settings] No settings found for $Mode mode" -ForegroundColor Yellow
        }
    } catch {
        Write-Host "  [Settings] Error loading settings: $($_.Exception.Message)" -ForegroundColor Red
    }
    
    return $null
}

function New-MigrationScheduledTask {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode,  # "SOURCE" or "DESTINATION"
        
        [Parameter(Mandatory=$true)]
        [string]$Time,  # "HH:mm" format (e.g., "21:00")
        
        [string]$SourceComputer = ""
    )
    
    $taskName = if ($Mode -eq "SOURCE") { $Script:TaskNameSource } else { $Script:TaskNameDestination }
    $installPath = "C:\ABCD-Migration"
    
    # Remove existing task first
    Remove-MigrationScheduledTask -Mode $Mode
    
    # Build the command
    $scriptPath = "$installPath\ABCD-Migration-Scheduled.ps1"
    $arguments = "-Mode $Mode"
    
    if ($Mode -eq "DESTINATION" -and $SourceComputer) {
        $arguments += " -SourceComputer `"$SourceComputer`""
    }
    
    # Create the scheduled task action
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-NoProfile -ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`" $arguments"
    
    # Parse time
    $timeParts = $Time -split ':'
    $hour = [int]$timeParts[0]
    $minute = [int]$timeParts[1]
    
    # Create daily trigger at specified time
    $trigger = New-ScheduledTaskTrigger -Daily -At "$($hour):$($minute.ToString('00'))"
    
    # Task settings - allow start on demand, don't stop on idle
    $taskSettings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -StartWhenAvailable -DontStopOnIdleEnd -ExecutionTimeLimit (New-TimeSpan -Hours 12)
    
    # For SOURCE mode, run as SYSTEM (local operations only)
    # For DESTINATION mode, run as SYSTEM but we'll use stored credentials in script
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    
    # Create the task
    try {
        $task = Register-ScheduledTask -TaskName $taskName -Action $action -Trigger $trigger -Principal $principal -Settings $taskSettings -Description "ABCD Systems Migration Tool - Automated $Mode" -Force
        
        Write-Host "  [Task] Created scheduled task: $taskName" -ForegroundColor Green
        Write-Host "  [Task] Runs daily at $Time" -ForegroundColor Gray
        Write-Host "  [Task] Task State: $($task.State)" -ForegroundColor Gray
        return $true
    }
    catch {
        Write-Host "  [Task] ERROR creating task: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Start-MigrationScheduledTask {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode  # "SOURCE" or "DESTINATION"
    )
    
    $taskName = if ($Mode -eq "SOURCE") { $Script:TaskNameSource } else { $Script:TaskNameDestination }
    
    try {
        $task = Get-ScheduledTask -TaskName $taskName -ErrorAction SilentlyContinue
        if ($task) {
            Start-ScheduledTask -TaskName $taskName
            Write-Host "  [Task] Started task: $taskName" -ForegroundColor Green
            return $true
        } else {
            Write-Host "  [Task] Task not found: $taskName" -ForegroundColor Yellow
            return $false
        }
    }
    catch {
        Write-Host "  [Task] ERROR starting task: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

function Remove-MigrationScheduledTask {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode  # "SOURCE" or "DESTINATION"
    )
    
    $taskName = if ($Mode -eq "SOURCE") { $Script:TaskNameSource } else { $Script:TaskNameDestination }
    
    try {
        $existingTask = Get-ScheduledTask -TaskName $taskName -ErrorAction SilentlyContinue
        if ($existingTask) {
            Unregister-ScheduledTask -TaskName $taskName -Confirm:$false
            Write-Host "  [Task] Removed scheduled task: $taskName" -ForegroundColor Yellow
            return $true
        }
    }
    catch {
        # Task doesn't exist, that's fine
    }
    
    return $false
}

function Get-MigrationScheduledTask {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode  # "SOURCE" or "DESTINATION"
    )
    
    $taskName = if ($Mode -eq "SOURCE") { $Script:TaskNameSource } else { $Script:TaskNameDestination }
    
    try {
        $task = Get-ScheduledTask -TaskName $taskName -ErrorAction SilentlyContinue
        if ($task) {
            $taskInfo = Get-ScheduledTaskInfo -TaskName $taskName -ErrorAction SilentlyContinue
            $trigger = $task.Triggers | Select-Object -First 1
            
            # Extract time from trigger
            $triggerTime = "Unknown"
            if ($trigger.StartBoundary) {
                $dt = [datetime]::Parse($trigger.StartBoundary)
                $triggerTime = $dt.ToString("HH:mm")
            }
            
            return @{
                Exists = $true
                TaskName = $taskName
                State = $task.State
                Time = $triggerTime
                LastRun = $taskInfo.LastRunTime
                NextRun = $taskInfo.NextRunTime
                LastResult = $taskInfo.LastTaskResult
            }
        }
    }
    catch {}
    
    return @{
        Exists = $false
        TaskName = $taskName
    }
}

function Test-MigrationScheduledTask {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Mode
    )
    
    $taskInfo = Get-MigrationScheduledTask -Mode $Mode
    return $taskInfo.Exists
}
