<#
.SYNOPSIS
    ABC&D Systems - Service Management Module
.DESCRIPTION
    Manages services and user sessions for clean backups/migrations.
    Stops all services EXCEPT VSS and Hyper-V which backup better while running.
.VERSION
    4.0
#>

# =============================================================================
# CONFIGURATION
# =============================================================================

# Services to NEVER stop (they help with backups)
$Script:ProtectedServices = @(
    "VSS",                      # Volume Shadow Copy - essential for backups
    "SDRSVC",                   # Windows Backup
    "vmms",                     # Hyper-V Virtual Machine Management
    "vmcompute",                # Hyper-V Host Compute Service
    "hvhost",                   # Hyper-V Host Service
    "vmicvss",                  # Hyper-V VSS Integration
    "vmicshutdown",             # Hyper-V Shutdown Service
    "vmicheartbeat",            # Hyper-V Heartbeat
    "vmicrdv",                  # Hyper-V Remote Desktop
    "vmictimesync",             # Hyper-V Time Sync
    "vmickvpexchange",          # Hyper-V KVP Exchange
    "vmicguestinterface",       # Hyper-V Guest Interface
    "vmicvmsession",            # Hyper-V PowerShell Direct
    "Winmgmt",                  # WMI - needed for remote management
    "RpcSs",                    # RPC - needed for everything
    "RpcEptMapper",             # RPC Endpoint Mapper
    "DcomLaunch",               # DCOM Server
    "EventLog",                 # Windows Event Log
    "PlugPlay",                 # Plug and Play
    "Power",                    # Power Service
    "ProfSvc",                  # User Profile Service
    "LanmanServer",             # Server (file sharing)
    "LanmanWorkstation",        # Workstation (network access)
    "Netlogon",                 # Network Logon
    "SamSs",                    # Security Accounts Manager
    "lsass",                    # Local Security Authority
    "RemoteRegistry",           # Remote Registry - needed for remote ops
    "Schedule",                 # Task Scheduler
    "Themes",                   # Desktop themes
    "AudioSrv",                 # Audio
    "Audiosrv"                  # Audio (alternate case)
)

# Services to prioritize stopping (file-locking apps)
$Script:PriorityStopServices = @(
    # QuickBooks (all versions)
    "QuickBooksDB*",
    "QBCFMonitorService",
    "QBIDPService",
    "IntuitUpdateService",
    "QBPOSDBService*",
    
    # SQL Server
    "MSSQLSERVER",
    "MSSQL`$*",
    "SQLAgent*",
    "SQLSERVERAGENT",
    "MSSQLServerOLAPService",
    "SQLBrowser",
    "MSSQLFDLauncher*",
    "SQLTELEMETRY*",
    
    # MySQL / MariaDB
    "MySQL*",
    "MariaDB*",
    
    # PostgreSQL
    "postgresql*",
    
    # Sage
    "Sage*Server*",
    "SAGEServer*",
    "Sage*Database*",
    
    # Pervasive (Sage, QuickBooks)
    "Pervasive*",
    "w3dbsmgr*",
    
    # Act! CRM
    "Act! Scheduler",
    "ActDiag",
    
    # Fishbowl
    "FishbowlMySQL",
    "FishbowlServer*",
    
    # Adobe
    "AGSService",
    "AdobeUpdateService",
    "AdobeARMservice",
    
    # Autodesk
    "AdskLicensing*",
    "FNPLicensingService*",
    
    # Backup Software (stop competing backups)
    "VeeamBackupSvc",
    "VeeamEndpointBackupSvc",
    "AcrSch2Svc",
    "AcronisAgent*",
    
    # Dynamics / Great Plains
    "Dynamics*",
    
    # Firebird
    "FirebirdServer*",
    "FirebirdGuardian*",
    
    # Cloud Sync (prevent conflicts)
    "DbxSvc",
    "OneSyncSvc*",
    "GoogleDriveFS",
    
    # Antivirus (can lock files)
    "MsMpSvc",              # Windows Defender
    "WinDefend",            # Windows Defender
    "Sophos*",
    "McAfee*",
    "Symantec*",
    "Norton*",
    "ESET*",
    "Kaspersky*",
    "Avast*",
    "AVG*",
    "Bitdefender*",
    "Trend*",
    "Webroot*",
    
    # Print Spooler (can lock files)
    "Spooler",
    
    # Search (indexes files)
    "WSearch",
    
    # MYOB
    "MYOB*"
)

# Processes to kill
$Script:ProcessesToKill = @(
    # QuickBooks
    "QBW32", "QBW", "QBDBMgrN", "QBDBMgr", "QBUpdate", "QBLaunch",
    
    # Sage
    "Peachw", "Sage*",
    
    # Microsoft Office
    "WINWORD", "EXCEL", "POWERPNT", "OUTLOOK", "MSACCESS", "ONENOTE",
    
    # Adobe
    "Acrobat", "AcroRd32", "Photoshop", "Illustrator", "InDesign",
    
    # Browsers (can lock cache/profile files)
    "chrome", "firefox", "msedge", "iexplore",
    
    # Cloud sync
    "Dropbox", "OneDrive", "GoogleDriveFS", "googledrivesync",
    
    # Communication
    "Teams", "ms-teams", "Slack", "Discord", "Zoom", "Skype",
    
    # Media
    "Spotify", "iTunes",
    
    # Development
    "devenv", "Code", "notepad++",
    
    # Database clients
    "ssms", "mysql", "pgadmin"
)

# Track what we stopped for restart
$Script:StoppedServices = @()
$Script:LoggedOffSessions = @()

# =============================================================================
# USER SESSION MANAGEMENT
# =============================================================================

function Get-LoggedOnUsers {
    <#
    .SYNOPSIS
        Gets list of logged on users (local or remote)
    #>
    param(
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null
    )
    
    $users = @()
    
    try {
        if ($ComputerName) {
            # Remote - use query session via WMI
            $scriptBlock = {
                $sessions = query session 2>$null | Select-Object -Skip 1
                foreach ($line in $sessions) {
                    if ($line -match '^\s*(\S+)\s+(\S+)\s+(\d+)\s+(\S+)') {
                        @{
                            SessionName = $matches[1]
                            UserName = $matches[2]
                            SessionId = $matches[3]
                            State = $matches[4]
                        }
                    }
                }
            }
            
            if ($Credential) {
                $users = Invoke-Command -ComputerName $ComputerName -Credential $Credential -ScriptBlock $scriptBlock -ErrorAction SilentlyContinue
            } else {
                $users = Invoke-Command -ComputerName $ComputerName -ScriptBlock $scriptBlock -ErrorAction SilentlyContinue
            }
        } else {
            # Local
            $sessions = query session 2>$null | Select-Object -Skip 1
            foreach ($line in $sessions) {
                if ($line -match '^\s*>?(\S+)\s+(\S+)\s+(\d+)\s+(\S+)') {
                    $users += @{
                        SessionName = $matches[1]
                        UserName = $matches[2]
                        SessionId = [int]$matches[3]
                        State = $matches[4]
                    }
                }
            }
        }
    } catch {
        Write-Host "  [Sessions] Could not query sessions: $_" -ForegroundColor Yellow
    }
    
    return $users
}

function Invoke-LogoffAllUsers {
    <#
    .SYNOPSIS
        Logs off all users except current session
    #>
    param(
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null,
        [switch]$Force
    )
    
    $target = if ($ComputerName) { $ComputerName } else { "local" }
    Write-Host "  [Sessions] ================================================" -ForegroundColor Cyan
    Write-Host "  [Sessions] Logging off users on $target..." -ForegroundColor Cyan
    
    $currentSessionId = [System.Diagnostics.Process]::GetCurrentProcess().SessionId
    $loggedOff = @()
    
    try {
        if ($ComputerName) {
            # Remote logoff
            $scriptBlock = {
                param($Force)
                $results = @()
                $sessions = query session 2>$null | Select-Object -Skip 1
                foreach ($line in $sessions) {
                    if ($line -match '^\s*>?(\S+)\s+(\S+)\s+(\d+)\s+(Active|Disc)') {
                        $sessionId = [int]$matches[3]
                        $userName = $matches[2]
                        if ($userName -ne "services" -and $userName -ne "65536") {
                            try {
                                logoff $sessionId /server:localhost
                                $results += $userName
                            } catch {}
                        }
                    }
                }
                return $results
            }
            
            if ($Credential) {
                $loggedOff = Invoke-Command -ComputerName $ComputerName -Credential $Credential -ScriptBlock $scriptBlock -ArgumentList $Force -ErrorAction SilentlyContinue
            } else {
                $loggedOff = Invoke-Command -ComputerName $ComputerName -ScriptBlock $scriptBlock -ArgumentList $Force -ErrorAction SilentlyContinue
            }
        } else {
            # Local logoff
            $sessions = query session 2>$null | Select-Object -Skip 1
            foreach ($line in $sessions) {
                if ($line -match '^\s*>?(\S+)\s+(\S+)\s+(\d+)\s+(Active|Disc)') {
                    $sessionId = [int]$matches[3]
                    $userName = $matches[2]
                    
                    # Don't log off current session or system sessions
                    if ($sessionId -ne $currentSessionId -and $userName -ne "services" -and $sessionId -ne 0) {
                        try {
                            Write-Host "  [Sessions] Logging off: $userName (Session $sessionId)" -ForegroundColor Yellow
                            logoff $sessionId
                            $loggedOff += $userName
                            Write-Host "  [Sessions]   Logged off successfully" -ForegroundColor Green
                        } catch {
                            Write-Host "  [Sessions]   Could not log off: $_" -ForegroundColor Red
                        }
                    }
                }
            }
        }
    } catch {
        Write-Host "  [Sessions] Error during logoff: $_" -ForegroundColor Red
    }
    
    $Script:LoggedOffSessions = $loggedOff
    Write-Host "  [Sessions] Logged off $($loggedOff.Count) user(s)" -ForegroundColor Green
    
    return $loggedOff
}

# =============================================================================
# SERVICE MANAGEMENT
# =============================================================================

function Test-IsProtectedService {
    param([string]$ServiceName)
    
    foreach ($protected in $Script:ProtectedServices) {
        if ($ServiceName -eq $protected -or $ServiceName -like $protected) {
            return $true
        }
    }
    return $false
}

function Stop-AllServicesForBackup {
    <#
    .SYNOPSIS
        Stops all non-essential services for clean backup.
        PRESERVES: VSS, Hyper-V, core Windows services
    #>
    param(
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null,
        [switch]$KillProcesses = $true
    )
    
    $target = if ($ComputerName) { $ComputerName } else { "local machine" }
    Write-Host "  [Services] ================================================" -ForegroundColor Cyan
    Write-Host "  [Services] Stopping services for backup on $target" -ForegroundColor Cyan
    Write-Host "  [Services] (Preserving VSS and Hyper-V)" -ForegroundColor Cyan
    Write-Host "  [Services] ================================================" -ForegroundColor Cyan
    
    $stoppedServices = @()
    
    # First, stop priority services (QuickBooks, SQL, etc.)
    Write-Host "  [Services] Phase 1: Stopping file-locking services..." -ForegroundColor Yellow
    
    foreach ($pattern in $Script:PriorityStopServices) {
        try {
            if ($ComputerName) {
                # Remote
                $filter = "Name LIKE '$($pattern.Replace('*','%'))' AND State = 'Running'"
                $wmiParams = @{
                    Class = "Win32_Service"
                    ComputerName = $ComputerName
                    Filter = $filter
                    ErrorAction = "SilentlyContinue"
                }
                if ($Credential) { $wmiParams.Credential = $Credential }
                
                $services = Get-WmiObject @wmiParams
                foreach ($svc in $services) {
                    if (!(Test-IsProtectedService $svc.Name)) {
                        Write-Host "  [Services] Stopping: $($svc.DisplayName)" -ForegroundColor Yellow
                        $result = $svc.StopService()
                        if ($result.ReturnValue -eq 0) {
                            $stoppedServices += $svc.Name
                            Write-Host "  [Services]   Stopped" -ForegroundColor Green
                        }
                    }
                }
            } else {
                # Local
                $services = Get-Service -Name $pattern -ErrorAction SilentlyContinue | Where-Object { $_.Status -eq 'Running' }
                foreach ($svc in $services) {
                    if (!(Test-IsProtectedService $svc.Name)) {
                        Write-Host "  [Services] Stopping: $($svc.DisplayName)" -ForegroundColor Yellow
                        Stop-Service -Name $svc.Name -Force -ErrorAction SilentlyContinue
                        $stoppedServices += $svc.Name
                        Write-Host "  [Services]   Stopped" -ForegroundColor Green
                    }
                }
            }
        } catch {
            # Continue on errors
        }
    }
    
    # Kill processes
    if ($KillProcesses) {
        Write-Host "  [Services] Phase 2: Terminating applications..." -ForegroundColor Yellow
        
        foreach ($procName in $Script:ProcessesToKill) {
            try {
                if ($ComputerName) {
                    $wmiParams = @{
                        Class = "Win32_Process"
                        ComputerName = $ComputerName
                        Filter = "Name LIKE '$procName%'"
                        ErrorAction = "SilentlyContinue"
                    }
                    if ($Credential) { $wmiParams.Credential = $Credential }
                    
                    $procs = Get-WmiObject @wmiParams
                    foreach ($p in $procs) {
                        Write-Host "  [Process] Terminating: $($p.Name)" -ForegroundColor Yellow
                        $p.Terminate() | Out-Null
                    }
                } else {
                    $procs = Get-Process -Name $procName -ErrorAction SilentlyContinue
                    foreach ($p in $procs) {
                        Write-Host "  [Process] Terminating: $($p.ProcessName)" -ForegroundColor Yellow
                        Stop-Process -Id $p.Id -Force -ErrorAction SilentlyContinue
                    }
                }
            } catch {
                # Continue
            }
        }
    }
    
    $Script:StoppedServices = $stoppedServices
    
    # Wait for services to fully stop
    Write-Host "  [Services] Waiting 5 seconds for services to release files..." -ForegroundColor Gray
    Start-Sleep -Seconds 5
    
    Write-Host "  [Services] Stopped $($stoppedServices.Count) service(s)" -ForegroundColor Green
    
    return $stoppedServices
}

function Start-ServicesAfterBackup {
    <#
    .SYNOPSIS
        Restarts services that were stopped for backup
    #>
    param(
        [string[]]$ServiceNames = $null,
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null
    )
    
    if (!$ServiceNames) {
        $ServiceNames = $Script:StoppedServices
    }
    
    if (!$ServiceNames -or $ServiceNames.Count -eq 0) {
        Write-Host "  [Services] No services to restart" -ForegroundColor Gray
        return
    }
    
    $target = if ($ComputerName) { $ComputerName } else { "local machine" }
    Write-Host "  [Services] ================================================" -ForegroundColor Cyan
    Write-Host "  [Services] Restarting services on $target..." -ForegroundColor Cyan
    Write-Host "  [Services] ================================================" -ForegroundColor Cyan
    
    $restarted = 0
    
    foreach ($svcName in $ServiceNames) {
        try {
            if ($ComputerName) {
                $wmiParams = @{
                    Class = "Win32_Service"
                    ComputerName = $ComputerName
                    Filter = "Name = '$svcName'"
                    ErrorAction = "SilentlyContinue"
                }
                if ($Credential) { $wmiParams.Credential = $Credential }
                
                $svc = Get-WmiObject @wmiParams
                if ($svc) {
                    Write-Host "  [Services] Starting: $($svc.DisplayName)" -ForegroundColor Yellow
                    $result = $svc.StartService()
                    if ($result.ReturnValue -eq 0) {
                        $restarted++
                        Write-Host "  [Services]   Started" -ForegroundColor Green
                    }
                }
            } else {
                $svc = Get-Service -Name $svcName -ErrorAction SilentlyContinue
                if ($svc) {
                    Write-Host "  [Services] Starting: $($svc.DisplayName)" -ForegroundColor Yellow
                    Start-Service -Name $svcName -ErrorAction SilentlyContinue
                    $restarted++
                    Write-Host "  [Services]   Started" -ForegroundColor Green
                }
            }
        } catch {
            Write-Host "  [Services] Could not start $svcName : $_" -ForegroundColor Red
        }
    }
    
    Write-Host "  [Services] Restarted $restarted service(s)" -ForegroundColor Green
    
    # Clear the list
    $Script:StoppedServices = @()
}

# =============================================================================
# WRAPPER FUNCTIONS FOR GUI
# =============================================================================

function Stop-RemoteServicesForCopy {
    <#
    .SYNOPSIS
        Stops services on remote SOURCE computer for migration
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$ComputerName,
        [System.Management.Automation.PSCredential]$Credential
    )
    
    # Log off users first
    Invoke-LogoffAllUsers -ComputerName $ComputerName -Credential $Credential -Force
    
    # Stop services
    return Stop-AllServicesForBackup -ComputerName $ComputerName -Credential $Credential -KillProcesses
}

function Start-RemoteStoppedServices {
    <#
    .SYNOPSIS
        Restarts services on remote computer after migration
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$ComputerName,
        [System.Management.Automation.PSCredential]$Credential,
        [string[]]$ServiceNames
    )
    
    Start-ServicesAfterBackup -ServiceNames $ServiceNames -ComputerName $ComputerName -Credential $Credential
}

function Stop-LocalServicesForBackup {
    <#
    .SYNOPSIS
        Stops local services for SOURCE export
    #>
    
    # Log off other users
    Invoke-LogoffAllUsers -Force
    
    # Stop services
    return Stop-AllServicesForBackup -KillProcesses
}

function Start-LocalServicesAfterBackup {
    <#
    .SYNOPSIS
        Restarts local services after SOURCE export
    #>
    param([string[]]$ServiceNames)
    
    Start-ServicesAfterBackup -ServiceNames $ServiceNames
}

# =============================================================================
# LEGACY COMPATIBILITY FUNCTIONS
# =============================================================================

function Get-RunningServicesForPatterns {
    param(
        [string[]]$ServicePatterns,
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null
    )
    
    $runningServices = @()
    
    foreach ($pattern in $ServicePatterns) {
        try {
            if ($ComputerName) {
                $filter = "Name LIKE '$($pattern.Replace('*','%'))' AND State = 'Running'"
                $wmiParams = @{
                    Class = "Win32_Service"
                    ComputerName = $ComputerName
                    Filter = $filter
                    ErrorAction = "SilentlyContinue"
                }
                if ($Credential) { $wmiParams.Credential = $Credential }
                
                $services = Get-WmiObject @wmiParams
                foreach ($svc in $services) {
                    $runningServices += @{
                        Name = $svc.Name
                        DisplayName = $svc.DisplayName
                        Status = $svc.State
                    }
                }
            } else {
                $services = Get-Service -Name $pattern -ErrorAction SilentlyContinue | Where-Object { $_.Status -eq 'Running' }
                foreach ($svc in $services) {
                    $runningServices += @{
                        Name = $svc.Name
                        DisplayName = $svc.DisplayName
                        Status = $svc.Status
                    }
                }
            }
        } catch {}
    }
    
    return $runningServices
}

function Stop-ServicesForFiles {
    param(
        [string[]]$FilePaths,
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null
    )
    
    # Just use the comprehensive stop
    return Stop-AllServicesForBackup -ComputerName $ComputerName -Credential $Credential
}

function Start-StoppedServices {
    param(
        [string[]]$ServiceNames,
        [string]$ComputerName = $null,
        [System.Management.Automation.PSCredential]$Credential = $null
    )
    
    Start-ServicesAfterBackup -ServiceNames $ServiceNames -ComputerName $ComputerName -Credential $Credential
}
