#Requires -RunAsAdministrator

# Helper function to create share using net share (works on all Windows editions)
function New-ShareNetCommand {
    param(
        [string]$ShareName,
        [string]$Path,
        [string]$Description = ""
    )
    
    # Remove existing share first
    $existingCheck = net share $ShareName 2>&1
    if ($existingCheck -notmatch "not found") {
        net share $ShareName /delete /y 2>&1 | Out-Null
    }
    
    # Create new share with full access for Everyone
    $result = net share "$ShareName=$Path" "/grant:Everyone,FULL" "/remark:$Description" 2>&1
    
    if ($LASTEXITCODE -eq 0) {
        return $true
    } else {
        Write-Host "  [Shares] net share error: $result" -ForegroundColor Yellow
        return $false
    }
}

# Helper function to remove share using net share
function Remove-ShareNetCommand {
    param([string]$ShareName)
    
    $result = net share $ShareName /delete /y 2>&1
    return ($LASTEXITCODE -eq 0)
}

# Check if SMB cmdlets are available
function Test-SmbCmdletsAvailable {
    $cmd = Get-Command New-SmbShare -ErrorAction SilentlyContinue
    return ($null -ne $cmd)
}

# Note: Migration share creation is handled inline in the GUI
# The functions below (Export/Import) are the only active share functions

function Export-NetworkShares {
    Write-Host "  [Shares] Exporting network shares..." -ForegroundColor Cyan
    
    $useSmbCmdlets = Test-SmbCmdletsAvailable
    $excludeShares = @('C$', 'D$', 'E$', 'F$', 'IPC$', 'ADMIN$', 'print$', 'ABCD-Migration$', 'ProgramFiles$', 'ProgramFilesX86$', 'ProgramData$', 'Users$')
    
    $exportData = @()
    
    if ($useSmbCmdlets) {
        $shares = Get-SmbShare | Where-Object { $_.Name -notin $excludeShares }
        
        foreach ($share in $shares) {
            try {
                $permissions = Get-SmbShareAccess -Name $share.Name -ErrorAction SilentlyContinue
                
                $shareData = @{
                    Name = $share.Name
                    Path = $share.Path
                    Description = $share.Description
                    Permissions = @()
                }
                
                foreach ($perm in $permissions) {
                    $shareData.Permissions += @{
                        AccountName = $perm.AccountName
                        AccessRight = $perm.AccessRight
                        AccessControlType = $perm.AccessControlType
                    }
                }
                
                $exportData += $shareData
                Write-Host "  [Shares] Exported: $($share.Name)" -ForegroundColor Green
            }
            catch {
                Write-Host "  [Shares] WARNING: Could not export share $($share.Name)" -ForegroundColor Yellow
            }
        }
    } else {
        # Use net share to list shares
        Write-Host "  [Shares] Using net share command..." -ForegroundColor Yellow
        $netShareOutput = net share 2>&1
        
        foreach ($line in $netShareOutput) {
            # Parse net share output (format: ShareName  Path  Remark)
            if ($line -match "^(\S+)\s+(\S+:\\.+?)\s*(.*)$") {
                $shareName = $matches[1]
                $sharePath = $matches[2].Trim()
                $description = $matches[3].Trim()
                
                if ($shareName -notin $excludeShares -and $sharePath -match "^[A-Z]:") {
                    $shareData = @{
                        Name = $shareName
                        Path = $sharePath
                        Description = $description
                        Permissions = @(@{AccountName="Everyone"; AccessRight="Full"; AccessControlType="Allow"})
                    }
                    $exportData += $shareData
                    Write-Host "  [Shares] Exported: $shareName" -ForegroundColor Green
                }
            }
        }
    }
    
    $shareCount = $exportData.Count
    Write-Host "  [Shares] Network share export complete - $shareCount shares" -ForegroundColor Green
    return $exportData
}

function Import-NetworkShares {
    param(
        [Parameter(Mandatory=$true)]
        [array]$Shares,
        
        [switch]$CreateFolders
    )
    
    Write-Host "  [Shares] Importing network shares..." -ForegroundColor Cyan
    
    $useSmbCmdlets = Test-SmbCmdletsAvailable
    if (-not $useSmbCmdlets) {
        Write-Host "  [Shares] Using net share command (Windows Home compatible)" -ForegroundColor Yellow
    }
    
    $imported = 0
    
    foreach ($share in $Shares) {
        try {
            # Check if share already exists
            if ($useSmbCmdlets) {
                $existing = Get-SmbShare -Name $share.Name -ErrorAction SilentlyContinue
                if ($existing) {
                    Write-Host "  [Shares] Share already exists: $($share.Name)" -ForegroundColor Gray
                    continue
                }
            } else {
                $existingCheck = net share $share.Name 2>&1
                if ($existingCheck -notmatch "not found") {
                    Write-Host "  [Shares] Share already exists: $($share.Name)" -ForegroundColor Gray
                    continue
                }
            }
            
            # Create folder if needed
            if ($CreateFolders -and !(Test-Path $share.Path)) {
                Write-Host "  [Shares] Creating folder: $($share.Path)" -ForegroundColor Yellow
                New-Item -Path $share.Path -ItemType Directory -Force | Out-Null
            }
            
            Write-Host "  [Shares] Creating share: $($share.Name)" -ForegroundColor Yellow
            
            if ($useSmbCmdlets) {
                New-SmbShare -Name $share.Name -Path $share.Path -Description $share.Description -FullAccess "Administrators" -ErrorAction Stop | Out-Null
                
                # Apply permissions
                foreach ($perm in $share.Permissions) {
                    try {
                        if ($perm.AccessControlType -eq "Allow") {
                            if ($perm.AccessRight -eq "Full") {
                                Grant-SmbShareAccess -Name $share.Name -AccountName $perm.AccountName -AccessRight Full -Force -ErrorAction SilentlyContinue | Out-Null
                            }
                            elseif ($perm.AccessRight -eq "Change") {
                                Grant-SmbShareAccess -Name $share.Name -AccountName $perm.AccountName -AccessRight Change -Force -ErrorAction SilentlyContinue | Out-Null
                            }
                            elseif ($perm.AccessRight -eq "Read") {
                                Grant-SmbShareAccess -Name $share.Name -AccountName $perm.AccountName -AccessRight Read -Force -ErrorAction SilentlyContinue | Out-Null
                            }
                        }
                    }
                    catch {
                        Write-Host "  [Shares] WARNING: Could not apply permission for $($perm.AccountName)" -ForegroundColor Yellow
                    }
                }
            } else {
                # Use net share command
                $desc = if ($share.Description) { $share.Description } else { "Migration share" }
                $result = net share "$($share.Name)=$($share.Path)" "/grant:Everyone,FULL" "/remark:$desc" 2>&1
                if ($LASTEXITCODE -ne 0) {
                    Write-Host "  [Shares] WARNING: net share failed: $result" -ForegroundColor Yellow
                    continue
                }
            }
            
            Write-Host "  [Shares] Imported: $($share.Name)" -ForegroundColor Green
            $imported++
        }
        catch {
            Write-Host "  [Shares] ERROR importing share $($share.Name): $($_.Exception.Message)" -ForegroundColor Red
        }
    }
    
    Write-Host "  [Shares] Network share import complete - $imported shares imported" -ForegroundColor Green
    return $imported
}
