#Requires -RunAsAdministrator

# =============================================================================
# SYSTEM TRAY MODULE - Background Operation Support
# =============================================================================
# Enables minimize to system tray functionality
# =============================================================================

$Script:NotifyIcon = $null
$Script:TrayContextMenu = $null
$Script:ParentForm = $null
$Script:IsMinimizedToTray = $false

function Initialize-SystemTray {
    <#
    .SYNOPSIS
        Initializes the system tray icon and context menu
    #>
    param(
        [Parameter(Mandatory=$true)]
        [System.Windows.Forms.Form]$Form,
        
        [string]$IconPath = "$Script:InstallPath\Assets\ABCD-Icon.ico",
        [string]$TooltipText = "ABCD Migration Tool"
    )
    
    $Script:ParentForm = $Form
    
    # Create NotifyIcon
    $Script:NotifyIcon = New-Object System.Windows.Forms.NotifyIcon
    $Script:NotifyIcon.Text = $TooltipText
    $Script:NotifyIcon.Visible = $false
    
    # Load icon
    if (Test-Path $IconPath) {
        try {
            $Script:NotifyIcon.Icon = New-Object System.Drawing.Icon($IconPath)
        }
        catch {
            $Script:NotifyIcon.Icon = [System.Drawing.SystemIcons]::Application
        }
    }
    else {
        $Script:NotifyIcon.Icon = [System.Drawing.SystemIcons]::Application
    }
    
    # Create context menu
    $Script:TrayContextMenu = New-Object System.Windows.Forms.ContextMenuStrip
    
    # Open menu item
    $menuOpen = New-Object System.Windows.Forms.ToolStripMenuItem
    $menuOpen.Text = "Open ABCD Migration Tool"
    $menuOpen.Font = New-Object System.Drawing.Font($menuOpen.Font, [System.Drawing.FontStyle]::Bold)
    $menuOpen.Add_Click({
        Restore-FromTray
    })
    $Script:TrayContextMenu.Items.Add($menuOpen)
    
    $Script:TrayContextMenu.Items.Add((New-Object System.Windows.Forms.ToolStripSeparator))
    
    # Status menu item
    $Script:MenuStatus = New-Object System.Windows.Forms.ToolStripMenuItem
    $Script:MenuStatus.Text = "Status: Idle"
    $Script:MenuStatus.Enabled = $false
    $Script:TrayContextMenu.Items.Add($Script:MenuStatus)
    
    $Script:TrayContextMenu.Items.Add((New-Object System.Windows.Forms.ToolStripSeparator))
    
    # View Logs
    $menuLogs = New-Object System.Windows.Forms.ToolStripMenuItem
    $menuLogs.Text = "View Logs Folder"
    $menuLogs.Add_Click({
        $logsPath = "$Script:InstallPath\Logs"
        if (Test-Path $logsPath) {
            Start-Process explorer.exe -ArgumentList $logsPath
        }
    })
    $Script:TrayContextMenu.Items.Add($menuLogs)
    
    # View Settings
    $menuSettings = New-Object System.Windows.Forms.ToolStripMenuItem
    $menuSettings.Text = "View Settings Folder"
    $menuSettings.Add_Click({
        $settingsPath = "$Script:InstallPath\Settings"
        if (Test-Path $settingsPath) {
            Start-Process explorer.exe -ArgumentList $settingsPath
        }
    })
    $Script:TrayContextMenu.Items.Add($menuSettings)
    
    $Script:TrayContextMenu.Items.Add((New-Object System.Windows.Forms.ToolStripSeparator))
    
    # Exit
    $menuExit = New-Object System.Windows.Forms.ToolStripMenuItem
    $menuExit.Text = "Exit"
    $menuExit.Add_Click({
        Exit-FromTray
    })
    $Script:TrayContextMenu.Items.Add($menuExit)
    
    $Script:NotifyIcon.ContextMenuStrip = $Script:TrayContextMenu
    
    # Double-click to restore
    $Script:NotifyIcon.Add_DoubleClick({
        Restore-FromTray
    })
    
    # Balloon tip click to restore
    $Script:NotifyIcon.Add_BalloonTipClicked({
        Restore-FromTray
    })
    
    return $Script:NotifyIcon
}

function Minimize-ToTray {
    <#
    .SYNOPSIS
        Minimizes the form to system tray
    #>
    if ($Script:ParentForm -and $Script:NotifyIcon) {
        $Script:ParentForm.WindowState = [System.Windows.Forms.FormWindowState]::Minimized
        $Script:ParentForm.ShowInTaskbar = $false
        $Script:NotifyIcon.Visible = $true
        $Script:IsMinimizedToTray = $true
        
        # Show balloon tip
        $Script:NotifyIcon.ShowBalloonTip(
            2000,
            "ABCD Migration Tool",
            "Running in background. Double-click to restore.",
            [System.Windows.Forms.ToolTipIcon]::Info
        )
    }
}

function Restore-FromTray {
    <#
    .SYNOPSIS
        Restores the form from system tray
    #>
    if ($Script:ParentForm -and $Script:NotifyIcon) {
        $Script:ParentForm.ShowInTaskbar = $true
        $Script:ParentForm.WindowState = [System.Windows.Forms.FormWindowState]::Normal
        $Script:NotifyIcon.Visible = $false
        $Script:IsMinimizedToTray = $false
        $Script:ParentForm.Activate()
    }
}

function Exit-FromTray {
    <#
    .SYNOPSIS
        Exits the application from tray
    #>
    if ($Script:NotifyIcon) {
        $Script:NotifyIcon.Visible = $false
        $Script:NotifyIcon.Dispose()
    }
    if ($Script:ParentForm) {
        $Script:ParentForm.Close()
    }
}

function Update-TrayStatus {
    <#
    .SYNOPSIS
        Updates the status shown in tray menu
    #>
    param(
        [string]$Status = "Idle"
    )
    
    if ($Script:MenuStatus) {
        $Script:MenuStatus.Text = "Status: $Status"
    }
    
    if ($Script:NotifyIcon) {
        $Script:NotifyIcon.Text = "ABCD Migration Tool - $Status"
    }
}

function Show-TrayNotification {
    <#
    .SYNOPSIS
        Shows a balloon notification from tray
    #>
    param(
        [string]$Title = "ABCD Migration Tool",
        [string]$Message,
        [System.Windows.Forms.ToolTipIcon]$Icon = [System.Windows.Forms.ToolTipIcon]::Info,
        [int]$Timeout = 3000
    )
    
    if ($Script:NotifyIcon -and $Script:NotifyIcon.Visible) {
        $Script:NotifyIcon.ShowBalloonTip($Timeout, $Title, $Message, $Icon)
    }
}

function Test-IsMinimizedToTray {
    <#
    .SYNOPSIS
        Returns whether the app is minimized to tray
    #>
    return $Script:IsMinimizedToTray
}

function Remove-SystemTray {
    <#
    .SYNOPSIS
        Cleans up system tray resources
    #>
    if ($Script:NotifyIcon) {
        $Script:NotifyIcon.Visible = $false
        $Script:NotifyIcon.Dispose()
        $Script:NotifyIcon = $null
    }
    if ($Script:TrayContextMenu) {
        $Script:TrayContextMenu.Dispose()
        $Script:TrayContextMenu = $null
    }
}
