#Requires -RunAsAdministrator

function Export-LocalUsers {
    Write-Host "  [Users] Scanning local user accounts..." -ForegroundColor Cyan
    
    $excludeUsers = @("Administrator", "Guest", "DefaultAccount", "WDAGUtilityAccount", "krbtgt")
    
    $users = Get-LocalUser | Where-Object { 
        $_.Enabled -eq $true -and $_.Name -notin $excludeUsers
    }
    
    $exportData = @()
    
    foreach ($user in $users) {
        try {
            $groups = @()
            $allGroups = Get-LocalGroup
            foreach ($group in $allGroups) {
                $members = Get-LocalGroupMember -Group $group.Name -ErrorAction SilentlyContinue
                if ($members.Name -contains "$env:COMPUTERNAME\$($user.Name)") {
                    $groups += $group.Name
                }
            }
            
            $userData = @{
                Name = $user.Name
                FullName = $user.FullName
                Description = $user.Description
                PasswordNeverExpires = $user.PasswordNeverExpires
                UserMayChangePassword = $user.UserMayChangePassword
                Groups = $groups
                SID = $user.SID.Value
            }
            
            $exportData += $userData
            Write-Host "  [Users] Exported: $($user.Name)" -ForegroundColor Green
        }
        catch {
            Write-Host "  [Users] WARNING: Could not export user $($user.Name)" -ForegroundColor Yellow
        }
    }
    
    $userCount = $exportData.Count
    Write-Host "  [Users] User export complete - $userCount users" -ForegroundColor Green
    return $exportData
}

# NOTE: Master password is ONLY applied when creating users on DESTINATION
# Source passwords are NOT modified - they cannot be exported anyway

function Import-LocalUsers {
    param(
        [Parameter(Mandatory=$true)]
        [array]$Users,
        
        [Parameter(Mandatory=$true)]
        [SecureString]$MasterPassword
    )
    
    Write-Host "  [Users] Importing user accounts..." -ForegroundColor Cyan
    
    # Get current admin user to skip
    $currentUser = $env:USERNAME
    Write-Host "  [Users] NOTE: Skipping current user ($currentUser) to prevent lockout" -ForegroundColor Yellow
    
    $imported = 0
    $skipped = 0
    
    foreach ($user in $Users) {
        # Skip current admin user
        if ($user.Name -eq $currentUser) {
            Write-Host "  [Users] SKIPPED: $($user.Name) (current admin - cannot modify)" -ForegroundColor Yellow
            $skipped++
            continue
        }
        
        # Skip built-in accounts
        if ($user.Name -match "^(Administrator|Guest|DefaultAccount|WDAGUtilityAccount)$") {
            Write-Host "  [Users] SKIPPED: $($user.Name) (built-in account)" -ForegroundColor Gray
            $skipped++
            continue
        }
        
        try {
            $existing = Get-LocalUser -Name $user.Name -ErrorAction SilentlyContinue
            if ($existing) {
                Write-Host "  [Users] User already exists: $($user.Name)" -ForegroundColor Gray
                $skipped++
                continue
            }
            
            Write-Host "  [Users] Creating user: $($user.Name)" -ForegroundColor Yellow
            
            $newUserParams = @{
                Name = $user.Name
                Password = $MasterPassword
                FullName = $user.FullName
                Description = $user.Description
                PasswordNeverExpires = $user.PasswordNeverExpires
                UserMayNotChangePassword = (-not $user.UserMayChangePassword)
            }
            
            New-LocalUser @newUserParams -ErrorAction Stop | Out-Null
            
            foreach ($groupName in $user.Groups) {
                try {
                    $group = Get-LocalGroup -Name $groupName -ErrorAction SilentlyContinue
                    if ($group) {
                        Add-LocalGroupMember -Group $groupName -Member $user.Name -ErrorAction SilentlyContinue
                        Write-Host "  [Users]   Added to group: $groupName" -ForegroundColor Gray
                    }
                }
                catch {
                    Write-Host "  [Users]   WARNING: Could not add to group $groupName" -ForegroundColor Yellow
                }
            }
            
            Write-Host "  [Users] Imported: $($user.Name) (password set to master)" -ForegroundColor Green
            $imported++
        }
        catch {
            Write-Host "  [Users] ERROR importing user $($user.Name)" -ForegroundColor Red
        }
    }
    
    Write-Host "  [Users] User import complete - $imported imported, $skipped skipped" -ForegroundColor Green
    return @{Imported=$imported; Skipped=$skipped}
}

function New-UserProfileViaRDS {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Username,
        
        [Parameter(Mandatory=$true)]
        [SecureString]$MasterPassword
    )
    
    Write-Host "  [RDS] Creating profile for: $Username" -ForegroundColor Yellow
    
    try {
        $BSTR = [System.Runtime.InteropServices.Marshal]::SecureStringToBSTR($MasterPassword)
        $plainPassword = [System.Runtime.InteropServices.Marshal]::PtrToStringAuto($BSTR)
        [System.Runtime.InteropServices.Marshal]::ZeroFreeBSTR($BSTR)
        
        $cmdkeyResult = cmdkey /generic:TERMSRV/127.0.0.1 /user:$Username /pass:$plainPassword 2>&1
        
        $rdpProcess = Start-Process mstsc -ArgumentList "/v:127.0.0.1 /w:100 /h:100" -WindowStyle Hidden -PassThru
        
        Write-Host "  [RDS]   Waiting for profile creation..." -ForegroundColor Gray
        Start-Sleep -Seconds 10
        
        if ($rdpProcess -and !$rdpProcess.HasExited) {
            $rdpProcess | Stop-Process -Force -ErrorAction SilentlyContinue
        }
        
        cmdkey /delete:TERMSRV/127.0.0.1 2>&1 | Out-Null
        
        $profilePath = "C:\Users\$Username"
        if (Test-Path $profilePath) {
            Write-Host "  [RDS] Profile created: $profilePath" -ForegroundColor Green
            return $true
        }
        else {
            Write-Host "  [RDS] WARNING: Profile folder not found: $profilePath" -ForegroundColor Yellow
            return $false
        }
    }
    catch {
        Write-Host "  [RDS] ERROR creating profile for $Username" -ForegroundColor Red
        cmdkey /delete:TERMSRV/127.0.0.1 2>&1 | Out-Null
        return $false
    }
}

function New-AllUserProfilesViaRDS {
    param(
        [Parameter(Mandatory=$true)]
        [array]$Users,
        
        [Parameter(Mandatory=$true)]
        [SecureString]$MasterPassword
    )
    
    Write-Host "  [RDS] Creating user profiles via RDS..." -ForegroundColor Cyan
    
    # Get current admin user to skip
    $currentUser = $env:USERNAME
    
    $successCount = 0
    $failCount = 0
    $skippedCount = 0
    $totalUsers = $Users.Count
    $currentUserNum = 0
    
    foreach ($user in $Users) {
        $currentUserNum++
        
        # Skip current admin user
        if ($user.Name -eq $currentUser) {
            Write-Host "  [RDS] SKIPPED: $($user.Name) (current admin)" -ForegroundColor Yellow
            $skippedCount++
            continue
        }
        
        # Skip built-in accounts
        if ($user.Name -match "^(Administrator|Guest|DefaultAccount|WDAGUtilityAccount)$") {
            Write-Host "  [RDS] SKIPPED: $($user.Name) (built-in account)" -ForegroundColor Gray
            $skippedCount++
            continue
        }
        
        # Check if profile already exists
        $profilePath = "C:\Users\$($user.Name)"
        if (Test-Path $profilePath) {
            Write-Host "  [RDS] SKIPPED: $($user.Name) (profile already exists)" -ForegroundColor Gray
            $skippedCount++
            continue
        }
        
        Write-Host "  [RDS] Processing $currentUserNum of $totalUsers - $($user.Name)..." -ForegroundColor Gray
        
        $result = New-UserProfileViaRDS -Username $user.Name -MasterPassword $MasterPassword
        
        if ($result) {
            $successCount++
        }
        else {
            $failCount++
        }
    }
    
    Write-Host "  [RDS] Profile creation complete: $successCount created, $skippedCount skipped, $failCount failed" -ForegroundColor Green
    return @{Success=$successCount; Skipped=$skippedCount; Failed=$failCount}
}
