#Requires -RunAsAdministrator

# =============================================================================
# VHD MIGRATION MODULE - Disk2VHD Integration
# =============================================================================
# Creates VHD/VHDX images of physical disks for backup or P2V migration
# Uses Microsoft Sysinternals Disk2VHD (INCLUDED - NOT DOWNLOADED)
# =============================================================================

$Script:Disk2VHDPath = $null

function Initialize-Disk2VHD {
    <#
    .SYNOPSIS
        Locates Disk2VHD in the Tools folder (bundled with installer)
    .DESCRIPTION
        This tool is INCLUDED with the installer - no download required.
        disk2vhd64.exe is located in the Tools folder.
    .RETURNS
        Path to disk2vhd64.exe or $null if not found
    #>
    param(
        [string]$ToolsPath = "$Script:InstallPath\Tools"
    )
    
    # Check Tools folder for bundled copy (64-bit preferred)
    $localPath64 = Join-Path $ToolsPath "disk2vhd64.exe"
    $localPath = Join-Path $ToolsPath "disk2vhd.exe"
    
    if (Test-Path $localPath64) {
        $Script:Disk2VHDPath = $localPath64
        Write-Host "  [VHD] Using bundled disk2vhd64.exe" -ForegroundColor Green
        return $Script:Disk2VHDPath
    }
    
    if (Test-Path $localPath) {
        $Script:Disk2VHDPath = $localPath
        Write-Host "  [VHD] Using bundled disk2vhd.exe" -ForegroundColor Green
        return $Script:Disk2VHDPath
    }
    
    # Check default install path
    $defaultPath = "C:\ABCD-Migration\Tools\disk2vhd64.exe"
    if (Test-Path $defaultPath) {
        $Script:Disk2VHDPath = $defaultPath
        Write-Host "  [VHD] Using disk2vhd64.exe from default install" -ForegroundColor Green
        return $Script:Disk2VHDPath
    }
    
    # Check if in system PATH (user may have installed separately)
    $systemPath64 = Get-Command "disk2vhd64.exe" -ErrorAction SilentlyContinue
    if ($systemPath64) {
        $Script:Disk2VHDPath = $systemPath64.Source
        Write-Host "  [VHD] Found disk2vhd64.exe in PATH" -ForegroundColor Green
        return $Script:Disk2VHDPath
    }
    
    $systemPath = Get-Command "disk2vhd.exe" -ErrorAction SilentlyContinue
    if ($systemPath) {
        $Script:Disk2VHDPath = $systemPath.Source
        Write-Host "  [VHD] Found disk2vhd.exe in PATH" -ForegroundColor Green
        return $Script:Disk2VHDPath
    }
    
    # Tool not found - show error (no download - bundled only)
    Write-Host "  [VHD] ERROR: disk2vhd64.exe not found!" -ForegroundColor Red
    Write-Host "  [VHD] This tool should be in: $ToolsPath" -ForegroundColor Red
    Write-Host "  [VHD] Please reinstall ABCD Admin Toolset to restore bundled tools." -ForegroundColor Yellow
    
    return $null
}

function Test-Disk2VHDAvailable {
    <#
    .SYNOPSIS
        Checks if Disk2VHD is available
    #>
    if ($Script:Disk2VHDPath -and (Test-Path $Script:Disk2VHDPath)) {
        return $true
    }
    
    $path = Initialize-Disk2VHD
    return ($null -ne $path)
}

function Get-PhysicalDisks {
    <#
    .SYNOPSIS
        Gets list of physical disks available for VHD creation
    #>
    
    $disks = @()
    
    try {
        $physicalDisks = Get-CimInstance -ClassName Win32_DiskDrive -ErrorAction SilentlyContinue
        
        foreach ($disk in $physicalDisks) {
            # Get associated partitions
            $partitions = Get-CimInstance -Query "ASSOCIATORS OF {Win32_DiskDrive.DeviceID='$($disk.DeviceID)'} WHERE AssocClass=Win32_DiskDriveToDiskPartition" -ErrorAction SilentlyContinue
            
            $volumes = @()
            foreach ($partition in $partitions) {
                $logicalDisks = Get-CimInstance -Query "ASSOCIATORS OF {Win32_DiskPartition.DeviceID='$($partition.DeviceID)'} WHERE AssocClass=Win32_LogicalDiskToPartition" -ErrorAction SilentlyContinue
                foreach ($logical in $logicalDisks) {
                    $volumes += $logical.DeviceID
                }
            }
            
            $disks += @{
                DeviceID = $disk.DeviceID
                Index = $disk.Index
                Model = $disk.Model
                Size = [math]::Round($disk.Size / 1GB, 2)
                Volumes = $volumes -join ", "
                InterfaceType = $disk.InterfaceType
            }
        }
    }
    catch {
        Write-Host "  [VHD] Error enumerating disks: $_" -ForegroundColor Red
    }
    
    return $disks
}

function New-DiskVHD {
    <#
    .SYNOPSIS
        Creates a VHD/VHDX from a physical disk or volume
    .PARAMETER SourceVolumes
        Array of volume letters to include (e.g., @("C:", "D:"))
    .PARAMETER DestinationPath
        Full path for the output VHD/VHDX file
    .PARAMETER UseVHDX
        If true, creates VHDX format (recommended). Default: true
    .PARAMETER UseVSS
        If true, uses Volume Shadow Copy for consistent snapshot. Default: true
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string[]]$SourceVolumes,
        
        [Parameter(Mandatory=$true)]
        [string]$DestinationPath,
        
        [bool]$UseVHDX = $true,
        [bool]$UseVSS = $true
    )
    
    # Ensure Disk2VHD is available
    if (-not (Test-Disk2VHDAvailable)) {
        return @{
            Success = $false
            Error = "Disk2VHD tool not found. Please reinstall ABCD Admin Toolset."
        }
    }
    
    # Validate source volumes
    foreach ($vol in $SourceVolumes) {
        $volLetter = $vol.TrimEnd(':') + ":"
        if (-not (Test-Path "$volLetter\")) {
            return @{
                Success = $false
                Error = "Volume $volLetter not found or not accessible"
            }
        }
    }
    
    # Ensure destination directory exists
    $destDir = Split-Path $DestinationPath -Parent
    if (-not (Test-Path $destDir)) {
        New-Item -Path $destDir -ItemType Directory -Force | Out-Null
    }
    
    # Ensure correct extension
    if ($UseVHDX -and -not $DestinationPath.EndsWith(".vhdx")) {
        $DestinationPath = [System.IO.Path]::ChangeExtension($DestinationPath, ".vhdx")
    }
    elseif (-not $UseVHDX -and -not $DestinationPath.EndsWith(".vhd")) {
        $DestinationPath = [System.IO.Path]::ChangeExtension($DestinationPath, ".vhd")
    }
    
    # Build command arguments
    # disk2vhd [-c] [-h] <volumes-to-include> <output-vhd>
    # -c = use volume shadow copy
    # -h = create VHDX (Hyper-V) format
    
    $volumes = ($SourceVolumes | ForEach-Object { $_.TrimEnd(':') + ":" }) -join " "
    
    $args = @()
    $args += "-accepteula"  # Accept EULA silently
    if ($UseVSS) { $args += "-c" }
    if ($UseVHDX) { $args += "-h" }
    $args += $volumes
    $args += "`"$DestinationPath`""
    
    Write-Host "  [VHD] Creating VHD from: $volumes" -ForegroundColor Cyan
    Write-Host "  [VHD] Output: $DestinationPath" -ForegroundColor Gray
    Write-Host "  [VHD] Format: $(if ($UseVHDX) { 'VHDX' } else { 'VHD' }), VSS: $UseVSS" -ForegroundColor Gray
    
    try {
        $startTime = Get-Date
        
        # Run Disk2VHD
        $process = Start-Process -FilePath $Script:Disk2VHDPath `
            -ArgumentList ($args -join " ") `
            -Wait -PassThru -NoNewWindow
        
        $duration = (Get-Date) - $startTime
        
        if ($process.ExitCode -eq 0 -and (Test-Path $DestinationPath)) {
            $fileSize = [math]::Round((Get-Item $DestinationPath).Length / 1GB, 2)
            Write-Host "  [VHD] Success! Created $fileSize GB in $([math]::Round($duration.TotalMinutes, 1)) minutes" -ForegroundColor Green
            
            return @{
                Success = $true
                Path = $DestinationPath
                SizeGB = $fileSize
                Duration = $duration
            }
        }
        else {
            Write-Host "  [VHD] Failed with exit code: $($process.ExitCode)" -ForegroundColor Red
            return @{
                Success = $false
                Error = "Disk2VHD exited with code $($process.ExitCode)"
            }
        }
    }
    catch {
        Write-Host "  [VHD] Error: $_" -ForegroundColor Red
        return @{
            Success = $false
            Error = $_.Exception.Message
        }
    }
}

function Show-Disk2VHDDialog {
    <#
    .SYNOPSIS
        Shows GUI dialog for Disk2VHD operations
    #>
    
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    # Initialize tool
    $toolPath = Initialize-Disk2VHD
    if (-not $toolPath) {
        [System.Windows.Forms.MessageBox]::Show(
            "Disk2VHD tool not found!`n`nThis tool should be bundled with the installer.`nPlease reinstall ABCD Admin Toolset.",
            "Tool Not Found",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Error
        )
        return
    }
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Disk to VHDX - ABCD Systems"
    $form.Size = New-Object System.Drawing.Size(500, 400)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(20, 20, 20)
    $form.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    
    # Source Selection
    $grpSource = New-Object System.Windows.Forms.GroupBox
    $grpSource.Text = "Source Volumes"
    $grpSource.Location = New-Object System.Drawing.Point(10, 10)
    $grpSource.Size = New-Object System.Drawing.Size(460, 150)
    $grpSource.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
    $grpSource.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.Controls.Add($grpSource)
    
    $lstVolumes = New-Object System.Windows.Forms.CheckedListBox
    $lstVolumes.Location = New-Object System.Drawing.Point(10, 20)
    $lstVolumes.Size = New-Object System.Drawing.Size(440, 120)
    $lstVolumes.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
    $lstVolumes.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $lstVolumes.CheckOnClick = $true
    $grpSource.Controls.Add($lstVolumes)
    
    # Populate volumes
    $volumes = Get-CimInstance -ClassName Win32_LogicalDisk -Filter "DriveType=3" -ErrorAction SilentlyContinue
    foreach ($vol in $volumes) {
        $sizeGB = [math]::Round($vol.Size / 1GB, 1)
        $freeGB = [math]::Round($vol.FreeSpace / 1GB, 1)
        $label = if ($vol.VolumeName) { $vol.VolumeName } else { "Local Disk" }
        $lstVolumes.Items.Add("$($vol.DeviceID) - $label ($sizeGB GB, $freeGB GB free)", ($vol.DeviceID -eq "C:"))
    }
    
    # Destination
    $grpDest = New-Object System.Windows.Forms.GroupBox
    $grpDest.Text = "Destination"
    $grpDest.Location = New-Object System.Drawing.Point(10, 170)
    $grpDest.Size = New-Object System.Drawing.Size(460, 80)
    $grpDest.BackColor = [System.Drawing.Color]::FromArgb(28, 28, 28)
    $grpDest.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $form.Controls.Add($grpDest)
    
    $lblDest = New-Object System.Windows.Forms.Label
    $lblDest.Text = "Output File:"
    $lblDest.Location = New-Object System.Drawing.Point(10, 25)
    $grpDest.Controls.Add($lblDest)
    
    $txtDest = New-Object System.Windows.Forms.TextBox
    $txtDest.Location = New-Object System.Drawing.Point(80, 22)
    $txtDest.Size = New-Object System.Drawing.Size(290, 22)
    $txtDest.Text = "E:\Backup\$env:COMPUTERNAME-$(Get-Date -Format 'yyyyMMdd').vhdx"
    $txtDest.BackColor = [System.Drawing.Color]::FromArgb(38, 38, 38)
    $txtDest.ForeColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    $grpDest.Controls.Add($txtDest)
    
    $btnBrowse = New-Object System.Windows.Forms.Button
    $btnBrowse.Text = "Browse..."
    $btnBrowse.Location = New-Object System.Drawing.Point(375, 21)
    $btnBrowse.Size = New-Object System.Drawing.Size(75, 24)
    $btnBrowse.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
    $btnBrowse.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpDest.Controls.Add($btnBrowse)
    
    # Options
    $chkVHDX = New-Object System.Windows.Forms.CheckBox
    $chkVHDX.Text = "VHDX format (recommended)"
    $chkVHDX.Location = New-Object System.Drawing.Point(10, 52)
    $chkVHDX.Size = New-Object System.Drawing.Size(180, 20)
    $chkVHDX.Checked = $true
    $grpDest.Controls.Add($chkVHDX)
    
    $chkVSS = New-Object System.Windows.Forms.CheckBox
    $chkVSS.Text = "Use VSS snapshot"
    $chkVSS.Location = New-Object System.Drawing.Point(200, 52)
    $chkVSS.Size = New-Object System.Drawing.Size(150, 20)
    $chkVSS.Checked = $true
    $grpDest.Controls.Add($chkVSS)
    
    # Status
    $lblStatus = New-Object System.Windows.Forms.Label
    $lblStatus.Text = "Tool: disk2vhd64.exe (bundled)"
    $lblStatus.Location = New-Object System.Drawing.Point(10, 260)
    $lblStatus.Size = New-Object System.Drawing.Size(460, 20)
    $lblStatus.ForeColor = [System.Drawing.Color]::FromArgb(180, 180, 180)
    $form.Controls.Add($lblStatus)
    
    # Progress
    $progressBar = New-Object System.Windows.Forms.ProgressBar
    $progressBar.Location = New-Object System.Drawing.Point(10, 285)
    $progressBar.Size = New-Object System.Drawing.Size(460, 20)
    $progressBar.Style = [System.Windows.Forms.ProgressBarStyle]::Marquee
    $progressBar.Visible = $false
    $form.Controls.Add($progressBar)
    
    # Buttons
    $btnCreate = New-Object System.Windows.Forms.Button
    $btnCreate.Text = "Create VHDX"
    $btnCreate.Location = New-Object System.Drawing.Point(270, 320)
    $btnCreate.Size = New-Object System.Drawing.Size(100, 30)
    $btnCreate.BackColor = [System.Drawing.Color]::FromArgb(76, 175, 80)
    $btnCreate.ForeColor = [System.Drawing.Color]::Black
    $btnCreate.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $form.Controls.Add($btnCreate)
    
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(380, 320)
    $btnClose.Size = New-Object System.Drawing.Size(80, 30)
    $btnClose.BackColor = [System.Drawing.Color]::FromArgb(45, 45, 45)
    $btnClose.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnClose.Add_Click({ $form.Close() })
    $form.Controls.Add($btnClose)
    
    # Event handlers
    $btnBrowse.Add_Click({
        $sfd = New-Object System.Windows.Forms.SaveFileDialog
        $sfd.Filter = "VHDX Files|*.vhdx|VHD Files|*.vhd"
        $sfd.FileName = [System.IO.Path]::GetFileName($txtDest.Text)
        if ($sfd.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $txtDest.Text = $sfd.FileName
        }
    })
    
    $btnCreate.Add_Click({
        # Get selected volumes
        $selectedVolumes = @()
        for ($i = 0; $i -lt $lstVolumes.Items.Count; $i++) {
            if ($lstVolumes.GetItemChecked($i)) {
                $item = $lstVolumes.Items[$i]
                $volume = $item.Split(" ")[0]
                $selectedVolumes += $volume
            }
        }
        
        if ($selectedVolumes.Count -eq 0) {
            [System.Windows.Forms.MessageBox]::Show("Please select at least one volume.", "No Selection")
            return
        }
        
        if ([string]::IsNullOrEmpty($txtDest.Text)) {
            [System.Windows.Forms.MessageBox]::Show("Please specify output file path.", "No Destination")
            return
        }
        
        $progressBar.Visible = $true
        $btnCreate.Enabled = $false
        $lblStatus.Text = "Creating VHD... This may take a while."
        $form.Refresh()
        
        $result = New-DiskVHD -SourceVolumes $selectedVolumes -DestinationPath $txtDest.Text -UseVHDX $chkVHDX.Checked -UseVSS $chkVSS.Checked
        
        $progressBar.Visible = $false
        $btnCreate.Enabled = $true
        
        if ($result.Success) {
            $lblStatus.Text = "Complete! $($result.SizeGB) GB created"
            [System.Windows.Forms.MessageBox]::Show(
                "VHD created successfully!`n`nPath: $($result.Path)`nSize: $($result.SizeGB) GB`nTime: $([math]::Round($result.Duration.TotalMinutes, 1)) minutes",
                "Success",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Information
            )
        } else {
            $lblStatus.Text = "Failed: $($result.Error)"
            [System.Windows.Forms.MessageBox]::Show(
                "VHD creation failed:`n`n$($result.Error)",
                "Error",
                [System.Windows.Forms.MessageBoxButtons]::OK,
                [System.Windows.Forms.MessageBoxIcon]::Error
            )
        }
    })
    
    $form.ShowDialog() | Out-Null
}
