#Requires -RunAsAdministrator

# =============================================================================
# VM EXPORT MODULE - Hyper-V Virtual Machine Backup
# =============================================================================
# Exports running VMs with snapshot cleanup and retention management
# =============================================================================

$Script:VMExportPath = "E:\Backup"
$Script:VMRetentionDays = 7

function Test-HyperVAvailable {
    <#
    .SYNOPSIS
        Checks if Hyper-V is available on this system
    #>
    try {
        $hyperv = Get-WindowsOptionalFeature -Online -FeatureName Microsoft-Hyper-V -ErrorAction SilentlyContinue
        if ($hyperv -and $hyperv.State -eq "Enabled") {
            return $true
        }
        
        # Also check for Hyper-V role on Server
        $role = Get-WindowsFeature -Name Hyper-V -ErrorAction SilentlyContinue
        if ($role -and $role.Installed) {
            return $true
        }
        
        return $false
    }
    catch {
        return $false
    }
}

function Get-VMList {
    <#
    .SYNOPSIS
        Gets list of all VMs on the system
    #>
    if (-not (Test-HyperVAvailable)) {
        Write-Host "  [VM] Hyper-V is not available on this system" -ForegroundColor Yellow
        return @()
    }
    
    try {
        $vms = Get-VM -ErrorAction Stop | Select-Object Name, State, 
            @{N='MemoryGB';E={[math]::Round($_.MemoryAssigned/1GB, 2)}},
            @{N='SizeGB';E={
                $size = 0
                $_.HardDrives | ForEach-Object {
                    if (Test-Path $_.Path) {
                        $size += (Get-Item $_.Path).Length
                    }
                }
                [math]::Round($size/1GB, 2)
            }},
            Uptime, Status
        
        return $vms
    }
    catch {
        Write-Host "  [VM] Error getting VM list: $($_.Exception.Message)" -ForegroundColor Red
        return @()
    }
}

function Export-VMWithCleanup {
    <#
    .SYNOPSIS
        Exports a VM to dated folder with optional snapshot cleanup
    .PARAMETER VMName
        Name of the VM to export (supports wildcards)
    .PARAMETER ExportPath
        Base path for exports (default: E:\Backup)
    .PARAMETER CleanSnapshots
        If true, removes all snapshots after export
    .PARAMETER RetentionDays
        Number of days to keep old exports
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$VMName,
        
        [string]$ExportPath = $Script:VMExportPath,
        
        [bool]$CleanSnapshots = $true,
        
        [int]$RetentionDays = $Script:VMRetentionDays
    )
    
    if (-not (Test-HyperVAvailable)) {
        Write-Host "  [VM] ERROR: Hyper-V is not available" -ForegroundColor Red
        return $false
    }
    
    # Get VMs matching the name
    $vms = Get-VM -Name $VMName -ErrorAction SilentlyContinue
    
    if (-not $vms) {
        Write-Host "  [VM] ERROR: No VMs found matching '$VMName'" -ForegroundColor Red
        return $false
    }
    
    # Create dated export folder
    $folderName = (Get-Date).ToString("dd-MM-yyyy")
    $exportFolder = Join-Path $ExportPath $folderName
    
    Write-Host "  [VM] ================================================" -ForegroundColor Cyan
    Write-Host "  [VM] Starting VM Export" -ForegroundColor Cyan
    Write-Host "  [VM] ================================================" -ForegroundColor Cyan
    Write-Host "  [VM] VMs to export: $($vms.Name -join ', ')" -ForegroundColor Gray
    Write-Host "  [VM] Export folder: $exportFolder" -ForegroundColor Gray
    
    # Create export folder
    if (-not (Test-Path $exportFolder)) {
        New-Item -ItemType Directory -Path $exportFolder -Force | Out-Null
        Write-Host "  [VM] Created export folder" -ForegroundColor Gray
    }
    
    $successCount = 0
    $errorCount = 0
    
    foreach ($vm in $vms) {
        try {
            Write-Host "  [VM] Exporting: $($vm.Name) (State: $($vm.State))..." -ForegroundColor Yellow
            
            # Export the VM
            Export-VM -Name $vm.Name -Path $exportFolder -ErrorAction Stop
            
            $successCount++
            Write-Host "  [VM]   Export complete" -ForegroundColor Green
            
            # Log to file
            if ($Script:LogFile) {
                $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
                "[$timestamp] [VM] Exported $($vm.Name) to $exportFolder" | Out-File -FilePath $Script:LogFile -Append -Encoding UTF8
            }
        }
        catch {
            $errorCount++
            Write-Host "  [VM]   ERROR: $($_.Exception.Message)" -ForegroundColor Red
        }
    }
    
    # Clean up snapshots if requested
    if ($CleanSnapshots) {
        Write-Host "  [VM] Cleaning up snapshots..." -ForegroundColor Yellow
        try {
            Get-VM -Name $VMName -ErrorAction SilentlyContinue | Get-VMSnapshot | Remove-VMSnapshot -ErrorAction SilentlyContinue
            Write-Host "  [VM] Snapshots removed" -ForegroundColor Green
        }
        catch {
            Write-Host "  [VM] WARNING: Could not remove some snapshots: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    }
    
    # Clean up old exports
    Write-Host "  [VM] Cleaning up exports older than $RetentionDays days..." -ForegroundColor Gray
    Clear-OldVMExports -ExportPath $ExportPath -RetentionDays $RetentionDays
    
    Write-Host "  [VM] ================================================" -ForegroundColor Cyan
    Write-Host "  [VM] Export complete: $successCount success, $errorCount errors" -ForegroundColor $(if ($errorCount -eq 0) { "Green" } else { "Yellow" })
    Write-Host "  [VM] ================================================" -ForegroundColor Cyan
    
    return ($errorCount -eq 0)
}

function Clear-OldVMExports {
    <#
    .SYNOPSIS
        Removes VM exports older than retention period
    #>
    param(
        [string]$ExportPath = $Script:VMExportPath,
        [int]$RetentionDays = $Script:VMRetentionDays
    )
    
    if (-not (Test-Path $ExportPath)) {
        return
    }
    
    $cutoffDate = (Get-Date).AddDays(-$RetentionDays)
    $removedCount = 0
    
    try {
        Get-ChildItem -Path $ExportPath -Directory | Where-Object {
            # Check if folder name matches date pattern (dd-MM-yyyy)
            $folderDate = $null
            [DateTime]::TryParseExact($_.Name, "dd-MM-yyyy", $null, [System.Globalization.DateTimeStyles]::None, [ref]$folderDate)
            $folderDate -and $folderDate -lt $cutoffDate
        } | ForEach-Object {
            Write-Host "  [VM] Removing old export: $($_.Name)" -ForegroundColor Gray
            Remove-Item -Path $_.FullName -Recurse -Force -ErrorAction SilentlyContinue
            $removedCount++
        }
        
        if ($removedCount -gt 0) {
            Write-Host "  [VM] Removed $removedCount old export folder(s)" -ForegroundColor Green
        }
    }
    catch {
        Write-Host "  [VM] Error during cleanup: $($_.Exception.Message)" -ForegroundColor Yellow
    }
}

function Start-VMsAfterExport {
    <#
    .SYNOPSIS
        Starts VMs in sequence after export (workflow-like behavior)
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string[]]$VMNames,
        
        [int]$DelaySeconds = 30
    )
    
    Write-Host "  [VM] Starting VMs in sequence..." -ForegroundColor Cyan
    
    foreach ($vmName in $VMNames) {
        try {
            $vm = Get-VM -Name $vmName -ErrorAction SilentlyContinue
            if ($vm) {
                if ($vm.State -ne 'Running') {
                    Write-Host "  [VM] Starting: $vmName" -ForegroundColor Yellow
                    Start-VM -Name $vmName -ErrorAction Stop
                    Write-Host "  [VM]   Started successfully" -ForegroundColor Green
                    
                    # Wait before starting next VM
                    if ($DelaySeconds -gt 0) {
                        Write-Host "  [VM]   Waiting $DelaySeconds seconds..." -ForegroundColor Gray
                        Start-Sleep -Seconds $DelaySeconds
                    }
                }
                else {
                    Write-Host "  [VM] $vmName is already running" -ForegroundColor Gray
                }
            }
            else {
                Write-Host "  [VM] VM not found: $vmName" -ForegroundColor Yellow
            }
        }
        catch {
            Write-Host "  [VM] ERROR starting $vmName : $($_.Exception.Message)" -ForegroundColor Red
        }
    }
    
    Write-Host "  [VM] VM startup sequence complete" -ForegroundColor Green
}

function Show-VMExportDialog {
    <#
    .SYNOPSIS
        Shows dialog for selecting VMs to export
    #>
    
    if (-not (Test-HyperVAvailable)) {
        [System.Windows.Forms.MessageBox]::Show(
            "Hyper-V is not available on this system.`n`nInstall Hyper-V to use VM Export features.",
            "Hyper-V Required",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Warning
        )
        return $null
    }
    
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $vms = Get-VMList
    
    if ($vms.Count -eq 0) {
        [System.Windows.Forms.MessageBox]::Show(
            "No virtual machines found on this system.",
            "No VMs",
            [System.Windows.Forms.MessageBoxButtons]::OK,
            [System.Windows.Forms.MessageBoxIcon]::Information
        )
        return $null
    }
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Export Hyper-V Virtual Machines"
    $form.Size = New-Object System.Drawing.Size(550, 500)
    $form.StartPosition = "CenterParent"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    
    # VM List
    $lblVMs = New-Object System.Windows.Forms.Label
    $lblVMs.Text = "Select VMs to export:"
    $lblVMs.Location = New-Object System.Drawing.Point(20, 20)
    $lblVMs.Size = New-Object System.Drawing.Size(200, 20)
    $form.Controls.Add($lblVMs)
    
    $lstVMs = New-Object System.Windows.Forms.CheckedListBox
    $lstVMs.Location = New-Object System.Drawing.Point(20, 45)
    $lstVMs.Size = New-Object System.Drawing.Size(490, 180)
    $lstVMs.CheckOnClick = $true
    
    foreach ($vm in $vms) {
        $displayText = "$($vm.Name) - $($vm.State) ($($vm.SizeGB)GB)"
        $lstVMs.Items.Add($displayText, $false)
    }
    
    $form.Controls.Add($lstVMs)
    
    # Export Path
    $lblPath = New-Object System.Windows.Forms.Label
    $lblPath.Text = "Export to:"
    $lblPath.Location = New-Object System.Drawing.Point(20, 240)
    $lblPath.Size = New-Object System.Drawing.Size(70, 20)
    $form.Controls.Add($lblPath)
    
    $txtPath = New-Object System.Windows.Forms.TextBox
    $txtPath.Text = $Script:VMExportPath
    $txtPath.Location = New-Object System.Drawing.Point(95, 237)
    $txtPath.Size = New-Object System.Drawing.Size(330, 22)
    $form.Controls.Add($txtPath)
    
    $btnBrowse = New-Object System.Windows.Forms.Button
    $btnBrowse.Text = "Browse..."
    $btnBrowse.Location = New-Object System.Drawing.Point(430, 235)
    $btnBrowse.Size = New-Object System.Drawing.Size(80, 25)
    $btnBrowse.Add_Click({
        $folderBrowser = New-Object System.Windows.Forms.FolderBrowserDialog
        $folderBrowser.SelectedPath = $txtPath.Text
        if ($folderBrowser.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $txtPath.Text = $folderBrowser.SelectedPath
        }
    })
    $form.Controls.Add($btnBrowse)
    
    # Options
    $grpOptions = New-Object System.Windows.Forms.GroupBox
    $grpOptions.Text = "Options"
    $grpOptions.Location = New-Object System.Drawing.Point(20, 275)
    $grpOptions.Size = New-Object System.Drawing.Size(490, 100)
    $form.Controls.Add($grpOptions)
    
    $chkCleanSnapshots = New-Object System.Windows.Forms.CheckBox
    $chkCleanSnapshots.Text = "Remove snapshots after export (recommended)"
    $chkCleanSnapshots.Location = New-Object System.Drawing.Point(15, 25)
    $chkCleanSnapshots.Size = New-Object System.Drawing.Size(350, 22)
    $chkCleanSnapshots.Checked = $true
    $grpOptions.Controls.Add($chkCleanSnapshots)
    
    $lblRetention = New-Object System.Windows.Forms.Label
    $lblRetention.Text = "Keep exports for:"
    $lblRetention.Location = New-Object System.Drawing.Point(15, 55)
    $lblRetention.Size = New-Object System.Drawing.Size(100, 20)
    $grpOptions.Controls.Add($lblRetention)
    
    $numRetention = New-Object System.Windows.Forms.NumericUpDown
    $numRetention.Location = New-Object System.Drawing.Point(120, 53)
    $numRetention.Size = New-Object System.Drawing.Size(60, 22)
    $numRetention.Minimum = 1
    $numRetention.Maximum = 30
    $numRetention.Value = $Script:VMRetentionDays
    $grpOptions.Controls.Add($numRetention)
    
    $lblDays = New-Object System.Windows.Forms.Label
    $lblDays.Text = "days"
    $lblDays.Location = New-Object System.Drawing.Point(185, 55)
    $lblDays.Size = New-Object System.Drawing.Size(40, 20)
    $grpOptions.Controls.Add($lblDays)
    
    $chkStartAfter = New-Object System.Windows.Forms.CheckBox
    $chkStartAfter.Text = "Start VMs after export (if they were running)"
    $chkStartAfter.Location = New-Object System.Drawing.Point(15, 75)
    $chkStartAfter.Size = New-Object System.Drawing.Size(350, 22)
    $chkStartAfter.Checked = $false
    $grpOptions.Controls.Add($chkStartAfter)
    
    # Buttons
    $btnExport = New-Object System.Windows.Forms.Button
    $btnExport.Text = "Export Selected"
    $btnExport.Location = New-Object System.Drawing.Point(320, 420)
    $btnExport.Size = New-Object System.Drawing.Size(100, 30)
    $btnExport.Add_Click({
        $selectedVMs = @()
        for ($i = 0; $i -lt $lstVMs.CheckedItems.Count; $i++) {
            $vmName = $lstVMs.CheckedItems[$i].Split(' - ')[0]
            $selectedVMs += $vmName
        }
        
        if ($selectedVMs.Count -eq 0) {
            [System.Windows.Forms.MessageBox]::Show("Please select at least one VM to export.", "No VMs Selected",
                [System.Windows.Forms.MessageBoxButtons]::OK, [System.Windows.Forms.MessageBoxIcon]::Warning)
            return
        }
        
        $form.Tag = @{
            VMs = $selectedVMs
            ExportPath = $txtPath.Text
            CleanSnapshots = $chkCleanSnapshots.Checked
            RetentionDays = [int]$numRetention.Value
            StartAfter = $chkStartAfter.Checked
        }
        $form.DialogResult = [System.Windows.Forms.DialogResult]::OK
        $form.Close()
    })
    $form.Controls.Add($btnExport)
    
    $btnCancel = New-Object System.Windows.Forms.Button
    $btnCancel.Text = "Cancel"
    $btnCancel.Location = New-Object System.Drawing.Point(430, 420)
    $btnCancel.Size = New-Object System.Drawing.Size(80, 30)
    $btnCancel.DialogResult = [System.Windows.Forms.DialogResult]::Cancel
    $form.Controls.Add($btnCancel)
    
    $result = $form.ShowDialog()
    
    if ($result -eq [System.Windows.Forms.DialogResult]::OK -and $form.Tag) {
        return $form.Tag
    }
    
    return $null
}

function Invoke-VMExport {
    <#
    .SYNOPSIS
        Performs VM export based on dialog settings
    #>
    param(
        [hashtable]$Settings
    )
    
    if (-not $Settings) {
        return $false
    }
    
    $Script:VMExportPath = $Settings.ExportPath
    $Script:VMRetentionDays = $Settings.RetentionDays
    
    $runningVMs = @()
    
    foreach ($vmName in $Settings.VMs) {
        $vm = Get-VM -Name $vmName -ErrorAction SilentlyContinue
        if ($vm -and $vm.State -eq 'Running') {
            $runningVMs += $vmName
        }
        
        $result = Export-VMWithCleanup -VMName $vmName -ExportPath $Settings.ExportPath `
            -CleanSnapshots $Settings.CleanSnapshots -RetentionDays $Settings.RetentionDays
    }
    
    # Start VMs if requested
    if ($Settings.StartAfter -and $runningVMs.Count -gt 0) {
        Start-VMsAfterExport -VMNames $runningVMs -DelaySeconds 30
    }
    
    return $true
}
