<#
.SYNOPSIS
    ABC&D Systems - Windows Update Management Module
.DESCRIPTION
    Comprehensive Windows Update management including:
    - Check for updates
    - Install updates with reboot handling
    - Autologon for multi-reboot update cycles
    - Scheduled update tasks
.VERSION
    4.0
#>

# =============================================================================
# CONFIGURATION
# =============================================================================

$Script:WUConfig = @{
    ToolsPath = "$PSScriptRoot\..\Tools"
    LogPath = "C:\ABCD-Migration\Logs"
    StatePath = "C:\ABCD-Migration\Settings\WUUpdateState.json"
    MaxRebootCycles = 5  # Prevent infinite reboot loops
}

# =============================================================================
# AUTOLOGON FUNCTIONS (shared with QuickBooks module)
# =============================================================================

function Set-WUAutoLogon {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Username,
        [Parameter(Mandatory=$true)]
        [string]$Password,
        [string]$Domain = $env:COMPUTERNAME
    )
    
    $autologonPath = Join-Path $Script:WUConfig.ToolsPath "Autologon64.exe"
    
    if (!(Test-Path $autologonPath)) {
        Write-Host "  [AutoLogon] ERROR: Autologon64.exe not found at $autologonPath" -ForegroundColor Red
        return $false
    }
    
    try {
        $process = Start-Process -FilePath $autologonPath -ArgumentList "`"$Username`" `"$Domain`" `"$Password`" /accepteula" -Wait -PassThru -NoNewWindow
        
        if ($process.ExitCode -eq 0) {
            Write-Host "  [AutoLogon] Configured for: $Domain\$Username" -ForegroundColor Green
            return $true
        } else {
            Write-Host "  [AutoLogon] Failed with exit code: $($process.ExitCode)" -ForegroundColor Red
            return $false
        }
    } catch {
        Write-Host "  [AutoLogon] ERROR: $_" -ForegroundColor Red
        return $false
    }
}

function Clear-WUAutoLogon {
    $autologonPath = Join-Path $Script:WUConfig.ToolsPath "Autologon64.exe"
    
    if (Test-Path $autologonPath) {
        Start-Process -FilePath $autologonPath -ArgumentList "/d" -Wait -NoNewWindow -ErrorAction SilentlyContinue
    }
    
    # Also clear via registry as backup
    try {
        $regPath = "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\Winlogon"
        Set-ItemProperty -Path $regPath -Name "AutoAdminLogon" -Value "0" -ErrorAction SilentlyContinue
        Remove-ItemProperty -Path $regPath -Name "DefaultPassword" -ErrorAction SilentlyContinue
    } catch {}
    
    Write-Host "  [AutoLogon] Cleared" -ForegroundColor Green
}

# =============================================================================
# STATE MANAGEMENT
# =============================================================================

function Save-WUState {
    param(
        [string]$Phase,
        [int]$RebootCount = 0,
        [string]$AdminUser,
        [string]$EncryptedPassword,
        [hashtable]$AdditionalData = @{}
    )
    
    $state = @{
        Phase = $Phase
        RebootCount = $RebootCount
        AdminUser = $AdminUser
        EncryptedPassword = $EncryptedPassword
        SavedAt = (Get-Date).ToString("o")
        Data = $AdditionalData
    }
    
    $stateDir = Split-Path $Script:WUConfig.StatePath -Parent
    if (!(Test-Path $stateDir)) {
        New-Item -Path $stateDir -ItemType Directory -Force | Out-Null
    }
    
    $state | ConvertTo-Json -Depth 5 | Out-File $Script:WUConfig.StatePath -Encoding UTF8
}

function Get-WUState {
    if (Test-Path $Script:WUConfig.StatePath) {
        try {
            return Get-Content $Script:WUConfig.StatePath -Raw | ConvertFrom-Json
        } catch { return $null }
    }
    return $null
}

function Clear-WUState {
    if (Test-Path $Script:WUConfig.StatePath) {
        Remove-Item $Script:WUConfig.StatePath -Force -ErrorAction SilentlyContinue
    }
}

# =============================================================================
# WINDOWS UPDATE FUNCTIONS
# =============================================================================

function Get-WindowsUpdateStatus {
    <#
    .SYNOPSIS
        Gets current Windows Update status
    #>
    
    $result = @{
        PendingUpdates = @()
        RebootRequired = $false
        LastCheck = $null
        ServiceStatus = "Unknown"
    }
    
    try {
        # Check Windows Update service
        $wuService = Get-Service -Name "wuauserv" -ErrorAction SilentlyContinue
        $result.ServiceStatus = if ($wuService) { $wuService.Status.ToString() } else { "Not Found" }
        
        # Check if reboot is required
        $rebootKeys = @(
            "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\RebootRequired",
            "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\RebootPending"
        )
        
        foreach ($key in $rebootKeys) {
            if (Test-Path $key) {
                $result.RebootRequired = $true
                break
            }
        }
        
        # Get pending updates using COM object
        try {
            $session = New-Object -ComObject Microsoft.Update.Session
            $searcher = $session.CreateUpdateSearcher()
            $searchResult = $searcher.Search("IsInstalled=0 and Type='Software'")
            
            foreach ($update in $searchResult.Updates) {
                $result.PendingUpdates += @{
                    Title = $update.Title
                    KB = ($update.KBArticleIDs | Select-Object -First 1)
                    Size = [math]::Round($update.MaxDownloadSize / 1MB, 2)
                    IsMandatory = $update.IsMandatory
                    RebootRequired = $update.RebootRequired
                }
            }
        } catch {
            Write-Host "  [WU] Could not query updates: $_" -ForegroundColor Yellow
        }
        
    } catch {
        Write-Host "  [WU] Status check error: $_" -ForegroundColor Red
    }
    
    return $result
}

function Install-WindowsUpdates {
    <#
    .SYNOPSIS
        Downloads and installs pending Windows Updates
    #>
    param(
        [switch]$AcceptAll,
        [switch]$IncludeOptional
    )
    
    Write-Host "  [WU] Checking for updates..." -ForegroundColor Cyan
    
    try {
        $session = New-Object -ComObject Microsoft.Update.Session
        $searcher = $session.CreateUpdateSearcher()
        
        $searchCriteria = "IsInstalled=0 and Type='Software'"
        if (-not $IncludeOptional) {
            $searchCriteria += " and IsHidden=0"
        }
        
        $searchResult = $searcher.Search($searchCriteria)
        
        if ($searchResult.Updates.Count -eq 0) {
            Write-Host "  [WU] No updates available" -ForegroundColor Green
            return @{ Success = $true; UpdatesInstalled = 0; RebootRequired = $false }
        }
        
        Write-Host "  [WU] Found $($searchResult.Updates.Count) update(s)" -ForegroundColor Yellow
        
        # Create update collection
        $updatesToDownload = New-Object -ComObject Microsoft.Update.UpdateColl
        $updatesToInstall = New-Object -ComObject Microsoft.Update.UpdateColl
        
        foreach ($update in $searchResult.Updates) {
            Write-Host "  [WU]   - $($update.Title)" -ForegroundColor Gray
            
            if ($AcceptAll -or (-not $update.EulaAccepted)) {
                $update.AcceptEula()
            }
            
            $updatesToDownload.Add($update) | Out-Null
            $updatesToInstall.Add($update) | Out-Null
        }
        
        # Download updates
        Write-Host "  [WU] Downloading updates..." -ForegroundColor Yellow
        $downloader = $session.CreateUpdateDownloader()
        $downloader.Updates = $updatesToDownload
        $downloadResult = $downloader.Download()
        
        if ($downloadResult.ResultCode -ne 2) {  # 2 = Succeeded
            Write-Host "  [WU] Download failed with code: $($downloadResult.ResultCode)" -ForegroundColor Red
            return @{ Success = $false; Error = "Download failed" }
        }
        
        # Install updates
        Write-Host "  [WU] Installing updates..." -ForegroundColor Yellow
        $installer = $session.CreateUpdateInstaller()
        $installer.Updates = $updatesToInstall
        $installResult = $installer.Install()
        
        $rebootRequired = $installResult.RebootRequired
        $installedCount = 0
        
        for ($i = 0; $i -lt $updatesToInstall.Count; $i++) {
            $updateResult = $installResult.GetUpdateResult($i)
            if ($updateResult.ResultCode -eq 2) {
                $installedCount++
                Write-Host "  [WU]   Installed: $($updatesToInstall.Item($i).Title)" -ForegroundColor Green
            } else {
                Write-Host "  [WU]   Failed: $($updatesToInstall.Item($i).Title) (Code: $($updateResult.ResultCode))" -ForegroundColor Red
            }
        }
        
        Write-Host "  [WU] Installed $installedCount of $($updatesToInstall.Count) update(s)" -ForegroundColor Cyan
        
        return @{
            Success = $true
            UpdatesInstalled = $installedCount
            RebootRequired = $rebootRequired
        }
        
    } catch {
        Write-Host "  [WU] Installation error: $_" -ForegroundColor Red
        return @{ Success = $false; Error = $_.Exception.Message }
    }
}

function Test-WURebootRequired {
    $rebootKeys = @(
        "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\RebootRequired",
        "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\RebootPending",
        "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\RebootInProgress"
    )
    
    foreach ($key in $rebootKeys) {
        if (Test-Path $key) { return $true }
    }
    
    # Check pending file rename operations
    try {
        $pending = Get-ItemProperty "HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager" -Name PendingFileRenameOperations -ErrorAction SilentlyContinue
        if ($pending.PendingFileRenameOperations) { return $true }
    } catch {}
    
    return $false
}

# =============================================================================
# FULL UPDATE WORKFLOW WITH REBOOT HANDLING
# =============================================================================

function Invoke-FullWindowsUpdate {
    <#
    .SYNOPSIS
        Complete Windows Update workflow with automatic reboot handling
    #>
    param(
        [Parameter(Mandatory=$true)]
        [string]$AdminUsername,
        
        [Parameter(Mandatory=$true)]
        [string]$AdminPassword,
        
        [switch]$IncludeOptional,
        [int]$MaxReboots = 5
    )
    
    $logFile = Join-Path $Script:WUConfig.LogPath "WUFullUpdate_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
    Start-Transcript -Path $logFile -ErrorAction SilentlyContinue
    
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host "ABCD Systems - Windows Update" -ForegroundColor Cyan
    Write-Host "========================================" -ForegroundColor Cyan
    
    # Check existing state (continuation after reboot)
    $state = Get-WUState
    $rebootCount = 0
    
    if ($state -and $state.Phase -eq "Rebooting") {
        $rebootCount = $state.RebootCount
        Write-Host "  [WU] Continuing after reboot ($rebootCount of $MaxReboots max)" -ForegroundColor Yellow
        
        # Clear autologon immediately
        Clear-WUAutoLogon
        
        if ($rebootCount -ge $MaxReboots) {
            Write-Host "  [WU] Maximum reboot cycles reached. Stopping." -ForegroundColor Red
            Clear-WUState
            Stop-Transcript -ErrorAction SilentlyContinue
            return @{ Success = $false; Error = "Max reboots reached" }
        }
    }
    
    # Install updates
    $result = Install-WindowsUpdates -AcceptAll -IncludeOptional:$IncludeOptional
    
    if (-not $result.Success) {
        Clear-WUState
        Stop-Transcript -ErrorAction SilentlyContinue
        return $result
    }
    
    # Check if reboot is needed
    if ($result.RebootRequired -or (Test-WURebootRequired)) {
        Write-Host "  [WU] Reboot required to complete updates" -ForegroundColor Yellow
        
        # Check for more pending updates (will need another cycle)
        $status = Get-WindowsUpdateStatus
        $morePending = $status.PendingUpdates.Count -gt 0
        
        if ($morePending -or $result.UpdatesInstalled -gt 0) {
            # Save state for continuation
            $encPassword = $AdminPassword | ConvertTo-SecureString -AsPlainText -Force | ConvertFrom-SecureString
            
            Save-WUState -Phase "Rebooting" -RebootCount ($rebootCount + 1) -AdminUser $AdminUsername -EncryptedPassword $encPassword
            
            # Set autologon
            Set-WUAutoLogon -Username $AdminUsername -Password $AdminPassword
            
            # Create continuation task
            Register-WUContinuationTask
            
            Write-Host "  [WU] Rebooting in 60 seconds..." -ForegroundColor Yellow
            Write-Host "  [WU] Updates will continue automatically after restart" -ForegroundColor Yellow
            
            Stop-Transcript -ErrorAction SilentlyContinue
            
            shutdown /r /t 60 /c "Windows Update requires restart. Updates will continue automatically."
            
            return @{ Success = $true; Status = "Rebooting"; RebootCount = $rebootCount + 1 }
        }
    }
    
    # All done!
    Write-Host "" -ForegroundColor Cyan
    Write-Host "========================================" -ForegroundColor Green
    Write-Host "Windows Update Complete!" -ForegroundColor Green
    Write-Host "Total reboot cycles: $rebootCount" -ForegroundColor Green
    Write-Host "========================================" -ForegroundColor Green
    
    Clear-WUState
    Clear-WUAutoLogon
    
    Stop-Transcript -ErrorAction SilentlyContinue
    return @{ Success = $true; Status = "Complete"; TotalReboots = $rebootCount }
}

function Register-WUContinuationTask {
    $taskName = "ABCD-WUContinuation"
    
    $scriptPath = "C:\ABCD-Migration\Scripts\WUContinuation.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    if (!(Test-Path $scriptDir)) {
        New-Item -Path $scriptDir -ItemType Directory -Force | Out-Null
    }
    
    $scriptContent = @'
# ABCD Windows Update Continuation
$ErrorActionPreference = 'Continue'
Start-Sleep -Seconds 90  # Wait for system to stabilize

$logFile = "C:\ABCD-Migration\Logs\WUContinuation_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
Start-Transcript -Path $logFile

# Import module
$modulePath = "C:\ABCD-Migration\Modules\WindowsUpdate-Management.ps1"
if (Test-Path $modulePath) {
    . $modulePath
}

# Get saved state
$state = Get-WUState

if ($state) {
    Write-Host "Continuing Windows Update (Reboot $($state.RebootCount))"
    
    # Decrypt password
    try {
        $securePass = $state.EncryptedPassword | ConvertTo-SecureString
        $cred = New-Object System.Management.Automation.PSCredential($state.AdminUser, $securePass)
        $password = $cred.GetNetworkCredential().Password
        
        # Continue update
        Invoke-FullWindowsUpdate -AdminUsername $state.AdminUser -AdminPassword $password
    } catch {
        Write-Host "Error: $_"
        Clear-WUAutoLogon
        Clear-WUState
    }
}

# Remove this task
Unregister-ScheduledTask -TaskName "ABCD-WUContinuation" -Confirm:$false -ErrorAction SilentlyContinue

Stop-Transcript
'@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8
    
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    $trigger = New-ScheduledTaskTrigger -AtLogOn
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries
    
    Register-ScheduledTask -TaskName $taskName -Trigger $trigger -Action $action -Principal $principal -Settings $settings -Description "ABCD Windows Update Continuation"
    
    Write-Host "  [Task] Created continuation task" -ForegroundColor Green
}

# =============================================================================
# SCHEDULED TASK FOR WINDOWS UPDATES
# =============================================================================

function Register-WindowsUpdateTask {
    param(
        [string]$Time = "03:00",
        [string]$AdminUsername,
        [string]$AdminPassword,
        [ValidateSet("Daily", "Weekly")]
        [string]$Frequency = "Weekly",
        [int]$DayOfWeek = 0  # 0 = Sunday
    )
    
    $taskName = "ABCD-WindowsUpdate"
    
    $scriptPath = "C:\ABCD-Migration\Scripts\WUScheduled.ps1"
    $scriptDir = Split-Path $scriptPath -Parent
    if (!(Test-Path $scriptDir)) {
        New-Item -Path $scriptDir -ItemType Directory -Force | Out-Null
    }
    
    $encPassword = $AdminPassword | ConvertTo-SecureString -AsPlainText -Force | ConvertFrom-SecureString
    
    $scriptContent = @"
# ABCD Scheduled Windows Update
`$ErrorActionPreference = 'Continue'
`$logFile = "C:\ABCD-Migration\Logs\WUScheduled_`$(Get-Date -Format 'yyyyMMdd_HHmmss').log"
Start-Transcript -Path `$logFile

# Import modules
. "C:\ABCD-Migration\Modules\WindowsUpdate-Management.ps1"
. "C:\ABCD-Migration\Modules\Service-Management.ps1"

`$encPassword = '$encPassword'
`$securePass = `$encPassword | ConvertTo-SecureString
`$cred = New-Object System.Management.Automation.PSCredential('$AdminUsername', `$securePass)
`$password = `$cred.GetNetworkCredential().Password

# Step 1: Stop ALL services (aggressive - like backup mode)
Write-Host "[WU Scheduled] Stopping all non-essential services..." -ForegroundColor Yellow
`$stoppedServices = Stop-AllServicesForBackup
Write-Host "[WU Scheduled] Stopped `$(`$stoppedServices.Services.Count) services" -ForegroundColor Green

# Step 2: Run Windows Updates (will reboot if needed)
Write-Host "[WU Scheduled] Running Windows Updates..." -ForegroundColor Cyan
`$result = Invoke-FullWindowsUpdate -AdminUsername '$AdminUsername' -AdminPassword `$password -MaxReboots 5

# Step 3: Restart services if no reboot pending
if (`$result.Status -ne "Rebooting") {
    Write-Host "[WU Scheduled] Restarting services..." -ForegroundColor Yellow
    Start-ServicesAfterBackup -StoppedServices `$stoppedServices.Services
    Write-Host "[WU Scheduled] Services restarted" -ForegroundColor Green
}

Stop-Transcript
"@
    
    $scriptContent | Out-File $scriptPath -Encoding UTF8
    
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    
    $timeParts = $Time.Split(':')
    $triggerTime = "$($timeParts[0]):$($timeParts[1])"
    
    switch ($Frequency) {
        "Daily" {
            $trigger = New-ScheduledTaskTrigger -Daily -At $triggerTime
        }
        "Weekly" {
            $daysOfWeek = @("Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday")
            $trigger = New-ScheduledTaskTrigger -Weekly -DaysOfWeek $daysOfWeek[$DayOfWeek] -At $triggerTime
        }
    }
    
    $action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument "-ExecutionPolicy Bypass -WindowStyle Hidden -File `"$scriptPath`""
    $principal = New-ScheduledTaskPrincipal -UserId "SYSTEM" -LogonType ServiceAccount -RunLevel Highest
    $settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -WakeToRun -ExecutionTimeLimit (New-TimeSpan -Hours 6)
    
    Register-ScheduledTask -TaskName $taskName -Trigger $trigger -Action $action -Principal $principal -Settings $settings -Description "ABCD Systems - Windows Update"
    
    Write-Host "  [Task] Created: $taskName ($Frequency at $Time)" -ForegroundColor Green
    return $true
}

# =============================================================================
# GUI DIALOG
# =============================================================================

function Show-WindowsUpdateDialog {
    Add-Type -AssemblyName System.Windows.Forms
    Add-Type -AssemblyName System.Drawing
    
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "Windows Update Management - ABCD Systems"
    $form.Size = New-Object System.Drawing.Size(500, 420)
    $form.StartPosition = "CenterScreen"
    $form.FormBorderStyle = "FixedDialog"
    $form.MaximizeBox = $false
    $form.BackColor = [System.Drawing.Color]::FromArgb(240, 240, 240)
    
    # =========================================================================
    # STATUS SECTION
    # =========================================================================
    
    $grpStatus = New-Object System.Windows.Forms.GroupBox
    $grpStatus.Text = "Current Status"
    $grpStatus.Location = New-Object System.Drawing.Point(10, 10)
    $grpStatus.Size = New-Object System.Drawing.Size(460, 100)
    $form.Controls.Add($grpStatus)
    
    $lblStatusText = New-Object System.Windows.Forms.Label
    $lblStatusText.Text = "Checking..."
    $lblStatusText.Location = New-Object System.Drawing.Point(10, 25)
    $lblStatusText.Size = New-Object System.Drawing.Size(440, 60)
    $grpStatus.Controls.Add($lblStatusText)
    
    $btnRefresh = New-Object System.Windows.Forms.Button
    $btnRefresh.Text = "Refresh"
    $btnRefresh.Location = New-Object System.Drawing.Point(380, 70)
    $btnRefresh.Size = New-Object System.Drawing.Size(70, 23)
    $btnRefresh.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpStatus.Controls.Add($btnRefresh)
    
    # =========================================================================
    # MANUAL UPDATE SECTION
    # =========================================================================
    
    $grpManual = New-Object System.Windows.Forms.GroupBox
    $grpManual.Text = "Manual Update"
    $grpManual.Location = New-Object System.Drawing.Point(10, 120)
    $grpManual.Size = New-Object System.Drawing.Size(460, 80)
    $form.Controls.Add($grpManual)
    
    $lblManualInfo = New-Object System.Windows.Forms.Label
    $lblManualInfo.Text = "Run updates now. System will reboot automatically if needed and continue updating."
    $lblManualInfo.Location = New-Object System.Drawing.Point(10, 20)
    $lblManualInfo.Size = New-Object System.Drawing.Size(440, 20)
    $grpManual.Controls.Add($lblManualInfo)
    
    $btnRunUpdate = New-Object System.Windows.Forms.Button
    $btnRunUpdate.Text = "Run Full Update Now"
    $btnRunUpdate.Location = New-Object System.Drawing.Point(10, 45)
    $btnRunUpdate.Size = New-Object System.Drawing.Size(140, 28)
    $btnRunUpdate.BackColor = [System.Drawing.Color]::FromArgb(0, 120, 215)
    $btnRunUpdate.ForeColor = [System.Drawing.Color]::Black
    $btnRunUpdate.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpManual.Controls.Add($btnRunUpdate)
    
    $chkOptional = New-Object System.Windows.Forms.CheckBox
    $chkOptional.Text = "Include optional updates"
    $chkOptional.Location = New-Object System.Drawing.Point(170, 50)
    $chkOptional.Size = New-Object System.Drawing.Size(180, 20)
    $grpManual.Controls.Add($chkOptional)
    
    # =========================================================================
    # SCHEDULED UPDATE SECTION
    # =========================================================================
    
    $grpSchedule = New-Object System.Windows.Forms.GroupBox
    $grpSchedule.Text = "Scheduled Updates"
    $grpSchedule.Location = New-Object System.Drawing.Point(10, 210)
    $grpSchedule.Size = New-Object System.Drawing.Size(460, 130)
    $form.Controls.Add($grpSchedule)
    
    $chkEnableSchedule = New-Object System.Windows.Forms.CheckBox
    $chkEnableSchedule.Text = "Enable scheduled Windows Updates"
    $chkEnableSchedule.Location = New-Object System.Drawing.Point(10, 22)
    $chkEnableSchedule.Size = New-Object System.Drawing.Size(250, 20)
    $grpSchedule.Controls.Add($chkEnableSchedule)
    
    $lblFreq = New-Object System.Windows.Forms.Label
    $lblFreq.Text = "Frequency:"
    $lblFreq.Location = New-Object System.Drawing.Point(10, 50)
    $grpSchedule.Controls.Add($lblFreq)
    
    $cboFrequency = New-Object System.Windows.Forms.ComboBox
    $cboFrequency.Location = New-Object System.Drawing.Point(80, 47)
    $cboFrequency.Size = New-Object System.Drawing.Size(100, 22)
    $cboFrequency.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    $cboFrequency.Items.AddRange(@("Weekly", "Daily"))
    $cboFrequency.SelectedIndex = 0
    $grpSchedule.Controls.Add($cboFrequency)
    
    $lblTime = New-Object System.Windows.Forms.Label
    $lblTime.Text = "Time:"
    $lblTime.Location = New-Object System.Drawing.Point(200, 50)
    $grpSchedule.Controls.Add($lblTime)
    
    $cboHour = New-Object System.Windows.Forms.ComboBox
    $cboHour.Location = New-Object System.Drawing.Point(240, 47)
    $cboHour.Size = New-Object System.Drawing.Size(50, 22)
    $cboHour.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    0..23 | ForEach-Object { $cboHour.Items.Add($_.ToString("00")) }
    $cboHour.SelectedItem = "03"
    $grpSchedule.Controls.Add($cboHour)
    
    $cboMinute = New-Object System.Windows.Forms.ComboBox
    $cboMinute.Location = New-Object System.Drawing.Point(295, 47)
    $cboMinute.Size = New-Object System.Drawing.Size(50, 22)
    $cboMinute.DropDownStyle = [System.Windows.Forms.ComboBoxStyle]::DropDownList
    @("00", "15", "30", "45") | ForEach-Object { $cboMinute.Items.Add($_) }
    $cboMinute.SelectedItem = "00"
    $grpSchedule.Controls.Add($cboMinute)
    
    $lblAdmin = New-Object System.Windows.Forms.Label
    $lblAdmin.Text = "Admin:"
    $lblAdmin.Location = New-Object System.Drawing.Point(10, 80)
    $grpSchedule.Controls.Add($lblAdmin)
    
    $txtAdminUser = New-Object System.Windows.Forms.TextBox
    $txtAdminUser.Location = New-Object System.Drawing.Point(60, 77)
    $txtAdminUser.Size = New-Object System.Drawing.Size(120, 22)
    $txtAdminUser.Text = $env:USERNAME
    $grpSchedule.Controls.Add($txtAdminUser)
    
    $lblPass = New-Object System.Windows.Forms.Label
    $lblPass.Text = "Password:"
    $lblPass.Location = New-Object System.Drawing.Point(190, 80)
    $grpSchedule.Controls.Add($lblPass)
    
    $txtAdminPass = New-Object System.Windows.Forms.TextBox
    $txtAdminPass.Location = New-Object System.Drawing.Point(255, 77)
    $txtAdminPass.Size = New-Object System.Drawing.Size(120, 22)
    $txtAdminPass.UseSystemPasswordChar = $true
    $grpSchedule.Controls.Add($txtAdminPass)
    
    $btnSave = New-Object System.Windows.Forms.Button
    $btnSave.Text = "Save"
    $btnSave.Location = New-Object System.Drawing.Point(385, 75)
    $btnSave.Size = New-Object System.Drawing.Size(65, 25)
    $btnSave.BackColor = [System.Drawing.Color]::LightGreen
    $btnSave.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $grpSchedule.Controls.Add($btnSave)
    
    # =========================================================================
    # CLOSE BUTTON
    # =========================================================================
    
    $btnClose = New-Object System.Windows.Forms.Button
    $btnClose.Text = "Close"
    $btnClose.Location = New-Object System.Drawing.Point(380, 350)
    $btnClose.Size = New-Object System.Drawing.Size(90, 30)
    $btnClose.FlatStyle = [System.Windows.Forms.FlatStyle]::Flat
    $btnClose.Add_Click({ $form.Close() })
    $form.Controls.Add($btnClose)
    
    # =========================================================================
    # EVENT HANDLERS
    # =========================================================================
    
    $refreshStatus = {
        $lblStatusText.Text = "Checking for updates..."
        $form.Refresh()
        
        $status = Get-WindowsUpdateStatus
        
        $text = "Service: $($status.ServiceStatus)`n"
        $text += "Pending Updates: $($status.PendingUpdates.Count)`n"
        $text += "Reboot Required: $(if ($status.RebootRequired) { 'YES' } else { 'No' })"
        
        if ($status.PendingUpdates.Count -gt 0) {
            $text += "`nUpdates: " + (($status.PendingUpdates | Select-Object -First 3 | ForEach-Object { $_.Title }) -join ", ")
        }
        
        $lblStatusText.Text = $text
    }
    
    $btnRefresh.Add_Click($refreshStatus)
    
    $btnRunUpdate.Add_Click({
        $credForm = New-Object System.Windows.Forms.Form
        $credForm.Text = "Admin Credentials for Auto-Reboot"
        $credForm.Size = New-Object System.Drawing.Size(350, 180)
        $credForm.StartPosition = "CenterParent"
        $credForm.FormBorderStyle = "FixedDialog"
        
        $lblU = New-Object System.Windows.Forms.Label
        $lblU.Text = "Username:"
        $lblU.Location = New-Object System.Drawing.Point(20, 20)
        $credForm.Controls.Add($lblU)
        
        $txtU = New-Object System.Windows.Forms.TextBox
        $txtU.Location = New-Object System.Drawing.Point(100, 17)
        $txtU.Size = New-Object System.Drawing.Size(200, 22)
        $txtU.Text = $env:USERNAME
        $credForm.Controls.Add($txtU)
        
        $lblP = New-Object System.Windows.Forms.Label
        $lblP.Text = "Password:"
        $lblP.Location = New-Object System.Drawing.Point(20, 55)
        $credForm.Controls.Add($lblP)
        
        $txtP = New-Object System.Windows.Forms.TextBox
        $txtP.Location = New-Object System.Drawing.Point(100, 52)
        $txtP.Size = New-Object System.Drawing.Size(200, 22)
        $txtP.UseSystemPasswordChar = $true
        $credForm.Controls.Add($txtP)
        
        $btnOK = New-Object System.Windows.Forms.Button
        $btnOK.Text = "Start Update"
        $btnOK.Location = New-Object System.Drawing.Point(100, 95)
        $btnOK.Size = New-Object System.Drawing.Size(100, 30)
        $btnOK.DialogResult = [System.Windows.Forms.DialogResult]::OK
        $credForm.Controls.Add($btnOK)
        
        if ($credForm.ShowDialog() -eq [System.Windows.Forms.DialogResult]::OK) {
            $form.Hide()
            Invoke-FullWindowsUpdate -AdminUsername $txtU.Text -AdminPassword $txtP.Text -IncludeOptional:$chkOptional.Checked
            $form.Close()
        }
    })
    
    $btnSave.Add_Click({
        if ($chkEnableSchedule.Checked) {
            if ([string]::IsNullOrEmpty($txtAdminPass.Text)) {
                [System.Windows.Forms.MessageBox]::Show("Password required for auto-reboot.", "Error")
                return
            }
            
            Register-WindowsUpdateTask `
                -Time "$($cboHour.SelectedItem):$($cboMinute.SelectedItem)" `
                -AdminUsername $txtAdminUser.Text `
                -AdminPassword $txtAdminPass.Text `
                -Frequency $cboFrequency.SelectedItem
            
            [System.Windows.Forms.MessageBox]::Show(
                "Windows Update scheduled!`n`nFrequency: $($cboFrequency.SelectedItem)`nTime: $($cboHour.SelectedItem):$($cboMinute.SelectedItem)",
                "Saved"
            )
        } else {
            Unregister-ScheduledTask -TaskName "ABCD-WindowsUpdate" -Confirm:$false -ErrorAction SilentlyContinue
            [System.Windows.Forms.MessageBox]::Show("Schedule removed.", "Removed")
        }
    })
    
    # Initial status check
    $form.Add_Shown({ & $refreshStatus })
    
    $form.ShowDialog() | Out-Null
}
