#Requires -RunAsAdministrator
<#
.SYNOPSIS
    ABCD Migration Tool - Master Backup Script
.DESCRIPTION
    Reads BackupConfig.json and executes configured backup operations
.PARAMETER Operation
    VMBackup, FolderBackup, Cleanup, VMCopy, or All
#>

param(
    [Parameter(Mandatory=$false)]
    [ValidateSet("VMBackup", "FolderBackup", "Cleanup", "VMCopy", "All")]
    [string]$Operation = "All"
)

$ErrorActionPreference = 'Continue'
$Script:InstallPath = "C:\ABCD-Migration"
$Script:LogFile = "$Script:InstallPath\Logs\Backup_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"

# Initialize logging
function Write-Log {
    param([string]$Message, [string]$Level = "INFO")
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $entry = "[$timestamp] [$Level] $Message"
    Write-Host $entry
    $entry | Out-File $Script:LogFile -Append -Encoding UTF8
}

Write-Log "=========================================="
Write-Log "ABCD Migration Tool - Backup Script"
Write-Log "Operation: $Operation"
Write-Log "=========================================="

# Load modules
try {
    . "$Script:InstallPath\Modules\Logging.ps1"
    . "$Script:InstallPath\Modules\Backup-Management.ps1"
    if (Test-Path "$Script:InstallPath\Modules\VMExport.ps1") {
        . "$Script:InstallPath\Modules\VMExport.ps1"
    }
    Write-Log "Modules loaded successfully"
}
catch {
    Write-Log "ERROR loading modules: $($_.Exception.Message)" -Level "ERROR"
    exit 1
}

# Load settings from BackupConfig.json (saved by Backup-Dialog)
$configPath = "$env:ProgramData\ABCD-Migration\BackupConfig.json"
if (-not (Test-Path $configPath)) {
    $configPath = "$Script:InstallPath\Settings\BackupConfig.json"
}

$settings = $null
if (Test-Path $configPath) {
    try {
        $settings = Get-Content $configPath -Raw | ConvertFrom-Json
        Write-Log "Settings loaded from: $configPath"
    }
    catch {
        Write-Log "ERROR reading config: $($_.Exception.Message)" -Level "ERROR"
    }
}

if (-not $settings) {
    Write-Log "ERROR: Could not load backup settings from $configPath" -Level "ERROR"
    exit 1
}

# ========================================
# Invoke-CustomFolderBackup
# ========================================
function Invoke-CustomFolderBackup {
    param(
        [array]$SourcePaths,
        [string]$BackupPath,
        [int]$KeepCopies = 7
    )
    
    if (-not $SourcePaths -or $SourcePaths.Count -eq 0) {
        Write-Log "  No folders configured for backup"
        return
    }
    
    $dateStamp = Get-Date -Format "yyyy-MM-dd_HHmmss"
    
    foreach ($sourcePath in $SourcePaths) {
        $folderName = Split-Path $sourcePath -Leaf
        Write-Log "  Backing up: $folderName ($sourcePath)"
        
        if (-not (Test-Path $sourcePath)) {
            Write-Log "  SKIPPED - Source not found: $sourcePath" -Level "WARN"
            continue
        }
        
        # Create destination
        $destFolder = Join-Path $BackupPath $folderName
        $destPath = Join-Path $destFolder $dateStamp
        
        if (-not (Test-Path $destPath)) {
            New-Item -ItemType Directory -Path $destPath -Force | Out-Null
        }
        
        # Get source size
        $sourceSize = (Get-ChildItem -Path $sourcePath -Recurse -File -ErrorAction SilentlyContinue |
            Measure-Object -Property Length -Sum).Sum
        $sourceSizeGB = [math]::Round($sourceSize / 1GB, 2)
        Write-Log "  Size: $sourceSizeGB GB -> $destPath"
        
        # Copy with robocopy
        $robocopyLog = "$destPath\_robocopy.log"
        $robocopyArgs = "`"$sourcePath`" `"$destPath`" /E /COPYALL /R:2 /W:3 /MT:8 /NP /LOG:`"$robocopyLog`""
        $exitCode = (Start-Process -FilePath "robocopy.exe" -ArgumentList $robocopyArgs -Wait -PassThru -NoNewWindow).ExitCode
        
        if ($exitCode -ge 8) {
            Write-Log "  PARTIAL - Some files skipped (exit code $exitCode, see log)" -Level "WARN"
        } else {
            Write-Log "  OK - Backup complete"
            Remove-Item $robocopyLog -Force -ErrorAction SilentlyContinue
        }
        
        # Cleanup old versions
        $existingVersions = Get-ChildItem -Path $destFolder -Directory -ErrorAction SilentlyContinue |
            Sort-Object Name -Descending
        
        if ($existingVersions.Count -gt $KeepCopies) {
            $toRemove = $existingVersions | Select-Object -Skip $KeepCopies
            foreach ($old in $toRemove) {
                Remove-Item -Path $old.FullName -Recurse -Force -ErrorAction SilentlyContinue
                Write-Log "  Removed old version: $($old.Name)"
            }
            Write-Log "  Kept $KeepCopies versions, removed $($toRemove.Count) old"
        }
    }
}

# ========================================
# VM Backup
# ========================================
if ($Operation -eq "VMBackup" -or $Operation -eq "All") {
    $vmCfg = $settings.VMExport
    if (-not $vmCfg) { $vmCfg = $settings.VMBackup }
    
    if ($vmCfg -and $vmCfg.Enabled) {
        Write-Log "Starting VM Backup..."
        
        try {
            Import-Module Hyper-V -ErrorAction Stop
            
            $vmNames = $vmCfg.VMNames
            $backupPath = $vmCfg.BackupPath
            $retentionDays = if ($vmCfg.RetentionDays) { $vmCfg.RetentionDays } else { 7 }
            $cleanSnapshots = $vmCfg.RemoveSnapshots
            
            $folderName = (Get-Date).ToString("dd-MM-yyyy")
            $exportPath = Join-Path $backupPath $folderName
            
            if (-not (Test-Path $exportPath)) {
                New-Item -ItemType Directory -Path $exportPath -Force | Out-Null
            }
            
            if ($vmNames.Count -eq 0 -or $vmNames -contains '*') {
                $vms = Get-VM
                Write-Log "Exporting ALL VMs ($($vms.Count) found)"
            } else {
                $vms = Get-VM | Where-Object { $vmNames -contains $_.Name }
                Write-Log "Exporting selected VMs: $(($vms | ForEach-Object { $_.Name }) -join ', ')"
            }
            
            foreach ($vm in $vms) {
                Write-Log "Exporting VM: $($vm.Name)..."
                try {
                    Export-VM -Name $vm.Name -Path $exportPath -ErrorAction Stop
                    Write-Log "  Export complete: $($vm.Name)"
                }
                catch {
                    Write-Log "  ERROR exporting $($vm.Name): $($_.Exception.Message)" -Level "ERROR"
                }
            }
            
            if ($cleanSnapshots) {
                Write-Log "Cleaning VM snapshots..."
                Get-VM | Get-VMSnapshot | Remove-VMSnapshot -ErrorAction SilentlyContinue
                Write-Log "Snapshots cleaned"
            }
            
            Write-Log "Cleaning old VM exports (retention: $retentionDays days)..."
            Invoke-BackupCleanup -Path $backupPath -RetentionDays $retentionDays
            
            Write-Log "VM Backup complete!"
        }
        catch {
            Write-Log "ERROR during VM Backup: $($_.Exception.Message)" -Level "ERROR"
        }
    }
    else {
        Write-Log "VM Backup is disabled, skipping"
    }
}

# ========================================
# Folder Backup
# ========================================
if ($Operation -eq "FolderBackup" -or $Operation -eq "All") {
    $fileCfg = $settings.FileBackup
    
    if ($fileCfg -and $fileCfg.Enabled) {
        $paths = @()
        if ($fileCfg.SourcePaths) { $paths = @($fileCfg.SourcePaths) }
        
        if ($paths.Count -gt 0) {
            Write-Log "Starting Folder Backup ($($paths.Count) folders)..."
            $destPath = if ($fileCfg.DestinationPath) { $fileCfg.DestinationPath } else { "$Script:InstallPath\Backup\Folders" }
            $keepCopies = if ($fileCfg.RetentionDays) { $fileCfg.RetentionDays } else { 7 }
            
            try {
                Invoke-CustomFolderBackup -SourcePaths $paths -BackupPath $destPath -KeepCopies $keepCopies
                Write-Log "Folder Backup complete!"
            }
            catch {
                Write-Log "ERROR during Folder Backup: $($_.Exception.Message)" -Level "ERROR"
            }
        } else {
            Write-Log "No folders configured for backup, skipping"
        }
    }
    else {
        Write-Log "Folder Backup is disabled, skipping"
    }
}

# ========================================
# Cleanup
# ========================================
if ($Operation -eq "Cleanup" -or $Operation -eq "All") {
    Write-Log "Cleaning old backup logs..."
    $logDir = "$Script:InstallPath\Logs"
    if (Test-Path $logDir) {
        Get-ChildItem -Path $logDir -Filter "Backup_*.log" -ErrorAction SilentlyContinue |
            Where-Object { $_.LastWriteTime -lt (Get-Date).AddDays(-30) } |
            ForEach-Object {
                Remove-Item $_.FullName -Force -ErrorAction SilentlyContinue
                Write-Log "  Removed old log: $($_.Name)"
            }
    }
    Write-Log "Log cleanup complete"
}

# ========================================
# VM Copy (staggered / network transfer)
# ========================================
if ($Operation -eq "VMCopy") {
    $netCfg = $settings.NetworkStaging
    if ($netCfg -and $netCfg.Enabled -and $netCfg.NetworkTargetPath) {
        Write-Log "Starting VM Copy to Network..."
        
        $vmCfg = $settings.VMExport
        if (-not $vmCfg) { $vmCfg = $settings.VMBackup }
        $sourcePath = if ($netCfg.LocalStagingPath) { $netCfg.LocalStagingPath } else { $vmCfg.BackupPath }
        
        try {
            if (Get-Command Copy-VMToNetwork -ErrorAction SilentlyContinue) {
                Copy-VMToNetwork -SourcePath $sourcePath -DestinationPath $netCfg.NetworkTargetPath `
                    -Verify:$netCfg.VerifyAfterTransfer -DeleteSource:$netCfg.DeleteLocalAfterTransfer
            } else {
                $robArgs = "`"$sourcePath`" `"$($netCfg.NetworkTargetPath)`" /E /COPYALL /R:3 /W:5 /MT:8 /NP"
                Start-Process -FilePath "robocopy.exe" -ArgumentList $robArgs -Wait -NoNewWindow
            }
            Write-Log "VM Copy complete!"
        }
        catch {
            Write-Log "ERROR during VM Copy: $($_.Exception.Message)" -Level "ERROR"
        }
    }
    else {
        Write-Log "VM Copy / Network Staging is disabled, skipping"
    }
}

Write-Log "=========================================="
Write-Log "Backup Script Complete"
Write-Log "Log file: $Script:LogFile"
Write-Log "=========================================="
