#Requires -RunAsAdministrator
<#
.SYNOPSIS
    ABCD Migration Tool - Scheduled Google Drive Sync
.DESCRIPTION
    Runs as a Windows Scheduled Task to sync local backup folder to Google Drive.
    Designed to run AFTER local backups/updates complete (e.g. 4 hours later).
    
    Behavior:
    - Mirror mode (sync): identical files skipped, new/changed uploaded, deleted removed
    - Additive mode (copy): identical files skipped, new/changed uploaded, nothing deleted
    - Optional retention: delete files on Drive older than N days
#>

$ErrorActionPreference = 'Continue'
$Script:InstallPath = "C:\ABCD-Migration"
$Script:LogFile = "$Script:InstallPath\Logs\GDriveSync_$(Get-Date -Format 'yyyyMMdd_HHmmss').log"

function Write-Log {
    param([string]$Message, [string]$Level = "INFO")
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $entry = "[$timestamp] [$Level] $Message"
    Write-Host $entry
    $entry | Out-File $Script:LogFile -Append -Encoding UTF8
}

Write-Log "=========================================="
Write-Log "ABCD Migration Tool - Google Drive Sync"
Write-Log "=========================================="

# Load required modules
try {
    . "$Script:InstallPath\Modules\Logging.ps1"
    . "$Script:InstallPath\Modules\GoogleDrive-Backup.ps1"
    
    # Load Licensing for Get-SavedCredentials (optional)
    $licPath = "$Script:InstallPath\Modules\Licensing.ps1"
    if (Test-Path $licPath) { . $licPath }
    
    Write-Log "Modules loaded"
} catch {
    Write-Log "Failed to load modules: $($_.Exception.Message)" -Level "ERROR"
    exit 1
}

# Load sync config
$configPath = "$Script:InstallPath\Settings\gdrive-sync-schedule.json"
if (-not (Test-Path $configPath)) {
    Write-Log "No schedule config found at $configPath" -Level "ERROR"
    exit 1
}

try {
    $config = Get-Content $configPath -Raw | ConvertFrom-Json
    Write-Log "Config loaded: Mode=$($config.Mode), Folder=$($config.FolderName)"
    Write-Log "  LocalPath: $($config.LocalPath)"
    Write-Log "  Mirror: $($config.MirrorMode)"
    Write-Log "  Retention: $(if ($config.RetentionEnabled) { "$($config.RetentionDays) days" } else { 'Keep all' })"
} catch {
    Write-Log "Failed to read config: $($_.Exception.Message)" -Level "ERROR"
    exit 1
}

# Verify Google Drive connection
$status = Test-GDriveConnection
if (-not $status.Connected) {
    Write-Log "Google Drive not connected. Skipping sync." -Level "ERROR"
    exit 1
}
Write-Log "Google Drive connected: $($status.Email)"

# Verify local path exists
if (-not (Test-Path $config.LocalPath)) {
    Write-Log "Local path does not exist: $($config.LocalPath)" -Level "WARNING"
    Write-Log "Nothing to sync."
    exit 0
}

# Run sync
$remotePath = "${Script:RemoteName}:$Script:DefaultBackupFolder/$($config.FolderName)"
$rcloneCmd = if ($config.MirrorMode) { "sync" } else { "copy" }

Write-Log "Starting $rcloneCmd : $($config.LocalPath) -> $remotePath"

$rArgs = @($rcloneCmd, $config.LocalPath, $remotePath)
$rArgs += "--progress"
$rArgs += "--stats-one-line"
$rArgs += "--stats", "30s"
$rArgs += "--transfers", "4"
$rArgs += "--checkers", "8"
$rArgs += "--retries", "3"
$rArgs += "--contimeout", "60s"
$rArgs += "--timeout", "300s"
$rArgs += "--low-level-retries", "10"
$rArgs += "--log-file", $Script:LogFile
$rArgs += "--log-level", "INFO"

$result = Invoke-Rclone -Arguments $rArgs -TimeoutSeconds 14400  # 4 hour max

if ($result.Success) {
    Write-Log "Sync completed successfully"
} else {
    Write-Log "Sync failed: $($result.Error)" -Level "ERROR"
}

# Retention cleanup (delete old files on Drive)
if ($config.RetentionEnabled -and $config.RetentionDays -gt 0) {
    Write-Log "Running retention cleanup: deleting files older than $($config.RetentionDays) days on Drive"
    
    $rArgs = @("delete", $remotePath, "--min-age", "$($config.RetentionDays)d")
    $rArgs += "--log-file", $Script:LogFile
    $rArgs += "--log-level", "INFO"
    
    $cleanResult = Invoke-Rclone -Arguments $rArgs -TimeoutSeconds 3600
    
    if ($cleanResult.Success) {
        Write-Log "Retention cleanup completed"
        
        # Also remove empty directories
        $rdirArgs = @("rmdirs", $remotePath, "--leave-root")
        $rdirArgs += "--log-file", $Script:LogFile
        Invoke-Rclone -Arguments $rdirArgs -TimeoutSeconds 300 | Out-Null
        Write-Log "Empty directories cleaned up"
    } else {
        Write-Log "Retention cleanup failed: $($cleanResult.Error)" -Level "WARNING"
    }
} else {
    Write-Log "Retention: Keep all files (no cleanup)"
}

# Record usage (optional)
try {
    if (Get-Command Get-SavedCredentials -ErrorAction SilentlyContinue) {
        $creds = Get-SavedCredentials
        if ($creds.LicenseKey) {
            $body = @{ 
                licenseKey = $creds.LicenseKey
                machineId = $creds.MachineId
                type = "backup"
                detail = "Scheduled GDrive sync: $($config.FolderName)"
            } | ConvertTo-Json
            Invoke-RestMethod -Uri "$Script:API_BASE/auth.php?action=record-usage" -Method POST -Body $body -ContentType "application/json" -ErrorAction SilentlyContinue
        }
    }
} catch { }

Write-Log "=========================================="
Write-Log "Google Drive Sync task complete"
Write-Log "=========================================="

# Clean old sync logs (keep 30 days)
Get-ChildItem "$Script:InstallPath\Logs\GDriveSync_*.log" -ErrorAction SilentlyContinue |
    Where-Object { $_.LastWriteTime -lt (Get-Date).AddDays(-30) } |
    Remove-Item -Force -ErrorAction SilentlyContinue
