#Requires -RunAsAdministrator
<#
.SYNOPSIS
    License Cleanup Tool for Server Migration
    
.DESCRIPTION
    Removes hardware-bound license data to allow fresh activation after migration.
    Supports: QuickBooks, Microsoft Office, Adobe, and more.
    
.NOTES
    Run this AFTER migration, BEFORE trying to activate software.
    Have your license keys ready!
#>

param(
    [switch]$QuickBooks,
    [switch]$Office365,
    [switch]$AdobeCC,
    [switch]$All,
    [switch]$ListOnly
)

# Colors
function Write-Header($text) { Write-Host "`n$text" -ForegroundColor Cyan }
function Write-Success($text) { Write-Host "  [OK] $text" -ForegroundColor Green }
function Write-Warning($text) { Write-Host "  [!] $text" -ForegroundColor Yellow }
function Write-Info($text) { Write-Host "  [i] $text" -ForegroundColor Gray }

Write-Host ""
Write-Host "============================================================" -ForegroundColor Cyan
Write-Host "       LICENSE CLEANUP TOOL - Server Migration" -ForegroundColor Cyan
Write-Host "============================================================" -ForegroundColor Cyan
Write-Host ""

if ($ListOnly) {
    Write-Host "Scanning for installed software requiring re-activation..." -ForegroundColor Yellow
    Write-Host ""
}

# ============================================================
# QUICKBOOKS CLEANUP
# ============================================================
function Clean-QuickBooks {
    Write-Header "QuickBooks License Cleanup"
    
    # Stop processes
    Write-Info "Stopping QuickBooks processes..."
    $qbProcesses = @("QBW32", "QBW", "qbupdate", "QBCFMonitorService", "QBDBMgrN", "QBDBMgr")
    foreach ($proc in $qbProcesses) {
        Stop-Process -Name $proc -Force -ErrorAction SilentlyContinue
    }
    Start-Sleep -Seconds 2
    
    # Remove entitlement files
    $entitlementPaths = @(
        "C:\ProgramData\Intuit\Entitlement Client",
        "C:\ProgramData\COMMON FILES\INTUIT",
        "C:\ProgramData\Intuit\Entitlement Client\v8",
        "C:\ProgramData\Intuit\Entitlement Client\v6"
    )
    
    foreach ($path in $entitlementPaths) {
        if (Test-Path $path) {
            Remove-Item -Path $path -Recurse -Force -ErrorAction SilentlyContinue
            Write-Success "Removed: $path"
        }
    }
    
    # Remove ecml files
    Get-ChildItem "C:\ProgramData\Intuit" -Filter "*.ecml" -Recurse -ErrorAction SilentlyContinue | ForEach-Object {
        Remove-Item $_.FullName -Force -ErrorAction SilentlyContinue
        Write-Success "Removed: $($_.FullName)"
    }
    
    # Remove qbregistration.dat files
    Get-ChildItem "C:\ProgramData\Intuit" -Filter "qbregistration.dat" -Recurse -ErrorAction SilentlyContinue | ForEach-Object {
        Remove-Item $_.FullName -Force -ErrorAction SilentlyContinue
        Write-Success "Removed: $($_.FullName)"
    }
    
    # Registry cleanup
    $regPaths = @(
        "HKLM:\SOFTWARE\Intuit\Entitlement Client",
        "HKLM:\SOFTWARE\WOW6432Node\Intuit\Entitlement Client",
        "HKCU:\SOFTWARE\Intuit\Entitlement Client"
    )
    
    foreach ($regPath in $regPaths) {
        if (Test-Path $regPath) {
            Remove-Item -Path $regPath -Recurse -Force -ErrorAction SilentlyContinue
            Write-Success "Removed registry: $regPath"
        }
    }
    
    Write-Host ""
    Write-Host "  QuickBooks cleanup complete!" -ForegroundColor Green
    Write-Host "   Open QuickBooks and enter your license key" -ForegroundColor Yellow
}

# ============================================================
# MICROSOFT OFFICE 365 / MICROSOFT 365 CLEANUP
# ============================================================
function Clean-Office365 {
    Write-Header "Microsoft Office 365 License Cleanup"
    
    # Stop Office processes
    Write-Info "Stopping Office processes..."
    $officeProcesses = @("WINWORD", "EXCEL", "POWERPNT", "OUTLOOK", "ONENOTE", "MSACCESS", "MSPUB", "lync", "Teams")
    foreach ($proc in $officeProcesses) {
        Stop-Process -Name $proc -Force -ErrorAction SilentlyContinue
    }
    Start-Sleep -Seconds 2
    
    # Remove Office licensing cache
    $officeLicensePaths = @(
        "$env:LOCALAPPDATA\Microsoft\Office\16.0\Licensing",
        "$env:LOCALAPPDATA\Microsoft\Office\Licenses",
        "$env:APPDATA\Microsoft\Office\Licenses"
    )
    
    foreach ($path in $officeLicensePaths) {
        if (Test-Path $path) {
            Remove-Item -Path $path -Recurse -Force -ErrorAction SilentlyContinue
            Write-Success "Removed: $path"
        }
    }
    
    # Clear Office identity cache
    $identityPaths = @(
        "HKCU:\SOFTWARE\Microsoft\Office\16.0\Common\Identity",
        "HKCU:\SOFTWARE\Microsoft\Office\16.0\Common\Licensing"
    )
    
    foreach ($regPath in $identityPaths) {
        if (Test-Path $regPath) {
            Remove-Item -Path $regPath -Recurse -Force -ErrorAction SilentlyContinue
            Write-Success "Removed registry: $regPath"
        }
    }
    
    # Clear Windows Credential Manager entries for Office
    Write-Info "Clearing Office credentials from Credential Manager..."
    cmdkey /list | Where-Object { $_ -match "MicrosoftOffice" } | ForEach-Object {
        $target = ($_ -split ":")[1].Trim()
        cmdkey /delete:$target 2>$null
        Write-Success "Removed credential: $target"
    }
    
    Write-Host ""
    Write-Host "  Office 365 cleanup complete!" -ForegroundColor Green
    Write-Host "   Open any Office app and sign in with your Microsoft account" -ForegroundColor Yellow
}

# ============================================================
# ADOBE CREATIVE CLOUD CLEANUP
# ============================================================
function Clean-AdobeCC {
    Write-Header "Adobe Creative Cloud License Cleanup"
    
    # Stop Adobe processes
    Write-Info "Stopping Adobe processes..."
    $adobeProcesses = @("Creative Cloud", "CCLibrary", "CCXProcess", "AdobeIPCBroker", "Adobe Desktop Service")
    foreach ($proc in $adobeProcesses) {
        Stop-Process -Name $proc -Force -ErrorAction SilentlyContinue
    }
    Start-Sleep -Seconds 2
    
    # Adobe licensing folders
    $adobePaths = @(
        "$env:APPDATA\Adobe\OOBE",
        "$env:LOCALAPPDATA\Adobe\OOBE",
        "C:\ProgramData\Adobe\SLStore",
        "C:\ProgramData\Adobe\FlexNet"
    )
    
    foreach ($path in $adobePaths) {
        if (Test-Path $path) {
            Remove-Item -Path $path -Recurse -Force -ErrorAction SilentlyContinue
            Write-Success "Removed: $path"
        }
    }
    
    Write-Host ""
    Write-Host "  Adobe CC cleanup complete!" -ForegroundColor Green
    Write-Host "   Open Creative Cloud and sign in with your Adobe ID" -ForegroundColor Yellow
}

# ============================================================
# OUTLOOK EMAIL PROFILE CLEANUP (Fresh Sign-in)
# ============================================================
function Clean-OutlookProfiles {
    Write-Header "Outlook Email Profile Cleanup"
    
    # Stop Outlook
    Write-Info "Stopping Outlook..."
    Stop-Process -Name "OUTLOOK" -Force -ErrorAction SilentlyContinue
    Start-Sleep -Seconds 2
    
    # Remove Outlook profiles (will require re-setup)
    $profilePath = "HKCU:\SOFTWARE\Microsoft\Office\16.0\Outlook\Profiles"
    if (Test-Path $profilePath) {
        # Backup first
        $backupPath = "C:\ABCD-Migration\Backup\OutlookProfiles_$(Get-Date -Format 'yyyyMMdd_HHmmss').reg"
        reg export "HKCU\SOFTWARE\Microsoft\Office\16.0\Outlook\Profiles" $backupPath /y 2>$null
        Write-Info "Backed up profiles to: $backupPath"
        
        Remove-Item -Path $profilePath -Recurse -Force -ErrorAction SilentlyContinue
        Write-Success "Removed Outlook profiles"
    }
    
    # Clear autodiscover cache
    $autodiscoverPaths = @(
        "HKCU:\SOFTWARE\Microsoft\Office\16.0\Outlook\AutoDiscover",
        "$env:LOCALAPPDATA\Microsoft\Outlook\RoamCache"
    )
    
    foreach ($path in $autodiscoverPaths) {
        if (Test-Path $path) {
            Remove-Item -Path $path -Recurse -Force -ErrorAction SilentlyContinue
            Write-Success "Removed: $path"
        }
    }
    
    Write-Host ""
    Write-Host "  Outlook cleanup complete!" -ForegroundColor Green
    Write-Host "   Open Outlook and set up your email account" -ForegroundColor Yellow
}

# ============================================================
# DETECT INSTALLED SOFTWARE
# ============================================================
function Get-InstalledLicensedSoftware {
    Write-Header "Detected Software Requiring Re-Activation"
    
    $found = @()
    
    # QuickBooks
    if (Test-Path "HKLM:\SOFTWARE\Intuit" -or Test-Path "HKLM:\SOFTWARE\WOW6432Node\Intuit") {
        $qbVersions = Get-ChildItem "C:\Program Files*\Intuit\QuickBooks*" -ErrorAction SilentlyContinue
        foreach ($qb in $qbVersions) {
            Write-Host "  [QB] $($qb.Name)" -ForegroundColor Yellow
            $found += "QuickBooks"
        }
    }
    
    # Microsoft Office
    if (Test-Path "HKLM:\SOFTWARE\Microsoft\Office" -or Test-Path "C:\Program Files\Microsoft Office") {
        $officeVersion = Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Office\ClickToRun\Configuration" -ErrorAction SilentlyContinue
        if ($officeVersion) {
            Write-Host "  [Office] Microsoft 365 / Office $($officeVersion.VersionToReport)" -ForegroundColor Yellow
            $found += "Office365"
        }
    }
    
    # Adobe
    if (Test-Path "C:\Program Files\Adobe" -or Test-Path "C:\Program Files (x86)\Adobe") {
        $adobeApps = Get-ChildItem "C:\Program Files*\Adobe\*" -Directory -ErrorAction SilentlyContinue | Where-Object { $_.Name -notmatch "Common" }
        foreach ($app in $adobeApps) {
            Write-Host "  [Adobe] $($app.Name)" -ForegroundColor Yellow
        }
        if ($adobeApps) { $found += "AdobeCC" }
    }
    
    # Outlook profiles
    if (Test-Path "HKCU:\SOFTWARE\Microsoft\Office\16.0\Outlook\Profiles") {
        $profiles = Get-ChildItem "HKCU:\SOFTWARE\Microsoft\Office\16.0\Outlook\Profiles" -ErrorAction SilentlyContinue
        Write-Host "  [Outlook] $($profiles.Count) email profile(s) found" -ForegroundColor Yellow
        $found += "Outlook"
    }
    
    if ($found.Count -eq 0) {
        Write-Host "  No licensed software detected" -ForegroundColor Gray
    }
    
    return $found
}

# ============================================================
# MAIN EXECUTION
# ============================================================

if ($ListOnly) {
    $detected = Get-InstalledLicensedSoftware
    Write-Host ""
    Write-Host "Run this script with switches to clean specific software:" -ForegroundColor Cyan
    Write-Host "  -QuickBooks    Clean QuickBooks licensing"
    Write-Host "  -Office365     Clean Office 365 licensing"
    Write-Host "  -AdobeCC       Clean Adobe Creative Cloud licensing"
    Write-Host "  -All           Clean all detected software"
    Write-Host ""
    exit
}

if ($All) {
    $detected = Get-InstalledLicensedSoftware
    if ($detected -contains "QuickBooks") { Clean-QuickBooks }
    if ($detected -contains "Office365") { Clean-Office365; Clean-OutlookProfiles }
    if ($detected -contains "AdobeCC") { Clean-AdobeCC }
}
else {
    if ($QuickBooks) { Clean-QuickBooks }
    if ($Office365) { Clean-Office365; Clean-OutlookProfiles }
    if ($AdobeCC) { Clean-AdobeCC }
}

Write-Host ""
Write-Host "============================================================" -ForegroundColor Cyan
Write-Host "  License cleanup complete!" -ForegroundColor Green
Write-Host "============================================================" -ForegroundColor Cyan
Write-Host ""
Write-Host "Next steps:" -ForegroundColor Yellow
Write-Host "  1. Open each application"
Write-Host "  2. Enter your license key or sign in"
Write-Host "  3. Complete activation"
Write-Host ""
