#!/usr/bin/env python3
"""
ABC&D Systems Backup and Migration Tools - Python Wrapper
Version 4.0

Handles authentication and licensing before launching the PowerShell GUI.

Build: pyinstaller --onefile --windowed --icon=../Assets/ABCD-Icon.ico --name="ABCD-Migration" --uac-admin wrapper.py
"""

import os
import sys
import json
import hashlib
import platform
import subprocess
import tkinter as tk
from tkinter import ttk, messagebox
from datetime import datetime, timedelta
from pathlib import Path
import winreg
import ctypes
from typing import Optional, Dict
import uuid

# Configuration
CONFIG = {
    "version": "4.0",
    "app_name": "ABC&D Systems Backup and Migration Tools",
    "api_base": "https://abcdtoolset.com/api",
    "reg_path": r"SOFTWARE\ABC&D Systems\Migration Tool",
    "install_path": r"C:\ABCD-Migration",
    "trial_days": 10,
    "pricing": {
        "basic_monthly": {"price": 10, "days": 30},
        "pro_monthly": {"price": 25, "days": 30},
        "pro_annual": {"price": 210, "days": 365},
    },
}

COLORS = {
    "gold": "#D4AF37",
    "navy": "#1a1a2e",
    "dark_bg": "#0a0a0f",
    "cream": "#F5F5DC",
    "white": "#FFFFFF",
    "gray": "#888888",
    "red": "#DC3545",
    "green": "#28A745",
}


def is_admin() -> bool:
    """Check if running with administrator privileges."""
    try:
        return ctypes.windll.shell32.IsUserAnAdmin()
    except:
        return False


def run_as_admin():
    """Restart with admin privileges."""
    ctypes.windll.shell32.ShellExecuteW(
        None, "runas", sys.executable, " ".join(sys.argv), None, 1
    )
    sys.exit()


def get_machine_id() -> str:
    """Generate unique machine identifier."""
    try:
        import wmi
        c = wmi.WMI()
        bios = c.Win32_BIOS()[0].SerialNumber or ""
        cpu = c.Win32_Processor()[0].ProcessorId or ""
        combined = f"{bios}-{cpu}"
        return hashlib.sha256(combined.encode()).hexdigest()[:32]
    except:
        return hashlib.sha256(f"{platform.node()}-{uuid.getnode()}".encode()).hexdigest()[:32]


def get_registry_value(name: str) -> Optional[str]:
    """Get value from registry."""
    try:
        key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, CONFIG["reg_path"], 0, winreg.KEY_READ)
        value, _ = winreg.QueryValueEx(key, name)
        winreg.CloseKey(key)
        return value
    except:
        return None


def set_registry_value(name: str, value: str) -> bool:
    """Set value in registry."""
    try:
        key = winreg.CreateKeyEx(winreg.HKEY_LOCAL_MACHINE, CONFIG["reg_path"], 0, winreg.KEY_WRITE)
        winreg.SetValueEx(key, name, 0, winreg.REG_SZ, value)
        winreg.CloseKey(key)
        return True
    except:
        return False


def save_license(token: str, expiry: str, tier: str, email: str):
    """Save license to registry."""
    set_registry_value("LicenseToken", token)
    set_registry_value("LicenseExpiry", expiry)
    set_registry_value("LicenseTier", tier)
    set_registry_value("UserEmail", email)
    set_registry_value("MachineId", get_machine_id())
    set_registry_value("ActivatedAt", datetime.now().strftime("%Y-%m-%d %H:%M:%S"))


def get_license() -> Optional[Dict]:
    """Get license from registry."""
    token = get_registry_value("LicenseToken")
    expiry = get_registry_value("LicenseExpiry")
    if token and expiry:
        return {
            "token": token,
            "expiry": expiry,
            "tier": get_registry_value("LicenseTier") or "basic",
            "email": get_registry_value("UserEmail") or "",
        }
    return None


def validate_license() -> Dict:
    """Validate license locally."""
    license_info = get_license()
    
    if not license_info:
        return {"valid": False, "reason": "No license found"}
    
    try:
        expiry = datetime.strptime(license_info["expiry"], "%Y-%m-%d %H:%M:%S")
        if expiry < datetime.now():
            return {"valid": False, "reason": f"License expired on {license_info['expiry']}", "can_renew": True}
        
        days_remaining = (expiry - datetime.now()).days
        return {
            "valid": True,
            "days_remaining": days_remaining,
            "tier": license_info["tier"],
            "email": license_info["email"],
            "show_warning": days_remaining <= 7
        }
    except:
        return {"valid": False, "reason": "Invalid license data"}


def launch_powershell_gui():
    """Launch the PowerShell GUI."""
    script_path = os.path.join(CONFIG["install_path"], "ABCD-Migration-GUI.ps1")
    
    if not os.path.exists(script_path):
        messagebox.showerror("Error", f"Migration tool not found at:\n{script_path}")
        return False
    
    try:
        subprocess.Popen([
            "powershell.exe", "-NoProfile", "-ExecutionPolicy", "Bypass",
            "-WindowStyle", "Hidden", "-File", script_path
        ], creationflags=subprocess.CREATE_NO_WINDOW)
        return True
    except Exception as e:
        messagebox.showerror("Error", f"Failed to launch: {e}")
        return False


class LoginWindow:
    """Login/Registration window."""
    
    def __init__(self, on_success):
        self.on_success = on_success
        self.root = tk.Tk()
        self.root.title("ABC&D Systems - Login")
        self.root.geometry("400x450")
        self.root.resizable(False, False)
        self.root.configure(bg=COLORS["navy"])
        self.root.eval('tk::PlaceWindow . center')
        self.setup_ui()
    
    def setup_ui(self):
        # Header
        tk.Label(self.root, text="ABC&D Systems", font=("Segoe UI", 20, "bold"),
                 fg=COLORS["gold"], bg=COLORS["navy"]).pack(pady=(30, 5))
        tk.Label(self.root, text="Backup and Migration Tools", font=("Segoe UI", 11),
                 fg=COLORS["cream"], bg=COLORS["navy"]).pack(pady=(0, 20))
        
        # Email
        tk.Label(self.root, text="Email:", fg=COLORS["cream"], bg=COLORS["navy"]).pack(pady=(10, 2))
        self.email_entry = tk.Entry(self.root, font=("Segoe UI", 11), width=30)
        self.email_entry.pack(pady=5)
        
        # Password
        tk.Label(self.root, text="Password:", fg=COLORS["cream"], bg=COLORS["navy"]).pack(pady=(10, 2))
        self.password_entry = tk.Entry(self.root, font=("Segoe UI", 11), width=30, show="•")
        self.password_entry.pack(pady=5)
        
        # Login button
        tk.Button(self.root, text="Login", font=("Segoe UI", 11, "bold"), bg=COLORS["gold"],
                  fg=COLORS["navy"], width=15, height=2, command=self.do_login).pack(pady=20)
        
        # Trial button
        tk.Button(self.root, text="Start Free Trial", font=("Segoe UI", 10), bg=COLORS["gray"],
                  fg=COLORS["white"], width=15, command=self.start_trial).pack(pady=5)
        
        # Status
        self.status_label = tk.Label(self.root, text="", fg=COLORS["red"], bg=COLORS["navy"])
        self.status_label.pack(pady=10)
        
        # Version
        tk.Label(self.root, text=f"v{CONFIG['version']}", fg=COLORS["gray"],
                 bg=COLORS["navy"]).pack(side="bottom", pady=10)
    
    def do_login(self):
        email = self.email_entry.get().strip()
        password = self.password_entry.get()
        
        if not email or not password:
            self.status_label.config(text="Please enter email and password")
            return
        
        # For demo: accept any login and create a trial
        self.start_trial_for_email(email)
    
    def start_trial(self):
        self.start_trial_for_email("trial@demo.local")
    
    def start_trial_for_email(self, email):
        expiry = (datetime.now() + timedelta(days=CONFIG["trial_days"])).strftime("%Y-%m-%d %H:%M:%S")
        token = hashlib.sha256(f"{email}-{datetime.now()}".encode()).hexdigest()[:32]
        save_license(token, expiry, "trial", email)
        
        self.status_label.config(text="Trial started!", fg=COLORS["green"])
        self.root.after(500, self.finish_login)
    
    def finish_login(self):
        self.root.destroy()
        self.on_success()
    
    def run(self):
        self.root.mainloop()


class PaywallWindow:
    """Subscription required window."""
    
    def __init__(self, reason=""):
        self.root = tk.Tk()
        self.root.title("Subscription Required")
        self.root.geometry("450x350")
        self.root.resizable(False, False)
        self.root.configure(bg=COLORS["navy"])
        self.root.eval('tk::PlaceWindow . center')
        self.setup_ui(reason)
    
    def setup_ui(self, reason):
        tk.Label(self.root, text="ABC&D Systems", font=("Segoe UI", 18, "bold"),
                 fg=COLORS["gold"], bg=COLORS["navy"]).pack(pady=(30, 5))
        
        if reason:
            tk.Label(self.root, text=reason, fg=COLORS["red"], bg=COLORS["navy"]).pack(pady=10)
        
        tk.Label(self.root, text="Your license has expired or is invalid.",
                 fg=COLORS["cream"], bg=COLORS["navy"]).pack(pady=20)
        
        tk.Button(self.root, text="Subscribe Now", font=("Segoe UI", 12, "bold"),
                  bg=COLORS["gold"], fg=COLORS["navy"], width=20, height=2,
                  command=lambda: __import__('webbrowser').open("https://abcdtoolset.com/")).pack(pady=20)
        
        tk.Button(self.root, text="Exit", bg=COLORS["gray"], fg=COLORS["white"],
                  width=10, command=self.root.destroy).pack(pady=10)
    
    def run(self):
        self.root.mainloop()


def main():
    """Main entry point."""
    if not is_admin():
        run_as_admin()
        return
    
    status = validate_license()
    
    if status.get("valid"):
        if status.get("show_warning"):
            messagebox.showwarning("License Expiring",
                f"Your license expires in {status.get('days_remaining', 0)} days.")
        launch_powershell_gui()
    else:
        license_info = get_license()
        if license_info:
            PaywallWindow(status.get("reason", "")).run()
        else:
            LoginWindow(lambda: launch_powershell_gui()).run()


if __name__ == "__main__":
    main()
